<?php
/**
 * Simply SEO Breadcrumbs
 *
 * Handles breadcrumb navigation output
 *
 * @package Simply_SEO
 * @since 1.0.2
 */

// Prevent direct access
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Breadcrumbs class
 */
class Simply_SEO_Breadcrumbs {

    /**
     * Default options
     */
    private $defaults = array(
        'separator'   => '›',
        'home_text'   => '',
        'home_icon'   => false,
        'show_home'   => true,
        'show_current'=> true,
        'before'      => '<nav class="simply-seo-breadcrumbs" aria-label="Breadcrumb"><ol itemscope itemtype="https://schema.org/BreadcrumbList">',
        'after'       => '</ol></nav>',
        'before_item' => '<li itemprop="itemListElement" itemscope itemtype="https://schema.org/ListItem">',
        'after_item'  => '</li>',
    );

    /**
     * Constructor
     */
    public function __construct() {
        // Load settings into defaults
        $this->load_settings();
        
        // Register shortcode
        add_shortcode( 'simply_seo_breadcrumbs', array( $this, 'shortcode' ) );
    }

    /**
     * Load settings from plugin options
     */
    private function load_settings() {
        $settings = Simply_SEO::get_settings();
        
        if ( isset( $settings['breadcrumbs_separator'] ) ) {
            $this->defaults['separator'] = $settings['breadcrumbs_separator'];
        }
        if ( isset( $settings['breadcrumbs_home_text'] ) ) {
            $this->defaults['home_text'] = $settings['breadcrumbs_home_text'];
        }
        if ( isset( $settings['breadcrumbs_home_icon'] ) ) {
            $this->defaults['home_icon'] = $settings['breadcrumbs_home_icon'];
        }
        if ( isset( $settings['breadcrumbs_show_current'] ) ) {
            $this->defaults['show_current'] = $settings['breadcrumbs_show_current'];
        }
    }

    /**
     * Flag to track if CSS has been output
     */
    private static $css_output = false;

    /**
     * Output inline CSS for breadcrumbs via wp_add_inline_style (only once, only when needed)
     */
    private function maybe_output_css() {
        if ( self::$css_output ) {
            return;
        }
        
        self::$css_output = true;
        
        $css = '
        .simply-seo-breadcrumbs {
            padding: 10px 0;
            font-size: 14px;
        }
        .simply-seo-breadcrumbs ol {
            list-style: none;
            padding: 0;
            margin: 0;
            display: flex;
            flex-wrap: wrap;
            align-items: center;
            gap: 0;
        }
        .simply-seo-breadcrumbs li {
            display: flex;
            align-items: center;
        }
        .simply-seo-breadcrumbs li:not(:last-child)::after {
            content: attr(data-separator);
            margin: 0 8px;
            color: #666;
        }
        .simply-seo-breadcrumbs a {
            color: #0073aa;
            text-decoration: none;
        }
        .simply-seo-breadcrumbs a:hover {
            text-decoration: underline;
        }
        .simply-seo-breadcrumbs .current {
            color: #666;
        }';
        
        // Register a dummy style handle if not already done
        if ( ! wp_style_is( 'simply-seo-breadcrumbs', 'registered' ) ) {
            wp_register_style( 'simply-seo-breadcrumbs', false, array(), SIMPLY_SEO_VERSION );
        }
        wp_enqueue_style( 'simply-seo-breadcrumbs' );
        wp_add_inline_style( 'simply-seo-breadcrumbs', $css );
    }

    /**
     * Shortcode handler
     */
    public function shortcode( $atts ) {
        $atts = shortcode_atts( array(
            'separator'    => $this->defaults['separator'],
            'home_text'    => $this->defaults['home_text'],
            'home_icon'    => $this->defaults['home_icon'] ? 'true' : 'false',
            'show_home'    => 'true',
            'show_current' => $this->defaults['show_current'] ? 'true' : 'false',
        ), $atts, 'simply_seo_breadcrumbs' );

        // Convert string booleans
        $atts['home_icon']    = filter_var( $atts['home_icon'], FILTER_VALIDATE_BOOLEAN );
        $atts['show_home']    = filter_var( $atts['show_home'], FILTER_VALIDATE_BOOLEAN );
        $atts['show_current'] = filter_var( $atts['show_current'], FILTER_VALIDATE_BOOLEAN );

        return $this->render( $atts );
    }

    /**
     * Render breadcrumbs
     */
    public function render( $args = array() ) {
        $args = wp_parse_args( $args, $this->defaults );

        // Don't show on front page unless forced
        if ( is_front_page() ) {
            return '';
        }

        $items = $this->get_items( $args );

        if ( empty( $items ) ) {
            return '';
        }

        // Enqueue CSS only once, only when breadcrumbs are actually used
        $this->maybe_output_css();

        $output = $args['before'];
        $count  = count( $items );
        $i      = 0;

        foreach ( $items as $item ) {
            $i++;
            $is_last  = ( $i === $count );
            $is_home  = ( 1 === $i && $args['show_home'] );

            $output .= str_replace(
                '<li',
                '<li data-separator="' . esc_attr( $args['separator'] ) . '"',
                $args['before_item']
            );

            // Determine display name (icon for home if enabled)
            $display_name = $item['name'];
            if ( $is_home && ! empty( $args['home_icon'] ) ) {
                $display_name = '<svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" style="vertical-align:middle;"><path d="M3 9l9-7 9 7v11a2 2 0 0 1-2 2H5a2 2 0 0 1-2-2z"></path><polyline points="9 22 9 12 15 12 15 22"></polyline></svg>';
            }

            if ( $is_last && $args['show_current'] ) {
                // Current item (no link)
                if ( $is_home && ! empty( $args['home_icon'] ) ) {
                    $output .= '<span class="current" itemprop="name">' . $display_name . '</span>';
                } else {
                    $output .= '<span class="current" itemprop="name">' . esc_html( $item['name'] ) . '</span>';
                }
            } else {
                // Linked item
                $output .= '<a href="' . esc_url( $item['url'] ) . '" itemprop="item">';
                if ( $is_home && ! empty( $args['home_icon'] ) ) {
                    $output .= '<span itemprop="name">' . $display_name . '<span class="screen-reader-text">' . esc_html( $item['name'] ) . '</span></span>';
                } else {
                    $output .= '<span itemprop="name">' . esc_html( $display_name ) . '</span>';
                }
                $output .= '</a>';
            }

            $output .= '<meta itemprop="position" content="' . $i . '">';
            $output .= $args['after_item'];
        }

        $output .= $args['after'];

        return $output;
    }

    /**
     * Get breadcrumb items
     */
    private function get_items( $args ) {
        $items = array();

        // Home
        if ( $args['show_home'] ) {
            $home_text = $args['home_text'] ? $args['home_text'] : __( 'Home', 'simply-seo' );
            $items[] = array(
                'name' => $home_text,
                'url'  => home_url( '/' ),
            );
        }

        // Singular posts/pages
        if ( is_singular() ) {
            $post = get_queried_object();
            
            // Post type archive link (for custom post types)
            if ( 'post' !== $post->post_type && 'page' !== $post->post_type ) {
                $post_type_obj = get_post_type_object( $post->post_type );
                if ( $post_type_obj && $post_type_obj->has_archive ) {
                    $items[] = array(
                        'name' => $post_type_obj->labels->name,
                        'url'  => get_post_type_archive_link( $post->post_type ),
                    );
                }
            }

            // Categories for posts
            if ( 'post' === $post->post_type ) {
                $categories = get_the_category( $post->ID );
                if ( ! empty( $categories ) ) {
                    $category = $categories[0];
                    
                    // Parent categories
                    $parents = get_ancestors( $category->term_id, 'category' );
                    $parents = array_reverse( $parents );
                    
                    foreach ( $parents as $parent_id ) {
                        $parent = get_term( $parent_id, 'category' );
                        if ( $parent && ! is_wp_error( $parent ) ) {
                            $items[] = array(
                                'name' => $parent->name,
                                'url'  => get_term_link( $parent ),
                            );
                        }
                    }
                    
                    $items[] = array(
                        'name' => $category->name,
                        'url'  => get_term_link( $category ),
                    );
                }
            }
            
            // Parent pages
            if ( 'page' === $post->post_type && $post->post_parent ) {
                $parents = get_ancestors( $post->ID, 'page' );
                $parents = array_reverse( $parents );
                
                foreach ( $parents as $parent_id ) {
                    $items[] = array(
                        'name' => get_the_title( $parent_id ),
                        'url'  => get_permalink( $parent_id ),
                    );
                }
            }

            // Current page/post
            if ( $args['show_current'] ) {
                $items[] = array(
                    'name' => get_the_title( $post->ID ),
                    'url'  => get_permalink( $post->ID ),
                );
            }

        // Category, Tag, Taxonomy archives
        } elseif ( is_category() || is_tag() || is_tax() ) {
            $term = get_queried_object();
            
            // Parent terms
            if ( $term->parent ) {
                $parents = get_ancestors( $term->term_id, $term->taxonomy );
                $parents = array_reverse( $parents );
                
                foreach ( $parents as $parent_id ) {
                    $parent = get_term( $parent_id, $term->taxonomy );
                    if ( $parent && ! is_wp_error( $parent ) ) {
                        $items[] = array(
                            'name' => $parent->name,
                            'url'  => get_term_link( $parent ),
                        );
                    }
                }
            }
            
            if ( $args['show_current'] ) {
                $items[] = array(
                    'name' => $term->name,
                    'url'  => get_term_link( $term ),
                );
            }

        // Post type archive
        } elseif ( is_post_type_archive() ) {
            $post_type = get_queried_object();
            if ( $args['show_current'] ) {
                $items[] = array(
                    'name' => $post_type->labels->name,
                    'url'  => get_post_type_archive_link( $post_type->name ),
                );
            }

        // Author archive
        } elseif ( is_author() ) {
            $author = get_queried_object();
            if ( $args['show_current'] ) {
                $items[] = array(
                    'name' => $author->display_name,
                    'url'  => get_author_posts_url( $author->ID ),
                );
            }

        // Date archives
        } elseif ( is_date() ) {
            if ( is_year() ) {
                $items[] = array(
                    'name' => get_the_date( 'Y' ),
                    'url'  => get_year_link( get_the_date( 'Y' ) ),
                );
            } elseif ( is_month() ) {
                $items[] = array(
                    'name' => get_the_date( 'Y' ),
                    'url'  => get_year_link( get_the_date( 'Y' ) ),
                );
                $items[] = array(
                    'name' => get_the_date( 'F' ),
                    'url'  => get_month_link( get_the_date( 'Y' ), get_the_date( 'm' ) ),
                );
            } elseif ( is_day() ) {
                $items[] = array(
                    'name' => get_the_date( 'Y' ),
                    'url'  => get_year_link( get_the_date( 'Y' ) ),
                );
                $items[] = array(
                    'name' => get_the_date( 'F' ),
                    'url'  => get_month_link( get_the_date( 'Y' ), get_the_date( 'm' ) ),
                );
                $items[] = array(
                    'name' => get_the_date( 'j' ),
                    'url'  => get_day_link( get_the_date( 'Y' ), get_the_date( 'm' ), get_the_date( 'd' ) ),
                );
            }

        // Search results
        } elseif ( is_search() ) {
            $items[] = array(
                'name' => sprintf(
                    /* translators: %s: search query */
                    __( 'Search: %s', 'simply-seo' ),
                    get_search_query()
                ),
                'url'  => get_search_link(),
            );

        // 404 page
        } elseif ( is_404() ) {
            $items[] = array(
                'name' => __( 'Page not found', 'simply-seo' ),
                'url'  => '',
            );
        }

        return $items;
    }
}

/**
 * Template function to display breadcrumbs
 *
 * @param array $args Optional arguments
 * @return void
 */
function simply_seo_breadcrumbs( $args = array() ) {
    $breadcrumbs = new Simply_SEO_Breadcrumbs();
    // Output is escaped within render method using esc_html, esc_url, and wp_kses
    // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Output escaped in render()
    echo $breadcrumbs->render( $args );
}
