<?php

/**
 * Plugin Name: Simple Upload Weight Limit
 * Description: Optimize server space by setting maximum upload size limits for specific user roles.
 * Version: 1.0.0
 * Author: SimpleWPlugins
 * Author URI: https://simplewplugins.com
 * Text Domain: simple-upload-weight-limit
 * License: GPLv2 or later
 */

defined('ABSPATH') || exit;

/**
 * Handle the upload limit logic
 */
add_filter('wp_handle_upload_prefilter', function ($file) {
    $options  = get_option('sul_settings');
    $limit_kb = ! empty($options['max_size']) ? intval($options['max_size']) : 1024;
    $limit_b  = $limit_kb * 1024;
    $block_admins = !empty($options['block_admins']);

    // Only allow admins to bypass if block_admins is not set
    if (!$block_admins && current_user_can('manage_options')) {
        return $file;
    }

    if ($file['size'] > $limit_b) {
        $file['error'] = sprintf(
            /* translators: %d: maximum file size in KB */
            esc_html__('File too large! Maximum allowed size is %d KB. Please compress your file.', 'simple-upload-weight-limit'),
            $limit_kb
        );
    }

    return $file;
});

/**
 * Register settings with proper sanitization
 */
add_action('admin_init', function () {
    register_setting('sul_group', 'sul_settings', [
        'sanitize_callback' => 'sul_sanitize_settings',
        'default'           => ['max_size' => 1024, 'block_admins' => 0]
    ]);
});

/**
 * Sanitize input data (Important for security review)
 */
function sul_sanitize_settings($input)
{
    $new_input = [];
    if (isset($input['max_size'])) {
        $new_input['max_size'] = absint($input['max_size']);
    }
    $new_input['block_admins'] = !empty($input['block_admins']) ? 1 : 0;
    return $new_input;
}

/**
 * Settings menu
 */
add_action('admin_menu', function () {
    add_options_page(
        __('Upload Weight Limit', 'simple-upload-weight-limit'),
        __('Upload Weight Limit', 'simple-upload-weight-limit'),
        'manage_options',
        'simple-upload-weight-limit',
        'sul_render_admin_page'
    );
});

/**
 * Admin UI
 */
function sul_render_admin_page()
{
    $options = get_option('sul_settings');
    $current = isset($options['max_size']) ? $options['max_size'] : 1024;
    $block_admins = !empty($options['block_admins']);
?>
    <div class="swp-wrap">
        <div class="swp-head">
            <div class="swp-brand">
                <div class="swp-dot"></div>SimpleWPlugins <span style="background:#FFF0EB; color:#FF5513; padding:4px 8px; border-radius:4px; font-size:11px; margin-left:10px;">Upload Weight Limit</span>
            </div>
            <a href="https://simplewplugins.com" target="_blank" style="text-decoration:none; color:#6b7280; font-size:14px;">Visit Website →</a>
        </div>
        <div class="swp-body">
            <form method="post" action="options.php">
                <?php settings_fields('sul_group'); ?>

                <div class="swp-box">
                    <label style="display:block; margin-bottom:15px; font-weight:600; color:#374151;"><?php echo esc_html__('Maximum File Size', 'simple-upload-weight-limit'); ?></label>
                    <input type="number" name="sul_settings[max_size]" class="swp-num" value="<?php echo esc_attr($current); ?>" min="10">
                    <span style="font-size: 22px; font-weight: 700; color: #9ca3af; margin-left: 10px;">KB</span>

                    <div style="margin-top:20px; text-align:left;">
                        <label>
                            <input type="checkbox" name="sul_settings[block_admins]" value="1" <?php checked($block_admins); ?>>
                            <?php echo esc_html__('Apply limit to Administrators too', 'simple-upload-weight-limit'); ?>
                        </label>
                    </div>

                    <p style="margin-top:15px; color:#6b7280; font-size:14px;">
                        <?php
                        $mb = round($current / 1024, 2);
                        /* translators: %s: size in Megabytes */
                        printf(esc_html__('Currently limited to %s MB per upload.', 'simple-upload-weight-limit'), esc_html($mb));
                        ?>
                    </p>
                </div>

                <p style="background:#f3f4f6; padding:15px; border-radius:8px; font-size:13px; color:#4b5563; margin-bottom:25px;">
                    <?php
                    if ($block_admins) {
                        echo esc_html__('Note: Administrators are also subject to this upload limit.', 'simple-upload-weight-limit');
                    } else {
                        echo esc_html__('Note: Administrators can always upload files of any size. This restriction only applies to non-admin roles.', 'simple-upload-weight-limit');
                    }
                    ?>
                </p>

                <?php submit_button(__('Save Limit Settings', 'simple-upload-weight-limit'), 'swp-btn'); ?>
            </form>
        </div>
    </div>
<?php
}

/**
 * Settings link in plugin list
 */
add_filter('plugin_action_links_' . plugin_basename(__FILE__), function ($links) {
    $url = admin_url('options-general.php?page=simple-upload-weight-limit');
    $link = '<a href="' . esc_url($url) . '">' . esc_html__('Settings', 'simple-upload-weight-limit') . '</a>';
    array_unshift($links, $link);
    return $links;
});

/**
 * Enqueue admin CSS only on plugin settings page
 */
function sul_enqueue_admin_css($hook)
{
    if ($hook !== 'settings_page_simple-upload-weight-limit') {
        return;
    }
    wp_enqueue_style(
        'sul-admin-css',
        plugin_dir_url(__FILE__) . 'assets/admin.css',
        [],
        '1.0'
    );
}
add_action('admin_enqueue_scripts', 'sul_enqueue_admin_css');
