<?php
/**
 * Plugin Name: Simple True/False Quiz
 * Description: Easily make true-false quizzes. Simple True/False Quiz is a lightweight plugin that adds a new block to create interactive true/false quiz questions. Also possible to add with a shortcode wherever you want in a page/post. 
 * Version: 2.0.1
 * Author: JEDIGITAL
 * Author URI: https://jedigital.net/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: simple-true-false-quiz
 * Requires at least: 6.0
 * Requires PHP: 7.4
 *
 * Usage as shortcode:
 * [quiz-slider question="Your question here?" correct="true" multiple="false"]
 * Parameters:
 * - question: The question text (required)
 * - correct: true/false - whether the correct answer is True (optional, default: false)
 * - multiple: true/false - allow multiple attempts (optional, default: true)
 */

declare(strict_types=1);

// If this file is called directly, abort.
if (!defined('ABSPATH')) {
    exit;
}

// Include SVG icons
require_once plugin_dir_path(__FILE__) . 'includes/svg-icons.php';

/**
 * Initialize the plugin
 */
function stfq_init() {
    // Register block
    register_block_type(
        plugin_dir_path(__FILE__) . 'block.json',
        array(
            'render_callback' => 'stfq_render_block'
        )
    );

    // Register shortcode
    add_shortcode('quiz-slider', 'stfq_shortcode_handler');

    // Fallback style registration
    if (!wp_style_is('stfq-style', 'registered')) {
        wp_register_style(
            'stfq-style',
            plugins_url('build/style-index.css', __FILE__),
            array(),
            filemtime(plugin_dir_path(__FILE__) . 'build/style-index.css')
        );
        wp_enqueue_style('stfq-style');
    }

    // Ensure frontend script is loaded for shortcodes
    if (!is_admin()) {
        wp_enqueue_script(
            'stfq-frontend',
            plugins_url('build/frontend.js', __FILE__),
            array(),
            filemtime(plugin_dir_path(__FILE__) . 'build/frontend.js'),
            true
        );
    }
}
add_action('init', 'stfq_init');

/**
 * Render callback for the block
 *
 * @param array    $attributes The block attributes
 * @param string   $content    The block content
 * @return string  The block HTML
 */
function stfq_render_block($attributes, $content) {
    // Debug logging
    error_log('STFQ Block Render Called');
    error_log('Attributes: ' . print_r($attributes, true));
    
    $question = $attributes['question'] ?? '';
    if (empty($question) && !empty($content)) {
        $question = $content;
    }
    
    $is_true = $attributes['isTrue'] ?? false;
    $allow_multiple = $attributes['allowMultipleAttempts'] ?? true;
    
    // Set different incorrect message based on multiple attempts setting
    $incorrect_message = $allow_multiple 
        ? __('Sorry, that\'s not right. Try again!', 'simple-true-false-quiz')
        : __('Sorry, that\'s not right.', 'simple-true-false-quiz');
    
    ob_start();
    ?>
    <div class="wp-block-simple-true-false-quiz" 
         data-is-true="<?php echo esc_attr($is_true ? '1' : '0'); ?>"
         data-multiple-attempts="<?php echo esc_attr($allow_multiple ? '1' : '0'); ?>"
         data-correct-message="<?php echo esc_attr__('Correct! Well done!', 'simple-true-false-quiz'); ?>"
         data-incorrect-message="<?php echo esc_attr($incorrect_message); ?>"
         data-reset-label="<?php echo esc_attr__('Reset question', 'simple-true-false-quiz'); ?>"
         role="region"
         aria-label="<?php esc_attr_e('True/False Quiz Question', 'simple-true-false-quiz'); ?>">
        
        <button type="button" class="stfq-icon false" 
                aria-label="<?php esc_attr_e('Select False', 'simple-true-false-quiz'); ?>">
            <?php echo STFQ_Icons::X_ICON; ?>
        </button>
        
        <div class="stfq-question">
            <?php echo wp_kses_post($question); ?>
        </div>
        
        <button type="button" class="stfq-icon true" 
                aria-label="<?php esc_attr_e('Select True', 'simple-true-false-quiz'); ?>">
            <?php echo STFQ_Icons::CHECK_ICON; ?>
        </button>
        
        <div class="stfq-feedback" role="alert" aria-hidden="true"></div>
    </div>
    <?php
    return ob_get_clean();
}

/**
 * Handle the quiz-slider shortcode
 *
 * @param array $atts Shortcode attributes
 * @return string Rendered HTML
 */
function stfq_shortcode_handler($atts) {
    // Parse and validate attributes
    $atts = shortcode_atts(
        array(
            'question' => '',
            'correct' => 'false',
            'multiple' => 'true'
        ),
        $atts,
        'quiz-slider'
    );

    // Validate question
    if (empty($atts['question'])) {
        return '';
    }

    // Convert attributes to block format
    $block_attributes = array(
        'question' => esc_html($atts['question']),
        'isTrue' => $atts['correct'] === 'true',
        'allowMultipleAttempts' => $atts['multiple'] === 'true'
    );

    // Use the same render function as the block
    return stfq_render_block($block_attributes, '');
} 