# WooCommerce Connection Settings - Using v1 Tokens

Simple guide for accessing WooCommerce connection settings using only your v1 API credentials (no Auth0 required).

## Overview

This integration uses a two-step process:
1. Exchange your v1 API credentials for a v3 Bearer token
2. Use the Bearer token to access connection settings

## Prerequisites

- Your TaxCloud v1 API credentials:
  - `apiLoginID` (e.g., `FE95570`)
  - `apiKey` (UUID format, e.g., `fb3e8a3a-057b-4628-a743-c89b4e37dfa8`)

## Step 1: Get v3 Bearer Token

Exchange your v1 credentials for a v3 Bearer token:

**Endpoint:** `POST /api/v3/auth/token`

**Request:**
```bash
curl -X POST \
  -H "Content-Type: application/json" \
  -d '{
    "apiLoginID": "FE95570",
    "apiKey": "fb3e8a3a-057b-4628-a743-c89b4e37dfa8"
  }' \
  https://staging-taxcloudapi.azurewebsites.net/api/v3/auth/token
```

**Response:**
```json
{
  "access_token": "eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9...",
  "access_token_validTo": "2025-11-26T00:52:55Z",
  "token_type": "Bearer",
  "scope": "v3_api",
  "merchant_id": 4,
  "contact_id": 11315,
  "connection_id": "59662"
}
```

## Step 2: Get Connection Settings

Use the Bearer token to access connection settings:

**Endpoint:** `GET /mgmt/connections/{apiKey}`

**Request:**
```bash
curl -X GET \
  -H "Authorization: Bearer {access_token}" \
  -H "Content-Type: application/json" \
  https://api.v3.taxcloud.net/mgmt/connections/fb3e8a3a-057b-4628-a743-c89b4e37dfa8
```

**Response:**
```json
{
  "id": "fb3e8a3a-057b-4628-a743-c89b4e37dfa8",
  "name": "asdasd1",
  "options": {
    "data_mover": {
      "type": "flag",
      "flag": true
    }
  }
}
```

## PHP Example

```php
function get_woo_connection_settings_v1($apiLoginID, $apiKey, $environment = 'staging') {
    // Environment URLs
    $baseUrls = [
        'staging' => 'https://staging-taxcloudapi.azurewebsites.net',
        'prod' => 'https://taxcloudapi.azurewebsites.net'
    ];
    
    $v3MgmtUrls = [
        'staging' => 'https://api.v3.taxcloud.net/mgmt',
        'prod' => 'https://api.v3.taxcloud.com/mgmt'
    ];
    
    $baseUrl = $baseUrls[$environment] ?? $baseUrls['staging'];
    $mgmtUrl = $v3MgmtUrls[$environment] ?? $v3MgmtUrls['staging'];
    
    // Step 1: Get Bearer token
    $tokenUrl = $baseUrl . '/api/v3/auth/token';
    $tokenData = [
        'apiLoginID' => $apiLoginID,
        'apiKey' => $apiKey
    ];
    
    $tokenResponse = wp_remote_post($tokenUrl, [
        'headers' => ['Content-Type' => 'application/json'],
        'body' => json_encode($tokenData),
        'timeout' => 30
    ]);
    
    if (is_wp_error($tokenResponse)) {
        error_log('Token request failed: ' . $tokenResponse->get_error_message());
        return null;
    }
    
    $tokenBody = json_decode(wp_remote_retrieve_body($tokenResponse), true);
    $accessToken = $tokenBody['access_token'] ?? null;
    
    if (!$accessToken) {
        error_log('Failed to get access token: ' . wp_remote_retrieve_body($tokenResponse));
        return null;
    }
    
    // Step 2: Get connection settings
    $settingsUrl = $mgmtUrl . '/connections/' . $apiKey;
    $settingsResponse = wp_remote_get($settingsUrl, [
        'headers' => [
            'Authorization' => 'Bearer ' . $accessToken,
            'Content-Type' => 'application/json'
        ],
        'timeout' => 30
    ]);
    
    if (is_wp_error($settingsResponse)) {
        error_log('Settings request failed: ' . $settingsResponse->get_error_message());
        return null;
    }
    
    $httpCode = wp_remote_retrieve_response_code($settingsResponse);
    if ($httpCode === 404) {
        // Connection settings don't exist yet (normal for new connections)
        return null;
    }
    
    if ($httpCode < 200 || $httpCode >= 300) {
        error_log('Settings request failed with HTTP ' . $httpCode . ': ' . wp_remote_retrieve_body($settingsResponse));
        return null;
    }
    
    return json_decode(wp_remote_retrieve_body($settingsResponse), true);
}

// Usage
$settings = get_woo_connection_settings_v1('FE95570', 'fb3e8a3a-057b-4628-a743-c89b4e37dfa8', 'staging');

if ($settings) {
    $wooDataMover = $settings['options']['woo_data_mover']['flag'] ?? false;
    $mode = $wooDataMover ? 'premium' : 'basic';
    echo "Current mode: " . $mode;
} else {
    echo "Connection settings not found or error occurred";
}
```

## Environments

- **staging**: `https://staging-taxcloudapi.azurewebsites.net` (default)
- **prod**: `https://taxcloudapi.azurewebsites.net`

## Connection Settings Structure

The `data_mover` flag determines the integration mode:
- `flag: false` = Premium mode (data mover enabled)
- `flag: true` = Basic mode (data mover disabled)

## Error Handling

- **400 Bad Request**: Invalid v1 credentials
- **404 Not Found**: Connection settings don't exist yet (normal for new connections)
- **401 Unauthorized**: Bearer token expired or invalid (get a new token)
- **422 Unprocessable Entity**: Invalid connection ID format (must be UUID)

## Notes

- Bearer tokens expire (check `access_token_validTo` in the token response)
- The `apiKey` is used as the `connection_id` in the v3 API
- Connection settings may not exist for new connections (404 is normal)
