<?php

if ( ! defined( 'ABSPATH' ) ) exit;

if ( ! function_exists( 'simpremed_get_template_part' ) ) {

	/**
	 * Load a template part for the Simple Restaurant Menu Display plugin.
	 *
	 * This function attempts to locate a template file in the active theme's
	 * `/simple-restaurant-menu-display/` directory. If not found, it falls back
	 * to the plugin's own `/templates/` directory.
	 *
	 * Variables are made available in the template via the `$args` array.
	 *
	 * Example usage:
	 * simpremed_get_template_part( 'simpremed-menu-item', [ 'currency' => '₹', 'quantities' => $items ] );
	 *
	 * @param string $template The template filename (without `.php`) to load.
	 *                         e.g., 'simpremed-menu-item' loads 'simpremed-menu-item.php'.
	 * @param array  $args     Optional. Associative array of variables to pass to the template.
	 *                         Accessible inside the template via $args['key'].
	 *
	 * @return void
	 */
	function simpremed_get_template_part( $template, $args = [] ) {
		$template_path   = 'simple-restaurant-menu-display/' . $template . '.php';
		$plugin_template = plugin_dir_path( __DIR__ ) . 'templates/' . $template . '.php';
		$theme_template  = locate_template( $template_path );

		$template_file = $theme_template ?: $plugin_template;

		if ( file_exists( $template_file ) ) {
			$args = is_array( $args ) ? $args : [];
			include $template_file;
		}
	}

}

/**
 * Adds a custom 'Menu Order' field to the 'simpremed_menu_section' taxonomy term add/edit forms,
 * saves the field value, and displays it in the admin term list.
 *
 * @package SRMD
 */

// Add the 'menu_order' field when creating a new simpremed_menu_section term.
add_action( 'simpremed_menu_section_add_form_fields', function() {
	?>
	<div class="form-field">
		<label for="menu_order"><?php esc_html_e( 'Menu Order', 'simple-restaurant-menu-display' ); ?></label>
		<input type="number" name="menu_order" id="menu_order" value="0" />
		<p class="description"><?php esc_html_e( 'Order to display this section (lower = earlier)', 'simple-restaurant-menu-display' ); ?></p>
		<?php wp_nonce_field( 'simpremed_menu_order_nonce_action', 'simpremed_menu_order_nonce' ); // Nonce field for security ?>
	</div>
	<?php
} );

/**
 * Adds the 'menu_order' field when editing an existing simpremed_menu_section term.
 *
 * @param WP_Term $term The current term object being edited.
 */
add_action( 'simpremed_menu_section_edit_form_fields', function( $term ) {
	$menu_order = get_term_meta( $term->term_id, 'menu_order', true );
	?>
	<tr class="form-field">
		<th scope="row" valign="top"><label for="menu_order"><?php esc_html_e( 'Menu Order', 'simple-restaurant-menu-display' ); ?></label></th>
		<td>
			<input type="number" name="menu_order" id="menu_order" value="<?php echo esc_attr( $menu_order ); ?>" />
			<p class="description"><?php esc_html_e( 'Order to display this section (lower = earlier)', 'simple-restaurant-menu-display' ); ?></p>
			<?php wp_nonce_field( 'simpremed_menu_order_nonce_action', 'simpremed_menu_order_nonce' ); // Nonce field for security ?>
		</td>
	</tr>
	<?php
}, 10, 1 );

/**
 * Save the custom menu order for a taxonomy term in 'simpremed_menu_section'.
 *
 * This function verifies a nonce for security and updates the term meta
 * with a sanitized menu order value, but only for the 'simpremed_menu_section' taxonomy.
 *
 * @param int $term_id The ID of the term being saved.
 */
function simpremed_save_menu_order( $term_id ) {
	$term = get_term( $term_id );

	// Only apply to the 'simpremed_menu_section' taxonomy.
	if ( ! $term || is_wp_error( $term ) || 'simpremed_menu_section' !== $term->taxonomy ) {
		return;
	}

	// check nonce
	$nonce = isset( $_POST['simpremed_menu_order_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['simpremed_menu_order_nonce'] ) ) : '';

	if ( $nonce && wp_verify_nonce( $nonce, 'simpremed_menu_order_nonce_action' ) ) {
		if ( isset( $_POST['menu_order'] ) ) {
			$menu_order = (int) sanitize_text_field( wp_unslash( $_POST['menu_order'] ) );
			update_term_meta( $term_id, 'menu_order', $menu_order );
		}
	} else {
		die( 'Security check failed.' );
	}
}

add_action( 'edited_simpremed_menu_section', 'simpremed_save_menu_order', 10, 1 );
add_action( 'create_simpremed_menu_section', 'simpremed_save_menu_order', 10, 1 );

/**
 * Adds a 'Menu Order' column to the simpremed_menu_section taxonomy list table.
 *
 * @param array $columns Existing column definitions.
 * @return array Modified column definitions.
 */
add_filter( 'manage_edit-simpremed_menu_section_columns', function( $columns ) {
	$columns['menu_order'] = __( 'Menu Order', 'simple-restaurant-menu-display' );
	return $columns;
} );

/**
 * Displays the value for the 'Menu Order' column in the taxonomy list table.
 *
 * @param string $out Output for the custom column.
 * @param string $column Column name.
 * @param int    $term_id Term ID.
 * @return string Modified output.
 */
add_filter( 'manage_simpremed_menu_section_custom_column', function( $out, $column, $term_id ) {
	if ( $column === 'menu_order' ) {
		$out = get_term_meta( $term_id, 'menu_order', true );
	}
	return $out;
}, 10, 3 );