<?php
/**
 * Shortcode handler for displaying the restaurant menu.
 *
 * Registers the [simple_restaurant_menu_display] shortcode and defines its behavior,
 * including optional filtering by taxonomy and meta fields.
 *
 * @package Simpremed
 */
if ( ! defined( 'ABSPATH' ) ) exit;

class Simpremed_Shortcodes {

	/**
	 * Constructor.
	 *
	 * Registers the shortcode [simple_restaurant_menu_display] on class initialization.
	 */
	public function __construct() {
		add_shortcode( 'simple_restaurant_menu_display', [ $this, 'simpremed_render_menu_shortcode' ] );
	}

	/**
	 * Renders the restaurant menu when the shortcode is used.
	 *
	 * Supports optional attributes to filter by simpremed_menu_section (taxonomy) and "special" status (custom meta).
	 * If no filters are applied, the default simpremed-menu-loop template displays all sections grouped.
	 *
	 * Shortcode attributes:
	 * - section (string) — The slug of the simpremed_menu_section taxonomy term to filter items.
	 * - special (string) — If set to "yes", filters items marked as special (meta key 'simpremed_is_special' = 1).
	 *
	 * Examples:
	 * - [simple_restaurant_menu_display] — shows full menu grouped by section.
	 * - [simple_restaurant_menu_display section="breakfast"] — filters items to the "breakfast" section.
	 * - [simple_restaurant_menu_display special="yes"] — shows only special items.
	 * - [simple_restaurant_menu_display section="lunch" special="yes"] — shows special lunch items.
	 *
	 * @param array $atts Shortcode attributes provided by the user.
	 * @return string HTML output of the rendered menu content.
	 */
	public function simpremed_render_menu_shortcode( $atts ) {

		// Merge user-defined shortcode attributes with default values.
		$atts = shortcode_atts([
			'section' => '',
			'special' => '',
		], $atts, 'simple_restaurant_menu_display');

		// Determine if any filters are applied.
		$has_filter = ! empty( $atts['section'] ) || strtolower( $atts['special'] ) === 'yes';

		ob_start();

		// If filters are set, prepare a filtered query.
		if ( $has_filter ) {
			$args = [
				'post_type'      => 'simpremed_menu_item',
				'post_status'    => 'publish',
				'posts_per_page' => -1,
				'orderby'        => 'menu_order',
				'order'          => 'ASC',
				//'tax_query'      => [],
				//'meta_query'     => [],
			];

			// Filter by taxonomy: menu_section (if section is provided).
			if ( ! empty( $atts['section'] ) ) {
				$args['tax_query'][] = [
					'taxonomy' => 'simpremed_menu_section',
					'field'    => 'slug',
					'terms'    => $atts['section'],
				];
			}

			// Filter by meta field: simpremed_is_special = 1 (if special="yes").
			if ( strtolower( $atts['special'] ) === 'yes' ) {
				$args['meta_query'][] = [
					'key'     => 'simpremed_is_special',
					'value'   => '1',
					'compare' => '=',
				];
			}

			// Run the query and pass it to the menu loop template.
			$query = new WP_Query( $args );
			simpremed_get_template_part( 'simpremed-menu-loop', [
				'query'        => $query,
				'section_slug' => $atts['section'],
				'is_special' => $atts['special'],
			] );
		} else {
			// No filters — render the default grouped menu view.
			simpremed_get_template_part( 'simpremed-menu-loop' );
		}

		return ob_get_clean();
	}
}
