<?php
/**
 * Core functionality for registering the custom post type and taxonomy,
 * adding meta boxes, saving metadata, and managing plugin settings.
 *
 * @package Simpremed
 */

 if ( ! defined( 'ABSPATH' ) ) exit;
 
/**
 * Register 'simpremed_menu_item' post type and 'simpremed_menu_section' taxonomy.
 */
add_action('init', 'simpremed_register_menu_item');
function simpremed_register_menu_item() {
	register_post_type('simpremed_menu_item', [
        'labels' => [
            'name'               => __('Menu Items', 'simple-restaurant-menu-display'),
            'singular_name'      => __('Menu Item', 'simple-restaurant-menu-display'),
            'add_new'            => __('Add New', 'simple-restaurant-menu-display'),
            'add_new_item'       => __('Add New Menu Item', 'simple-restaurant-menu-display'),
            'edit_item'          => __('Edit Menu Item', 'simple-restaurant-menu-display'),
            'new_item'           => __('New Menu Item', 'simple-restaurant-menu-display'),
            'view_item'          => __('View Menu Item', 'simple-restaurant-menu-display'),
            'search_items'       => __('Search Menu Items', 'simple-restaurant-menu-display'),
            'not_found'          => __('No menu items found.', 'simple-restaurant-menu-display'),
            'not_found_in_trash' => __('No menu items found in Trash.', 'simple-restaurant-menu-display'),
            'menu_name'          => __('Menu Items', 'simple-restaurant-menu-display'),
        ],
        'public'       => true,
        'menu_icon'    => 'dashicons-carrot',
        'supports'     => ['title', 'editor', 'thumbnail'],
        'show_in_rest' => true,
        'query_var' => true,
    ]);

	register_taxonomy('simpremed_menu_section', 'simpremed_menu_item', [
		'label'        => __('Menu Sections', 'simple-restaurant-menu-display'),
		'hierarchical' => true,
		'public'       => true,
		'show_in_rest' => true,
        'query_var' => true,
        'rewrite'   => ['slug' => 'simpremed-menu-section'],
	]);
}

/**
 * Add the repeater field meta box for quantity and prices.
 */
add_action('add_meta_boxes', function() {
	add_meta_box(
		'simpremed_quantities',
		__('Quantity & Prices', 'simple-restaurant-menu-display'),
		'simpremed_render_quantity_box',
		'simpremed_menu_item',
		'normal',
		'default'
	);
});

/**
 * Render the repeater fields inside the meta box.
 *
 * @param WP_Post $post The post object.
 */
function simpremed_render_quantity_box($post) {
    
	$values = get_post_meta($post->ID, '_simpremed_quantities', true);
    wp_nonce_field('simpremed_save_quantities', 'simpremed_quantities_nonce');
	echo '<div id="simpremed-repeater-wrapper">';
	echo '<button type="button" class="button" id="simpremed-add-row">Add Row</button>';
	echo '<table class="widefat simpremed-repeater"><thead><tr><th>Label</th><th>Price</th><th>Remove</th></tr></thead><tbody>';

	if ($values) {
		foreach (json_decode($values, true) as $row) {
			echo '<tr>';
			echo '<td><input type="text" name="simpremed_label[]" value="' . esc_attr($row['label']) . '" /></td>';
			echo '<td><input type="text" name="simpremed_price[]" value="' . esc_attr($row['price']) . '" /></td>';
			echo '<td><button class="button remove-row">X</button></td>';
			echo '</tr>';
		}
	}

	echo '</tbody></table></div>';
}

/**
 * Save repeater field data when the menu item post is saved.
 *
 * @param int $post_id The ID of the current post.
 */
/**
 * Save custom quantity labels and prices for the 'menu_item' post type.
 *
 * This function verifies the nonce, then collects and sanitizes user-submitted
 * labels and prices. Only non-empty labels with valid float prices are saved.
 * The data is stored in the '_simpremed_quantities' post meta as a JSON string.
 *
 * @param int $post_id The ID of the post being saved.
 */
add_action('save_post_simpremed_menu_item', 'simpremed_save_quantities_func', 20, 1);
function simpremed_save_quantities_func( $post_id ) {

    // return if empty, useful when blank API call runs
    if ( empty( $_POST ) ) {
        return;
    }

    // Ensure we are not running this on autosave
    if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
        return;
    }

	// Verify the nonce securely.
	$nonce = isset( $_POST['simpremed_quantities_nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['simpremed_quantities_nonce'] ) ) : '';
	if ( ! $nonce || ! wp_verify_nonce( $nonce, 'simpremed_save_quantities' ) ) {
		return;
	}

    // Check user permissions.
	if ( isset( $_POST['post_type'] ) && 'simpremed_menu_item' === $_POST['post_type'] && ! current_user_can( 'edit_post', $post_id ) ) {
		return;
	}


	// Check if 'simpremed_label' and 'simpremed_price' are set and are arrays
    $raw_labels = isset( $_POST['simpremed_label'] ) ? wp_unslash( $_POST['simpremed_label'] ) : [];
    $raw_prices = isset( $_POST['simpremed_price'] ) ? wp_unslash( $_POST['simpremed_price'] ) : [];
    $sanitized_labels = array_map( 'sanitize_text_field', $raw_labels );
    $sanitized_prices = array_map( 'sanitize_text_field', $raw_prices );
    if ( ! empty( $sanitized_labels ) && is_array( $sanitized_labels ) && ! empty( $sanitized_prices ) && is_array( $sanitized_prices ) ) {
		$quantities = [];

		foreach ( $sanitized_labels as $index => $label ) {
			$price = isset( $sanitized_prices[ $index ] ) ? $sanitized_prices[ $index ] : '';

			// Ensure label is not empty and price is a valid float
			if ( ! empty( $label ) && $price !== '' && is_numeric( $price ) ) {
				$quantities[] = [
					'label' => $label,
					'price' => floatval( $price ),
				];
			}
		}

		update_post_meta( $post_id, '_simpremed_quantities', wp_json_encode( $quantities ) );
	}
   
}

/**
 * Register plugin settings for currency symbol.
 */
/**
 * Add a submenu item under the "Menu Items" CPT for plugin settings.
 */
add_action('admin_menu', 'simpremed_add_settings_submenu');
function simpremed_add_settings_submenu() {
    add_submenu_page(
        'edit.php?post_type=simpremed_menu_item',                           // Parent menu slug
        __('Menu Settings', 'simple-restaurant-menu-display'),    // Page title
        __('Settings', 'simple-restaurant-menu-display'),         // Menu title
        'manage_options',                                         // Capability
        'simpremed-settings',                                          // Menu slug
        'simpremed_render_settings_page'                               // Callback function
    );
}

/**
 * Render the settings page content.
 */
function simpremed_render_settings_page() {
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Restaurant Menu Settings', 'simple-restaurant-menu-display'); ?></h1>
        <form method="post" action="options.php">
            <?php
            settings_fields('simpremed_settings_group');
            do_settings_sections('simpremed-settings');
            submit_button();
            ?>
        </form>
    </div>
    <?php
}

/**
 * Register custom plugin settings for the settings page.
 */
add_action('admin_init', 'simpremed_register_custom_settings');
function simpremed_register_custom_settings() {
    register_setting('simpremed_settings_group', 'simpremed_currency_symbol', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => '₹'
    ]);

    add_settings_section(
        'simpremed_settings_section',                                    // ID
        __('Currency Settings', 'simple-restaurant-menu-display'),  // Title
        null,                                                       // Callback
        'simpremed-settings'                                             // Page
    );

    add_settings_field(
        'simpremed_currency_symbol',                                     // ID
        __('Currency Symbol', 'simple-restaurant-menu-display'),    // Title
        'simpremed_currency_symbol_field_html',                          // Callback
        'simpremed-settings',                                            // Page
        'simpremed_settings_section'                                     // Section
    );
}

/**
 * Render the input field for currency symbol setting.
 */
function simpremed_currency_symbol_field_html() {
	$value = esc_attr(get_option('simpremed_currency_symbol', '₹'));
	echo '<input type="text" name="simpremed_currency_symbol" value="' . esc_attr($value) . '" class="regular-text" />';
}

/**
 * Get the currency symbol with filter support.
 *
 * @return string Filtered currency symbol.
 */
function simpremed_get_currency_symbol() {
	$symbol = get_option('simpremed_currency_symbol', '₹');
	return apply_filters('simpremed_currency_symbol', $symbol);
}

// Load additional meta box logic (if needed).
require_once plugin_dir_path(__FILE__) . 'simpremed-meta-boxes.php';
