/**
 * Gutenberg block for inserting the [simple_restaurant_menu_display] shortcode.
 * Allows selecting a menu section and toggling "special only" filter.
 *
 * @since 1.0.0
 */

(function (blocks, element, editor, components, i18n) {
	const { registerBlockType } = blocks;
	const { createElement: el } = element;
	const { InspectorControls } = editor;
	const { PanelBody, TextControl, ToggleControl } = components;
	const { __ } = i18n;

	/**
	 * Register the block type.
	 */
	registerBlockType('simpremed/menu-shortcode', {
		title: __('Simple Restaurant Menu', 'simple-restaurant-menu-display'),
		icon: 'clipboard',
		category: 'widgets',
		supports: {
			html: false,
		},
		attributes: {
			/**
			 * Section slug to filter menu items by.
			 * @type {string}
			 */
			section: {
				type: 'string',
				default: '',
			},
			/**
			 * Whether to show only special items.
			 * @type {boolean}
			 */
			special: {
				type: 'boolean',
				default: false,
			},
		},

		/**
		 * Edit function for the block editor.
		 * @param {Object} props Block properties.
		 * @returns {Element} Block UI
		 */
		edit: (props) => {
			const {
				attributes: { section, special },
				setAttributes,
			} = props;

			// Construct shortcode preview
			const shortcode = `[simple_restaurant_menu_display${section ? ` section="${section}"` : ''}${special ? ` special="yes"` : ''}]`;

			return el(
				'div',
				{},
				el(
					InspectorControls,
					{},
					el(
						PanelBody,
						{ title: __('Menu Settings', 'simple-restaurant-menu-display'), initialOpen: true },
						el(TextControl, {
							label: __('Section (slug)', 'simple-restaurant-menu-display'),
							value: section,
							onChange: (value) => setAttributes({ section: value }),
						}),
						el(ToggleControl, {
							label: __('Show Special Items Only', 'simple-restaurant-menu-display'),
							checked: special,
							onChange: (value) => setAttributes({ special: value }),
						})
					)
				),
				el('p', {}, __('This block will display:', 'simple-restaurant-menu-display')),
				el('code', {}, shortcode)
			);
		},

		/**
		 * Save function returns null since block is rendered server-side.
		 * @returns {null}
		 */
		save: () => null,
	});
})(
	window.wp.blocks,
	window.wp.element,
	window.wp.blockEditor || window.wp.editor,
	window.wp.components,
	window.wp.i18n
);
