<?php
/**
 * Plugin Name: Simple Redirect 404 to Homepage
 * Plugin URI: https://wordpress.org/plugins/simple-redirect-404-to-homepage/
 * Description: Redirect all 404 pages to homepage with configurable redirect type and optional logging. GDPR compliant with privacy considerations.
 * Version: 2.0
 * Author: YoGha
 * Author URI: https://wpagency.top/
 * License: GPLv2 or later
 * Text Domain: simple-redirect-404-to-homepage
 * Requires at least: 5.0
 * Tested up to: 6.8
 * Requires PHP: 7.4
 *
 * @package Simple_Redirect_404
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
define( 'SIMPLE_REDIRECT_404_VERSION', '2.0' );
define( 'SIMPLE_REDIRECT_404_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SIMPLE_REDIRECT_404_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SIMPLE_REDIRECT_404_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );

/**
 * Main plugin class.
 */
class Simple_Redirect_404 {

	/**
	 * Plugin settings.
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->load_settings();
		$this->init_hooks();
	}

	/**
	 * Initialize hooks.
	 */
	private function init_hooks() {
		// Initialize plugin.
		add_action( 'init', array( $this, 'init' ) );

		// Main redirect functionality.
		add_action( 'template_redirect', array( $this, 'handle_404_redirect' ), 1 );

		// Admin functionality.
		add_action( 'admin_init', array( $this, 'admin_init' ) );
		add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );

		// Plugin settings.
		add_filter( 'plugin_action_links_' . SIMPLE_REDIRECT_404_PLUGIN_BASENAME, array( $this, 'add_settings_link' ) );

		// Privacy policy content.
		add_action( 'admin_init', array( $this, 'add_privacy_policy_content' ) );
	}

	/**
	 * Initialize plugin.
	 */
	public function init() {
		load_plugin_textdomain(
			'simple-redirect-404-to-homepage',
			false,
			dirname( SIMPLE_REDIRECT_404_PLUGIN_BASENAME ) . '/languages'
		);
	}

	/**
	 * Load plugin settings.
	 */
	private function load_settings() {
		$default_settings = array(
			'redirect_type'    => 301,
			'enable_logging'   => false,
			'exclude_patterns' => '',
			'exclude_ajax'     => true,
		);

		$this->settings = wp_parse_args(
			get_option( 'simple_redirect_404_settings', array() ),
			$default_settings
		);
	}

	/**
	 * Handle 404 redirect with enhanced logic.
	 */
	public function handle_404_redirect() {
		if ( ! is_404() ) {
			return;
		}

		// Don't redirect AJAX requests if disabled.
		if ( $this->settings['exclude_ajax'] && wp_doing_ajax() ) {
			return;
		}

		// Don't redirect REST API requests.
		if ( defined( 'REST_REQUEST' ) && REST_REQUEST ) {
			return;
		}

		// Don't redirect in admin area.
		if ( is_admin() ) {
			return;
		}

		// Check for excluded URL patterns.
		if ( $this->is_url_excluded() ) {
			return;
		}

		// Log the redirect if enabled.
		if ( $this->settings['enable_logging'] ) {
			$this->log_redirect();
		}

		// Perform the redirect.
		$redirect_url = apply_filters( 'simple_redirect_404_redirect_url', home_url() );
		$status_code  = apply_filters( 'simple_redirect_404_status_code', absint( $this->settings['redirect_type'] ) );

		// Validate status code.
		if ( ! in_array( $status_code, array( 301, 302, 303, 307 ), true ) ) {
			$status_code = 301;
		}

		wp_safe_redirect( $redirect_url, $status_code );
		exit;
	}

	/**
	 * Check if current URL should be excluded from redirect.
	 *
	 * @return bool
	 */
	private function is_url_excluded() {
		$current_url = $_SERVER['REQUEST_URI'] ?? '';

		if ( empty( $this->settings['exclude_patterns'] ) || empty( $current_url ) ) {
			return false;
		}

		$patterns = array_map( 'trim', explode( "\n", $this->settings['exclude_patterns'] ) );

		foreach ( $patterns as $pattern ) {
			if ( empty( $pattern ) ) {
				continue;
			}

			// Simple string matching for basic patterns.
			if ( false !== strpos( $current_url, $pattern ) ) {
				return true;
			}

			// Regex matching for advanced patterns.
			if ( $this->is_regex_pattern( $pattern ) ) {
				if ( @preg_match( $pattern, $current_url ) ) { // phpcs:ignore WordPress.PHP.NoSilencedErrors.Discouraged
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Check if pattern is a regex pattern.
	 *
	 * @param string $pattern The pattern to check.
	 * @return bool
	 */
	private function is_regex_pattern( $pattern ) {
		return ( strpos( $pattern, '^' ) === 0 || substr( $pattern, -1 ) === '$' || strpos( $pattern, '*' ) !== false );
	}

	/**
	 * Log redirect for debugging purposes.
	 */
	private function log_redirect() {
		$log_entry = sprintf(
			'[%s] 404 Redirect: %s -> %s',
			wp_date( 'Y-m-d H:i:s' ),
			esc_url_raw( $_SERVER['REQUEST_URI'] ?? 'unknown' ),
			home_url()
		);

		// Log to WordPress debug log.
		error_log( $log_entry ); // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
	}

	/**
	 * Admin initialization.
	 */
	public function admin_init() {
		$this->register_settings();
	}

	/**
	 * Register plugin settings.
	 */
	private function register_settings() {
		register_setting(
			'simple_redirect_404_settings_group',
			'simple_redirect_404_settings',
			array(
				'sanitize_callback' => array( $this, 'sanitize_settings' ),
				'default'           => array(),
			)
		);

		add_settings_section(
			'simple_redirect_404_main_section',
			__( 'Redirect Settings', 'simple-redirect-404-to-homepage' ),
			array( $this, 'settings_section_callback' ),
			'simple-redirect-404-settings'
		);

		add_settings_field(
			'redirect_type',
			__( 'Redirect Type', 'simple-redirect-404-to-homepage' ),
			array( $this, 'redirect_type_field_callback' ),
			'simple-redirect-404-settings',
			'simple_redirect_404_main_section'
		);

		add_settings_field(
			'enable_logging',
			__( 'Enable Logging', 'simple-redirect-404-to-homepage' ),
			array( $this, 'enable_logging_field_callback' ),
			'simple-redirect-404-settings',
			'simple_redirect_404_main_section'
		);

		add_settings_field(
			'exclude_patterns',
			__( 'Exclude URL Patterns', 'simple-redirect-404-to-homepage' ),
			array( $this, 'exclude_patterns_field_callback' ),
			'simple-redirect-404-settings',
			'simple_redirect_404_main_section'
		);

		add_settings_field(
			'exclude_ajax',
			__( 'Exclude AJAX Requests', 'simple-redirect-404-to-homepage' ),
			array( $this, 'exclude_ajax_field_callback' ),
			'simple-redirect-404-settings',
			'simple_redirect_404_main_section'
		);
	}

	/**
	 * Sanitize settings.
	 *
	 * @param array $input Settings input.
	 * @return array Sanitized settings.
	 */
	public function sanitize_settings( $input ) {
		$sanitized = array();

		if ( isset( $input['redirect_type'] ) ) {
			$sanitized['redirect_type'] = absint( $input['redirect_type'] );
			if ( ! in_array( $sanitized['redirect_type'], array( 301, 302, 303, 307 ), true ) ) {
				$sanitized['redirect_type'] = 301;
			}
		}

		if ( isset( $input['enable_logging'] ) ) {
			$sanitized['enable_logging'] = (bool) $input['enable_logging'];
		}

		if ( isset( $input['exclude_patterns'] ) ) {
			$sanitized['exclude_patterns'] = sanitize_textarea_field( $input['exclude_patterns'] );
		}

		if ( isset( $input['exclude_ajax'] ) ) {
			$sanitized['exclude_ajax'] = (bool) $input['exclude_ajax'];
		}

		return $sanitized;
	}

	/**
	 * Settings section callback.
	 */
	public function settings_section_callback() {
		echo '<p>' . esc_html__( 'Configure how 404 pages should be redirected to the homepage.', 'simple-redirect-404-to-homepage' ) . '</p>';
	}

	/**
	 * Redirect type field callback.
	 */
	public function redirect_type_field_callback() {
		$options = array(
			301 => __( '301 Moved Permanently', 'simple-redirect-404-to-homepage' ),
			302 => __( '302 Found', 'simple-redirect-404-to-homepage' ),
			303 => __( '303 See Other', 'simple-redirect-404-to-homepage' ),
			307 => __( '307 Temporary Redirect', 'simple-redirect-404-to-homepage' ),
		);

		echo '<select name="simple_redirect_404_settings[redirect_type]">';
		foreach ( $options as $value => $label ) {
			printf(
				'<option value="%s" %s>%s</option>',
				esc_attr( $value ),
				selected( $this->settings['redirect_type'], $value, false ),
				esc_html( $label )
			);
		}
		echo '</select>';
		echo '<p class="description">' . esc_html__( 'Choose the HTTP status code for the redirect. 301 is recommended for SEO.', 'simple-redirect-404-to-homepage' ) . '</p>';
	}

	/**
	 * Enable logging field callback.
	 */
	public function enable_logging_field_callback() {
		printf(
			'<input type="checkbox" name="simple_redirect_404_settings[enable_logging]" value="1" %s />',
			checked( $this->settings['enable_logging'], true, false )
		);
		echo '<span class="description">' . esc_html__( 'Log redirects to the WordPress debug log. Requires WP_DEBUG to be enabled.', 'simple-redirect-404-to-homepage' ) . '</span>';
	}

	/**
	 * Exclude patterns field callback.
	 */
	public function exclude_patterns_field_callback() {
		printf(
			'<textarea name="simple_redirect_404_settings[exclude_patterns]" rows="5" cols="50" class="large-text code">%s</textarea>',
			esc_textarea( $this->settings['exclude_patterns'] )
		);
		echo '<p class="description">' . esc_html__( 'Enter URL patterns (one per line) that should not be redirected. You can use simple string matching or regex patterns.', 'simple-redirect-404-to-homepage' ) . '</p>';
	}

	/**
	 * Exclude AJAX field callback.
	 */
	public function exclude_ajax_field_callback() {
		printf(
			'<input type="checkbox" name="simple_redirect_404_settings[exclude_ajax]" value="1" %s />',
			checked( $this->settings['exclude_ajax'], true, false )
		);
		echo '<span class="description">' . esc_html__( 'Exclude AJAX requests from being redirected.', 'simple-redirect-404-to-homepage' ) . '</span>';
	}

	/**
	 * Add admin menu.
	 */
	public function add_admin_menu() {
		add_options_page(
			__( '404 Redirect Settings', 'simple-redirect-404-to-homepage' ),
			__( '404 Redirect', 'simple-redirect-404-to-homepage' ),
			'manage_options',
			'simple-redirect-404-settings',
			array( $this, 'admin_page_callback' )
		);
	}

	/**
	 * Admin page callback.
	 */
	public function admin_page_callback() {
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}
		?>
		<div class="wrap">
			<h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
			<form action="options.php" method="post">
				<?php
				settings_fields( 'simple_redirect_404_settings_group' );
				do_settings_sections( 'simple-redirect-404-settings' );
				submit_button();
				?>
			</form>
		</div>
		<?php
	}

	/**
	 * Add settings link to plugins page.
	 *
	 * @param array $links Existing plugin action links.
	 * @return array Modified plugin action links.
	 */
	public function add_settings_link( $links ) {
		$settings_link = sprintf(
			'<a href="%s">%s</a>',
			esc_url( admin_url( 'options-general.php?page=simple-redirect-404-settings' ) ),
			__( 'Settings', 'simple-redirect-404-to-homepage' )
		);
		array_unshift( $links, $settings_link );
		return $links;
	}

	/**
	 * Add privacy policy content.
	 */
	public function add_privacy_policy_content() {
		if ( ! function_exists( 'wp_add_privacy_policy_content' ) ) {
			return;
		}

		$content = sprintf(
			'<h2>%s</h2><p>%s</p>',
			__( 'Simple Redirect 404 to Homepage', 'simple-redirect-404-to-homepage' ),
			__( 'This plugin redirects 404 (page not found) errors to your homepage. It does not collect, store, or process any personal user data. The plugin may log redirect activity for debugging purposes, but these logs do not contain personally identifiable information and are stored temporarily in server logs only when logging is explicitly enabled in settings.', 'simple-redirect-404-to-homepage' )
		);

		wp_add_privacy_policy_content(
			__( 'Simple Redirect 404 to Homepage', 'simple-redirect-404-to-homepage' ),
			wp_kses_post( wpautop( $content, false ) )
		);
	}
}

// Initialize the plugin.
new Simple_Redirect_404();

/**
 * Activation hook.
 */
register_activation_hook( __FILE__, 'simple_redirect_404_activate' );
function simple_redirect_404_activate() {
	// Set default settings on activation.
	$default_settings = array(
		'redirect_type'    => 301,
		'enable_logging'   => false,
		'exclude_patterns' => '',
		'exclude_ajax'     => true,
	);

	if ( ! get_option( 'simple_redirect_404_settings' ) ) {
		add_option( 'simple_redirect_404_settings', $default_settings );
	}
}

/**
 * Deactivation hook.
 */
register_deactivation_hook( __FILE__, 'simple_redirect_404_deactivate' );
function simple_redirect_404_deactivate() {
	// Clean up if needed.
}

/**
 * Uninstall hook.
 */
register_uninstall_hook( __FILE__, 'simple_redirect_404_uninstall' );
function simple_redirect_404_uninstall() {
	delete_option( 'simple_redirect_404_settings' );
}
