<?php

if (!defined('ABSPATH')) {
    exit;
} // Exit if accessed directly

/**
 * Simple_Job_Board_Shortcode_Jobpost Class
 * 
 * This class lists the jobs on frontend for [jobpost] shortcode.
 * 
 * @link        https://wordpress.org/plugins/simple-job-board
 * @since       2.2.3
 * @since       2.4.0   Revised Inputs & Outputs Sanitization & Escaping
 * @since       2.8.0   Removed 'sjb_job_listing_view' hook to make the global listing view settings independent
 * @since       2.13.9  Modify shortcode to add functionality to remove pagination and seach form on the job listing 
 * @since       2.13.9  Modify shortcode to add functionality to display jobs by specific tags 
 *
 * @package     Simple_Job_Board
 * @subpackage  Simple_Job_Board/includes/shortcodes
 * @author      PressTigers <support@presstigers.com>
 */
class Simple_Job_Board_Shortcode_Jobpost {

    /**
     * Constructor
     */
    public function __construct() {

        // Hook -> Add Job Listing ShortCode
        add_shortcode('jobpost', array($this, 'jobpost_shortcode'));
    }

    /**
     * List all Jobs.
     *
     * @since   1.0.0
     * @since   2.9.5   Attached filter sjb_posts_per_page
     * 
     * @param   array   $atts    Shortcode attribute
     * @return  HTML    $html    Job Listing HTML Structure.
     */
    public function jobpost_shortcode($atts, $content) {

        /**
         * Enqueue Frontend Scripts.
         * 
         * @since   2.2.4
         */
        do_action('sjb_enqueue_scripts');

        ob_start();

        global $job_query;

        // Shortcode Default Array, show_search_form, show_pagination added after version 2.13.8
        $shortcode_args = array(
            'posts' => apply_filters('sjb_posts_per_page', 15),
            'category' => '',
            'type' => '',
            'location' => '',
            'keywords' => '',
            'order' => 'DESC',
            'search' => 'true',
            'layout' => '',
            'show_search_form' => 'yes', 
            'show_pagination'  => 'yes', 
            'tag' => '',
        );

        $atts = is_array($atts) ? apply_filters('sjb_shortcode_atts', array_map('sanitize_text_field', $atts)) : '';
        
        // Combine User Defined Shortcode Attributes with Known Attributes
        $shortcode_args = shortcode_atts(apply_filters('sjb_output_jobs_defaults', $shortcode_args, $atts), $atts);

        
        // Get paged variable.
        if (get_query_var('paged')) {
            $paged = (int) get_query_var('paged');
        } elseif (get_query_var('page')) {
            $paged = (int) get_query_var('page');
        } else {
            $paged = 1;
        }


        // WP Query Default Arguments
        $args = apply_filters(
            'sjb_output_jobs_args', array(
                'post_status' => 'publish',
                'posts_per_page' => sanitize_text_field($shortcode_args['posts']),
                'post_type' => 'jobpost',
                'paged' => $paged,
                'order' => sanitize_text_field($shortcode_args['order']),
                'jobpost_category' => (!empty($_GET['selected_category']) && -1 != $_GET['selected_category'] ) ? sanitize_text_field($_GET['selected_category']) : sanitize_text_field($shortcode_args['category']),
                'jobpost_job_type' => (!empty($_GET['selected_jobtype']) && -1 != $_GET['selected_jobtype'] ) ? sanitize_text_field($_GET['selected_jobtype']) : sanitize_text_field($shortcode_args['type']),
                'jobpost_location' => (!empty($_GET['selected_location']) && -1 != $_GET['selected_location'] ) ? sanitize_text_field($_GET['selected_location']) : sanitize_text_field($shortcode_args['location']),
                's' => ( NULL != filter_input(INPUT_GET, 'search_keywords') ) ? sanitize_text_field($_GET['search_keywords']) : '',
                'jobpost_tag' => (!empty($_GET['selected_tag'])) ? sanitize_text_field($_GET['selected_tag']) : '',
            ), $atts
        );

        /**
         * Add support for filtering by tags attribute (comma-separated)
         */
        if (!empty($shortcode_args['tag'])) {
            $tags_array = array_map('trim', explode(',', $shortcode_args['tag']));
            $args['tax_query'][] = array(
                'taxonomy' => 'jobpost_tag',
                'field'    => 'slug',
                'terms'    => $tags_array,
            );
        }

        // Job Query
        $job_query = new WP_Query($args);
        /**
         * Fires before listing jobs on job listing page.
         * 
         * @since   2.2.0
         */
        do_action('sjb_job_filters_before');

        /**
         * Template -> Job Listing Wrapper Start:
         * 
         * - SJB Starting of Job Listing Wrapper
         */
        get_simple_job_board_template('listing/listing-wrapper-start.php');

        if ( isset( $shortcode_args['show_search_form'] ) && strtolower( $shortcode_args['show_search_form'] ) != 'no' ) {
            if ('false' != strtolower($shortcode_args['search']) && !empty($shortcode_args['search'])):

                /**
                 * Template -> Job Filters:
                 * 
                 * - Keywords Search.
                 * - Job Category Filter.
                 * - Job Type Filter.
                 * - Job Location Filter.
                 * 
                 * Search jobs by keywords, category, location & type.
                 */
                if (!wp_doing_ajax()) {
                    get_simple_job_board_template('job-filters.php', array(
                        'per_page' => $shortcode_args['posts'],
                        'order' => $shortcode_args['order'],
                        'categories' => $shortcode_args['category'],
                        'job_types' => $shortcode_args['type'],
                        'atts' => $atts,
                        'location' => $shortcode_args['location'],
                        'keywords' => $shortcode_args['keywords']
                    ));
                }
            endif;
        }

        /**
         * Template -> Job Listing Start:
         * 
         * - SJB Starting of Job List
         */
        get_simple_job_board_template('listing/job-listings-start.php');

        /**
         * Fires before listing jobs on job listing page.
         * 
         * @since   2.2.0
         */
        do_action('sjb_job_listing_before');

        if ($job_query->have_posts()) {
            
            global $counter, $post_count;
            $counter = 1;
            $post_count = $job_query->post_count;
            do_action("display_job_count_on_shortcode_page", $job_query, $shortcode_args);
            
            
            while ($job_query->have_posts()): $job_query->the_post();
                //Backward Compatibility
                if ($shortcode_args['layout']) {
                    // Display the user defined job listing view
                    if ('grid' === $shortcode_args['layout']) {
                        get_simple_job_board_template('content-job-listing-grid-view.php');
                    } elseif ('list' === $shortcode_args['layout']) {
                        get_simple_job_board_template('content-job-listing-list-view.php');
                    }
                } elseif (FALSE !== get_option('job_board_listing_view')) {
                    $job_board_listing_view = get_option('job_board_listing_view');
                    // Display the user defined job listing view
                    if ('grid-view' === $job_board_listing_view) {
                        get_simple_job_board_template('content-job-listing-grid-view.php');
                    } elseif ('list-view' === $job_board_listing_view) {
                        get_simple_job_board_template('content-job-listing-list-view.php');
                    }
                } else {
                    get_simple_job_board_template('content-job-listing-list-view.php');
                }
            endwhile;

            if ( isset( $shortcode_args['show_pagination'] ) && strtolower( $shortcode_args['show_pagination'] ) != 'no'  ) {
                /**
                 * Template -> Pagination:
                 * 
                 * - Add Pagination to Resulted Jobs.
                 */
                get_simple_job_board_template('listing/job-pagination.php', array('job_query' => $job_query));
            }

        } else {

            /**
             * Template -> No Job Found:
             * 
             * - Display Message on No Job Found.
             */
            get_simple_job_board_template_part('listing/content-no-jobs-found');
        }

        wp_reset_postdata();

        /**
         * Fires after listing jobs on job listing page.
         * 
         * @since   2.2.0
         */
        do_action('sjb_job_listing_after');

        /**
         * Template -> Job Listing End:
         * 
         * - SJB Ending of Job List.
         */
        get_simple_job_board_template('listing/job-listings-end.php');

        do_action('add_faq_section_below_job_listing', $job_query, $shortcode_args, $args);

        /**
         * Template -> Job Listing Wrapper End:
         * 
         * - SJB Endding of Job Listing Wrapper
         */
        get_simple_job_board_template('listing/listing-wrapper-end.php');

        $html = ob_get_clean();

        /**
         * Filter -> Modify the Job Listing Shortcode
         * 
         * @since   2.2.0
         * 
         * @param   HTML    $html    Job Listing HTML Structure.
         */
        return apply_filters('sjb_job_listing_shortcode', $html . do_shortcode($content), $atts);
    }


}
