<?php
/**
 * Plugin Name: Simple Booking System
 * Description: A user-friendly WordPress plugin for handling appointment bookings.
 * Version: 1.0
 * Author: Sachinraj CP
 * Text Domain: simple-booking-system
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.6
 * Requires PHP: 7.4
 * Tested up to: 6.8
 */

if (!defined('ABSPATH')) exit;

define('SIMPLE_BOOKING_SYSTEM_VERSION', '1.0');
define('SIMPLE_BOOKING_SYSTEM_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('SIMPLE_BOOKING_SYSTEM_PLUGIN_URL', plugin_dir_url(__FILE__));

// Enqueue Frontend Scripts and Styles
function simple_booking_system_enqueue_scripts() {
    wp_enqueue_script('jquery');
    
    // Bootstrap CSS
    wp_enqueue_style(
        'simple-booking-system-bootstrap-css', 
        SIMPLE_BOOKING_SYSTEM_PLUGIN_URL . 'assets/css/bootstrap.min.css',
        array(),
        '5.3.0'
    );
    
    // Plugin CSS
    wp_enqueue_style(
        'simple-booking-system-style', 
        SIMPLE_BOOKING_SYSTEM_PLUGIN_URL . 'assets/css/style.css',
        array(),
        SIMPLE_BOOKING_SYSTEM_VERSION
    );
    
    // Bootstrap JS
    wp_enqueue_script(
        'simple-booking-system-bootstrap-js', 
        SIMPLE_BOOKING_SYSTEM_PLUGIN_URL . 'assets/js/bootstrap.bundle.min.js', 
        array('jquery'), 
        '5.3.0', 
        true
    );
    
    // Plugin JS
    wp_enqueue_script(
        'simple-booking-system-script', 
        SIMPLE_BOOKING_SYSTEM_PLUGIN_URL . 'assets/js/script.js', 
        array('jquery'), 
        SIMPLE_BOOKING_SYSTEM_VERSION, 
        true
    );
    
    // Localize script data
    wp_localize_script('simple-booking-system-script', 'simple_booking_system_ajax', array(
        'ajax_url' => admin_url('admin-ajax.php'),
        'nonce' => wp_create_nonce('simple_booking_system_nonce'),
        'success_message' => __('Your appointment has been booked successfully!', 'simple-booking-system'),
        'weekend_message' => __('Please select a weekday (Monday to Friday).', 'simple-booking-system')
    ));
}
add_action('wp_enqueue_scripts', 'simple_booking_system_enqueue_scripts');

// Enqueue Admin Scripts and Styles
function simple_booking_system_admin_scripts($hook) {
    if (strpos($hook, 'simple-booking-system') === false) {
        return;
    }
    
    // Admin CSS
    wp_enqueue_style(
        'simple-booking-system-admin-css',
        SIMPLE_BOOKING_SYSTEM_PLUGIN_URL . 'assets/css/style.css',
        array(),
        SIMPLE_BOOKING_SYSTEM_VERSION
    );
    
    // Admin JS
    wp_enqueue_script(
        'simple-booking-system-admin-js', 
        SIMPLE_BOOKING_SYSTEM_PLUGIN_URL . 'assets/js/admin.js', 
        array('jquery'), 
        SIMPLE_BOOKING_SYSTEM_VERSION, 
        true
    );
}
add_action('admin_enqueue_scripts', 'simple_booking_system_admin_scripts');

// Plugin Activation Hook
function simple_booking_system_activate_plugin() {
    add_option('simple_booking_system_emails', 'admin@example.com');
    add_option('simple_booking_system_time_start', '08:00');
    add_option('simple_booking_system_time_end', '17:00');
    add_option('simple_booking_system_time_interval', 30);
    add_option('simple_booking_system_custom_fields', array());
    add_option('simple_booking_system_booked_slots', array());
}
register_activation_hook(__FILE__, 'simple_booking_system_activate_plugin');

// Admin Menu
function simple_booking_system_admin_menu() {
    add_menu_page(
        __('Booking Settings', 'simple-booking-system'),
        __('Booking System', 'simple-booking-system'),
        'manage_options',
        'simple-booking-system-settings',
        'simple_booking_system_settings_page',
        'dashicons-calendar-alt'
    );
    
    add_submenu_page(
        'simple-booking-system-settings',
        __('Booking Settings', 'simple-booking-system'),
        __('Settings', 'simple-booking-system'),
        'manage_options',
        'simple-booking-system-settings',
        'simple_booking_system_settings_page'
    );
    
    add_submenu_page(
        'simple-booking-system-settings',
        __('Custom Fields', 'simple-booking-system'),
        __('Custom Fields', 'simple-booking-system'),
        'manage_options',
        'simple-booking-system-custom-fields',
        'simple_booking_system_custom_fields_page'
    );
    
    add_submenu_page(
        'simple-booking-system-settings',
        __('Bookings', 'simple-booking-system'),
        __('Bookings', 'simple-booking-system'),
        'manage_options',
        'simple-booking-system-bookings',
        'simple_booking_system_bookings_page'
    );
}
add_action('admin_menu', 'simple_booking_system_admin_menu');

// Settings Page
function simple_booking_system_settings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'simple-booking-system'));
    }

    if (isset($_POST['simple_booking_system_settings_save']) && check_admin_referer('simple_booking_system_settings_nonce')) {
        update_option('simple_booking_system_emails', sanitize_text_field($_POST['simple_booking_system_emails']));
        update_option('simple_booking_system_time_start', sanitize_text_field($_POST['simple_booking_system_time_start']));
        update_option('simple_booking_system_time_end', sanitize_text_field($_POST['simple_booking_system_time_end']));
        update_option('simple_booking_system_time_interval', intval($_POST['simple_booking_system_time_interval']));
        
        echo '<div class="notice notice-success"><p>' . esc_html__('Settings saved.', 'simple-booking-system') . '</p></div>';
    }
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Booking System Settings', 'simple-booking-system'); ?></h1>
        
        <div class="card mb-4">
            <div class="card-header">
                <h2><?php esc_html_e('How to Use the Booking System', 'simple-booking-system'); ?></h2>
            </div>
            <div class="card-body">
                <p><?php esc_html_e('To display the booking form on any page or post, simply use the following shortcode:', 'simple-booking-system'); ?></p>
                <div class="alert alert-info" style="font-family: monospace; font-size: 16px;">
                    [simple_booking_system]
                </div>
                <p><?php esc_html_e('You can add this shortcode to:', 'simple-booking-system'); ?></p>
                <ul>
                    <li><?php esc_html_e('Page content using the WordPress editor', 'simple-booking-system'); ?></li>
                    <li><?php esc_html_e('Widget areas using the Shortcode widget', 'simple-booking-system'); ?></li>
                    <li><?php esc_html_e('Template files using', 'simple-booking-system'); ?> <code>do_shortcode('[simple_booking_system]')</code></li>
                </ul>
            </div>
        </div>
        
        <form method="post">
            <?php wp_nonce_field('simple_booking_system_settings_nonce'); ?>
            <div class="card">
                <div class="card-header">
                    <h2><?php esc_html_e('System Configuration', 'simple-booking-system'); ?></h2>
                </div>
                <div class="card-body">
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e('Notification Emails', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="text" name="simple_booking_system_emails" value="<?php echo esc_attr(get_option('simple_booking_system_emails')); ?>" class="regular-text" placeholder="<?php esc_attr_e('comma separated emails', 'simple-booking-system'); ?>" />
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Start Time', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="time" name="simple_booking_system_time_start" value="<?php echo esc_attr(get_option('simple_booking_system_time_start')); ?>" />
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('End Time', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="time" name="simple_booking_system_time_end" value="<?php echo esc_attr(get_option('simple_booking_system_time_end')); ?>" />
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Interval (minutes)', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="number" name="simple_booking_system_time_interval" value="<?php echo esc_attr(get_option('simple_booking_system_time_interval')); ?>" min="5" max="60" />
                            </td>
                        </tr>
                    </table>
                </div>
                <div class="card-footer text-end">
                    <input type="submit" name="simple_booking_system_settings_save" value="<?php esc_attr_e('Save Settings', 'simple-booking-system'); ?>" class="button button-primary" />
                </div>
            </div>
        </form>
    </div>
    <?php
}

// Custom Fields Page
function simple_booking_system_custom_fields_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'simple-booking-system'));
    }

    $custom_fields = get_option('simple_booking_system_custom_fields', array());
    
    if (isset($_POST['simple_booking_system_add_field']) && check_admin_referer('simple_booking_system_add_field_nonce')) {
        $new_field = array(
            'name' => sanitize_text_field($_POST['field_name']),
            'label' => sanitize_text_field($_POST['field_label']),
            'type' => sanitize_text_field($_POST['field_type']),
            'required' => isset($_POST['field_required']) ? 1 : 0,
            'options' => isset($_POST['field_options']) ? sanitize_text_field($_POST['field_options']) : ''
        );
        $custom_fields[] = $new_field;
        update_option('simple_booking_system_custom_fields', $custom_fields);
        echo '<div class="notice notice-success"><p>' . esc_html__('Field added successfully.', 'simple-booking-system') . '</p></div>';
    }
    
    if (isset($_GET['delete_field']) && check_admin_referer('simple_booking_system_delete_field_nonce')) {
        $index = intval($_GET['delete_field']);
        if (isset($custom_fields[$index])) {
            unset($custom_fields[$index]);
            update_option('simple_booking_system_custom_fields', array_values($custom_fields));
            echo '<div class="notice notice-success"><p>' . esc_html__('Field deleted successfully.', 'simple-booking-system') . '</p></div>';
        }
    }
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Custom Form Fields', 'simple-booking-system'); ?></h1>
        
        <div class="card">
            <div class="card-header">
                <h2><?php esc_html_e('Add New Field', 'simple-booking-system'); ?></h2>
            </div>
            <div class="card-body">
                <form method="post">
                    <?php wp_nonce_field('simple_booking_system_add_field_nonce'); ?>
                    <table class="form-table">
                        <tr>
                            <th><?php esc_html_e('Field Name', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="text" name="field_name" class="regular-text" required placeholder="<?php esc_attr_e('e.g. phone_number', 'simple-booking-system'); ?>" />
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Field Label', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="text" name="field_label" class="regular-text" required placeholder="<?php esc_attr_e('e.g. Phone Number', 'simple-booking-system'); ?>" />
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Field Type', 'simple-booking-system'); ?></th>
                            <td>
                                <select name="field_type" class="regular-text" id="field-type-selector">
                                    <option value="text"><?php esc_html_e('Text', 'simple-booking-system'); ?></option>
                                    <option value="email"><?php esc_html_e('Email', 'simple-booking-system'); ?></option>
                                    <option value="tel"><?php esc_html_e('Phone', 'simple-booking-system'); ?></option>
                                    <option value="select"><?php esc_html_e('Dropdown', 'simple-booking-system'); ?></option>
                                    <option value="textarea"><?php esc_html_e('Text Area', 'simple-booking-system'); ?></option>
                                    <option value="checkbox"><?php esc_html_e('Checkbox', 'simple-booking-system'); ?></option>
                                </select>
                            </td>
                        </tr>
                        <tr id="options-row" style="display:none;">
                            <th><?php esc_html_e('Dropdown Options', 'simple-booking-system'); ?></th>
                            <td>
                                <input type="text" name="field_options" class="regular-text" placeholder="<?php esc_attr_e('Comma separated options (e.g. Option 1,Option 2)', 'simple-booking-system'); ?>" />
                                <p class="description"><?php esc_html_e('Enter options separated by commas', 'simple-booking-system'); ?></p>
                            </td>
                        </tr>
                        <tr>
                            <th><?php esc_html_e('Required', 'simple-booking-system'); ?></th>
                            <td><input type="checkbox" name="field_required" value="1" /></td>
                        </tr>
                    </table>
                    <input type="submit" name="simple_booking_system_add_field" value="<?php esc_attr_e('Add Field', 'simple-booking-system'); ?>" class="button button-primary" />
                </form>
            </div>
        </div>
        
        <div class="card mt-4">
            <div class="card-header">
                <h2><?php esc_html_e('Existing Fields', 'simple-booking-system'); ?></h2>
            </div>
            <div class="card-body">
                <?php if (empty($custom_fields)): ?>
                    <p><?php esc_html_e('No custom fields added yet.', 'simple-booking-system'); ?></p>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Name', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Label', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Type', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Required', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Actions', 'simple-booking-system'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($custom_fields as $index => $field): ?>
                            <tr>
                                <td><?php echo esc_html($field['name']); ?></td>
                                <td><?php echo esc_html($field['label']); ?></td>
                                <td><?php echo esc_html($field['type']); ?></td>
                                <td><?php echo $field['required'] ? esc_html__('Yes', 'simple-booking-system') : esc_html__('No', 'simple-booking-system'); ?></td>
                                <td>
                                    <a href="<?php echo esc_url(wp_nonce_url(add_query_arg('delete_field', $index), 'simple_booking_system_delete_field_nonce')); ?>" class="button button-secondary" onclick="return confirm('<?php esc_attr_e('Are you sure?', 'simple-booking-system'); ?>')">
                                        <?php esc_html_e('Delete', 'simple-booking-system'); ?>
                                    </a>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php
}

// Bookings Page
function simple_booking_system_bookings_page() {
    if (!current_user_can('manage_options')) {
        wp_die(__('You do not have sufficient permissions to access this page.', 'simple-booking-system'));
    }

    $booked_slots = get_option('simple_booking_system_booked_slots', array());
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Bookings', 'simple-booking-system'); ?></h1>
        
        <div class="card">
            <div class="card-body">
                <?php if (empty($booked_slots)): ?>
                    <p><?php esc_html_e('No bookings yet.', 'simple-booking-system'); ?></p>
                <?php else: ?>
                    <table class="wp-list-table widefat fixed striped">
                        <thead>
                            <tr>
                                <th><?php esc_html_e('Date', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Time', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Name', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Email', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Phone', 'simple-booking-system'); ?></th>
                                <th><?php esc_html_e('Actions', 'simple-booking-system'); ?></th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($booked_slots as $date => $times): ?>
                                <?php foreach ($times as $time => $booking): ?>
                                <tr>
                                    <td><?php echo esc_html(date_i18n(get_option('date_format'), strtotime($date))); ?></td>
                                    <td><?php echo esc_html(date_i18n(get_option('time_format'), strtotime($time))); ?></td>
                                    <td><?php echo esc_html($booking['name'] ?? __('N/A', 'simple-booking-system')); ?></td>
                                    <td><?php echo esc_html($booking['email'] ?? __('N/A', 'simple-booking-system')); ?></td>
                                    <td><?php echo esc_html($booking['simple_booking_system_phone'] ?? __('N/A', 'simple-booking-system')); ?></td>
                                    <td>
                                        <a href="#" class="button button-secondary"><?php esc_html_e('View Details', 'simple-booking-system'); ?></a>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>
    <?php
}

// Booking Form Shortcode
function simple_booking_system_booking_form() {
    $custom_fields = get_option('simple_booking_system_custom_fields', array());
    ob_start(); ?>
    <div class="container mt-4">
        <form id="simple-booking-system-booking-form" class="row g-3">
            <input type="hidden" name="simple_booking_system_nonce" value="<?php echo esc_attr(wp_create_nonce('simple_booking_system_booking_nonce')); ?>">
            
            <div class="col-md-6">
                <label class="form-label"><?php esc_html_e('First Name:', 'simple-booking-system'); ?></label>
                <input type="text" class="form-control" name="simple_booking_system_first_name" required>
            </div>
            <div class="col-md-6">
                <label class="form-label"><?php esc_html_e('Email:', 'simple-booking-system'); ?></label>
                <input type="email" class="form-control" name="simple_booking_system_email" required>
            </div>
            <div class="col-md-6">
                <label class="form-label"><?php esc_html_e('Phone Number:', 'simple-booking-system'); ?></label>
                <input type="tel" class="form-control" name="simple_booking_system_phone" required>
            </div>
            
            <?php foreach ($custom_fields as $field): ?>
                <div class="col-md-6">
                    <label class="form-label"><?php echo esc_html($field['label']); ?>:</label>
                    <?php if ($field['type'] === 'textarea'): ?>
                        <textarea class="form-control" name="<?php echo esc_attr($field['name']); ?>" <?php echo $field['required'] ? 'required' : ''; ?>></textarea>
                    <?php elseif ($field['type'] === 'select'): ?>
                        <select class="form-select" name="<?php echo esc_attr($field['name']); ?>" <?php echo $field['required'] ? 'required' : ''; ?>>
                            <option value=""><?php esc_html_e('Select an option', 'simple-booking-system'); ?></option>
                            <?php if (!empty($field['options'])): ?>
                                <?php $options = explode(',', $field['options']); ?>
                                <?php foreach ($options as $option): ?>
                                    <option value="<?php echo esc_attr(trim($option)); ?>"><?php echo esc_html(trim($option)); ?></option>
                                <?php endforeach; ?>
                            <?php endif; ?>
                        </select>
                    <?php elseif ($field['type'] === 'checkbox'): ?>
                        <div class="form-check">
                            <input type="checkbox" class="form-check-input" name="<?php echo esc_attr($field['name']); ?>" value="1" <?php echo $field['required'] ? 'required' : ''; ?>>
                            <label class="form-check-label"><?php echo esc_html($field['label']); ?></label>
                        </div>
                    <?php else: ?>
                        <input type="<?php echo esc_attr($field['type']); ?>" class="form-control" name="<?php echo esc_attr($field['name']); ?>" <?php echo $field['required'] ? 'required' : ''; ?>>
                    <?php endif; ?>
                </div>
            <?php endforeach; ?>
            
            <div class="col-md-6">
                <label class="form-label"><?php esc_html_e('Requested Date:', 'simple-booking-system'); ?></label>
                <input type="date" class="form-control" name="simple_booking_system_date" required id="simple-booking-system-date">
            </div>
            <div class="col-md-6">
                <label class="form-label"><?php esc_html_e('Requested Time:', 'simple-booking-system'); ?></label>
                <select class="form-select" name="simple_booking_system_time" required id="simple-booking-system-time"></select>
            </div>
            
            <div class="col-12 text-center">
                <button type="submit" class="btn btn-primary"><?php esc_html_e('Submit Appointment', 'simple-booking-system'); ?></button>
            </div>
        </form>
        <div id="simple-booking-system-response" class="mt-3" style="display: none;"></div>
    </div>
    <?php return ob_get_clean();
}
add_shortcode('simple_booking_system', 'simple_booking_system_booking_form');

// Get Available Time Slots
function simple_booking_system_get_available_times() {
    check_ajax_referer('simple_booking_system_nonce', 'nonce');
    
    if (!isset($_POST['date'])) {
        wp_send_json_error(array('message' => __('Invalid request.', 'simple-booking-system')));
    }

    $date = sanitize_text_field($_POST['date']);
    $day = date('w', strtotime($date));
    if ($day == 0 || $day == 6) {
        echo '<option value="">' . esc_html__('Closed on weekends', 'simple-booking-system') . '</option>';
        wp_die();
    }

    $booked_slots = get_option('simple_booking_system_booked_slots', []);
    $start_time = strtotime(get_option('simple_booking_system_time_start', '08:00'));
    $end_time = strtotime(get_option('simple_booking_system_time_end', '17:00'));
    $interval = get_option('simple_booking_system_time_interval', 30) * 60;
    $options = '';

    while ($start_time < $end_time) {
        $time_value = date('H:i', $start_time);
        $time_display = date_i18n(get_option('time_format'), $start_time);
        if (empty($booked_slots[$date]) || !array_key_exists($time_value, $booked_slots[$date])) {
            $options .= '<option value="' . esc_attr($time_value) . '">' . esc_html($time_display) . '</option>';
        }
        $start_time += $interval;
    }

    if (empty($options)) {
        $options = '<option value="">' . esc_html__('No slots available', 'simple-booking-system') . '</option>';
    }

    echo $options;
    wp_die();
}
add_action('wp_ajax_simple_booking_system_get_available_times', 'simple_booking_system_get_available_times');
add_action('wp_ajax_nopriv_simple_booking_system_get_available_times', 'simple_booking_system_get_available_times');

// Handle Form Submission
function simple_booking_system_handle_booking() {
    check_ajax_referer('simple_booking_system_booking_nonce', 'simple_booking_system_nonce');
    
    if (empty($_POST['simple_booking_system_date']) || 
        empty($_POST['simple_booking_system_time']) || 
        empty($_POST['simple_booking_system_first_name']) || 
        empty($_POST['simple_booking_system_email']) || 
        empty($_POST['simple_booking_system_phone'])) {
        wp_send_json_error(array('message' => __('All required fields must be filled.', 'simple-booking-system')));
    }

    $date = sanitize_text_field($_POST['simple_booking_system_date']);
    $time = sanitize_text_field($_POST['simple_booking_system_time']);
    $slots = get_option('simple_booking_system_booked_slots', []);

    if (!empty($slots[$date]) && array_key_exists($time, $slots[$date])) {
        wp_send_json_error(array('message' => __('This time slot is already booked.', 'simple-booking-system')));
    }

    // Store all form data
    $booking_data = array(
        'name' => sanitize_text_field($_POST['simple_booking_system_first_name']),
        'email' => sanitize_email($_POST['simple_booking_system_email']),
        'simple_booking_system_phone' => sanitize_text_field($_POST['simple_booking_system_phone']),
        'date' => $date,
        'time' => $time
    );

    // Add custom fields data
    $custom_fields = get_option('simple_booking_system_custom_fields', array());
    foreach ($custom_fields as $field) {
        $field_name = $field['name'];
        if (isset($_POST[$field_name])) {
            $booking_data[$field_name] = is_array($_POST[$field_name]) ? 
                array_map('sanitize_text_field', $_POST[$field_name]) : 
                sanitize_text_field($_POST[$field_name]);
        }
    }

    $slots[$date][$time] = $booking_data;
    update_option('simple_booking_system_booked_slots', $slots);

    $message = "<h3>" . __('New Appointment Request', 'simple-booking-system') . "</h3>";
    $message .= "<p><strong>" . __('Name:', 'simple-booking-system') . "</strong> " . esc_html($booking_data['name']) . "</p>";
    $message .= "<p><strong>" . __('Email:', 'simple-booking-system') . "</strong> " . esc_html($booking_data['email']) . "</p>";
    $message .= "<p><strong>" . __('Phone:', 'simple-booking-system') . "</strong> " . esc_html($booking_data['simple_booking_system_phone']) . "</p>";
    $message .= "<p><strong>" . __('Date:', 'simple-booking-system') . "</strong> " . esc_html(date_i18n(get_option('date_format'), strtotime($date))) . "</p>";
    $message .= "<p><strong>" . __('Time:', 'simple-booking-system') . "</strong> " . esc_html(date_i18n(get_option('time_format'), strtotime($time))) . "</p>";
    
    // Add custom fields to email
    foreach ($custom_fields as $field) {
        $field_name = $field['name'];
        if (isset($booking_data[$field_name])) {
            $value = is_array($booking_data[$field_name]) ? 
                implode(', ', $booking_data[$field_name]) : 
                $booking_data[$field_name];
            $message .= "<p><strong>" . esc_html($field['label']) . ":</strong> " . esc_html($value) . "</p>";
        }
    }

    $headers = array('Content-Type: text/html; charset=UTF-8');
    $emails = explode(',', get_option('simple_booking_system_emails'));
    $subject = __('New Booking:', 'simple-booking-system') . ' ' . esc_html($booking_data['name']) . ' - ' . esc_html(date_i18n(get_option('date_format'), strtotime($date))) . ' ' . __('at', 'simple-booking-system') . ' ' . esc_html(date_i18n(get_option('time_format'), strtotime($time)));
    $success = wp_mail($emails, $subject, $message, $headers);

    if ($success) {
        wp_send_json_success(array('message' => __('Your appointment has been booked successfully!', 'simple-booking-system')));
    } else {
        wp_send_json_error(array('message' => __('Booking was saved but email notification failed.', 'simple-booking-system')));
    }
}
add_action('wp_ajax_simple_booking_system_handle_booking', 'simple_booking_system_handle_booking');
add_action('wp_ajax_nopriv_simple_booking_system_handle_booking', 'simple_booking_system_handle_booking');