<?php
/*
 * Plugin Name: Simple Bike Rental
 * Plugin URI: https://www.simplebikerental.com
 * Description: A lightweight plugin to manage bike rentals with daily and hourly booking options.
 * Version: 1.0.7
 * Author: Rodolfo Rizzo
 * Author URI: https://www.rodolforizzo.it
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: simple-bike-rental
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) {
    exit;
}

define('SIMPBIRE_VERSION', '1.0.7');
define('SIMPBIRE_PLUGIN_PATH', plugin_dir_path(__FILE__));

// NOTE FOR REVIEWERS:
// This is the main bootstrap file. All functionality is split across includes/ (admin, AJAX, shortcodes).
// Nonces are used for all AJAX and POST operations (see ajax.php and admin.php).

// === Core file inclusion ===
require_once SIMPBIRE_PLUGIN_PATH . 'includes/database.php';
require_once SIMPBIRE_PLUGIN_PATH . 'includes/admin.php';
require_once SIMPBIRE_PLUGIN_PATH . 'includes/functions.php';
require_once SIMPBIRE_PLUGIN_PATH . 'includes/ajax.php';
require_once SIMPBIRE_PLUGIN_PATH . 'includes/shortcodes.php';
require_once SIMPBIRE_PLUGIN_PATH . 'includes/notifiche.php';


// === Activation/Deactivation Hook ===
register_activation_hook(__FILE__, 'simpbire_plugin_activate');
register_uninstall_hook(__FILE__, 'simpbire_plugin_uninstall');


// === Backend: style and calendar ===
function simpbire_admin_enqueue_scripts($hook) {
    // This value is only used to conditionally load scripts and styles, no data is being processed or stored.
    // Nonce verification is not required here.
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
    $screen = isset($_GET['page']) ? wp_unslash($_GET['page']) : '';

    // Global CSS always
    wp_enqueue_style('simpbire-style', plugin_dir_url(__FILE__) . 'assets/css/style.css', [], null, 'all');

    // Flatpickr (if required)
    $usa_flatpickr = ['prenotazioni_biciclette', 'aggiungi_prenotazione', 'modifica_prenotazione', 'aggiungi_biciclette', 'esporta_prenotazioni'];
    if (in_array($screen, $usa_flatpickr, true)) {
        wp_enqueue_script('flatpickr', plugin_dir_url(__FILE__) . 'assets/lib/flatpickr/flatpickr.min.js', [], '4.6.13', true);
        wp_enqueue_style('flatpickr-style', plugin_dir_url(__FILE__) . 'assets/lib/flatpickr/flatpickr.min.css', [], '4.6.13');
        simpbire_enqueue_flatpickr_locale();
    }

    if ($screen === 'esporta_prenotazioni') {
        wp_enqueue_script(
            'simpbire-admin-esporta-prenotazioni',
            plugin_dir_url(__FILE__) . 'assets/js/admin-esporta-prenotazioni.js',
            ['flatpickr'],
            SIMPBIRE_VERSION,
            true
        );
    }

    // FullCalendar for reservations
    if ($screen === 'prenotazioni_biciclette') {
        wp_enqueue_script('fullcalendar-core', plugin_dir_url(__FILE__) . 'assets/lib/fullcalendar/core/index.global.min.js', [], '6.1.8', true);
        wp_enqueue_script('fullcalendar-daygrid', plugin_dir_url(__FILE__) . 'assets/lib/fullcalendar/daygrid/index.global.min.js', ['fullcalendar-core'], '6.1.8', true);
        wp_enqueue_script('fullcalendar-timegrid', plugin_dir_url(__FILE__) . 'assets/lib/fullcalendar/timegrid/index.global.min.js', ['fullcalendar-core'], '6.1.8', true);
        wp_enqueue_script('fullcalendar-interaction', plugin_dir_url(__FILE__) . 'assets/lib/fullcalendar/interaction/index.global.min.js', ['fullcalendar-core'], '6.1.8', true);
        wp_enqueue_script('fullcalendar-locales', plugin_dir_url(__FILE__) . 'assets/lib/fullcalendar/core/locales-all.global.min.js', ['fullcalendar-core'], '6.1.8', true);

        wp_enqueue_script('calendar-js', plugin_dir_url(__FILE__) . 'assets/js/calendar.js', [
            'fullcalendar-core',
            'fullcalendar-daygrid',
            'fullcalendar-timegrid',
            'fullcalendar-interaction',
            'fullcalendar-locales',
        ], time(), true);

        $locale = substr(get_locale(), 0, 2);
        wp_localize_script('calendar-js', 'CalendarAjax', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'locale'  => $locale,
            'nonce'   => wp_create_nonce('simpbire_elimina_prenotazione_ajax'),
            'nonce_modifica' => wp_create_nonce('simpbire_modifica_prenotazione'),
            'security' => wp_create_nonce('simpbire_load_bookings_nonce'),
            'i18n'    => [
                'conferma_eliminazione' => __('Are you sure you want to cancel this reservation?', 'simple-bike-rental'),
                'errore_eliminazione'   => __('Unable to cancel the reservation.', 'simple-bike-rental'),
                'errore_generico'       => __('Unexpected error.', 'simple-bike-rental')
            ]
        ]);

        wp_enqueue_script(
            'simpbire-prenotazioni-tabs',
            plugin_dir_url(__FILE__) . 'assets/js/admin-prenotazioni-tabs.js',
            [],
            SIMPBIRE_VERSION,
            true
        );
    }

    // Script for the "Add Bicycle" page
    if ($screen === 'aggiungi_biciclette') {
        wp_enqueue_media();
        wp_enqueue_script(
            'simpbire-admin-aggiungi-bicicletta',
            plugin_dir_url(__FILE__) . 'assets/js/admin-aggiungi-bicicletta.js',
            ['jquery', 'media-upload', 'thickbox'],
            time(),
            true
        );
        wp_localize_script('simpbire-admin-aggiungi-bicicletta', 'simpbireAdminVars', [
            'selectImageTitle'  => __('Choose an image', 'simple-bike-rental'),
            'selectImageButton' => __('Choose Image', 'simple-bike-rental'),
        ]);
    }

    // Script for the "Add Booking" page
    if ($screen === 'aggiungi_prenotazione') {
        wp_enqueue_script(
            'simpbire-admin-aggiungi-prenotazione',
            plugin_dir_url(__FILE__) . 'assets/js/admin-aggiungi-prenotazione.js',
            ['jquery', 'flatpickr'],
            time(),
            true
        );
        
        wp_localize_script('simpbire-admin-aggiungi-prenotazione', 'simpbireAggiungiPrenotazioneVars', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce'   => wp_create_nonce('simpbire_disponibili_nonce'),
            'i18n' => [
                'nessuna_bici' => __('No bikes available', 'simple-bike-rental'),
                'caricamento'  => __('Loading...', 'simple-bike-rental'),
            ]
        ]);
    }

    // Script for the "Edit Booking" page
    if ($screen === 'modifica_prenotazione') {
        wp_enqueue_script(
            'simpbire-admin-modifica-prenotazione',
            plugin_dir_url(__FILE__) . 'assets/js/admin-modifica-prenotazione.js',
            ['jquery', 'flatpickr'],
            time(),
            true
        );
        wp_localize_script('simpbire-admin-modifica-prenotazione', 'simpbireModificaPrenotazioneVars', [
            'ajaxurl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('simpbire_disponibili_nonce'),
            // This ID is only used to retrieve and display reservation data in the admin interface.
            // No write operations are performed, so nonce verification is not required.
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
            'prenotazione_id' => isset($_GET['id']) ? intval(wp_unslash($_GET['id'])) : 0,
            'bicicletta_id' => isset($prenotazione->bicicletta_id) ? $prenotazione->bicicletta_id : '',
            'i18n' => [
                'nessuna_bici' => __('No bikes available', 'simple-bike-rental'),
                'caricamento'  => __('Loading...', 'simple-bike-rental'),
            ]
        ]);
    }

    if ($hook === 'admin_page_modifica_bicicletta') {
        wp_enqueue_media();

        wp_enqueue_script(
            'simpbire-admin-modifica-bicicletta',
            plugin_dir_url(__FILE__) . 'assets/js/admin-modifica-bicicletta.js',
            ['jquery', 'media-editor'],
            SIMPBIRE_VERSION,
            true
        );

        wp_localize_script('simpbire-admin-modifica-bicicletta', 'simpleBikeRental', [
            'titolo'  => esc_html__('Select an image', 'simple-bike-rental'),
            'bottone' => esc_html__('Use this image', 'simple-bike-rental'),
        ]);
    }

    // Other specific scripts
}
add_action('admin_enqueue_scripts', 'simpbire_admin_enqueue_scripts');


function simpbire_enqueue_flatpickr_locale() {
    $locale = determine_locale();
    $flatpickr_lang = substr($locale, 0, 2);

    if ($flatpickr_lang === 'en') return;

    wp_enqueue_script(
        'flatpickr-locale',
        plugin_dir_url(__FILE__) . "assets/lib/flatpickr/l10n/{$flatpickr_lang}.js",
        ['flatpickr'],
        '4.6.13',
        true
    );

    $inline = "if (window.flatpickr && window.flatpickr.l10ns && window.flatpickr.l10ns.{$flatpickr_lang}) {
        flatpickr.localize(flatpickr.l10ns.{$flatpickr_lang});
    }";
    wp_add_inline_script('flatpickr-locale', $inline, 'after');
}

add_action('admin_post_simpbire_aggiungi_bicicletta_admin', 'simpbire_aggiungi_bicicletta_admin');
add_action( 'admin_post_simpbire_modifica_bicicletta_admin', 'simpbire_modifica_bicicletta_admin' );
add_action('admin_post_simpbire_elimina_bicicletta_admin', 'simpbire_elimina_bicicletta_admin');
add_action('admin_post_simpbire_aggiungi_prenotazione_admin', 'simpbire_aggiungi_prenotazione_admin');
add_action('admin_post_simpbire_modifica_prenotazione_admin', 'simpbire_modifica_prenotazione_admin');
add_action('admin_post_simpbire_salva_notifica_admin', 'simpbire_salva_notifica_admin');
add_action('simpbire_pulizia_prenotazioni_scadute', 'simpbire_elimina_prenotazioni_scadute');

// === Frontend ===
function simpbire_enqueue_frontend_scripts() {
    if (!is_singular()) return;

    wp_enqueue_script('flatpickr', plugin_dir_url(__FILE__) . 'assets/lib/flatpickr/flatpickr.min.js', [], '4.6.13', true);
    wp_enqueue_style('flatpickr-style', plugin_dir_url(__FILE__) . 'assets/lib/flatpickr/flatpickr.min.css', [], '4.6.13');

    simpbire_enqueue_flatpickr_locale();

    wp_enqueue_script('simpbire-frontend', plugin_dir_url(__FILE__) . 'assets/js/frontend.js', [], time(), true);
    wp_enqueue_style('simpbire-frontend-style', plugin_dir_url(__FILE__) . 'assets/css/frontend.css', [], time());

    // NOTE: Nonces passed here (simpbire_frontend_nonce and simpbire_disponibili_nonce)
    // are verified in ajax.php and protect frontend booking actions.
    wp_localize_script('simpbire-frontend', 'SIMPBIREFrontend', [
        'ajaxurl' => admin_url('admin-ajax.php'),
        'nonce'   => wp_create_nonce('simpbire_frontend_nonce'),
        'nonce_disponibili' => wp_create_nonce('simpbire_disponibili_nonce'),
        'i18n' => [
            'caricamento'         => __('Loading...', 'simple-bike-rental'),
            'nessuna_bici'        => __('No bicycles available for the selected interval.', 'simple-bike-rental'),
            'errore_bici'         => __('Error loading bicycles.', 'simple-bike-rental'),
            'prenotazione_ok'     => __('Reservation saved!', 'simple-bike-rental'),
            'prenotazione_errore' => __('Booking error.', 'simple-bike-rental'),
            'attendere'           => __('Please wait...', 'simple-bike-rental'),
        ]
    ]);
}
add_action('wp_enqueue_scripts', 'simpbire_enqueue_frontend_scripts');
