<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Central function to record available notifications
function simpbire_get_notifiche_registrate() {
    return [
        'prenotazione_inviata' => [
            'nome' => __('Booking confirmed (customer)', 'simple-bike-rental'),
            'descrizione' => __('Email sent to the user after booking.', 'simple-bike-rental'),
            'destinatario' => 'user',
            'segnaposto' => ['{name}', '{email}', '{phone}', '{rental_period}', '{bike_name}']
        ],
        'prenotazione_admin' => [
            'nome' => __('New Booking Notification (Admin)', 'simple-bike-rental'),
            'descrizione' => __('Email sent to one or more specified addresses for each new reservation.', 'simple-bike-rental'),
            'destinatario' => 'custom',
            'segnaposto' => ['{name}', '{email}', '{phone}', '{rental_period}', '{bike_name}']
        ]
    ];
}


// Callback named to set the HTML content-type in emails.
if (!function_exists('simpbire_set_html_mail_content_type')) {
    function simpbire_set_html_mail_content_type() {
        return 'text/html';
    }
}

// NOTE FOR REVIEWERS: This function sends emails to the user or custom recipients.
// All email addresses are sanitized and validated with `is_email`.
// Email body is escaped with `wp_kses_post()` and placeholders are replaced safely.
function simpbire_invia_notifica($slug, $dati = []) {
    $notifiche = simpbire_get_notifiche_registrate();

    // Verify that the "slug" exists in the logged notifications
    if (!isset($notifiche[$slug])) {
        return false;
    }

    $notifica = $notifiche[$slug];

    // Check if the notification is active from options
    $attiva = get_option("simpbire_notifica_attiva_$slug", '');
    if (!$attiva) {
        return false;
    }

    // Load object and body from settings
    $oggetto = get_option("simpbire_notifica_oggetto_$slug", '');
    $corpo = get_option("simpbire_notifica_contenuto_$slug", '');

    // Replacing placeholder {key} with the values ​​passed in $data
    foreach ($dati as $chiave => $valore) {
        $corpo = str_replace('{' . $chiave . '}', $valore, $corpo);
        $oggetto = str_replace('{' . $chiave . '}', $valore, $oggetto);
    }

    // Determine recipients based on type
    if ($notifica['destinatario'] === 'user') {
        $destinatari = isset($dati['email']) ? [sanitize_email($dati['email'])] : [];
    } elseif ($notifica['destinatario'] === 'custom') {
        $lista = get_option("simpbire_notifica_destinatari_$slug", '');
        $raw = explode(',', $lista);
        $destinatari = [];

        foreach ($raw as $email) {
            $email = trim($email);
            if (is_email($email)) {
                $destinatari[] = $email;
            }
        }
    } else {
        return false;
    }

    if (empty($destinatari)) {
        return false;
    }

    // temporarily set the content-type to 'text/html'
    add_filter('wp_mail_content_type', 'simpbire_set_html_mail_content_type');

    // Normalize the body:
    // - if there are no HTML tags, we convert the \n to <br>
    // - otherwise, we use wpautop to honor line breaks
    if (wp_strip_all_tags($corpo) === $corpo) {
        $corpo = nl2br($corpo);
    } else {
        $corpo = wpautop($corpo);
    }

    // The object must not contain HTML tags
    $oggetto = wp_strip_all_tags($oggetto);

    // Send to all recipients
    $success = true;
    foreach ($destinatari as $to) {
        if (!wp_mail($to, $oggetto, $corpo)) {
            $success = false;
        }
    }

    // Reset the default content type by removing the filter
    remove_filter('wp_mail_content_type', 'simpbire_set_html_mail_content_type');

    return $success;
}


// NOTE FOR REVIEWERS: This admin form handler is protected by `current_user_can('manage_options')` and `wp_verify_nonce()`.
// All inputs are sanitized before being saved with `update_option()`.
function simpbire_salva_notifica_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You don\'t have permission to edit this notification.', 'simple-bike-rental'));
    }

    if (
        !isset($_POST['simpbire_salva_notifica_nonce_field']) ||
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['simpbire_salva_notifica_nonce_field'])), 'simpbire_salva_notifica_nonce')
    ) {
        wp_die(esc_html__('Security error: invalid nonce.', 'simple-bike-rental'));
    }

    if (!isset($_POST['notifica_slug'])) {
        wp_die(esc_html__('Missing notification slug.', 'simple-bike-rental'));
    }

    $slug = sanitize_key($_POST['notifica_slug']);
    $notifiche = simpbire_get_notifiche_registrate();

    if (!isset($notifiche[$slug])) {
        wp_die(esc_html__('Notification not found.', 'simple-bike-rental'));
    }

    $attiva_key = "simpbire_notifica_attiva_$slug";
    $oggetto_key = "simpbire_notifica_oggetto_$slug";
    $contenuto_key = "simpbire_notifica_contenuto_$slug";
    $destinatari_key = "simpbire_notifica_destinatari_$slug";

    if ($notifiche[$slug]['destinatario'] === 'custom') {
        // Sanitize and validate every email
        $raw_emails = explode(',', wp_unslash($_POST['destinatari_email'] ?? '')); // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Emails are sanitized below
        $pulite = [];

        foreach ($raw_emails as $email) {
            $email = sanitize_email(trim($email));
            if (is_email($email)) {
                $pulite[] = $email;
            }
        }

        update_option($destinatari_key, implode(', ', $pulite));
    }

    update_option($attiva_key, isset($_POST['attiva_notifica']) ? 1 : 0);
    update_option($oggetto_key, sanitize_text_field(wp_unslash($_POST['oggetto_email'] ?? '')));
    update_option($contenuto_key, wp_kses_post(wp_unslash($_POST['corpo_email'] ?? '')));

    $redirect_url = add_query_arg(
        array(
            'page'       => 'noleggio_notifiche',
            'modificato' => 1,
        ),
        admin_url( 'admin.php' )
    );

    wp_safe_redirect( $redirect_url );
    exit;
}
