<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

function simpbire_is_giornaliero($data_inizio, $data_fine) {
    return (substr($data_inizio, 11) === '00:00:00' && substr($data_fine, 11) === '23:59:59');
}

function simpbire_get_currency_symbol() {
    $currency = get_option('simpbire_valuta_pagamento', 'eur');

    $currency_symbols = [
        'eur' => '€',
        'usd' => '$',
        'gbp' => '£',
        'cad' => '$',
        'aud' => '$',
        'chf' => 'CHF',
        'sek' => 'kr',
        'dkk' => 'kr',
        'nok' => 'kr',
        'pln' => 'zł',
        'czk' => 'Kč',
    ];

    $legacy_symbol = get_option('simpbire_currency_symbol', '');
    $symbol = $legacy_symbol ? $legacy_symbol : ($currency_symbols[$currency] ?? '€');

    /**
     * Filter the symbol used to display the currency on the frontend.
     *
     * @param string $symbol   The symbol derived from the selected currency or legacy option.
     * @param string $currency The selected currency code (e.g., eur, usd).
     */
    $symbol = apply_filters('simpbire_currency_symbol', $symbol, $currency);

    return sanitize_text_field($symbol);
}

function simpbire_formatta_periodo($data_inizio, $data_fine) {
    $inizio = strtotime($data_inizio);
    $fine   = strtotime($data_fine);

    if (simpbire_is_giornaliero($data_inizio, $data_fine)) {
        return date_i18n(get_option('date_format'), $inizio) . ' – ' . __('Daily duration', 'simple-bike-rental');
    }

    return date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $inizio)
    . ' → ' .
    date_i18n(get_option('date_format') . ' ' . get_option('time_format'), $fine);
}


function simpbire_formatta_stato($stato) {
    switch ($stato) {
        case 'attiva':
            return __('Active', 'simple-bike-rental');
        case 'in_attesa':
            return __('Pending', 'simple-bike-rental');
        case 'in_attesa_pagamento':
            return __('Waiting for payment', 'simple-bike-rental');
        case 'annullata':
            return __('Cancelled', 'simple-bike-rental');
        default:
            return ucfirst(str_replace('_', ' ', $stato)); // fallback generico
    }
}


function simpbire_get_stato_prenotazione_con_label($stato, $data_fine) {
    $ora_corrente = current_time('mysql');

    if ($stato === 'attiva' && $data_fine < $ora_corrente) {
        return __('Completed', 'simple-bike-rental');
    }

    return simpbire_formatta_stato($stato);
}


function simpbire_get_all_bicycles_cached() {
    $biciclette = get_transient('simpbire_all_bicycles');

    if ($biciclette === false) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query cached via transients. Direct query is required for custom table.
        $biciclette = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}simpbire_bicycles");
        set_transient('simpbire_all_bicycles', $biciclette, 300); // 5 minuti
    }

    return $biciclette;
}


function simpbire_get_all_reservations_cached() {
    $prenotazioni = get_transient('simpbire_all_reservations');

    if ($prenotazioni === false) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query cached via transients. Direct query is required for custom reservations table.
        $prenotazioni = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}simpbire_reservations");
        set_transient('simpbire_all_reservations', $prenotazioni, 300); // 5 minuti
    }

    return $prenotazioni;
}


function simpbire_get_bicicletta_by_id_cached($id) {
    $id = intval($id);
    $cache_key = 'simpbire_bicicletta_' . $id;
    $bicicletta = get_transient($cache_key);

    if ($bicicletta === false) {
        global $wpdb;
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query cached via transients. Secure prepared query for custom bicycle lookup.
        $bicicletta = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$wpdb->prefix}simpbire_bicycles WHERE id = %d", $id
        ));

        if ($bicicletta) {
            set_transient($cache_key, $bicicletta, 300); // cache 5 minuti
        }
    }

    return $bicicletta;
}


function simpbire_verifica_token_prenotazione() {
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (!isset($_GET['simpbire_confirm'])) {
        return;
    }

    // Verify the nonce for protection
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'simpbire_verifica_token')) {
        wp_die(esc_html__('Invalid or expired link.', 'simple-bike-rental'));
    }

    global $wpdb, $simpbire_token_message;

    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Recommended
    $token = sanitize_text_field(wp_unslash($_GET['simpbire_confirm']));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $prenotazione = $wpdb->get_row($wpdb->prepare("
        SELECT * FROM {$wpdb->prefix}simpbire_reservations
        WHERE token = %s
        LIMIT 1
    ", $token));

    if (!$prenotazione || $prenotazione->stato !== 'in_attesa') {
        $simpbire_token_message = '
        <div class="messaggio-prenotazione error" role="alert" aria-live="polite">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                <circle cx="12" cy="12" r="10" stroke-width="2" stroke="currentColor" fill="none"/>
                <path d="M15 9l-6 6M9 9l6 6" stroke-width="2" stroke="currentColor" fill="none" stroke-linecap="round"/>
            </svg>
            <span>' . esc_html__('The link is invalid or the booking has already been confirmed.', 'simple-bike-rental') . '</span>
        </div>';
        return;
    }

    if (strtotime($prenotazione->token_scadenza) < time()) {
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Recommended
        $simpbire_token_message = '
        <div class="messaggio-prenotazione error" role="alert" aria-live="polite">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                <circle cx="12" cy="12" r="10" stroke-width="2" stroke="currentColor" fill="none"/>
                <path d="M15 9l-6 6M9 9l6 6" stroke-width="2" stroke="currentColor" fill="none" stroke-linecap="round"/>
            </svg>
            <span>' . esc_html__('The confirmation link has expired. We invite you to make a new reservation.', 'simple-bike-rental') . '</span>
        </div>';
        return;
    }

    // Check availability
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $conflitti = (int) $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations
        WHERE bicicletta_id = %d
        AND stato = 'attiva'
        AND id != %d
        AND (
            data_inizio < %s
            AND data_fine > %s
        )
    ",
        (int) $prenotazione->bicicletta_id,
        (int) $prenotazione->id,
        $prenotazione->data_fine,
        $prenotazione->data_inizio
    ));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $quantita = (int) $wpdb->get_var($wpdb->prepare("
        SELECT quantita FROM {$wpdb->prefix}simpbire_bicycles WHERE id = %d
    ", (int) $prenotazione->bicicletta_id));

    if ($conflitti >= $quantita) {
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.NonceVerification.Recommended
        $simpbire_token_message = '
        <div class="messaggio-prenotazione error" role="alert" aria-live="polite">
            <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
                <circle cx="12" cy="12" r="10" stroke-width="2" stroke="currentColor" fill="none"/>
                <path d="M15 9l-6 6M9 9l6 6" stroke-width="2" stroke="currentColor" fill="none" stroke-linecap="round"/>
            </svg>
            <span>' . esc_html__('Sorry, availability has changed and it is no longer possible to confirm the reservation.', 'simple-bike-rental') . '</span>
        </div>';
        return;
    }

    // Everything ok → confirm the booking
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->update(
        "{$wpdb->prefix}simpbire_reservations",
        ['stato' => 'attiva', 'token' => null],
        ['id' => (int) $prenotazione->id],
        ['%s', '%s'],
        ['%d']
    );

    $nome_bici = simpbire_get_nome_bicicletta((int) $prenotazione->bicicletta_id);
    $periodo = simpbire_formatta_periodo($prenotazione->data_inizio, $prenotazione->data_fine);

    simpbire_invia_notifica('prenotazione_inviata', [
        'name'          => sanitize_text_field($prenotazione->nome),
        'email'         => sanitize_email($prenotazione->email),
        'phone'         => sanitize_text_field($prenotazione->telefono),
        'rental_period' => $periodo,
        'bike_name'     => sanitize_text_field($nome_bici)
    ]);

    simpbire_invia_notifica('prenotazione_admin', [
        'name'          => sanitize_text_field($prenotazione->nome),
        'email'         => sanitize_email($prenotazione->email),
        'phone'         => sanitize_text_field($prenotazione->telefono),
        'rental_period' => $periodo,
        'bike_name'     => sanitize_text_field($nome_bici)
    ]);

    $simpbire_token_message = '
    <div class="messaggio-prenotazione success" role="alert" aria-live="polite">
        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24">
            <circle cx="12" cy="12" r="10" stroke-width="2" stroke="currentColor" fill="none"/>
            <path d="M16 8l-6.5 7L8 12.5" stroke-width="2" stroke="currentColor" fill="none" stroke-linecap="round" stroke-linejoin="round"/>
        </svg>
        <span>' . esc_html__('Booking confirmed! Thank you for verifying your email.', 'simple-bike-rental') . '</span>
    </div>';
}
add_action('init', 'simpbire_verifica_token_prenotazione');


// Redirect function to the selected page
function simpbire_redirect_to_token_page() {
    $pagina_id = get_option('simpbire_pagina_messaggi_token_id');
    if ($pagina_id) {
        $url_pagina = get_permalink($pagina_id);
        wp_safe_redirect($url_pagina);
        exit;
    }
}
add_action('init', 'simpbire_verifica_token_prenotazione');



function simpbire_get_nome_bicicletta($bicicletta_id) {
    global $wpdb;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    return $wpdb->get_var($wpdb->prepare(
        "SELECT nome FROM {$wpdb->prefix}simpbire_bicycles WHERE id = %d",
        $bicicletta_id
    ));
}


function simpbire_elimina_prenotazioni_scadute() {
    global $wpdb;
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $wpdb->query("
        DELETE FROM {$wpdb->prefix}simpbire_reservations
        WHERE stato = 'in_attesa'
        AND token IS NOT NULL
        AND token_scadenza < NOW()
    ");
}


// Custom email filters
add_filter('wp_mail_from_name', function($name) {
    $custom_name = get_option('simpbire_email_nome_mittente');
    return !empty($custom_name) ? $custom_name : $name;
});

add_filter('wp_mail_from', function($email) {
    $custom_email = get_option('simpbire_email_indirizzo_mittente');
    return (is_email($custom_email)) ? $custom_email : $email;
});
