<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Plugin activation function
// NOTE FOR REVIEWERS: This function safely creates the plugin tables using dbDelta().
// Table names are prefixed and the collation matches the current database settings.
function simpbire_plugin_activate() {
    global $wpdb;

    $table_bicycles = $wpdb->prefix . 'simpbire_bicycles';
    $table_reservations = $wpdb->prefix . 'simpbire_reservations';
    $collation = $wpdb->get_charset_collate();

    // SQL to create the bicycle table
    $sql_bicycles = "CREATE TABLE $table_bicycles (
        id INT NOT NULL AUTO_INCREMENT,
        nome VARCHAR(255) NOT NULL,
        descrizione TEXT,
        tipo VARCHAR(100) NOT NULL,
        quantita INT DEFAULT 1,
        tariffa_giornaliera DECIMAL(10, 2),
        tariffa_oraria DECIMAL(10, 2),
        immagine VARCHAR(255),
        data_inserimento TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id)
        ) $collation;";

    // SQL to create reservations table
    $sql_reservations = "CREATE TABLE $table_reservations (
        id INT NOT NULL AUTO_INCREMENT,
        bicicletta_id INT NOT NULL,
        utente_id INT DEFAULT NULL,
        nome VARCHAR(255) DEFAULT NULL,
        email VARCHAR(255) DEFAULT NULL,
        telefono VARCHAR(50) DEFAULT NULL,
        data_inizio DATETIME NOT NULL,
        data_fine DATETIME NOT NULL,
        stato VARCHAR(50) DEFAULT 'attiva',
        token VARCHAR(100) DEFAULT NULL,
        token_scadenza DATETIME DEFAULT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        FOREIGN KEY (bicicletta_id) REFERENCES $table_bicycles(id) ON DELETE CASCADE
    ) $collation;";

    // Run create-table queries
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql_bicycles);
    dbDelta($sql_reservations);

    if (!wp_next_scheduled('simpbire_pulizia_prenotazioni_scadute')) {
        wp_schedule_event(time(), 'hourly', 'simpbire_pulizia_prenotazioni_scadute');
    }
}


function simpbire_plugin_deactivate() {
    wp_clear_scheduled_hook('simpbire_pulizia_prenotazioni_scadute');
}
register_deactivation_hook(__FILE__, 'simpbire_plugin_deactivate');


// Safe plugin uninstall feature
// NOTE FOR REVIEWERS: This uninstall routine is conditional. It only drops tables if the user explicitly enabled the "Delete tables on uninstall" option.
function simpbire_plugin_uninstall() {
    if (get_option('simpbire_elimina_tabelle_plugin') != 1) {
        return;
    }

    global $wpdb;

    $table_bicycles = $wpdb->prefix . 'simpbire_bicycles';
    $table_reservations = $wpdb->prefix . 'simpbire_reservations';

    // Removing any foreign key (dynamic name)
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Required direct query to information_schema for dynamic foreign key removal.
    $foreign_key = $wpdb->get_var("
    SELECT CONSTRAINT_NAME
    FROM information_schema.KEY_COLUMN_USAGE
    WHERE TABLE_NAME = '{$wpdb->prefix}simpbire_reservations'
    AND TABLE_SCHEMA = DATABASE()
    AND REFERENCED_TABLE_NAME IS NOT NULL
    LIMIT 1
    ");

    if ($foreign_key) {
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Required schema modification to drop dynamic foreign key safely during uninstall.
        $wpdb->query("ALTER TABLE {$wpdb->prefix}simpbire_reservations DROP FOREIGN KEY " . esc_sql($foreign_key));
    }

    // Delete the tables
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Required schema change to safely remove plugin tables during uninstall.
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_reservations));
    
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.SchemaChange -- Required schema change to safely remove plugin tables during uninstall.
    $wpdb->query("DROP TABLE IF EXISTS " . esc_sql($table_bicycles));

    // Removes the control option
    delete_option('simpbire_elimina_tabelle_plugin');
}
// Uninstall Hook Registration
register_uninstall_hook(__FILE__, 'simpbire_plugin_uninstall');
