<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

function simpbire_get_biciclette_disponibili() {
    // NOTE FOR REVIEWERS: This AJAX endpoint uses a nonce ('simpbire_disponibili_nonce') passed via wp_localize_script.
    // It is validated with wp_verify_nonce and no sensitive data is modified here.
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- AJAX request protected by nonce check below.
    if (
        !isset($_GET['security']) ||
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['security'])), 'simpbire_disponibili_nonce')
    ) {
        wp_send_json_error(__('Security check failed', 'simple-bike-rental'));
    }

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Frontend public function that returns non-sensitive data and does not modify anything.
    if (!isset($_GET['inizio']) || !isset($_GET['fine'])) {
        wp_send_json_error(__('Missing parameters', 'simple-bike-rental'));
    }

    global $wpdb;

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Public input used only for a safe read-only query.
    $inizio = sanitize_text_field(wp_unslash($_GET['inizio']));

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Same as above.
    $fine = sanitize_text_field(wp_unslash($_GET['fine']));

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Rental type used only to tailor the response payload.
    $tipo_noleggio = isset($_GET['tipo']) ? sanitize_text_field(wp_unslash($_GET['tipo'])) : '';

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Optional public input used for safe read-only filtering logic.
    $id_corrente = isset($_GET['id_prenotazione']) ? intval($_GET['id_prenotazione']) : 0;

    // Normalizza i valori per sicurezza e per MySQL DATETIME
    $inizio_timestamp = strtotime($inizio);
    $fine_timestamp = strtotime($fine);

    if (empty($inizio) || empty($fine) || $inizio_timestamp === false || $fine_timestamp === false) {
        wp_send_json_error(__('Invalid dates', 'simple-bike-rental'));
    }

    // Convert to standard DATETIME format (Y-m-d H:i:s)
    $inizio = wp_date('Y-m-d H:i:s', $inizio_timestamp);
    $fine = wp_date('Y-m-d H:i:s', $fine_timestamp);

    $intervallo_minuti = intval(get_option('simpbire_minuti_intervallo_prenotazioni', 0));

    // Detect if it is a daily booking (es. end at 23:59:59)
    $ora_fine = substr($fine, 11);
    $usa_intervallo = ($ora_fine !== '23:59:59');

    $durata_ore = max(($fine_timestamp - $inizio_timestamp) / HOUR_IN_SECONDS, 0);
    $simbolo_valuta = simpbire_get_currency_symbol();

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Custom query on live availability, not suitable for caching.
    $biciclette = $wpdb->get_results("SELECT * FROM {$wpdb->prefix}simpbire_bicycles WHERE quantita > 0");
    $disponibili = [];

    foreach ($biciclette as $bici) {
        if ($usa_intervallo && $intervallo_minuti > 0) {
            $inizio_nuova = $inizio;
            $fine_nuova = $fine;

            $buffer_fine = wp_date('Y-m-d H:i:s', strtotime($fine_nuova) + ($intervallo_minuti * 60));

            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Real-time availability check requires fresh data.
            $conflitti = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations
                WHERE bicicletta_id = %d
                AND stato = 'attiva'
                AND id != %d
                AND (
                    data_inizio < %s
                    AND data_inizio < %s
                    AND data_inizio >= %s
                    OR
                    data_inizio < %s
                    AND DATE_ADD(data_fine, INTERVAL %d MINUTE) > %s
                )
            ",
                $bici->id, $id_corrente,
                $buffer_fine, $buffer_fine, $fine_nuova,
                $fine_nuova, $intervallo_minuti, $inizio_nuova
            ));
        } else {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Real-time check for overlapping reservations. Cannot be cached.
            $conflitti = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations
                WHERE bicicletta_id = %d
                AND stato = 'attiva'
                AND id != %d
                AND (
                    data_inizio < %s
                    AND data_fine > %s
                )
            ",
                $bici->id, $id_corrente,
                $fine, $inizio
            ));
        }

        if ($conflitti < $bici->quantita) {
            $img_url = $bici->immagine ? wp_get_attachment_image_url($bici->immagine, 'medium') : '';

            $prezzo = ($tipo_noleggio === 'orario')
                ? floatval($bici->tariffa_oraria) * $durata_ore
                : floatval($bici->tariffa_giornaliera);

            $prezzo = max( $prezzo, 0 );
            $etichetta_prezzo = sanitize_text_field(
                trim( $simbolo_valuta . ' ' . number_format_i18n( $prezzo, 2 ) )
            );

            $disponibili[] = [
                'id' => $bici->id,
                'nome' => sanitize_text_field($bici->nome),
                'immagine' => esc_url($img_url),
                'tipo' => sanitize_text_field($bici->tipo),
                'prezzo' => $prezzo,
                'prezzo_formattato' => $prezzo_formattato,
                'etichetta_prezzo' => $etichetta_prezzo,
            ];
        }
    }

    wp_send_json_success($disponibili);
}
add_action('wp_ajax_simpbire_get_biciclette_disponibili', 'simpbire_get_biciclette_disponibili');
add_action('wp_ajax_nopriv_simpbire_get_biciclette_disponibili', 'simpbire_get_biciclette_disponibili');


// NOTE FOR REVIEWERS: This AJAX endpoint is used by FullCalendar in the admin panel.
// It is read-only and does not perform any write operations.
function simpbire_carica_prenotazioni_callback() {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions.', 'simple-bike-rental'), 403);
    }

    check_ajax_referer('simpbire_load_bookings_nonce', 'security');
    
    global $wpdb;
    $prenotazioni = simpbire_get_all_reservations_cached(); // Cache da transient
    $eventi = [];

    static $cache_biciclette = [];

    // Recuperiamo il timezone impostato su WordPress
    $wp_timezone = wp_timezone_string();
    $tz = new DateTimeZone($wp_timezone ? $wp_timezone : 'UTC'); // fallback UTC

    foreach ($prenotazioni as $p) {
        $giornaliero = (substr($p->data_inizio, 11) === '00:00:00' && substr($p->data_fine, 11) === '23:59:59');

        // Creiamo DateTime con il fuso WordPress per formattare il titolo
        $start_dt = new DateTime($p->data_inizio, $tz);
        $end_dt   = new DateTime($p->data_fine, $tz);

        $titolo = $p->nome . ($giornaliero
            ? ' – ' . __('Daily', 'simple-bike-rental')
            : ' – ' . $start_dt->format('H:i') . ' / ' . $end_dt->format('H:i'));

        if (!isset($cache_biciclette[$p->bicicletta_id])) {
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
            $cache_biciclette[$p->bicicletta_id] = $wpdb->get_row($wpdb->prepare(
                "SELECT nome, immagine FROM {$wpdb->prefix}simpbire_bicycles WHERE id = %d", 
                $p->bicicletta_id
            ));
        }

        $bicicletta = $cache_biciclette[$p->bicicletta_id];
        $nome_bici = $bicicletta ? $bicicletta->nome : '';
        $url_img = $bicicletta && $bicicletta->immagine
            ? wp_get_attachment_image_url($bicicletta->immagine, 'medium')
            : '';

        $classi = [];
        $ora_attuale = current_time('mysql');

        if ($p->stato === 'annullata') {
            $classi[] = 'annullata';
            $colore = '#cccccc';
        } elseif ($p->stato === 'attiva' && $p->data_fine < $ora_attuale) {
            $classi[] = 'completata';
            $colore = '#999999';
        } else {
            $classi[] = $giornaliero ? 'giornaliero' : 'orario';
            $colore = $giornaliero ? '#271002' : '#ffffff';
        }

        $eventi[] = [
            'id' => $p->id,
            'title' => esc_html($titolo),
            'start' => $p->data_inizio, // WordPress salva in fuso locale
            'end'   => $p->data_fine,
            'backgroundColor' => $colore,
            'classNames' => $classi,
            'extendedProps' => [
                'email' => $p->email,
                'telefono' => $p->telefono,
                'periodo' => simpbire_formatta_periodo($p->data_inizio, $p->data_fine),
                // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
                'link_modifica' => admin_url('admin.php?page=modifica_prenotazione&id=' . $p->id),
                'nome_bicicletta' => $nome_bici,
                'immagine_bicicletta' => $url_img
            ]
        ];
    }

    wp_send_json($eventi);
}
add_action( 'wp_ajax_simpbire_carica_prenotazioni', 'simpbire_carica_prenotazioni_callback' );


// NOTE FOR REVIEWERS: This AJAX handler is used for saving new reservations from the frontend.
// It performs write operations and is protected by `check_ajax_referer('simpbire_frontend_nonce', 'security')`.
function simpbire_salva_prenotazione_frontend() {
    // Nonce verification
    check_ajax_referer('simpbire_frontend_nonce', 'security');

    global $wpdb;

    // Validation and sanitization
    $bicicletta_id = isset($_POST['bicicletta_id']) ? intval($_POST['bicicletta_id']) : 0;
    $nome = isset($_POST['nome']) ? sanitize_text_field(wp_unslash($_POST['nome'])) : '';
    $email = isset($_POST['email']) ? sanitize_email(wp_unslash($_POST['email'])) : '';
    $telefono = isset($_POST['telefono']) ? sanitize_text_field(wp_unslash($_POST['telefono'])) : '';
    $tipo_noleggio = isset($_POST['tipo_noleggio']) ? sanitize_text_field(wp_unslash($_POST['tipo_noleggio'])) : '';

    if ($tipo_noleggio === 'giornaliero') {
        $data = isset($_POST['data_noleggio']) ? sanitize_text_field(wp_unslash($_POST['data_noleggio'])) : '';
        $data_inizio = $data . ' 00:00:00';
        $data_fine   = $data . ' 23:59:59';
    } else {
        $data_inizio = isset($_POST['data_inizio_orario']) ? sanitize_text_field(wp_unslash($_POST['data_inizio_orario'])) : '';
        $data_fine   = isset($_POST['data_fine_orario']) ? sanitize_text_field(wp_unslash($_POST['data_fine_orario'])) : '';
    }

    if (!$bicicletta_id || !$data_inizio || !$data_fine || !$nome || !$email) {
        wp_send_json_error(['message' => __('Missing or invalid fields.', 'simple-bike-rental')]);
    }

    // Stato iniziale: in attesa di conferma
    $stato = 'in_attesa';
    $token = wp_generate_password(32, false, false);
    $token_scadenza = gmdate('Y-m-d H:i:s', time() + (30 * 60));

    // Inserisci prenotazione
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe insert into custom plugin table using $wpdb->insert().
    $wpdb->insert("{$wpdb->prefix}simpbire_reservations", [
        'bicicletta_id' => $bicicletta_id,
        'nome' => $nome,
        'email' => $email,
        'telefono' => $telefono,
        'data_inizio' => $data_inizio,
        'data_fine' => $data_fine,
        'stato' => $stato,
        'token' => $token,
        'token_scadenza' => $token_scadenza
    ], [
        '%d', '%s', '%s', '%s', '%s', '%s', '%s', '%s', '%s'
    ]);

    $prenotazione_id = $wpdb->insert_id;

    if ($prenotazione_id && $token) {
        // Recupera la pagina scelta per i messaggi di conferma
        $pagina_id = get_option('simpbire_pagina_messaggi_token_id');
        $url_base = $pagina_id ? get_permalink($pagina_id) : site_url();

        $link_conferma = esc_url_raw(add_query_arg([
            'simpbire_confirm' => $token,
            '_wpnonce' => wp_create_nonce('simpbire_verifica_token')
        ], $url_base));

        // Filtri per soggetto e messaggio HTML
        $subject = apply_filters('simpbire_email_subject', __('Confirm your reservation', 'simple-bike-rental'), $nome);

        $button_label = esc_html__( 'Confirm your reservation', 'simple-bike-rental' );

        $message = apply_filters(
            'simpbire_email_message',
            sprintf(
                '<p>Hi %1$s,</p>
                <p>Thank you for your booking!</p>
                <p>To confirm availability and finalize your rental, please click the button below:</p>
                <p>
                    <a href="%2$s"
                    style="display:inline-block;padding:10px 18px;background-color:#2271b1;color:#ffffff;text-decoration:none;border-radius:4px;">
                        %3$s
                    </a>
                </p>
                <p>If the button does not work, copy and paste this link into your browser:</p>
                <p><a href="%2$s">%2$s</a></p>
                <p>If you did not make this request, you can ignore this email.</p>',
                esc_html( $nome ),
                esc_url( $link_conferma ),
                $button_label
            ),
            $nome,
            $link_conferma
        );

        $headers = ['Content-Type: text/html; charset=UTF-8'];

        wp_mail(sanitize_email($email), $subject, $message, $headers);
    }

    // Cancella la cache delle prenotazioni
    delete_transient('simpbire_all_reservations');

    // Redirezione finale
    $pagina_id = get_option('simpbire_pagina_redirect_id');
    $redirect_url = $pagina_id ? get_permalink($pagina_id) : '';

    wp_send_json_success([
        'message' => __('Reservation successfully saved.', 'simple-bike-rental'),
        'redirect_url' => $redirect_url
    ]);

    wp_die();
}
add_action('wp_ajax_simpbire_salva_prenotazione_frontend', 'simpbire_salva_prenotazione_frontend');
add_action('wp_ajax_nopriv_simpbire_salva_prenotazione_frontend', 'simpbire_salva_prenotazione_frontend');


// NOTE FOR REVIEWERS: This AJAX handler deletes a reservation. 
// It is protected by `check_ajax_referer('simpbire_elimina_prenotazione_ajax')` and restricted to administrators via `current_user_can('manage_options')`.
function simpbire_elimina_prenotazione() {
    // Verifica i permessi
    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Insufficient permissions.', 'simple-bike-rental'));
    }

    // Verifica il nonce di sicurezza
    check_ajax_referer('simpbire_elimina_prenotazione_ajax');

    // Verifica la presenza dell'ID
    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    if ($id <= 0) {
        wp_send_json_error(__('Invalid ID.', 'simple-bike-rental'));
    }

    global $wpdb;
    $table = $wpdb->prefix . 'simpbire_reservations';

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe deletion from custom table using $wpdb->delete(). No caching needed for write operations.
    $deleted = $wpdb->delete($table, ['id' => $id]);

    if ($deleted) {
        delete_transient('simpbire_all_reservations');
        wp_send_json_success(__('Reservation successfully deleted.', 'simple-bike-rental'));
    } else {
        wp_send_json_error(__('Error while deleting the reservation.', 'simple-bike-rental'));
    }
}
add_action('wp_ajax_simpbire_elimina_prenotazione', 'simpbire_elimina_prenotazione');
