<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Show notification messages in the backend (successes and errors)
 * based on the type of content (bicycle, reservation, notification).
 */
function simpbire_mostra_messaggi($tipo) {
    $labels = [
        'bicicletta' => [
            'successo' => [
                'inserito'   => esc_html__('Bicycle added successfully!', 'simple-bike-rental'),
                'aggiornato' => esc_html__('Bicycle successfully modified!', 'simple-bike-rental'),
                'eliminato'  => esc_html__('Bicycle successfully removed!', 'simple-bike-rental'),
            ],
            'errore' => [
                'errore_inserimento' => esc_html__('Error occurred while adding the bicycle.', 'simple-bike-rental'),
                'errore_modifica'    => esc_html__('Error occurred while modifying the bicycle.', 'simple-bike-rental'),
                'errore_eliminazione' => esc_html__('Error occurred while deleting the bicycle.', 'simple-bike-rental'),
                'errore_eliminazione_con_prenotazioni' => esc_html__('Unable to delete: the bicycle has active reservations.', 'simple-bike-rental'),
            ],
        ],
        'prenotazione' => [
            'successo' => [
                'inserito'         => esc_html__('Reservation successfully created!', 'simple-bike-rental'),
                'aggiornato'       => esc_html__('Reservation successfully modified!', 'simple-bike-rental'),
                'eliminato'        => esc_html__('Reservation successfully deleted!', 'simple-bike-rental'),
                'annullato'        => esc_html__('Booking canceled successfully', 'simple-bike-rental'),
                'riattivato'       => esc_html__('Booking successfully reactivated!', 'simple-bike-rental'),
                'nessuna_modifica' => esc_html__('No changes made.', 'simple-bike-rental'),
                'eliminati_passati' => esc_html__('Past bookings successfully deleted!', 'simple-bike-rental'),
            ],
            'errore' => [
                'errore_inserimento'   => esc_html__('Error while creating the reservation.', 'simple-bike-rental'),
                'errore_modifica'      => esc_html__('Error while editing the reservation.', 'simple-bike-rental'),
                'errore_eliminazione'  => esc_html__('Error while deleting the reservation.', 'simple-bike-rental'),
                'dati_mancanti'        => esc_html__('Missing or invalid data: please check and try again.', 'simple-bike-rental'),
                'errore_annullamento'  => esc_html__('Error while canceling the booking.', 'simple-bike-rental'),
                'errore_riattivazione' => esc_html__('Error during the reactivation of the reservation.', 'simple-bike-rental'),
                'errore_eliminazione_passate' => esc_html__('Error occurred while deleting past bookings.', 'simple-bike-rental'),
            ],
        ],
        'notifica' => [
            'successo' => [
                'modificato' => esc_html__('Notification successfully updated!', 'simple-bike-rental'),
            ],
            'errore' => [
                'errore_modifica' => esc_html__('Error occurred while editing the notification.', 'simple-bike-rental'),
            ],
        ],
    ];

    if (!isset($labels[$tipo])) {
        return; // Unmanaged type
    }

    // Show successful messages
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    foreach ($labels[$tipo]['successo'] as $chiave => $testo) {
        if (isset($_GET[$chiave]) && $_GET[$chiave] == 1) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html($testo) . '</p></div>';
        }
    }

    // Show error messages
    foreach ($labels[$tipo]['errore'] as $chiave => $testo) {
        if (isset($_GET[$chiave]) && $_GET[$chiave] == 1) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($testo) . '</p></div>';
        }
    }
}


/**
 * Bicycle management administration page
 */
function simpbire_bicicletta_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }

    $biciclette = simpbire_get_all_bicycles_cached();

    include plugin_dir_path(__FILE__) . '../templates/admin/elenco-biciclette.php';
}


function simpbire_aggiungi_bicicletta_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }
    
    if (isset($_SERVER['REQUEST_METHOD']) && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['nome_bicicletta'])) {
        if (
            !isset($_POST['simpbire_nonce']) ||
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verification doesn't require sanitization
            !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['simpbire_nonce'])), 'simpbire_aggiungi_bicicletta')
        ) {
            wp_die(esc_html__('Security failed. Please try again.', 'simple-bike-rental'));
        }

        $nome = sanitize_text_field(wp_unslash($_POST['nome_bicicletta'] ?? ''));
        $descrizione = sanitize_textarea_field(wp_unslash($_POST['descrizione'] ?? ''));
        $tipo = sanitize_text_field(wp_unslash($_POST['tipo_bicicletta'] ?? ''));
        $tariffa_giornaliera = floatval(str_replace(',', '.', sanitize_text_field(wp_unslash($_POST['tariffa_giornaliera'] ?? '0'))));
        $tariffa_oraria     = floatval(str_replace(',', '.', sanitize_text_field(wp_unslash($_POST['tariffa_oraria'] ?? '0'))));
        $immagine_id = intval(wp_unslash($_POST['immagine_id'] ?? 0));
        $quantita = intval(wp_unslash($_POST['quantita'] ?? 1));


        global $wpdb;

        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe insert into custom table with validated data
        $esito = $wpdb->insert(
            "{$wpdb->prefix}simpbire_bicycles",
            [
                'nome' => $nome,
                'descrizione' => $descrizione,
                'tipo' => $tipo,
                'tariffa_giornaliera' => $tariffa_giornaliera,
                'tariffa_oraria' => $tariffa_oraria,
                'immagine' => $immagine_id > 0 ? $immagine_id : null,
                'quantita' => $quantita,
            ]
        );

        if ($esito === false) {
            if (defined('WP_DEBUG') && WP_DEBUG && defined('WP_DEBUG_LOG') && WP_DEBUG_LOG) {
                // phpcs:ignore WordPress.PHP.DevelopmentFunctions.error_log_error_log
                error_log('Errore WPDB: ' . $wpdb->last_error);
            }
            wp_die(esc_html__('Error while entering the bicycle.', 'simple-bike-rental'));
        }

        $redirect_url = add_query_arg(
            array(
                'page'     => 'gestione_biciclette',
                'inserito' => 1,
            ),
            admin_url( 'admin.php' )
        );

        wp_safe_redirect( $redirect_url );
        exit;

    }

    wp_enqueue_media();
    include plugin_dir_path(__FILE__) . '../templates/admin/aggiungi-bicicletta.php';
}


function simpbire_aggiungi_bicicletta_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    if (
        !isset($_POST['simpbire_nonce']) ||
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce verification does not require sanitization
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['simpbire_nonce'])), 'simpbire_aggiungi_bicicletta')
    ) {
        wp_die(esc_html__('Security failed. Please try again.', 'simple-bike-rental'));
    }

    global $wpdb;

    $nome = sanitize_text_field(wp_unslash($_POST['nome_bicicletta'] ?? ''));
    $descrizione = sanitize_textarea_field(wp_unslash($_POST['descrizione'] ?? ''));
    $tipo = sanitize_text_field(wp_unslash($_POST['tipo_bicicletta'] ?? ''));
    $tariffa_giornaliera = floatval(str_replace(',', '.', sanitize_text_field(wp_unslash($_POST['tariffa_giornaliera'] ?? '0'))));
    $tariffa_oraria = floatval(str_replace(',', '.', sanitize_text_field(wp_unslash($_POST['tariffa_oraria'] ?? '0'))));
    $immagine_id = intval(wp_unslash($_POST['immagine_id'] ?? 0));
    $quantita = intval(wp_unslash($_POST['quantita'] ?? 1));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery -- Safe insert into custom table with validated and sanitized data
    $result = $wpdb->insert("{$wpdb->prefix}simpbire_bicycles", [
        'nome' => $nome,
        'descrizione' => $descrizione,
        'tipo' => $tipo,
        'tariffa_giornaliera' => $tariffa_giornaliera,
        'tariffa_oraria' => $tariffa_oraria,
        'immagine' => $immagine_id > 0 ? $immagine_id : null,
        'quantita' => $quantita,
    ]);

    if ( false !== $result ) {
        delete_transient( 'simpbire_all_bicycles' );

        $redirect_url = add_query_arg(
            array(
                'page'     => 'gestione_biciclette',
                'inserito' => 1,
            ),
            admin_url( 'admin.php' )
        );
    } else {
        $redirect_url = add_query_arg(
            array(
                'page'   => 'aggiungi_biciclette',
                'errore' => 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}


/**
 * Pagina per la modifica di una bicicletta esistente
 */
function simpbire_modifica_bicicletta_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }

    global $wpdb;

    // Carica il media uploader di WordPress
    wp_enqueue_media();

    // Verifica che sia stato passato un ID valido
    $id = isset($_GET['id']) ? intval($_GET['id']) : 0;
    $nonce = isset($_GET['_wpnonce']) ? sanitize_text_field(wp_unslash($_GET['_wpnonce'])) : '';

    if (
        !$id ||
        empty($nonce) ||
        !wp_verify_nonce($nonce, 'simpbire_modifica_bicicletta_' . $id)
    ) {
        echo '<div class="notice notice-error"><p>' . esc_html__('Invalid ID or nonce.', 'simple-bike-rental') . '</p></div>';
        return;
    }

    $id = intval($_GET['id']);

    // Recupera i dati della bicicletta
    $bicicletta = simpbire_get_bicicletta_by_id_cached($id);

    if (!$bicicletta) {
        echo '<div class="notice notice-error"><p>' . esc_html__('Bicycle not found.', 'simple-bike-rental') . '</p></div>';
        return;
    }

    include plugin_dir_path(__FILE__) . '../templates/admin/modifica-bicicletta.php';
}


function simpbire_modifica_bicicletta_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    $nonce = isset($_POST['simpbire_nonce']) ? sanitize_text_field(wp_unslash($_POST['simpbire_nonce'])) : '';

    if (empty($nonce) || !wp_verify_nonce($nonce, 'simpbire_modifica_bicicletta')) {
        wp_die(esc_html__('Security failed. Please try again.', 'simple-bike-rental'));
    }

    if (!isset($_POST['bicicletta_id']) || !is_numeric($_POST['bicicletta_id'])) {
        wp_die(esc_html__('Missing or invalid bicycle ID.', 'simple-bike-rental'));
    }

    global $wpdb;

    $id = intval(wp_unslash($_POST['bicicletta_id']));
    $nome = sanitize_text_field(wp_unslash($_POST['nome_bicicletta'] ?? ''));
    $descrizione = sanitize_textarea_field(wp_unslash($_POST['descrizione'] ?? ''));
    $tipo = sanitize_text_field(wp_unslash($_POST['tipo_bicicletta'] ?? ''));
    $raw_tariffa = isset($_POST['tariffa_giornaliera']) ? sanitize_text_field(wp_unslash($_POST['tariffa_giornaliera'])) : '0';
    $tariffa_giornaliera = floatval(str_replace(',', '.', $raw_tariffa));
    $raw_tariffa_oraria = isset($_POST['tariffa_oraria']) ? sanitize_text_field(wp_unslash($_POST['tariffa_oraria'])) : '0';
    $tariffa_oraria = floatval(str_replace(',', '.', $raw_tariffa_oraria));

    $immagine_id = intval(wp_unslash($_POST['immagine_id'] ?? 0));
    $quantita = intval(wp_unslash($_POST['quantita'] ?? 1));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct UPDATE on custom table with sanitized and validated data, and cache is cleared manually
    $wpdb->update(
        "{$wpdb->prefix}simpbire_bicycles",
        [
            'nome' => $nome,
            'descrizione' => $descrizione,
            'tipo' => $tipo,
            'tariffa_giornaliera' => $tariffa_giornaliera,
            'tariffa_oraria' => $tariffa_oraria,
            'immagine' => $immagine_id > 0 ? $immagine_id : null,
            'quantita' => $quantita
        ],
        [ 'id' => $id ]
    );

    delete_transient('simpbire_all_bicycles');
    delete_transient('simpbire_bicicletta_' . $id);

    $redirect_url = add_query_arg(
        array(
            'page'       => 'gestione_biciclette',
            'aggiornato' => 1,
        ),
        admin_url( 'admin.php' )
    );

    wp_safe_redirect( $redirect_url );
    exit;
}


function simpbire_elimina_bicicletta($id) {
    if ( ! current_user_can('manage_options') ) {
        return new WP_Error('unauthorized', __('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    global $wpdb;

    $id = intval($id);
    if ($id <= 0) {
        return new WP_Error('invalid_id', __('Invalid ID.', 'simple-bike-rental'));
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query sicura su tabella personalizzata per controllo logico prima della cancellazione
    $ha_prenotazioni = $wpdb->get_var($wpdb->prepare(
        "SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations WHERE bicicletta_id = %d AND stato = 'attiva'",
        $id
    ));

    if ($ha_prenotazioni > 0) {
        return new WP_Error('has_reservations', __('Unable to delete: there are active bookings.', 'simple-bike-rental'));
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Cancellazione sicura di un record specifico da tabella personalizzata
    $result = $wpdb->delete("{$wpdb->prefix}simpbire_bicycles", ['id' => $id]);

    if ($result === false) {
        return new WP_Error('db_error', __('Error while deleting from the database.', 'simple-bike-rental'));
    }

    delete_transient('simpbire_all_bicycles');

    return true;
}


function simpbire_elimina_bicicletta_admin() {
    if ( ! current_user_can('manage_options') ) {
        wp_die(esc_html__('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    if (
        ! isset($_POST['id'], $_POST['simpbire_elimina_bicicletta_nonce']) ||
        ! is_numeric($_POST['id'])
    ) {
        wp_die(esc_html__('Invalid nonce or missing ID.', 'simple-bike-rental'));
    }

    $id = intval($_POST['id']);

    $nonce = sanitize_text_field(wp_unslash($_POST['simpbire_elimina_bicicletta_nonce']));

    if ( ! wp_verify_nonce($nonce, 'simpbire_elimina_bicicletta_' . $id) ) {
        wp_die(esc_html__('Invalid nonce.', 'simple-bike-rental'));
    }

    $esito = simpbire_elimina_bicicletta($id);

    $args = array(
        'page' => 'gestione_biciclette',
    );

    if ( true === $esito ) {
        $args['eliminato'] = 1;

    } elseif ( $esito instanceof WP_Error ) {
        $code = $esito->get_error_code();

        switch ( $code ) {
            case 'has_reservations':
                $args['errore_eliminazione_con_prenotazioni'] = 1;
                break;

            case 'unauthorized':
                $args['errore_permesso'] = 1;
                break;

            default:
                $args['errore_eliminazione'] = 1;
                break;
        }

    } else {
        $args['errore_eliminazione'] = 1;
    }

    $redirect_url = add_query_arg( $args, admin_url( 'admin.php' ) );

    wp_safe_redirect( $redirect_url );
    exit;
}


/**
 * Administration page for viewing reservations
 */
function simpbire_prenotazioni_page() {
    ?>
    <div class="wrap">
        <?php simpbire_mostra_messaggi('prenotazione'); ?>

        <h1><?php esc_html_e('Bookings', 'simple-bike-rental'); ?></h1>

        <!-- Button to add a new reservation -->
        <div style="margin-bottom: 15px; display: flex; gap: 10px; align-items: center; margin:20px 0;">
            <a href="<?php echo esc_url(admin_url('admin.php?page=aggiungi_prenotazione')); ?>" class="button button-primary">
                <?php esc_html_e('+ Add Booking', 'simple-bike-rental'); ?>
            </a>

            <a href="<?php echo esc_url(admin_url('admin.php?page=esporta_prenotazioni')); ?>" class="button button-secondary">
                <?php esc_html_e('Export', 'simple-bike-rental'); ?>
            </a>
        </div>

        <!-- Tab Switcher -->
        <div class="prenotazioni-tabs">
            <a href="#" class="tab-link active" data-target="calendar-tab"><?php esc_html_e('Calendar View', 'simple-bike-rental'); ?></a>
            <a href="#" class="tab-link" data-target="table-tab"><?php esc_html_e('Table View', 'simple-bike-rental'); ?></a>
        </div>

        <!-- Calendar -->
        <div id="calendar-tab" class="tab-content active">
            <div id="calendar" style="background: #fff; padding: 20px; border-radius: 8px;"></div>
        </div>

        <!-- Table -->
        <div id="table-tab" class="tab-content">
            <?php
            global $wpdb;
            $ora_corrente = current_time('mysql');

            // Check if there are past bookings
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query per mostrare pulsante "elimina prenotazioni passate"; caching non necessario
            $ci_sono_passate = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations WHERE data_fine < %s",
                $ora_corrente
            ));
            ?>

            <?php if ($ci_sono_passate > 0) : ?>
                <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" onsubmit="return confirm('<?php esc_attr_e('Are you sure you want to delete all past bookings?', 'simple-bike-rental'); ?>');">
                    <?php wp_nonce_field('simpbire_elimina_prenotazioni_passate', 'simpbire_elimina_prenotazioni_passate_nonce'); ?>
                    <input type="hidden" name="action" value="simpbire_elimina_prenotazioni_passate">
                    <input type="submit" class="button button-secondary" style="margin-bottom:20px;" value="<?php esc_attr_e('Delete all past bookings', 'simple-bike-rental'); ?>">
                </form>
            <?php endif; ?>
            <?php
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Query for tabular view of reservations in admin; caching not required and all data is displayed immediately
            $prenotazioni = $wpdb->get_results("
                SELECT p.*, b.nome AS nome_bici, b.immagine AS immagine_bici
                FROM {$wpdb->prefix}simpbire_reservations p
                LEFT JOIN {$wpdb->prefix}simpbire_bicycles b ON p.bicicletta_id = b.id
                ORDER BY p.data_inizio DESC
            ");
            ?>
            <?php if (!empty($prenotazioni)) : ?>
                <table class="widefat fixed striped">
                    <thead>
                        <tr>
                            <th class="prenotazione-id"><?php esc_html_e('ID', 'simple-bike-rental'); ?></th>
                            <th class="bicicletta-img"><?php esc_html_e('Image', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('Name', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('Email', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('Phone', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('Start Date', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('End Date', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('Status', 'simple-bike-rental'); ?></th>
                            <th><?php esc_html_e('Actions', 'simple-bike-rental'); ?></th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php foreach ($prenotazioni as $p) : ?>
                            <tr>
                                <td><?php echo esc_html($p->id); ?></td>
                                <td>
                                    <?php if ($p->immagine_bici) : ?>
                                        <img src="<?php echo esc_url(wp_get_attachment_image_url($p->immagine_bici, 'thumbnail')); ?>" alt="" style="max-width: 60px; height: auto; border-radius: 4px;" />
                                    <?php else : ?>
                                        —
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html($p->nome); ?></td>
                                <td><?php echo esc_html($p->email); ?></td>
                                <td><?php echo esc_html($p->telefono); ?></td>
                                <td><?php echo esc_html($p->data_inizio); ?></td>
                                <td><?php echo esc_html($p->data_fine); ?></td>
                                <td>
                                    <?php
                                    echo '<span style="font-weight: bold;">' . esc_html(simpbire_get_stato_prenotazione_con_label($p->stato, $p->data_fine)) . '</span>';
                                    ?>
                                </td>
                                <td>
                                    <div class="noleggio-icone-wrapper">
                                        <?php
                                            $url_modifica = wp_nonce_url(
                                                admin_url('admin.php?page=modifica_prenotazione&id=' . intval($p->id)),
                                                'simpbire_modifica_prenotazione'
                                            );
                                        ?>
                                        <a href="<?php echo esc_url($url_modifica); ?>" class="noleggio-icona-btn modifica" title="<?php esc_html_e('Edit', 'simple-bike-rental'); ?>">
                                            <span class="dashicons dashicons-edit"></span>
                                        </a>

                                        <?php if ($p->stato !== 'annullata') : ?>
                                            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                                <?php wp_nonce_field('simpbire_annulla_prenotazione_' . $p->id, 'simpbire_annulla_prenotazione_nonce'); ?>
                                                <input type="hidden" name="action" value="simpbire_annulla_prenotazione_admin">
                                                <input type="hidden" name="id" value="<?php echo esc_attr($p->id); ?>">
                                                <button type="submit" class="noleggio-icona-btn annulla" title="<?php esc_html_e('Cancel', 'simple-bike-rental'); ?>" onclick="return confirm('<?php esc_html_e('Are you sure you want to cancel this booking?', 'simple-bike-rental'); ?>');">
                                                    <span class="dashicons dashicons-no-alt"></span>
                                                </button>
                                            </form>
                                        <?php else : ?>
                                            <span class="noleggio-icona-btn disattiva" title="<?php esc_html_e('Reservation canceled', 'simple-bike-rental'); ?>">
                                                <span class="dashicons dashicons-yes-alt"></span>
                                            </span>
                                        <?php endif; ?>

                                        <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" style="display:inline;">
                                            <?php wp_nonce_field('simpbire_elimina_prenotazione_admin_' . $p->id, 'simpbire_elimina_prenotazione_admin_nonce'); ?>
                                            <input type="hidden" name="action" value="simpbire_elimina_prenotazione_admin">
                                            <input type="hidden" name="id" value="<?php echo esc_attr($p->id); ?>">
                                            <button type="submit" class="noleggio-icona-btn elimina" title="<?php esc_html_e('Delete', 'simple-bike-rental'); ?>" onclick="return confirm('<?php esc_html_e('Are you sure you want to permanently delete this booking?', 'simple-bike-rental'); ?>');">
                                                <span class="dashicons dashicons-trash"></span>
                                            </button>
                                        </form>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    </tbody>
                </table>
            <?php else : ?>
                <p class="empty-table-msg"><?php esc_html_e('No reservations found.', 'simple-bike-rental'); ?></p>
            <?php endif; ?>
        </div>

        <!-- Modale Calendario -->
        <?php include SIMPBIRE_PLUGIN_PATH . 'templates/modale-prenotazione.php'; ?>
    </div>
    <?php
}


/**
 * Page for manually adding a reservation from the backend
 */
function simpbire_aggiungi_prenotazione_page() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to access this page.', 'simple-bike-rental'));
    }

    wp_enqueue_media(); // For any future use

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Add Booking', 'simple-bike-rental') . '</h1>';

    include SIMPBIRE_PLUGIN_PATH . 'templates/admin/form-aggiungi-prenotazione.php';

    echo '</div>';
}


/**
 * Managing the insertion of a reservation from the backend
 */
function simpbire_aggiungi_prenotazione_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    $nonce = isset($_POST['simpbire_aggiungi_prenotazione_admin_nonce']) ? sanitize_text_field(wp_unslash($_POST['simpbire_aggiungi_prenotazione_admin_nonce'])) : '';

    if (empty($nonce) || !wp_verify_nonce($nonce, 'simpbire_aggiungi_prenotazione_admin_action')) {
        wp_die(esc_html__('Invalid nonce. Please reload the page and try again.', 'simple-bike-rental'));
    }

    global $wpdb;

    $nome = sanitize_text_field(wp_unslash($_POST['nome_utente'] ?? ''));
    $email = sanitize_email(wp_unslash($_POST['email_utente'] ?? ''));
    $telefono = sanitize_text_field(wp_unslash($_POST['telefono_utente'] ?? ''));
    $tipo_noleggio = sanitize_text_field(wp_unslash($_POST['tipo_noleggio'] ?? ''));
    $bicicletta_id = intval(wp_unslash($_POST['bicicletta_id'] ?? 0));

    // URL di redirect per dati mancanti
    $missing_data_redirect = add_query_arg(
        array(
            'page'          => 'aggiungi_prenotazione',
            'dati_mancanti' => 1,
        ),
        admin_url( 'admin.php' )
    );

    if ( 'giornaliero' === $tipo_noleggio ) {
        $data_raw    = sanitize_text_field( wp_unslash( $_POST['data_giornaliera'] ?? '' ) );
        $data_inizio = $data_raw ? $data_raw . ' 00:00:00' : '';
        $data_fine   = $data_raw ? $data_raw . ' 23:59:59' : '';

    } elseif ( 'orario' === $tipo_noleggio ) {
        $data_inizio = sanitize_text_field( wp_unslash( $_POST['data_inizio_oraria'] ?? '' ) );
        $data_fine   = sanitize_text_field( wp_unslash( $_POST['data_fine_oraria'] ?? '' ) );

    } else {
        wp_safe_redirect( $missing_data_redirect );
        exit;
    }

    if ( ! $nome || ! $email || ! $bicicletta_id || ! $data_inizio || ! $data_fine ) {
        wp_safe_redirect( $missing_data_redirect );
        exit;
    }

    $intervallo = intval(get_option('simpbire_minuti_intervallo_prenotazioni', 0));

    // Verifica conflitti rispetto alla quantità
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Real-time availability conflict check on custom table; caching is not applicable for time-sensitive validation
    $conflitti = $wpdb->get_var($wpdb->prepare("
    SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations
    WHERE bicicletta_id = %d
    AND stato = 'attiva'
    AND (
        (data_inizio <= %s AND DATE_ADD(data_fine, INTERVAL %d MINUTE) > %s)
    OR (data_inizio >= %s AND DATE_ADD(data_fine, INTERVAL %d MINUTE) <= %s)
    OR (data_inizio <= %s AND DATE_ADD(data_fine, INTERVAL %d MINUTE) >= %s)
    )
    ",
    $bicicletta_id,
    $data_inizio, $intervallo, $data_inizio,
    $data_inizio, $intervallo, $data_fine,
    $data_inizio, $intervallo, $data_fine
    ));

    // Total quantity available
    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Real-time retrieval of available bike quantity for validation; caching is not suitable
    $quantita = $wpdb->get_var($wpdb->prepare("
    SELECT quantita FROM {$wpdb->prefix}simpbire_bicycles WHERE id = %d
    ", $bicicletta_id));

    if ( (int) $quantita <= $conflitti ) {
        $redirect_url = add_query_arg(
            array(
                'page'              => 'aggiungi_prenotazione',
                'errore_inserimento'=> 1,
            ),
            admin_url( 'admin.php' )
        );

        wp_safe_redirect( $redirect_url );
        exit;
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe insert into custom table with sanitized and validated data; caching not applicable
    $result = $wpdb->insert("{$wpdb->prefix}simpbire_reservations", [
        'bicicletta_id' => $bicicletta_id,
        'nome' => $nome,
        'email' => $email,
        'telefono' => $telefono,
        'data_inizio' => $data_inizio,
        'data_fine' => $data_fine,
        'stato' => 'attiva',
    ]);

    if ( false !== $result ) {
        delete_transient( 'simpbire_all_reservations' );

        $redirect_url = add_query_arg(
            array(
                'page'     => 'prenotazioni_biciclette',
                'inserito' => 1,
            ),
            admin_url( 'admin.php' )
        );

    } else {
        $redirect_url = add_query_arg(
            array(
                'page'              => 'aggiungi_prenotazione',
                'errore_inserimento'=> 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}
add_action( 'admin_post_simpbire_aggiungi_prenotazione_admin', 'simpbire_aggiungi_prenotazione_admin' );

/**
 * Pagina per modificare una prenotazione esistente
 */
function simpbire_modifica_prenotazione_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }

    // Verify nonce
    if (
        !isset($_GET['_wpnonce']) ||
        !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'simpbire_modifica_prenotazione')
    ) {
        wp_die(esc_html__('Invalid or expired link.', 'simple-bike-rental'));
    }

    global $wpdb;

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- This is a read-only check to load a reservation form.
    if (!isset($_GET['id'])) {
        echo '<div class="notice notice-error"><p>' . esc_html__('Missing booking ID.', 'simple-bike-rental') . '</p></div>';
        return;
    }

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Safe usage: retrieving reservation ID to display form.
    $id = intval($_GET['id']);

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe SELECT on custom table using prepared statement; real-time lookup required, caching not applicable.
    $prenotazione = $wpdb->get_row(
        $wpdb->prepare("SELECT * FROM {$wpdb->prefix}simpbire_reservations WHERE id = %d", $id)
    );

    if (!$prenotazione) {
        echo '<div class="notice notice-error"><p>' . esc_html__('Reservation not found.', 'simple-bike-rental') . '</p></div>';
        return;
    }

    if ($prenotazione->stato === 'annullata') {
        echo '<div class="notice notice-warning"><p>' . esc_html__('This reservation is canceled.', 'simple-bike-rental') . '</p></div>';
        echo '<form method="post" action="' . esc_url(admin_url('admin-post.php')) . '">';
        wp_nonce_field('simpbire_riattiva_prenotazione_' . $prenotazione->id, 'simpbire_riattiva_prenotazione_nonce');
        echo '<input type="hidden" name="action" value="simpbire_riattiva_prenotazione_admin">';
        echo '<input type="hidden" name="id" value="' . esc_attr($prenotazione->id) . '">';
        submit_button(esc_html__('Reactivate Reservation', 'simple-bike-rental'), 'secondary');
        echo '</form>';
    }

    $biciclette = simpbire_get_all_bicycles_cached();

    $is_giornaliero = (substr($prenotazione->data_inizio, 11) === '00:00:00' && substr($prenotazione->data_fine, 11) === '23:59:59');
    $tipo_noleggio = $is_giornaliero ? 'giornaliero' : 'orario';
    ?>
    <div class="wrap">
        <h1><?php esc_html_e('Edit Reservation', 'simple-bike-rental'); ?></h1>
        <p>
            <a href="<?php echo esc_url(admin_url('admin.php?page=prenotazioni_biciclette')); ?>">
            <?php esc_html_e('Back to Reservations', 'simple-bike-rental'); ?>
            </a>
        </p>

        <?php
        include plugin_dir_path(__FILE__) . '../templates/admin/form-modifica-prenotazione.php';
        ?>
    </div>
    <?php
}


function simpbire_modifica_prenotazione_admin() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to perform this operation.', 'simple-bike-rental' ) );
    }

    global $wpdb;

    $id = isset( $_POST['id'] ) ? absint( wp_unslash( $_POST['id'] ) ) : 0;

    if ( 0 === $id ) {
        $redirect_url = add_query_arg(
            array(
                'page'            => 'prenotazioni_biciclette',
                'errore_modifica' => 1,
            ),
            admin_url( 'admin.php' )
        );

        wp_safe_redirect( $redirect_url );
        exit;
    }

    $nonce = isset($_POST['simpbire_modifica_prenotazione_admin_nonce']) ? sanitize_text_field(wp_unslash($_POST['simpbire_modifica_prenotazione_admin_nonce'])) : '';

    if (empty($nonce) || !wp_verify_nonce($nonce, 'simpbire_modifica_prenotazione_admin_action')) {
        wp_die(esc_html__('Invalid nonce. Please reload the page and try again.', 'simple-bike-rental'));
    }

    $nome = sanitize_text_field(wp_unslash($_POST['nome'] ?? ''));
    $email = sanitize_email(wp_unslash($_POST['email'] ?? ''));
    $telefono = sanitize_text_field(wp_unslash($_POST['telefono'] ?? ''));
    $tipo_noleggio = sanitize_text_field(wp_unslash($_POST['tipo_noleggio'] ?? ''));
    $bicicletta_id = intval(wp_unslash($_POST['bicicletta_id'] ?? 0));

    // URL di redirect per dati mancanti in modifica prenotazione.
    $missing_data_redirect = add_query_arg(
        array(
            'page'          => 'modifica_prenotazione',
            'id'            => (int) $id,
            'dati_mancanti' => 1,
        ),
        admin_url( 'admin.php' )
    );

    if ( 'giornaliero' === $tipo_noleggio ) {
        $data_raw    = sanitize_text_field( wp_unslash( $_POST['data_giornaliera'] ?? '' ) );
        $data_inizio = $data_raw ? $data_raw . ' 00:00:00' : '';
        $data_fine   = $data_raw ? $data_raw . ' 23:59:59' : '';

    } elseif ( 'orario' === $tipo_noleggio ) {
        $data_inizio = sanitize_text_field( wp_unslash( $_POST['data_inizio'] ?? '' ) );
        $data_fine   = sanitize_text_field( wp_unslash( $_POST['data_fine'] ?? '' ) );

    } else {
        wp_safe_redirect( $missing_data_redirect );
        exit;
    }

    if ( ! $nome || ! $email || ! $bicicletta_id || ! $data_inizio || ! $data_fine ) {
        wp_safe_redirect( $missing_data_redirect );
        exit;
    }

    $intervallo = intval(get_option('simpbire_minuti_intervallo_prenotazioni', 0));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct SELECT query with prepared statement to check for reservation conflicts; real-time validation required, not suitable for caching.
    $conflitti = $wpdb->get_var($wpdb->prepare("
    SELECT COUNT(*) FROM {$wpdb->prefix}simpbire_reservations
    WHERE bicicletta_id = %d
    AND id != %d
    AND stato = 'attiva'
    AND (
        (data_inizio <= %s AND DATE_ADD(data_fine, INTERVAL %d MINUTE) > %s)
    OR (data_inizio >= %s AND DATE_ADD(data_fine, INTERVAL %d MINUTE) <= %s)
    OR (data_inizio <= %s AND DATE_ADD(data_fine, INTERVAL %d MINUTE) >= %s)
    )
    ",
    $bicicletta_id, $id,
    $data_inizio, $intervallo, $data_inizio,
    $data_inizio, $intervallo, $data_fine,
    $data_inizio, $intervallo, $data_fine
    ));

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct SELECT query to retrieve current bike quantity for real-time availability check; not suitable for caching.
    $quantita = $wpdb->get_var($wpdb->prepare("
    SELECT quantita FROM {$wpdb->prefix}simpbire_bicycles WHERE id = %d
    ", $bicicletta_id)); 

    if ( (int) $quantita <= $conflitti ) {
        $redirect_url = add_query_arg(
            array(
                'page'            => 'modifica_prenotazione',
                'id'              => (int) $id,
                'errore_modifica' => 1,
            ),
            admin_url( 'admin.php' )
        );

        wp_safe_redirect( $redirect_url );
        exit;
    }

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct UPDATE on custom reservations table; caching not applicable for write operations.
    $updated = $wpdb->update(
        "{$wpdb->prefix}simpbire_reservations",
        [
            'nome' => $nome,
            'email' => $email,
            'telefono' => $telefono,
            'bicicletta_id' => $bicicletta_id,
            'data_inizio' => $data_inizio,
            'data_fine' => $data_fine,
        ],
        [ 'id' => $id ]
    );

    if ( false !== $updated && 0 < $updated ) {
        delete_transient( 'simpbire_all_reservations' );

        $redirect_url = add_query_arg(
            array(
                'page'       => 'prenotazioni_biciclette',
                'aggiornato' => 1,
            ),
            admin_url( 'admin.php' )
        );

    } elseif ( 0 === $updated ) {

        $redirect_url = add_query_arg(
            array(
                'page'             => 'prenotazioni_biciclette',
                'nessuna_modifica' => 1,
            ),
            admin_url( 'admin.php' )
        );

    } else {

        $redirect_url = add_query_arg(
            array(
                'page'            => 'modifica_prenotazione',
                'id'              => (int) $id,
                'errore_modifica' => 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}
add_action('admin_post_simpbire_modifica_prenotazione_admin', 'simpbire_modifica_prenotazione_admin');


function simpbire_annulla_prenotazione_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'simple-bike-rental'));
    }

    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;

    $nonce = isset($_POST['simpbire_annulla_prenotazione_nonce']) ? sanitize_text_field(wp_unslash($_POST['simpbire_annulla_prenotazione_nonce'])) : '';

    if (!$id || !wp_verify_nonce($nonce, 'simpbire_annulla_prenotazione_' . $id)) {
        wp_die(esc_html__('Invalid request.', 'simple-bike-rental'));
    }

    global $wpdb;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct UPDATE to change reservation status; caching not applicable to write operations.
    $aggiornato = $wpdb->update(
        $wpdb->prefix . 'simpbire_reservations',
        ['stato' => 'annullata'],
        ['id' => $id]
    );

    if ( false !== $aggiornato ) {
        delete_transient( 'simpbire_all_reservations' );

        $redirect_url = add_query_arg(
            array(
                'page'      => 'prenotazioni_biciclette',
                'annullato' => 1,
            ),
            admin_url( 'admin.php' )
        );

    } else {
        $redirect_url = add_query_arg(
            array(
                'page'               => 'prenotazioni_biciclette',
                'errore_annullamento'=> 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}
add_action('admin_post_simpbire_annulla_prenotazione_admin', 'simpbire_annulla_prenotazione_admin');


function simpbire_riattiva_prenotazione_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('Insufficient permissions.', 'simple-bike-rental'));
    }

    $id = isset($_POST['id']) ? intval($_POST['id']) : 0;
    $nonce = isset($_POST['simpbire_riattiva_prenotazione_nonce']) ? sanitize_text_field(wp_unslash($_POST['simpbire_riattiva_prenotazione_nonce'])) : '';

    if (
        !$id || 
        !wp_verify_nonce($nonce, 'simpbire_riattiva_prenotazione_' . $id)
    ) {
        wp_die(esc_html__('Invalid request.', 'simple-bike-rental'));
    }

    global $wpdb;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct UPDATE to set reservation status; caching not applicable to write operations.
    $aggiornato = $wpdb->update(
        $wpdb->prefix . 'simpbire_reservations',
        ['stato' => 'attiva'],
        ['id' => $id]
    );

    if ( false !== $aggiornato ) {
        delete_transient( 'simpbire_all_reservations' );

        $redirect_url = add_query_arg(
            array(
                'page'       => 'prenotazioni_biciclette',
                'riattivato' => 1,
            ),
            admin_url( 'admin.php' )
        );

    } else {
        $redirect_url = add_query_arg(
            array(
                'page'                 => 'prenotazioni_biciclette',
                'errore_riattivazione' => 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}
add_action('admin_post_simpbire_riattiva_prenotazione_admin', 'simpbire_riattiva_prenotazione_admin');


function simpbire_elimina_prenotazione_admin() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    $id = intval($_POST['id'] ?? 0);

    $nonce = isset($_POST['simpbire_elimina_prenotazione_admin_nonce']) ? sanitize_text_field(wp_unslash($_POST['simpbire_elimina_prenotazione_admin_nonce'])) : '';

    if (
        !$id ||
        empty($nonce) ||
        !wp_verify_nonce($nonce, 'simpbire_elimina_prenotazione_admin_' . $id)
    ) {
        wp_die(esc_html__('Unauthorized action or missing data.', 'simple-bike-rental'));
    }

    global $wpdb;

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Safe direct DELETE from custom table; caching not applicable to write operations.
    $deleted = $wpdb->delete("{$wpdb->prefix}simpbire_reservations", ['id' => $id]);

    if ( true === $deleted ) {
        delete_transient( 'simpbire_all_reservations' );

        $redirect_url = add_query_arg(
            array(
                'page'      => 'prenotazioni_biciclette',
                'eliminato' => 1,
            ),
            admin_url( 'admin.php' )
        );

    } else {
        $redirect_url = add_query_arg(
            array(
                'page'               => 'prenotazioni_biciclette',
                'errore_eliminazione'=> 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}
add_action('admin_post_simpbire_elimina_prenotazione_admin', 'simpbire_elimina_prenotazione_admin');


function simpbire_elimina_prenotazioni_passate() {
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have permission to perform this operation.', 'simple-bike-rental'));
    }

    if (
        !isset($_POST['simpbire_elimina_prenotazioni_passate_nonce']) ||
        !wp_verify_nonce(
            sanitize_text_field(wp_unslash($_POST['simpbire_elimina_prenotazioni_passate_nonce'])),
            'simpbire_elimina_prenotazioni_passate'
        )
    ) {
        wp_die(esc_html__('Invalid nonce.', 'simple-bike-rental'));
    }

    global $wpdb;
    $ora_corrente = current_time('mysql');

    // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
    $eliminati = $wpdb->query(
        $wpdb->prepare(
            "DELETE FROM {$wpdb->prefix}simpbire_reservations WHERE data_fine < %s",
            $ora_corrente
        )
    );

    if ( false === $eliminati ) {
        $redirect_url = add_query_arg(
            array(
                'page'                         => 'prenotazioni_biciclette',
                'errore_eliminazione_passate'  => 1,
            ),
            admin_url( 'admin.php' )
        );

    } else {
        delete_transient( 'simpbire_all_reservations' );

        $redirect_url = add_query_arg(
            array(
                'page'               => 'prenotazioni_biciclette',
                'eliminati_passati'  => 1,
            ),
            admin_url( 'admin.php' )
        );
    }

    wp_safe_redirect( $redirect_url );
    exit;
}
add_action('admin_post_simpbire_elimina_prenotazioni_passate', 'simpbire_elimina_prenotazioni_passate');


function simpbire_pagina_elenco_notifiche() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }

    // Recorded notifications (currently static)
    $notifiche = simpbire_get_notifiche_registrate();

    echo '<div class="wrap">';

    simpbire_mostra_messaggi('notifica');

    echo '<h1>' . esc_html__('Notifications', 'simple-bike-rental') . '</h1>';
    echo '<p>' . esc_html__('Configuration of emails sent by the rental system.', 'simple-bike-rental') . '</p>';

    echo '<table class="widefat fixed striped">';
    echo '<thead><tr>';
    echo '<th>' . esc_html__('Notification', 'simple-bike-rental') . '</th>';
    echo '<th>' . esc_html__('Recipient', 'simple-bike-rental') . '</th>';
    echo '<th>' . esc_html__('Status', 'simple-bike-rental') . '</th>';
    echo '<th>' . esc_html__('Actions', 'simple-bike-rental') . '</th>';
    echo '</tr></thead>';
    echo '<tbody>';

    foreach ($notifiche as $slug => $notifica) {
        $attiva = get_option("simpbire_notifica_attiva_$slug", 0); // Default: disabled
        $stato = $attiva ? esc_html__('Active', 'simple-bike-rental') : esc_html__('Deactivated', 'simple-bike-rental');
        $url_modifica = wp_nonce_url(
            admin_url("admin.php?page=modifica_notifica&notifica=" . rawurlencode($slug)),
            'simpbire_modifica_notifica'
        );

        echo '<tr>';
        echo '<td>' . esc_html($notifica['nome']) . '</td>';
        echo '<td>' . esc_html($notifica['destinatario']) . '</td>';
        echo '<td>' . esc_html($stato) . '</td>';
        echo '<td><a href="' . esc_url($url_modifica) . '" class="button">' . esc_html__('Edit', 'simple-bike-rental') . '</a></td>';
        echo '</tr>';
    }

    echo '</tbody>';
    echo '</table>';
    echo '</div>';
}


function simpbire_modifica_notifica_page() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }

    // Verifica nonce
    // phpcs:ignore WordPress.Security.NonceVerification.Recommended
    if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'simpbire_modifica_notifica')) {
        wp_die(esc_html__('Invalid or expired link.', 'simple-bike-rental'));
    }

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Safe usage: only reading GET input to determine which notification to edit.
    if (empty($_GET['notifica'])) {
        echo '<div class="notice notice-error"><p>' . esc_html__('Notification not specified.', 'simple-bike-rental') . '</p></div>';
        return;
    }

    // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Safe usage: sanitized with sanitize_key below.
    $slug = sanitize_key($_GET['notifica']);
    $notifiche = simpbire_get_notifiche_registrate();

    if (!isset($notifiche[$slug])) {
        echo '<div class="notice notice-error"><p>' . esc_html__('Notification not found.', 'simple-bike-rental') . '</p></div>';
        return;
    }

    $notifica = $notifiche[$slug];
    $attiva_key = "simpbire_notifica_attiva_$slug";
    $oggetto_key = "simpbire_notifica_oggetto_$slug";
    $contenuto_key = "simpbire_notifica_contenuto_$slug";
    $destinatari_key = "simpbire_notifica_destinatari_$slug";

    $attiva = get_option($attiva_key, '');
    $oggetto = get_option($oggetto_key, '');
    $contenuto = get_option($contenuto_key, '');
    $destinatari_valore = get_option($destinatari_key, '');

    ?>
    <div class="wrap">
        <h1>
            <?php
            // translators: %s is the name of the notification.
            echo sprintf(esc_html__('Edit Notification: %s', 'simple-bike-rental'), esc_html($notifica['nome']));
            ?>
        </h1>

        <?php
        include plugin_dir_path(__FILE__) . '../templates/admin/form-modifica-notifica.php';
        ?>
    </div>
    <?php
}


function simpbire_pagina_impostazioni_noleggio() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'You do not have permission to access this page.', 'simple-bike-rental' ) );
    }

    echo '<div class="wrap">';
    echo '<h1>' . esc_html__('Settings', 'simple-bike-rental') . '</h1>';

    // Nav tab non più necessaria, ma lasciamo struttura coerente per eventuali espansioni future
    echo '<nav class="nav-tab-wrapper">';
    echo '<a href="#" class="nav-tab nav-tab-active">' . esc_html__('Settings', 'simple-bike-rental') . '</a>';
    echo '</nav>';

    // Include la pagina delle impostazioni generali
    $base_path = dirname(__DIR__) . '/templates/';
    include $base_path . 'impostazioni-generali.php';

    echo '</div>';
}


// Gestione export CSV prima di qualsiasi output
add_action('admin_init', 'simpbire_verifica_export_csv');

function simpbire_verifica_export_csv() {
    if (
        is_admin() &&
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce is verified inside simpbire_handle_export_csv()
        isset($_GET['page']) && $_GET['page'] === 'esporta_prenotazioni' &&
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Nonce is verified inside simpbire_handle_export_csv()
        isset($_GET['export_csv']) && $_GET['export_csv'] === '1' &&
        current_user_can('manage_options')
    ) {
        simpbire_handle_export_csv();
        exit;
    }
}


// Callback function
function simpbire_esporta_prenotazioni() {
    include SIMPBIRE_PLUGIN_PATH . 'templates/admin/esporta-prenotazioni.php';
}


// Export to CSV format
function simpbire_handle_export_csv() {
    if (
        ! isset( $_GET['simpbire_export_csv_nonce'] ) ||
        // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash, WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Nonce is verified directly with wp_verify_nonce(), no unslashing needed.
        ! wp_verify_nonce( $_GET['simpbire_export_csv_nonce'], 'simpbire_export_csv_action' )
    ) {
        wp_die( esc_html__( 'Security check failed.', 'simple-bike-rental' ) );
    }

    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied.', 'simple-bike-rental' ) );
    }

    global $wpdb;

    // Raccolta e sanificazione
    $stato   = isset($_GET['stato']) ? sanitize_text_field(wp_unslash($_GET['stato'])) : '';
    $tipo    = isset($_GET['tipo']) ? sanitize_text_field(wp_unslash($_GET['tipo'])) : '';
    $data_da = isset($_GET['data_da']) ? sanitize_text_field(wp_unslash($_GET['data_da'])) : '';
    $data_a  = isset($_GET['data_a']) ? sanitize_text_field(wp_unslash($_GET['data_a'])) : '';

    $where = [];
    $params = [];

    if ( $data_da !== '' ) {
        $where[] = 'data_inizio >= %s';
        $params[] = $data_da . ' 00:00:00';
    }

    if ( $data_a !== '' ) {
        $where[] = 'data_fine <= %s';
        $params[] = $data_a . ' 23:59:59';
    }

    $where_sql = ! empty( $where ) ? 'WHERE ' . implode(' AND ', $where) : '';

    // phpcs:disable PluginCheck.Security.DirectDB.UnescapedDBParameter

    // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared -- Query is built safely and only prepared if needed.
    $sql = "SELECT id, utente_id, bicicletta_id, nome, email, telefono, data_inizio, data_fine, stato 
            FROM {$wpdb->prefix}simpbire_reservations
            $where_sql
            ORDER BY data_inizio DESC";

    if ( ! empty( $params ) ) {
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query is safe, uses prepare(), and is executed once for export.
        $results = $wpdb->get_results( $wpdb->prepare( $sql, $params ), ARRAY_A );
    } else {
        // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared, WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching -- Direct query has no placeholders, safe and used only for export.
        $results = $wpdb->get_results( $sql, ARRAY_A );
    }

    // phpcs:enable PluginCheck.Security.DirectDB.UnescapedDBParameter

    if ( empty( $results ) ) {
        wp_die( esc_html__( 'No reservations found to export.', 'simple-bike-rental' ) );
    }

    $now_ts = strtotime( current_time('mysql') );
    $final = [];

    foreach ( $results as $riga ) {
        $data_inizio = strtotime($riga['data_inizio']);
        $data_fine   = strtotime($riga['data_fine']);

        $tipo_noleggio = (
            gmdate('H:i:s', $data_inizio) === '00:00:00' &&
            gmdate('H:i:s', $data_fine)   === '23:59:59' &&
            gmdate('Y-m-d', $data_inizio) === gmdate('Y-m-d', $data_fine)
        ) ? 'daily' : 'hourly';

        $stato_effettivo = match (true) {
            $data_fine < $now_ts && $riga['stato'] === 'attiva' => 'completed',
            $riga['stato'] === 'attiva' => 'active',
            $riga['stato'] === 'in_attesa' => 'pending',
            $riga['stato'] === 'annullata' => 'cancelled',
            default => $riga['stato'],
        };

        if ( $tipo && $tipo !== $tipo_noleggio ) {
            continue;
        }

        if ( $stato && $stato !== $stato_effettivo ) {
            continue;
        }

        $final[] = [
            'ID'            => $riga['id'],
            'Name'          => $riga['nome'],
            'Email'         => $riga['email'],
            'Phone'         => $riga['telefono'],
            'Start Date'    => $riga['data_inizio'],
            'End Date'      => $riga['data_fine'],
            'Rental Type'   => $tipo_noleggio,
            'Status'        => $stato_effettivo,
        ];
    }

    if ( empty( $final ) ) {
        wp_die( esc_html__( 'No reservations matched the selected filters.', 'simple-bike-rental' ) );
    }

    $filename_parts = [ gmdate('Y-m-d') ];
    if ( $stato ) $filename_parts[] = $stato;
    if ( $tipo )  $filename_parts[] = $tipo;
    $filename = 'reservations-' . implode('-', $filename_parts) . '.csv';

    header('Content-Type: text/csv; charset=UTF-8');
    header('Content-Disposition: attachment; filename="' . $filename . '"');

    $output = fopen('php://output', 'w');
    fputcsv($output, array_keys($final[0]));
    foreach ( $final as $row ) {
        fputcsv($output, $row);
    }

    // phpcs:ignore WordPress.WP.AlternativeFunctions.file_system_operations_fclose -- fclose is safe and appropriate for php://output
    fclose($output);
    exit;
}


// Add pages to the admin menu
function simpbire_aggiungi_menu_biciclette() {
    // Main menu
    add_menu_page(
        esc_html__('Bicycle Management', 'simple-bike-rental'),
        esc_html__('Bicycles', 'simple-bike-rental'),
        'manage_options',
        'gestione_biciclette',
        'simpbire_bicicletta_page',
        'dashicons-store',
        20
    );

    // Visible subpages
    add_submenu_page(
        'gestione_biciclette',
        esc_html__('Add Bicycles', 'simple-bike-rental'),
        esc_html__('Add Bicycle', 'simple-bike-rental'),
        'manage_options',
        'aggiungi_biciclette',
        'simpbire_aggiungi_bicicletta_page'
    );

    add_submenu_page(
        'gestione_biciclette',
        esc_html__('Bookings', 'simple-bike-rental'),
        esc_html__('Bookings', 'simple-bike-rental'),
        'manage_options',
        'prenotazioni_biciclette',
        'simpbire_prenotazioni_page'
    );

    add_submenu_page(
        'gestione_biciclette',
        esc_html__('Notifications', 'simple-bike-rental'),
        esc_html__('Notifications', 'simple-bike-rental'),
        'manage_options',
        'noleggio_notifiche',
        'simpbire_pagina_elenco_notifiche'
    );

    add_submenu_page(
        'gestione_biciclette',
        esc_html__('Settings', 'simple-bike-rental'),
        esc_html__('Settings', 'simple-bike-rental'),
        'manage_options',
        'impostazioni_noleggio',
        'simpbire_pagina_impostazioni_noleggio',
        99
    );

    // Hidden subpages (not visible in menu)
    add_submenu_page(
        'noleggio_biciclette_hidden',
        esc_html__('Edit Bicycle', 'simple-bike-rental'),
        '',
        'manage_options',
        'modifica_bicicletta',
        'simpbire_modifica_bicicletta_page'
    );

    add_submenu_page(
        'noleggio_biciclette_hidden',
        esc_html__('Add Booking', 'simple-bike-rental'),
        '',
        'manage_options',
        'aggiungi_prenotazione',
        'simpbire_aggiungi_prenotazione_page'
    );

    add_submenu_page(
        'noleggio_biciclette_hidden',
        esc_html__('Edit Booking', 'simple-bike-rental'),
        '',
        'manage_options',
        'modifica_prenotazione',
        'simpbire_modifica_prenotazione_page'
    );

    add_submenu_page(
        'noleggio_biciclette_hidden',
        esc_html__('Edit Notification', 'simple-bike-rental'),
        '',
        'manage_options',
        'modifica_notifica',
        'simpbire_modifica_notifica_page'
    );

    add_submenu_page(
        'noleggio_biciclette_hidden',
        esc_html__('Export Reservations', 'simple-bike-rental'),
        '',
        'manage_options',
        'esporta_prenotazioni',
        'simpbire_esporta_prenotazioni'
    );
}
add_action('admin_menu', 'simpbire_aggiungi_menu_biciclette');
