<?php
/**
 * Plugin Name: Simple Auto Post Scheduler
 * Plugin URI: https://wordpress.org/plugins/simple-auto-post-scheduler/
 * Description: Schedule posts to be published at specific times and intervals with an easy-to-use interface.
 * Version: 1.0.0
 * Author: Mehedi Hasan
 * License: GPL v2 or later
 * Text Domain: simple-auto-post-scheduler
 * Requires at least: 5.0
 * Requires PHP: 7.0
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
if (!defined('POST_SCHEDULER_VERSION')) {
    define('POST_SCHEDULER_VERSION', '1.0.0');
}
if (!defined('POST_SCHEDULER_PLUGIN_DIR')) {
    define('POST_SCHEDULER_PLUGIN_DIR', plugin_dir_path(__FILE__));
}
if (!defined('POST_SCHEDULER_PLUGIN_URL')) {
    define('POST_SCHEDULER_PLUGIN_URL', plugin_dir_url(__FILE__));
}

/**
 * Include required files
 */
function post_scheduler_include_files() {
    $includes_path = POST_SCHEDULER_PLUGIN_DIR . 'includes/post-scheduler-functions.php';
    $admin_path = POST_SCHEDULER_PLUGIN_DIR . 'admin/post-scheduler-admin.php';
    
    if (file_exists($includes_path)) {
        require_once $includes_path;
    }
    
    if (file_exists($admin_path)) {
        require_once $admin_path;
    }
}
add_action('plugins_loaded', 'post_scheduler_include_files');

/**
 * Activation hook
 */
function post_scheduler_activate() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'post_scheduler';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE IF NOT EXISTS $table_name (
        id mediumint(9) NOT NULL AUTO_INCREMENT,
        post_id bigint(20) NOT NULL,
        scheduled_time datetime NOT NULL,
        interval_type varchar(50) DEFAULT NULL,
        interval_value int(11) DEFAULT NULL,
        status varchar(20) DEFAULT 'pending',
        created_at datetime DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        KEY post_id (post_id),
        KEY scheduled_time (scheduled_time),
        KEY status (status)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);

    // Schedule cron event
    if (!wp_next_scheduled('post_scheduler_cron_hook')) {
        wp_schedule_event(time(), 'every_minute', 'post_scheduler_cron_hook');
    }
    
    // Set activation flag
    add_option('post_scheduler_activated', 'yes');
}
register_activation_hook(__FILE__, 'post_scheduler_activate');

/**
 * Deactivation hook
 */
function post_scheduler_deactivate() {
    $timestamp = wp_next_scheduled('post_scheduler_cron_hook');
    if ($timestamp) {
        wp_unschedule_event($timestamp, 'post_scheduler_cron_hook');
    }
    
    // Clear all scheduled events
    wp_clear_scheduled_hook('post_scheduler_cron_hook');
}
register_deactivation_hook(__FILE__, 'post_scheduler_deactivate');

/**
 * Add custom cron schedule
 */
function post_scheduler_cron_schedules($schedules) {
    if (!isset($schedules['every_minute'])) {
        $schedules['every_minute'] = array(
            'interval' => 60,
            'display' => esc_html__('Every Minute', 'simple-auto-post-scheduler')
        );
    }
    return $schedules;
}
add_filter('cron_schedules', 'post_scheduler_cron_schedules');

/**
 * Cron job to check and publish scheduled posts
 */
function post_scheduler_check_scheduled_posts() {
    global $wpdb;
    
    $table_name = $wpdb->prefix . 'post_scheduler';
    $current_time = current_time('mysql');

    // Get posts that need to be published
    $scheduled_posts = $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}post_scheduler 
        WHERE scheduled_time <= %s 
        AND status = %s 
        ORDER BY scheduled_time ASC 
        LIMIT 10",
        $current_time,
        'pending'
    ));

    if (empty($scheduled_posts)) {
        return;
    }

    foreach ($scheduled_posts as $scheduled_post) {
        // Get the post
        $post = get_post($scheduled_post->post_id);
        
        if (!$post) {
            // Post doesn't exist, mark as failed
            $wpdb->update(
                $table_name,
                array('status' => 'failed'),
                array('id' => $scheduled_post->id),
                array('%s'),
                array('%d')
            );
            continue;
        }
        
        // Only publish if post is draft
        if ($post->post_status === 'draft') {
            $result = wp_update_post(array(
                'ID' => $scheduled_post->post_id,
                'post_status' => 'publish',
                'post_date' => $scheduled_post->scheduled_time,
                'post_date_gmt' => get_gmt_from_date($scheduled_post->scheduled_time)
            ), true);

            if (is_wp_error($result)) {
                // Mark as failed
                $wpdb->update(
                    $table_name,
                    array('status' => 'failed'),
                    array('id' => $scheduled_post->id),
                    array('%s'),
                    array('%d')
                );
                continue;
            }

            // If recurring, schedule next occurrence
            if (!empty($scheduled_post->interval_type) && !empty($scheduled_post->interval_value)) {
                $next_time = post_scheduler_calculate_next_time(
                    $scheduled_post->scheduled_time,
                    $scheduled_post->interval_type,
                    $scheduled_post->interval_value
                );

                $wpdb->update(
                    $table_name,
                    array(
                        'scheduled_time' => $next_time,
                        'status' => 'pending'
                    ),
                    array('id' => $scheduled_post->id),
                    array('%s', '%s'),
                    array('%d')
                );
            } else {
                // Mark as completed if not recurring
                $wpdb->update(
                    $table_name,
                    array('status' => 'completed'),
                    array('id' => $scheduled_post->id),
                    array('%s'),
                    array('%d')
                );
            }
        } else {
            // Post is already published or in another state
            $wpdb->update(
                $table_name,
                array('status' => 'completed'),
                array('id' => $scheduled_post->id),
                array('%s'),
                array('%d')
            );
        }
    }
}
add_action('post_scheduler_cron_hook', 'post_scheduler_check_scheduled_posts');

/**
 * Enqueue admin styles and scripts
 */
function post_scheduler_enqueue_admin_assets($hook) {
    // Only load on our plugin pages
    if (strpos($hook, 'simple-auto-post-scheduler') === false) {
        return;
    }

    // Enqueue CSS
    $css_file = POST_SCHEDULER_PLUGIN_URL . 'assets/css/style.css';
    if (file_exists(POST_SCHEDULER_PLUGIN_DIR . 'assets/css/style.css')) {
        wp_enqueue_style(
            'simple-auto-post-scheduler-admin',
            $css_file,
            array(),
            POST_SCHEDULER_VERSION
        );
    }

    // Enqueue jQuery UI
    wp_enqueue_script('jquery');
    wp_enqueue_script('jquery-ui-datepicker');
}
add_action('admin_enqueue_scripts', 'post_scheduler_enqueue_admin_assets');

/**
 * Calculate next scheduled time based on interval
 */
function post_scheduler_calculate_next_time($current_time, $interval_type, $interval_value) {
    $timestamp = strtotime($current_time);
    
    switch ($interval_type) {
        case 'hours':
            $next_timestamp = strtotime("+{$interval_value} hours", $timestamp);
            break;
        case 'days':
            $next_timestamp = strtotime("+{$interval_value} days", $timestamp);
            break;
        case 'weeks':
            $next_timestamp = strtotime("+{$interval_value} weeks", $timestamp);
            break;
        case 'months':
            $next_timestamp = strtotime("+{$interval_value} months", $timestamp);
            break;
        default:
            $next_timestamp = $timestamp;
    }
    
    return gmdate('Y-m-d H:i:s', $next_timestamp);
}

/**
 * Plugin initialization
 */
