<?php
/**
 * Helper functions for Simple Auto Post Scheduler
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Get scheduled post by ID
 *
 * @param int $schedule_id Schedule ID
 * @return object|null Scheduled post object or null
 */
function post_scheduler_get_schedule($schedule_id) {
    global $wpdb;
    
    return $wpdb->get_row($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}post_scheduler WHERE id = %d",
        intval($schedule_id)
    ));
}

/**
 * Get all scheduled posts for a specific post ID
 *
 * @param int $post_id Post ID
 * @return array Array of scheduled post objects
 */
function post_scheduler_get_schedules_by_post($post_id) {
    global $wpdb;
    
    return $wpdb->get_results($wpdb->prepare(
        "SELECT * FROM {$wpdb->prefix}post_scheduler WHERE post_id = %d ORDER BY scheduled_time ASC",
        intval($post_id)
    ));
}

/**
 * Delete schedule by ID
 *
 * @param int $schedule_id Schedule ID
 * @return bool Success status
 */
function post_scheduler_delete_schedule($schedule_id) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'post_scheduler';
    
    return $wpdb->delete(
        $table_name,
        array('id' => intval($schedule_id)),
        array('%d')
    );
}

/**
 * Get count of pending scheduled posts
 *
 * @return int Count of pending posts
 */
function post_scheduler_get_pending_count() {
    global $wpdb;
    
    return (int) $wpdb->get_var(
        "SELECT COUNT(*) FROM {$wpdb->prefix}post_scheduler WHERE status = 'pending'"
    );
}

/**
 * Get upcoming scheduled posts (next 7 days)
 *
 * @param int $limit Number of posts to retrieve
 * @return array Array of scheduled post objects
 */
function post_scheduler_get_upcoming($limit = 10) {
    global $wpdb;
    $current_time = current_time('mysql');
    $week_later = date('Y-m-d H:i:s', strtotime('+7 days', strtotime($current_time)));
    
    return $wpdb->get_results($wpdb->prepare(
        "SELECT s.*, p.post_title 
        FROM {$wpdb->prefix}post_scheduler s 
        LEFT JOIN {$wpdb->posts} p ON s.post_id = p.ID 
        WHERE s.status = 'pending' 
        AND s.scheduled_time BETWEEN %s AND %s 
        ORDER BY s.scheduled_time ASC 
        LIMIT %d",
        $current_time,
        $week_later,
        intval($limit)
    ));
}

/**
 * Clean up completed schedules older than 30 days
 */
function post_scheduler_cleanup_old_schedules() {
    global $wpdb;
    $thirty_days_ago = gmdate('Y-m-d H:i:s', time() - (30 * 24 * 60 * 60));
    
    $wpdb->query($wpdb->prepare(
        "DELETE FROM {$wpdb->prefix}post_scheduler 
        WHERE status = 'completed' 
        AND scheduled_time < %s",
        $thirty_days_ago
    ));
}

/**
 * Add meta box to post editor
 */
function post_scheduler_add_meta_box() {
    add_meta_box(
        'post_scheduler_meta',
        esc_html__('Simple Auto Post Scheduler', 'simple-auto-post-scheduler'),
        'post_scheduler_meta_box_callback',
        'post',
        'side',
        'high'
    );
}
add_action('add_meta_boxes', 'post_scheduler_add_meta_box');

/**
 * Meta box callback
 */
function post_scheduler_meta_box_callback($post) {
    if (!$post || !isset($post->ID)) {
        return;
    }
    
    $schedules = post_scheduler_get_schedules_by_post($post->ID);
    
    ?>
    <div class="simple-auto-post-scheduler-meta-box">
        <?php if (empty($schedules)) : ?>
            <p><?php echo esc_html__('No schedules for this post.', 'simple-auto-post-scheduler'); ?></p>
        <?php else : ?>
            <ul class="simple-auto-post-scheduler-list">
                <?php foreach ($schedules as $schedule) : ?>
                    <li>
                        <strong><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($schedule->scheduled_time))); ?></strong>
                        <br>
                        <span class="status-<?php echo esc_attr($schedule->status); ?>">
                            <?php echo esc_html(ucfirst($schedule->status)); ?>
                        </span>
                        <?php if ($schedule->interval_type) : ?>
                            <br>
                            <em><?php 
                            // Translators: %1$d is the interval value, %2$s is the interval type
                            printf(
                                esc_html__('Repeats every %1$d %2$s', 'simple-auto-post-scheduler'),
                                esc_html($schedule->interval_value),
                                esc_html($schedule->interval_type)
                            ); 
                            ?></em>
                        <?php endif; ?>
                    </li>
                <?php endforeach; ?>
            </ul>
        <?php endif; ?>
        
        <p>
            <a href="<?php echo esc_url(admin_url('admin.php?page=simple-auto-post-scheduler-add')); ?>" class="button button-secondary">
                <?php echo esc_html__('Add Schedule', 'simple-auto-post-scheduler'); ?>
            </a>
        </p>
    </div>
    <?php
}

/**
 * Add dashboard widget
 */
function post_scheduler_dashboard_widget() {
    wp_add_dashboard_widget(
        'post_scheduler_widget',
        esc_html__('Upcoming Scheduled Posts', 'simple-auto-post-scheduler'),
        'post_scheduler_dashboard_widget_callback'
    );
}
add_action('wp_dashboard_setup', 'post_scheduler_dashboard_widget');

/**
 * Dashboard widget callback
 */
function post_scheduler_dashboard_widget_callback() {
    $upcoming = post_scheduler_get_upcoming(5);
    
    if (empty($upcoming)) {
        echo '<p>' . esc_html__('No upcoming scheduled posts.', 'simple-auto-post-scheduler') . '</p>';
        return;
    }
    
    echo '<ul>';
    foreach ($upcoming as $schedule) {
        echo '<li>';
        echo '<strong>' . esc_html($schedule->post_title ? $schedule->post_title : __('(No title)', 'simple-auto-post-scheduler')) . '</strong><br>';
        echo '<small>' . esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($schedule->scheduled_time))) . '</small>';
        echo '</li>';
    }
    echo '</ul>';
    
    echo '<p><a href="' . esc_url(admin_url('admin.php?page=simple-auto-post-scheduler')) . '">' . esc_html__('View All Schedules', 'simple-auto-post-scheduler') . '</a></p>';
}

/**
 * Weekly cleanup cron job
 */
function post_scheduler_weekly_cleanup() {
    post_scheduler_cleanup_old_schedules();
}
add_action('wp_scheduled_delete', 'post_scheduler_weekly_cleanup');

/**
 * Add admin notice for activation
 */
function post_scheduler_activation_notice() {
    $screen = get_current_screen();
    
    if (get_option('post_scheduler_activated') === 'yes') {
        ?>
        <div class="notice notice-success is-dismissible">
            <p><?php echo esc_html__('Simple Auto Simple Auto Post Scheduler plugin activated successfully!', 'simple-auto-post-scheduler'); ?> 
            <a href="<?php echo esc_url(admin_url('admin.php?page=simple-auto-post-scheduler')); ?>"><?php echo esc_html__('Get Started', 'simple-auto-post-scheduler'); ?></a></p>
        </div>
        <?php
        delete_option('post_scheduler_activated');
    }
}
add_action('admin_notices', 'post_scheduler_activation_notice');