<?php
/**
 * Admin functionality for Simple Auto Post Scheduler
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add admin menu
 */
function post_scheduler_admin_menu() {
    add_menu_page(
        esc_html__('Simple Auto Post Scheduler', 'simple-auto-post-scheduler'),
        esc_html__('Simple Auto Post Scheduler', 'simple-auto-post-scheduler'),
        'edit_posts',
        'simple-auto-post-scheduler',
        'post_scheduler_admin_page',
        'dashicons-clock',
        26
    );

    add_submenu_page(
        'simple-auto-post-scheduler',
        esc_html__('Add New Schedule', 'simple-auto-post-scheduler'),
        esc_html__('Add New', 'simple-auto-post-scheduler'),
        'edit_posts',
        'simple-auto-post-scheduler-add',
        'post_scheduler_add_page'
    );
}
add_action('admin_menu', 'post_scheduler_admin_menu');

/**
 * Enqueue admin scripts
 */
function post_scheduler_enqueue_admin_scripts($hook) {
    // Only load on our plugin pages
    if (strpos($hook, 'simple-auto-post-scheduler') === false) {
        return;
    }

    // Enqueue our admin JavaScript
    wp_enqueue_script(
        'simple-auto-post-scheduler-admin',
        POST_SCHEDULER_PLUGIN_URL . 'assets/js/admin-script.js',
        array('jquery', 'jquery-ui-datepicker'),
        POST_SCHEDULER_VERSION,
        true
    );
}
add_action('admin_enqueue_scripts', 'post_scheduler_enqueue_admin_scripts');

/**
 * Main admin page - List scheduled posts
 */
function post_scheduler_admin_page() {
    // Check user capabilities
    if (!current_user_can('edit_posts')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'simple-auto-post-scheduler'));
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'post_scheduler';

    // Handle delete action
    if (isset($_GET['action']) && $_GET['action'] === 'delete' && isset($_GET['id'])) {
        $schedule_id = intval($_GET['id']);
        
        // Verify nonce - FIXED: Added sanitize_text_field()
        if (!isset($_GET['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_GET['_wpnonce'])), 'post_scheduler_delete_' . $schedule_id)) {
            wp_die(esc_html__('Security check failed.', 'simple-auto-post-scheduler'));
        }
        
        $wpdb->delete($table_name, array('id' => $schedule_id), array('%d'));
        echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__('Schedule deleted successfully.', 'simple-auto-post-scheduler') . '</p></div>';
    }

    // Get all scheduled posts
    $scheduled_posts = $wpdb->get_results(
        "SELECT s.*, p.post_title, p.post_status 
        FROM {$wpdb->prefix}post_scheduler s 
        LEFT JOIN {$wpdb->posts} p ON s.post_id = p.ID 
        ORDER BY s.scheduled_time DESC"
    );

    ?>
    <div class="wrap">
        <h1 class="wp-heading-inline"><?php echo esc_html__('Scheduled Posts', 'simple-auto-post-scheduler'); ?></h1>
        <a href="<?php echo esc_url(admin_url('admin.php?page=simple-auto-post-scheduler-add')); ?>" class="page-title-action">
            <?php echo esc_html__('Add New', 'simple-auto-post-scheduler'); ?>
        </a>
        <hr class="wp-header-end">

        <?php if (empty($scheduled_posts)) : ?>
            <div class="simple-auto-post-scheduler-empty-state">
                <p><?php echo esc_html__('No scheduled posts found.', 'simple-auto-post-scheduler'); ?></p>
                <p><a href="<?php echo esc_url(admin_url('admin.php?page=simple-auto-post-scheduler-add')); ?>" class="button button-primary">
                    <?php echo esc_html__('Schedule Your First Post', 'simple-auto-post-scheduler'); ?>
                </a></p>
            </div>
        <?php else : ?>
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php echo esc_html__('Post Title', 'simple-auto-post-scheduler'); ?></th>
                        <th><?php echo esc_html__('Scheduled Time', 'simple-auto-post-scheduler'); ?></th>
                        <th><?php echo esc_html__('Interval', 'simple-auto-post-scheduler'); ?></th>
                        <th><?php echo esc_html__('Status', 'simple-auto-post-scheduler'); ?></th>
                        <th><?php echo esc_html__('Actions', 'simple-auto-post-scheduler'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($scheduled_posts as $schedule) : ?>
                        <tr>
                            <td>
                                <strong><?php echo esc_html($schedule->post_title ? $schedule->post_title : __('(No title)', 'simple-auto-post-scheduler')); ?></strong>
                                <br>
                                <small><?php echo esc_html__('ID:', 'simple-auto-post-scheduler') . ' ' . esc_html($schedule->post_id); ?></small>
                            </td>
                            <td><?php echo esc_html(date_i18n(get_option('date_format') . ' ' . get_option('time_format'), strtotime($schedule->scheduled_time))); ?></td>
                            <td>
                                <?php 
                                if ($schedule->interval_type && $schedule->interval_value) {
                                    // Translators: %1$d is the interval value, %2$s is the interval type
                                    printf(
                                        esc_html__('Every %1$d %2$s', 'simple-auto-post-scheduler'),
                                        esc_html($schedule->interval_value),
                                        esc_html($schedule->interval_type)
                                    );
                                } else {
                                    echo esc_html__('One-time', 'simple-auto-post-scheduler');
                                }
                                ?>
                            </td>
                            <td>
                                <span class="schedule-status-<?php echo esc_attr($schedule->status); ?>">
                                    <?php echo esc_html(ucfirst($schedule->status)); ?>
                                </span>
                            </td>
                            <td>
                                <a href="<?php echo esc_url(wp_nonce_url(admin_url('admin.php?page=simple-auto-post-scheduler&action=delete&id=' . $schedule->id), 'post_scheduler_delete_' . $schedule->id)); ?>" 
                                   class="button button-small" 
                                   onclick="return confirm('<?php echo esc_js(__('Are you sure you want to delete this schedule?', 'simple-auto-post-scheduler')); ?>');">
                                    <?php echo esc_html__('Delete', 'simple-auto-post-scheduler'); ?>
                                </a>
                            </td>
                        </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php endif; ?>
    </div>
    <?php
}

/**
 * Add new schedule page
 */
function post_scheduler_add_page() {
    // Check user capabilities
    if (!current_user_can('edit_posts')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'simple-auto-post-scheduler'));
    }
    
    global $wpdb;
    $table_name = $wpdb->prefix . 'post_scheduler';
    $message = '';
    $error = '';

    // Handle form submission
    if (isset($_POST['post_scheduler_submit'])) {
        // Verify nonce - FIXED: Added sanitize_text_field()
        if (!isset($_POST['_wpnonce']) || !wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['_wpnonce'])), 'post_scheduler_add_nonce')) {
            wp_die(esc_html__('Security check failed.', 'simple-auto-post-scheduler'));
        }

        $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
        $scheduled_date = isset($_POST['scheduled_date']) ? sanitize_text_field(wp_unslash($_POST['scheduled_date'])) : '';
        $scheduled_time = isset($_POST['scheduled_time']) ? sanitize_text_field(wp_unslash($_POST['scheduled_time'])) : '';
        $interval_type = isset($_POST['interval_type']) ? sanitize_text_field(wp_unslash($_POST['interval_type'])) : 'none';
        $interval_value = isset($_POST['interval_value']) ? intval($_POST['interval_value']) : 1;

        // Combine date and time
        $scheduled_datetime = $scheduled_date . ' ' . $scheduled_time . ':00';

        // Validate
        if (empty($post_id) || empty($scheduled_date) || empty($scheduled_time)) {
            $error = esc_html__('Please fill in all required fields.', 'simple-auto-post-scheduler');
        } else {
            // Validate datetime
            $timestamp = strtotime($scheduled_datetime);
            if ($timestamp === false) {
                $error = esc_html__('Invalid date or time format.', 'simple-auto-post-scheduler');
            } else {
                // Insert into database
                $result = $wpdb->insert(
                    $table_name,
                    array(
                        'post_id' => $post_id,
                        'scheduled_time' => $scheduled_datetime,
                        'interval_type' => $interval_type !== 'none' ? $interval_type : null,
                        'interval_value' => $interval_value > 0 ? $interval_value : null,
                        'status' => 'pending'
                    ),
                    array('%d', '%s', '%s', '%d', '%s')
                );

                if ($result) {
                    $message = esc_html__('Post scheduled successfully!', 'simple-auto-post-scheduler');
                } else {
                    $error = esc_html__('Failed to schedule post. Please try again.', 'simple-auto-post-scheduler');
                }
            }
        }
    }

    // Get draft posts
    $draft_posts = get_posts(array(
        'post_status' => 'draft',
        'numberposts' => -1,
        'post_type' => 'post',
        'orderby' => 'date',
        'order' => 'DESC'
    ));

    ?>
    <div class="wrap">
        <h1><?php echo esc_html__('Add New Schedule', 'simple-auto-post-scheduler'); ?></h1>

        <?php if ($message) : ?>
            <div class="notice notice-success is-dismissible"><p><?php echo esc_html($message); ?></p></div>
        <?php endif; ?>

        <?php if ($error) : ?>
            <div class="notice notice-error is-dismissible"><p><?php echo esc_html($error); ?></p></div>
        <?php endif; ?>

        <form method="post" action="" class="simple-auto-post-scheduler-form">
            <?php wp_nonce_field('post_scheduler_add_nonce'); ?>

            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="post_id"><?php echo esc_html__('Select Post', 'simple-auto-post-scheduler'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <select name="post_id" id="post_id" required class="regular-text">
                            <option value=""><?php echo esc_html__('-- Select a Draft Post --', 'simple-auto-post-scheduler'); ?></option>
                            <?php foreach ($draft_posts as $post) : ?>
                                <option value="<?php echo esc_attr($post->ID); ?>">
                                    <?php echo esc_html($post->post_title ? $post->post_title : '(No title)'); ?> (ID: <?php echo esc_html($post->ID); ?>)
                                </option>
                            <?php endforeach; ?>
                        </select>
                        <p class="description"><?php echo esc_html__('Select a draft post to schedule. Only draft posts are shown.', 'simple-auto-post-scheduler'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="scheduled_date"><?php echo esc_html__('Scheduled Date', 'simple-auto-post-scheduler'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <input type="text" name="scheduled_date" id="scheduled_date" required class="regular-text datepicker" 
                               placeholder="YYYY-MM-DD" autocomplete="off">
                        <p class="description"><?php echo esc_html__('Select the date when you want to publish the post.', 'simple-auto-post-scheduler'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="scheduled_time"><?php echo esc_html__('Scheduled Time', 'simple-auto-post-scheduler'); ?> <span class="required">*</span></label>
                    </th>
                    <td>
                        <input type="time" name="scheduled_time" id="scheduled_time" required class="regular-text" value="12:00">
                        <p class="description"><?php echo esc_html__('Select the time when you want to publish the post.', 'simple-auto-post-scheduler'); ?></p>
                    </td>
                </tr>

                <tr>
                    <th scope="row">
                        <label for="interval_type"><?php echo esc_html__('Repeat Interval', 'simple-auto-post-scheduler'); ?></label>
                    </th>
                    <td>
                        <select name="interval_type" id="interval_type" class="regular-text">
                            <option value="none"><?php echo esc_html__('One-time (No Repeat)', 'simple-auto-post-scheduler'); ?></option>
                            <option value="hours"><?php echo esc_html__('Hours', 'simple-auto-post-scheduler'); ?></option>
                            <option value="days"><?php echo esc_html__('Days', 'simple-auto-post-scheduler'); ?></option>
                            <option value="weeks"><?php echo esc_html__('Weeks', 'simple-auto-post-scheduler'); ?></option>
                            <option value="months"><?php echo esc_html__('Months', 'simple-auto-post-scheduler'); ?></option>
                        </select>
                        <p class="description"><?php echo esc_html__('Choose if you want this post to be republished at regular intervals.', 'simple-auto-post-scheduler'); ?></p>
                    </td>
                </tr>

                <tr id="interval_value_row" style="display: none;">
                    <th scope="row">
                        <label for="interval_value"><?php echo esc_html__('Interval Value', 'simple-auto-post-scheduler'); ?></label>
                    </th>
                    <td>
                        <input type="number" name="interval_value" id="interval_value" min="1" value="1" class="small-text">
                        <p class="description"><?php echo esc_html__('Enter the number of intervals (e.g., "2" for every 2 hours/days).', 'simple-auto-post-scheduler'); ?></p>
                    </td>
                </tr>
            </table>

            <p class="submit">
                <input type="submit" name="post_scheduler_submit" class="button button-primary" value="<?php echo esc_attr__('Schedule Post', 'simple-auto-post-scheduler'); ?>">
                <a href="<?php echo esc_url(admin_url('admin.php?page=simple-auto-post-scheduler')); ?>" class="button"><?php echo esc_html__('Cancel', 'simple-auto-post-scheduler'); ?></a>
            </p>
        </form>
    </div>
    <?php
}