<?php
/**
 * Plugin Name:       Simon's Simple Contact Form
 * Plugin URI:        https://simonward.net/simons-simple-contact-form/
 * Description:       Lightweight WordPress contact form plugin with 18 themes, SMTP support, Google reCAPTCHA or internal captcha, and instant theme switching. Add shortcode [sscfp_contact_form] to any post or page.
 * Version:           1.0.3
 * Author:            Simon Ward
 * Author URI:        https://simonward.net
 * License:           GPL-2.0-or-later
 * License URI:       https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain:       simons-simple-contact-form
 * Requires PHP:      7.4
 * Requires at least: 5.6
 */


defined('ABSPATH') or exit;

define('SSCFP_VERSION', '3.1.0');
define('SSCFP_DIR', plugin_dir_path(__FILE__));
define('SSCFP_URL', plugin_dir_url(__FILE__));
// Addons uploaded by admins must not be stored in the plugin directory.
// Store them in uploads so they survive plugin updates.
$sscfp_uploads = wp_upload_dir(null, false);
define('SSCFP_ADDONS_DIR', trailingslashit($sscfp_uploads['basedir'] . '/sscfp-addons'));
define('SSCFP_ADDONS_URL', trailingslashit($sscfp_uploads['baseurl'] . '/sscfp-addons'));

/**
 * Ensure addons directory exists.
 */
function sscfp_ensure_addons_dir() {
    if (!file_exists(SSCFP_ADDONS_DIR)) {
        wp_mkdir_p(SSCFP_ADDONS_DIR);
    }
}

register_activation_hook(__FILE__, 'sscfp_ensure_addons_dir');

/**
 * Include themes list + theme sanitizing.
 * (This is the ONLY PHP file you edit when adding new themes.)
 */
require_once SSCFP_DIR . 'scf-themes.php';

/**
 * One-time migration from legacy option names (scf_*) to the new, more unique
 * (sscfp_*) prefix.
 */
add_action('plugins_loaded', function () {
    if (get_option('sscfp_migrated', '0') === '1') {
        return;
    }

    $map = [
        'scf_theme'                => 'sscfp_theme',
        'scf_sender'               => 'sscfp_sender',
        'scf_reply_to'             => 'sscfp_reply_to',
        'scf_smtp'                 => 'sscfp_smtp',
        'scf_captcha_type'         => 'sscfp_captcha_type',
        'scf_recaptcha_site_key'   => 'sscfp_recaptcha_site_key',
        'scf_recaptcha_secret_key' => 'sscfp_recaptcha_secret_key',
    ];

    foreach ($map as $old => $new) {
        if (get_option($new, null) !== null) {
            continue;
        }
        $val = get_option($old, null);
        if ($val !== null) {
            update_option($new, $val);
        }
    }

    update_option('sscfp_migrated', '1');
});

/* --------------------------------------------------- Addons: Theme Packs --------------------------------------------------- */

/**
 * Read and validate all installed addons.
 *
 * Addon format:
 *  - /addons/<folder>/addon.json
 *
 * addon.json supports:
 *  {
 *    "addon_slug": "theme-pack-1",
 *    "addon_name": "Theme Pack 1",
 *    "addon_type": "themes",
 *    "version": "1.0.0",
 *    "themes": [{"slug":"aurora","label":"Aurora"}, ...],
 *    "assets": {"css": "themes.css"},
 *    "bootstrap": "bootstrap.php",
 *    "admin": {"file":"admin.php","render":"my_render_callback"}
 *  }
 */
function sscfp_get_addons_all(): array {
    static $cache = null;
    if ($cache !== null) return $cache;

    $cache = [];
    if (!is_dir(SSCFP_ADDONS_DIR)) return $cache;

    $dirs = glob(trailingslashit(SSCFP_ADDONS_DIR) . '*', GLOB_ONLYDIR);
    if (!is_array($dirs)) return $cache;

    foreach ($dirs as $dir) {
        $dir = (string)$dir;
        $folder = basename($dir);

        $json_path = trailingslashit($dir) . 'addon.json';
        if (!is_file($json_path)) {
            // Legacy extracted folder with no manifest
            $cache[] = [
                'folder'  => sanitize_key($folder),
                'slug'    => sanitize_key($folder),
                'name'    => ucwords(str_replace(['-', '_'], ' ', sanitize_key($folder))),
                'version' => '1.0.0',
                'type'    => 'unknown',
                'css'     => '',
                'themes'  => [],
                'bootstrap' => '',
                'admin_file' => '',
                'admin_render' => '',
            ];
            continue;
        }

        $raw = file_get_contents($json_path);
        if (!is_string($raw) || trim($raw) === '') continue;

        $data = json_decode($raw, true);
        if (!is_array($data)) continue;

        $addon_type = !empty($data['addon_type']) ? sanitize_key((string)$data['addon_type']) : 'unknown';

        $addon_slug = !empty($data['addon_slug']) ? sanitize_key((string)$data['addon_slug']) : sanitize_key($folder);
        $addon_name = !empty($data['addon_name']) ? sanitize_text_field((string)$data['addon_name']) : $addon_slug;
        $version    = !empty($data['version']) ? sanitize_text_field((string)$data['version']) : '1.0.0';

        // CSS asset (optional)
        $css_rel = '';
        if (isset($data['assets']['css'])) {
            $css_rel = (string)$data['assets']['css'];
        }
        $css_rel = ltrim(str_replace(['..', '\\'], ['', '/'], $css_rel), '/');
        if ($css_rel !== '') {
            $css_path = trailingslashit($dir) . $css_rel;
            if (!is_file($css_path)) $css_rel = '';
        }

        // Themes list (optional)
        $parsed_themes = [];
        $themes = isset($data['themes']) && is_array($data['themes']) ? $data['themes'] : [];
        foreach ($themes as $t) {
            if (!is_array($t) || empty($t['slug'])) continue;
            $slug  = sanitize_key((string)$t['slug']);
            if ($slug === '') continue;
            $label = !empty($t['label']) ? sanitize_text_field((string)$t['label']) : ucwords(str_replace(['-', '_'], ' ', $slug));
            $parsed_themes[$slug] = $label;
        }

        // Bootstrap (optional) - autoloaded by SCF
        $bootstrap = '';
        if (!empty($data['bootstrap'])) {
            $bootstrap = ltrim(str_replace(['..', '\\'], ['', '/'], (string)$data['bootstrap']), '/');
            if ($bootstrap !== '' && !is_file(trailingslashit($dir) . $bootstrap)) $bootstrap = '';
        }

        // Admin renderer (optional) - shown inside the addon tab
        $admin_file = '';
        $admin_render = '';
        if (isset($data['admin']) && is_array($data['admin'])) {
            if (!empty($data['admin']['file'])) {
                $admin_file = ltrim(str_replace(['..', '\\'], ['', '/'], (string)$data['admin']['file']), '/');
                if ($admin_file !== '' && !is_file(trailingslashit($dir) . $admin_file)) $admin_file = '';
            }
            if (!empty($data['admin']['render'])) {
                $admin_render = sanitize_text_field((string)$data['admin']['render']);
            }
        }

        $cache[] = [
            'folder'  => sanitize_key($folder),
            'slug'    => $addon_slug,
            'name'    => $addon_name,
            'version' => $version,
            'type'    => $addon_type,
            'css'     => $css_rel,
            'themes'  => $parsed_themes,
            'bootstrap' => $bootstrap,
            'admin_file' => $admin_file,
            'admin_render' => $admin_render,
        ];
    }

    return $cache;
}

/**
 * Autoload addon bootstrap files (optional).
 * This allows addons to register filters, enqueue inline CSS, add admin-post handlers, etc.
 */
add_action('plugins_loaded', function () {
    foreach (sscfp_get_addons_all() as $addon) {
        if (empty($addon['bootstrap'])) continue;

        $folder = (string)$addon['folder'];
        $file   = (string)$addon['bootstrap'];

        // Only allow .php
        if (substr($file, -4) !== '.php') continue;

        $path = trailingslashit(SSCFP_ADDONS_DIR) . $folder . '/' . ltrim($file, '/');
        if (is_file($path)) {
            include_once $path;
        }
    }
}, 9);


/**
 * Read and validate all installed addons that provide themes.
 *
 * Addon format:
 *  - /addons/<folder>/addon.json
 *  - /addons/<folder>/<css file>
 *
 * addon.json expects:
 *  {
 *    "addon_type": "themes",
 *    "themes": [{"slug":"aurora","label":"Aurora"}, ...],
 *    "assets": {"css": "themes-pack.css"}
 *  }
 */
function sscfp_get_theme_addons(): array {
    $out = [];
    foreach (sscfp_get_addons_all() as $addon) {
        if (empty($addon['type']) || $addon['type'] !== 'themes') continue;
        if (empty($addon['themes']) || !is_array($addon['themes'])) continue;
        $out[] = $addon;
    }
    return $out;
}

/** Merge addon themes into the core theme list */
add_filter('sscfp_themes_list', function ($themes) {
    $themes = is_array($themes) ? $themes : [];
    foreach (sscfp_get_theme_addons() as $addon) {
        if (empty($addon['themes']) || !is_array($addon['themes'])) continue;
        $themes = array_merge($themes, array_keys($addon['themes']));
    }
    return $themes;
});

/** Theme labels (used in admin UI) */
function sscfp_theme_labels(): array {
    static $labels = null;
    if ($labels !== null) return $labels;

    $labels = [];
    foreach (sscfp_themes_list() as $slug) {
        $labels[$slug] = ucwords(str_replace(['-', '_'], ' ', (string)$slug));
    }

    foreach (sscfp_get_theme_addons() as $addon) {
        foreach ((array)$addon['themes'] as $slug => $label) {
            $labels[$slug] = $label;
        }
    }

    $labels = apply_filters('sscfp_theme_labels', $labels);
    return $labels;
}

function sscfp_theme_label(string $slug): string {
    $labels = sscfp_theme_labels();
    return isset($labels[$slug]) ? (string)$labels[$slug] : ucwords(str_replace(['-', '_'], ' ', $slug));
}

/* --------------------------------------------------- Helpers --------------------------------------------------- */
function sscfp_sanitize_email_or_empty($value) {
    $value = trim((string)$value);
    if ($value === '') return '';
    $email = sanitize_email($value);
    return is_email($email) ? $email : '';
}

function sscfp_sanitize_smtp($value) {
    $old = get_option('sscfp_smtp', []);
    $value = is_array($value) ? $value : [];

    $new = [];
    $new['enabled'] = !empty($value['enabled']) ? 1 : 0;
    $new['host']    = sanitize_text_field(isset($value['host']) ? $value['host'] : '');
    $new['port']    = absint(isset($value['port']) ? $value['port'] : 0);
    $new['user']    = sanitize_text_field(isset($value['user']) ? $value['user'] : '');
    $new['secure']  = sanitize_text_field(isset($value['secure']) ? $value['secure'] : '');

    if (!in_array($new['secure'], ['', 'ssl', 'tls'], true)) {
        $new['secure'] = '';
    }

    // Only overwrite stored password if a new one is provided.
    $incoming_pass = isset($value['pass']) ? (string)$value['pass'] : '';
    if (trim($incoming_pass) !== '') {
        $new['pass'] = $incoming_pass;
    } else {
        $new['pass'] = isset($old['pass']) ? (string)$old['pass'] : '';
    }

    return $new;
}


function sscfp_admin_nonce_action() { 
    return 'sscfp_admin_action';
}

/* --------------------------------------------------- Enqueue CSS (Admin + Frontend) --------------------------------------------------- */
function sscfp_enqueue_shared_css() {
    wp_enqueue_style(
        'sscfp-styles',
        SSCFP_URL . 'css/scf-styles.css',
        [],
        SSCFP_VERSION
    );

    // Load theme pack addon CSS (if any)
    foreach (sscfp_get_theme_addons() as $addon) {
        if (empty($addon['css'])) continue;
        $folder = (string)$addon['folder'];
        $css    = (string)$addon['css'];
        $rel    = rawurlencode($folder) . '/' . ltrim($css, '/');

        $handle = 'sscfp-addon-' . (!empty($addon['slug']) ? $addon['slug'] : $folder);
        $ver    = (!empty($addon['version']) ? $addon['version'] : '1.0.0');

        wp_enqueue_style(
            $handle,
            trailingslashit(SSCFP_ADDONS_URL) . $rel,
            ['sscfp-styles'],
            $ver
        );
    }
}

// Frontend CSS (safe to load sitewide; tiny)
add_action('wp_enqueue_scripts', function () {
    sscfp_enqueue_shared_css();

    // Frontend JS (kept small; required for reCAPTCHA v3 token injection)
    wp_enqueue_script(
        'sscfp-frontend',
        SSCFP_URL . 'js/sscfp-frontend.js',
        [],
        SSCFP_VERSION,
        true
    );

    // Google reCAPTCHA (optional)
    $type = get_option('sscfp_captcha_type', 'internal');

    if ($type === 'recaptcha_v2') {
        wp_enqueue_script(
            'sscfp-recaptcha-v2',
            'https://www.google.com/recaptcha/api.js',
            [],
            SSCFP_VERSION,
            true
        );
    }

    if ($type === 'recaptcha_v3') {
        $site_key = trim((string)get_option('sscfp_recaptcha_site_key', ''));
        if ($site_key !== '') {
            wp_enqueue_script(
                'sscfp-recaptcha-v3',
                'https://www.google.com/recaptcha/api.js?render=' . rawurlencode($site_key),
                [],
                SSCFP_VERSION,
                true
            );

            // Provide site key to the frontend helper.
            wp_localize_script('sscfp-frontend', 'sscfpFront', [
                'recaptchaV3SiteKey' => $site_key,
            ]);
        }
    }

});

// Admin CSS only on our settings page
add_action('admin_enqueue_scripts', function ($hook) {
    if ($hook !== 'settings_page_sscfp-settings') return;
    sscfp_enqueue_shared_css();

    wp_enqueue_script(
        'sscfp-admin',
        SSCFP_URL . 'js/sscfp-admin.js',
        [],
        SSCFP_VERSION,
        true
    );

    wp_localize_script('sscfp-admin', 'sscfpAdmin', [
        'ajaxUrl'      => admin_url('admin-ajax.php'),
        'ajaxAction'   => 'sscfp_set_theme',
        'nonce'        => wp_create_nonce(sscfp_admin_nonce_action()),
        'failText'     => 'Theme switch failed.',
        'okTextPrefix' => 'Theme set to: ',
    ]);
});

/* --------------------------------------------------- Admin Settings --------------------------------------------------- */
add_action('admin_init', function () {
    register_setting('sscfp_settings', 'sscfp_theme', [
        'type'              => 'string',
        'sanitize_callback' => 'sscfp_sanitize_theme',
        'default'           => 'light',
    ]);

    register_setting('sscfp_settings', 'sscfp_sender', [
        'type'              => 'string',
        'sanitize_callback' => 'sscfp_sanitize_email_or_empty',
        'default'           => '',
    ]);

    register_setting('sscfp_settings', 'sscfp_reply_to', [
        'type'              => 'string',
        'sanitize_callback' => 'sscfp_sanitize_email_or_empty',
        'default'           => '',
    ]);

    register_setting('sscfp_settings', 'sscfp_smtp', [
        'type'              => 'array',
        'sanitize_callback' => 'sscfp_sanitize_smtp',
        'default'           => [],
    ]);

    // Captcha (Internal / Google reCAPTCHA v2 / v3)
    register_setting('sscfp_settings', 'sscfp_captcha_type', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => 'internal',
    ]);

    register_setting('sscfp_settings', 'sscfp_recaptcha_site_key', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => '',
    ]);

    register_setting('sscfp_settings', 'sscfp_recaptcha_secret_key', [
        'type'              => 'string',
        'sanitize_callback' => 'sanitize_text_field',
        'default'           => '',
    ]);


});

add_action('admin_menu', function () {
    add_options_page(
        "Simon's Simple Contact Form",
        "Simon's Simple Contact Form",
        'manage_options',
        'sscfp-settings',
        'sscfp_settings_page'
    );
});

/* --------------------------------------------------- Admin AJAX Theme Switch (1 click) --------------------------------------------------- */
add_action('wp_ajax_sscfp_set_theme', function () {
    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Forbidden'], 403);
    }

    // Verify nonce for this AJAX action.
    check_ajax_referer(sscfp_admin_nonce_action(), 'nonce');

    $theme_raw = isset($_POST['theme']) ? sanitize_text_field( wp_unslash( $_POST['theme'] ) ) : '';
    $theme     = $theme_raw !== '' ? sscfp_sanitize_theme($theme_raw) : 'light';
    update_option('sscfp_theme', $theme);

    wp_send_json_success(['theme' => $theme]);
});

/* --------------------------------------------------- Admin Addon Upload --------------------------------------------------- */
function sscfp_zip_is_safe($zip_path) {
    if (!class_exists('ZipArchive')) return true; // fall back to WordPress unzip protections

    $zip = new ZipArchive();
    if ($zip->open($zip_path) !== true) return false;

    for ($i = 0; $i < $zip->numFiles; $i++) {
        $name = (string)$zip->getNameIndex($i);

        // Basic zip-slip protection
        if ($name === '') { $zip->close(); return false; }
        if (strpos($name, '..') !== false) { $zip->close(); return false; }
        if ($name[0] === '/' || $name[0] === '\\') { $zip->close(); return false; }
        if (preg_match('#^[a-zA-Z]:#', $name)) { $zip->close(); return false; }
    }

    $zip->close();
    return true;
}

function sscfp_unique_addon_slug($filename) {
    $base = sanitize_title(pathinfo($filename, PATHINFO_FILENAME));
    if ($base === '') $base = 'addon';

    $slug = $base;
    $n = 2;
    while (file_exists(trailingslashit(SSCFP_ADDONS_DIR) . $slug)) {
        $slug = $base . '-' . $n;
        $n++;
    }
    return $slug;
}

add_action('admin_post_sscfp_upload_addon', function () {
    if (!current_user_can('manage_options')) {
        wp_die('Forbidden', 'Forbidden', ['response' => 403]);
    }

    check_admin_referer('sscfp_upload_addon', 'sscfp_upload_addon_nonce');
    sscfp_ensure_addons_dir();

    // Grab uploaded file (validated and sanitized below).
    // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- $_FILES is a special case; values are validated and sanitized below.
    $file = isset( $_FILES['sscfp_addon_zip'] ) && is_array( $_FILES['sscfp_addon_zip'] ) ? $_FILES['sscfp_addon_zip'] : null;
    if ( empty( $file ) ) {
        wp_safe_redirect( add_query_arg( 'sscfp_addon_upload', 'missing', admin_url( 'options-general.php?page=sscfp-settings' ) ) );
        exit;
    }

    // Sanitize filename to keep checks predictable.
    $file_name = isset($file['name']) ? sanitize_file_name(wp_unslash($file['name'])) : '';
    $file['name'] = $file_name;
    if (!empty($file['error'])) {
        wp_safe_redirect(add_query_arg('sscfp_addon_upload', 'error', admin_url('options-general.php?page=sscfp-settings')));
        exit;
    }

    $check = wp_check_filetype_and_ext($file['tmp_name'], $file['name'], ['zip' => 'application/zip']);
    if (empty($check['ext']) || $check['ext'] !== 'zip') {
        wp_safe_redirect(add_query_arg('sscfp_addon_upload', 'type', admin_url('options-general.php?page=sscfp-settings')));
        exit;
    }

    // Move upload into WP uploads dir
    require_once ABSPATH . 'wp-admin/includes/file.php';
    $overrides = [
        'test_form' => false,
        'mimes'     => ['zip' => 'application/zip'],
    ];
    $uploaded = wp_handle_upload($file, $overrides);
    if (!is_array($uploaded) || !empty($uploaded['error']) || empty($uploaded['file'])) {
        wp_safe_redirect(add_query_arg('sscfp_addon_upload', 'uploadfail', admin_url('options-general.php?page=sscfp-settings')));
        exit;
    }

    $zip_path = (string)$uploaded['file'];
    if (!sscfp_zip_is_safe($zip_path)) {
        wp_delete_file($zip_path);
        wp_safe_redirect(add_query_arg('sscfp_addon_upload', 'unsafe', admin_url('options-general.php?page=sscfp-settings')));
        exit;
    }

    $slug = sscfp_unique_addon_slug($file['name']);
    $dest = trailingslashit(SSCFP_ADDONS_DIR) . $slug;
    wp_mkdir_p($dest);

    // Unzip (load only the core helpers we immediately use).
    require_once ABSPATH . 'wp-admin/includes/file.php';
    WP_Filesystem();

    $unzipped = unzip_file($zip_path, $dest);
    wp_delete_file($zip_path);

    if (is_wp_error($unzipped)) {
        wp_safe_redirect(add_query_arg('sscfp_addon_upload', 'unzipfail', admin_url('options-general.php?page=sscfp-settings')));
        exit;
    }

    wp_safe_redirect(add_query_arg(['sscfp_addon_upload' => 'ok', 'sscfp_addon' => rawurlencode($slug)], admin_url('options-general.php?page=sscfp-settings')));
    exit;
});

/* --------------------------------------------------- Admin Page UI --------------------------------------------------- */
function sscfp_settings_page() {
    $current = get_option('sscfp_theme', 'light');
    $sender  = get_option('sscfp_sender', '');
    $replyto = get_option('sscfp_reply_to', '');

    $smtp = get_option('sscfp_smtp', []);
    $smtp_enabled = !empty($smtp['enabled']);
    $smtp_host    = isset($smtp['host']) ? (string)$smtp['host'] : '';
    $smtp_port    = isset($smtp['port']) ? (string)$smtp['port'] : '';
    $smtp_user    = isset($smtp['user']) ? (string)$smtp['user'] : '';
    $smtp_secure  = isset($smtp['secure']) ? (string)$smtp['secure'] : '';

    $nonce = wp_create_nonce(sscfp_admin_nonce_action());

    // Navigation only (view state). Sanitize and avoid direct superglobal access.
    $tab_raw = filter_input(INPUT_GET, 'tab', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
    $tab = $tab_raw ? sanitize_key((string)$tab_raw) : 'themes';
    if ($tab === '') $tab = 'themes';

    $addons = sscfp_get_addons_all();
    $addon_by_tab = [];

    $tabs = [
        'themes'   => 'Themes',
        'settings' => 'Settings',
        'addons'   => 'Addons',
    ];

    foreach ($addons as $a) {
        $key = 'addon-' . sanitize_key((string)$a['slug']);
        if ($key === 'addon-') continue;
        $tabs[$key] = (string)$a['name'];
        $addon_by_tab[$key] = $a;
    }

    if (!isset($tabs[$tab])) {
        $tab = 'themes';
    }
    ?>
    <div class="wrap">
        <h1>Simon's Simple Contact Form</h1>

        <?php
        // Addon upload notices
        $addon_upload_raw = filter_input(INPUT_GET, 'sscfp_addon_upload', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        // filter_input() returns null when the key is missing; don't show notices on fresh installs.
        if ($addon_upload_raw !== null && $addon_upload_raw !== false && $addon_upload_raw !== '') {
            // Status codes only.
            $code = sanitize_key((string)$addon_upload_raw);
            if ($code === 'ok') {
                $addon_raw = filter_input(INPUT_GET, 'sscfp_addon', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
                $addon = $addon_raw ? sanitize_text_field((string)$addon_raw) : '';
                echo '<div class="notice notice-success is-dismissible"><p>' . esc_html('Addon uploaded' . ($addon ? ' (' . $addon . ')' : '') . '.') . '</p></div>';
            } else {
                $map = [
                    'missing'    => 'No file was selected.',
                    'error'      => 'Upload error. Please try again.',
                    'type'       => 'Please upload a .zip file.',
                    'uploadfail' => 'Upload failed. Please try again.',
                    'unsafe'     => 'Zip was rejected for safety (invalid paths).',
                    'unzipfail'  => 'Could not extract the zip file.',
                ];
                $msg = isset($map[$code]) ? $map[$code] : 'Addon upload failed.';
                echo '<div class="notice notice-error is-dismissible"><p>' . esc_html($msg) . '</p></div>';
            }
        }

        // Generic "updated" notice for addon admin tools
        $notice_raw = filter_input(INPUT_GET, 'sscfp_notice', FILTER_SANITIZE_FULL_SPECIAL_CHARS);
        if ($notice_raw !== null && $notice_raw !== false && $notice_raw !== '' && sanitize_key((string)$notice_raw) === 'saved') {
            echo '<div class="notice notice-success is-dismissible"><p>' . esc_html('Saved.') . '</p></div>';
        }
        ?>

        <h2 class="nav-tab-wrapper">
            <?php foreach ($tabs as $key => $label):
                $url = add_query_arg(['page' => 'sscfp-settings', 'tab' => $key], admin_url('options-general.php'));
                $cls = 'nav-tab' . (($tab === $key) ? ' nav-tab-active' : '');
            ?>
                <a href="<?php echo esc_url($url); ?>" class="<?php echo esc_attr($cls); ?>"><?php echo esc_html($label); ?></a>
            <?php endforeach; ?>
        </h2>

        <?php if ($tab === 'themes'): ?>
            <p class="scf-admin-note">Click a theme to activate it instantly (AJAX). The thumbnails are template previews.</p>

            <?php
            $themes = sscfp_themes_list();
            ?>
            <div class="scf-theme-grid">
                <?php foreach ($themes as $theme): ?>
                    <button
                        type="button"
                        class="scf-theme-thumb <?php echo ($current === $theme) ? 'active' : ''; ?>"
                        data-theme="<?php echo esc_attr($theme); ?>"
                        aria-pressed="<?php echo ($current === $theme) ? 'true' : 'false'; ?>"
                    >
                        <div class="scf-thumb-preview scf-theme-<?php echo esc_attr($theme); ?>">
                            <div class="scf-thumb-title">Contact Us</div>

                            <div class="scf-thumb-row">
                                <span class="scf-thumb-pill">Name</span>
                                <span class="scf-thumb-pill">Email</span>
                            </div>

                            <div class="scf-thumb-row">
                                <span class="scf-thumb-line"></span>
                            </div>

                            <div class="scf-thumb-row">
                                <span class="scf-thumb-btn">Send</span>
                            </div>
                        </div>

                        <span class="scf-thumb-label"><?php echo esc_html(sscfp_theme_label((string)$theme)); ?></span>
                    </button>
                <?php endforeach; ?>
            </div>

            <div id="scf-theme-status" class="scf-theme-status"></div>

        <?php elseif ($tab === 'settings'): ?>

            <form method="post" action="options.php" style="margin-top:12px;">
                <?php settings_fields('sscfp_settings'); ?>

                <h2>Email Options</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="sscfp_sender">Sender (From)</label></th>
                        <td>
                            <input id="sscfp_sender" type="email" name="sscfp_sender" value="<?php echo esc_attr($sender); ?>" placeholder="no-reply@example.com" class="regular-text">
                            <p class="description">Optional. If blank, the plugin will use a safe default no-reply address for your domain.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sscfp_reply_to">Reply-To (Override)</label></th>
                        <td>
                            <input id="sscfp_reply_to" type="email" name="sscfp_reply_to" value="<?php echo esc_attr($replyto); ?>" placeholder="support@example.com" class="regular-text">
                            <p class="description">Optional. If blank, Reply-To will be set to the visitor’s email.</p>
                        </td>
                    </tr>
                </table>

                <h2>SMTP (Optional)</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row">Enable SMTP</th>
                        <td>
                            <label>
                                <input type="checkbox" name="sscfp_smtp[enabled]" value="1" <?php checked($smtp_enabled); ?>>
                                Use SMTP for sending
                            </label>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sscfp_smtp_host">Host</label></th>
                        <td><input id="sscfp_smtp_host" name="sscfp_smtp[host]" value="<?php echo esc_attr($smtp_host); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sscfp_smtp_port">Port</label></th>
                        <td><input id="sscfp_smtp_port" name="sscfp_smtp[port]" value="<?php echo esc_attr($smtp_port); ?>" class="small-text" placeholder="587"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sscfp_smtp_user">Username</label></th>
                        <td><input id="sscfp_smtp_user" name="sscfp_smtp[user]" value="<?php echo esc_attr($smtp_user); ?>" class="regular-text"></td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sscfp_smtp_pass">Password</label></th>
                        <td>
                            <input id="sscfp_smtp_pass" type="password" name="sscfp_smtp[pass]" value="" class="regular-text" autocomplete="new-password">
                            <p class="description">Leave blank to keep the existing saved password.</p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="sscfp_smtp_secure">Encryption</label></th>
                        <td>
                            <select id="sscfp_smtp_secure" name="sscfp_smtp[secure]">
                                <option value="" <?php selected($smtp_secure, ''); ?>>None</option>
                                <option value="ssl" <?php selected($smtp_secure, 'ssl'); ?>>SSL</option>
                                <option value="tls" <?php selected($smtp_secure, 'tls'); ?>>TLS</option>
                            </select>
                        </td>
                    </tr>
                </table>

                <h2>Captcha Settings</h2>
                <table class="form-table">
                    <tr>
                        <th scope="row"><label for="sscfp_captcha_type">Captcha Type</label></th>
                        <td>
                            <?php $sscfp_captcha_type = get_option('sscfp_captcha_type', 'internal'); ?>
                            <select id="sscfp_captcha_type" name="sscfp_captcha_type">
                                <option value="internal" <?php selected($sscfp_captcha_type, 'internal'); ?>>Internal Captcha</option>
                                <option value="recaptcha_v2" <?php selected($sscfp_captcha_type, 'recaptcha_v2'); ?>>Google reCAPTCHA v2</option>
                                <option value="recaptcha_v3" <?php selected($sscfp_captcha_type, 'recaptcha_v3'); ?>>Google reCAPTCHA v3</option>
                            </select>
                            <p class="description">If you select Google reCAPTCHA, it will override (disable) the internal captcha.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="sscfp_recaptcha_site_key">reCAPTCHA Site Key</label></th>
                        <td>
                            <input id="sscfp_recaptcha_site_key" type="text" name="sscfp_recaptcha_site_key" value="<?php echo esc_attr(get_option('sscfp_recaptcha_site_key', '')); ?>" class="regular-text">
                            <p class="description">Required for reCAPTCHA v2 and v3.</p>
                        </td>
                    </tr>

                    <tr>
                        <th scope="row"><label for="sscfp_recaptcha_secret_key">reCAPTCHA Secret Key</label></th>
                        <td>
                            <input id="sscfp_recaptcha_secret_key" type="text" name="sscfp_recaptcha_secret_key" value="<?php echo esc_attr(get_option('sscfp_recaptcha_secret_key', '')); ?>" class="regular-text">
                            <p class="description">Required for server-side verification.</p>
                        </td>
                    </tr>
                </table>

                <?php submit_button('Save Changes'); ?>
            </form>

        <?php elseif ($tab === 'addons'): ?>

            <p class="scf-admin-note">Upload .zip addon packages (theme packs, theme editor tools, etc).</p>

            <p style="margin: 10px 0 0;"><a class="button button-primary" href="https://simonward.net/product-category/plugin-addons/" target="_blank" rel="noopener noreferrer">Purchase Addons</a></p>

            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" enctype="multipart/form-data" style="margin: 12px 0 18px;">
                <input type="hidden" name="action" value="sscfp_upload_addon">
                <?php wp_nonce_field('sscfp_upload_addon', 'sscfp_upload_addon_nonce'); ?>
                <input type="file" name="sscfp_addon_zip" accept=".zip" required>
                <?php submit_button('Upload Addon', 'secondary', 'submit', false); ?>
            </form>

            <?php
            if (!empty($addons)) {
                echo '<h3 style="margin-top:10px;">Installed addons</h3>';
                echo '<table class="widefat striped" style="max-width: 900px;">';
                echo '<thead><tr><th>Name</th><th>Type</th><th>Version</th><th>Folder</th></tr></thead><tbody>';
                foreach ($addons as $a) {
                    echo '<tr>';
                    echo '<td>' . esc_html((string)$a['name']) . '</td>';
                    echo '<td>' . esc_html((string)$a['type']) . '</td>';
                    echo '<td>' . esc_html((string)$a['version']) . '</td>';
                    echo '<td>' . esc_html((string)$a['folder']) . '</td>';
                    echo '</tr>';
                }
                echo '</tbody></table>';
            } else {
                echo '<p class="description">No addons installed yet.</p>';
            }
            ?>

        <?php else: ?>

            <?php
            $addon = isset($addon_by_tab[$tab]) ? $addon_by_tab[$tab] : null;
            if (!$addon) {
                echo '<p>Addon not found.</p>';
            } else {
                $type = !empty($addon['type']) ? (string)$addon['type'] : 'unknown';
                ?>
                <h2 style="margin-top:16px;"><?php echo esc_html((string)$addon['name']); ?></h2>
                <p class="description">
                    <?php echo esc_html('Type: ' . $type . ' · Version: ' . (string)$addon['version'] . ' · Folder: ' . (string)$addon['folder']); ?>
                </p>

                <?php
                // Custom admin renderer (preferred)
                if (!empty($addon['admin_file']) && !empty($addon['admin_render'])) {
                    $admin_path = trailingslashit(SSCFP_ADDONS_DIR) . (string)$addon['folder'] . '/' . ltrim((string)$addon['admin_file'], '/');
                    if (is_file($admin_path)) {
                        include_once $admin_path;
                        $cb = (string)$addon['admin_render'];
                        if (function_exists($cb)) {
                            call_user_func($cb, $addon);
                        } else {
                            echo '<div class="notice notice-error"><p>Addon admin callback not found: ' . esc_html($cb) . '</p></div>';
                        }
                    } else {
                        echo '<div class="notice notice-error"><p>Addon admin file missing.</p></div>';
                    }
                } elseif ($type === 'themes' && !empty($addon['themes'])) {
                    // Theme pack listing
                    $pack_themes = array_keys((array)$addon['themes']);
                    echo '<h3 style="margin-top:16px;">Themes in this pack (' . esc_html((string)count($pack_themes)) . ')</h3>';
                    ?>
                    <div class="scf-theme-grid">
                        <?php foreach ($pack_themes as $theme): ?>
                            <button
                                type="button"
                                class="scf-theme-thumb <?php echo ($current === $theme) ? 'active' : ''; ?>"
                                data-theme="<?php echo esc_attr($theme); ?>"
                                aria-pressed="<?php echo ($current === $theme) ? 'true' : 'false'; ?>"
                            >
                                <div class="scf-thumb-preview scf-theme-<?php echo esc_attr($theme); ?>">
                                    <div class="scf-thumb-title">Contact Us</div>

                                    <div class="scf-thumb-row">
                                        <span class="scf-thumb-pill">Name</span>
                                        <span class="scf-thumb-pill">Email</span>
                                    </div>

                                    <div class="scf-thumb-row">
                                        <span class="scf-thumb-line"></span>
                                    </div>

                                    <div class="scf-thumb-row">
                                        <span class="scf-thumb-btn">Send</span>
                                    </div>
                                </div>

                                <span class="scf-thumb-label"><?php echo esc_html(sscfp_theme_label((string)$theme)); ?></span>
                            </button>
                        <?php endforeach; ?>
                    </div>
                    <div id="scf-theme-status" class="scf-theme-status"></div>
                    <?php
                } else {
                    echo '<p class="description">This addon has no admin UI.</p>';
                }
            }
            ?>

        <?php endif; ?>

    </div>

    <!-- Admin JS is enqueued via admin_enqueue_scripts. -->
    <?php
}

/* --------------------------------------------------- Shortcode --------------------------------------------------- */
function sscfp_render_contact_form_shortcode() {
    $theme = sscfp_sanitize_theme(get_option('sscfp_theme', 'light'));
    $captcha_type = get_option('sscfp_captcha_type', 'internal');
    $site_key     = trim((string)get_option('sscfp_recaptcha_site_key', ''));

    // Internal captcha only (Google reCAPTCHA overrides this)
    $a = $b = 0;
    $key = '';
    if ($captcha_type === 'internal') {
        $a = wp_rand(1, 9);
        $b = wp_rand(1, 9);
        $key = wp_generate_uuid4();
        set_transient('sscfp_captcha_' . $key, $a + $b, 10 * MINUTE_IN_SECONDS);
    }
    // UI feedback flags only. Sanitize and avoid direct superglobal access.
    $success = (bool) absint((string) filter_input(INPUT_GET, 'sscfp_success', FILTER_SANITIZE_NUMBER_INT));
    $error   = (bool) absint((string) filter_input(INPUT_GET, 'sscfp_error', FILTER_SANITIZE_NUMBER_INT));

    ob_start(); ?>
        <form class="scf-wrap scf-theme-<?php echo esc_attr($theme); ?>" method="post">
            <?php wp_nonce_field('sscfp_submit', 'sscfp_nonce'); ?>
            <?php if ($captcha_type === 'internal'): ?>
                <input type="hidden" name="sscfp_key" value="<?php echo esc_attr($key); ?>">
            <?php endif; ?>

            <label for="sscfp_name">Name</label>
            <input id="sscfp_name" type="text" name="sscfp_name" required>

            <label for="sscfp_email">Email</label>
            <input id="sscfp_email" type="email" name="sscfp_email" required>

            <label for="sscfp_subject">Subject</label>
            <input id="sscfp_subject" type="text" name="sscfp_subject" required>

            <label for="sscfp_message">Message</label>
            <textarea id="sscfp_message" name="sscfp_message" rows="5" required></textarea>
            <?php if ($captcha_type === 'internal'): ?>
                <label for="sscfp_captcha">Solve: <?php echo esc_html("$a + $b"); ?></label>
                <input
                  id="sscfp_captcha"
                  type="text"
                  name="sscfp_captcha"
                  inputmode="numeric"
                  pattern="[0-9]*"
                  autocomplete="off"
                  required
                >
            <?php elseif ($captcha_type === 'recaptcha_v2'): ?>
                <?php if ($site_key === ''): ?>
                    <div class="scf-msg scf-error">reCAPTCHA is enabled but not configured (missing Site Key).</div>
                <?php else: ?>
                    <div class="g-recaptcha" data-sitekey="<?php echo esc_attr($site_key); ?>"></div>
                <?php endif; ?>
            <?php elseif ($captcha_type === 'recaptcha_v3'): ?>
                <?php if ($site_key === ''): ?>
                    <div class="scf-msg scf-error">reCAPTCHA is enabled but not configured (missing Site Key).</div>
                <?php else: ?>
                    <input type="hidden" name="sscfp_recaptcha_token" id="sscfp_recaptcha_token">
                <?php endif; ?>
            <?php endif; ?>

            <button type="submit" name="sscfp_submit">Send Message</button>

            <?php if ($success): ?>
                <div class="scf-msg scf-success">Message sent successfully.</div>
            <?php endif; ?>

            <?php if ($error): ?>
                <div class="scf-msg scf-error">Captcha verification failed. Please try again.</div>
            <?php endif; ?>
        </form>
    <?php
    return ob_get_clean();
}

// New preferred shortcode.
add_shortcode('sscfp_contact_form', 'sscfp_render_contact_form_shortcode');

// Backwards compatibility for legacy content without registering a generic shortcode name.
add_filter('the_content', function ($content) {
    if (is_string($content) && strpos($content, '[simple_contact_form') !== false) {
        $content = str_replace('[simple_contact_form', '[sscfp_contact_form', $content);
    }
    return $content;
}, 9);

/* --------------------------------------------------- Optional SMTP --------------------------------------------------- */
add_action('phpmailer_init', function ($phpmailer) {
    $smtp = get_option('sscfp_smtp', []);
    if (empty($smtp['enabled']) || empty($smtp['host'])) return;

    $phpmailer->isSMTP();
    $phpmailer->Host = (string)$smtp['host'];

    if (!empty($smtp['port'])) {
        $phpmailer->Port = (int)$smtp['port'];
    }

    if (!empty($smtp['user'])) {
        $phpmailer->SMTPAuth = true;
        $phpmailer->Username = (string)$smtp['user'];
        $phpmailer->Password = (string)(isset($smtp['pass']) ? $smtp['pass'] : '');
    } else {
        $phpmailer->SMTPAuth = false;
    }

    if (in_array((isset($smtp['secure']) ? (string)$smtp['secure'] : ''), ['ssl', 'tls'], true)) {
        $phpmailer->SMTPSecure = $smtp['secure'];
    }
});

/* --------------------------------------------------- Submission Handler --------------------------------------------------- */

/* --------------------------------------------------- Google reCAPTCHA --------------------------------------------------- */
function sscfp_verify_recaptcha_request($token, $expected_action = "", $min_score = 0.0) {
    $secret = trim((string)get_option('sscfp_recaptcha_secret_key', ''));
    if ($secret === '' || $token === '') return false;

    $resp = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', [
        'timeout' => 10,
        'body'    => [
            'secret'   => $secret,
            'response' => $token,
            'remoteip' => isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '',
        ],
    ]);

    if (is_wp_error($resp)) return false;

    $data = json_decode((string)wp_remote_retrieve_body($resp), true);
    if (empty($data['success'])) return false;

    // v3 returns score + action; v2 usually doesn't (so we only enforce if provided)
    if ($expected_action !== '' && !empty($data['action']) && $data['action'] !== $expected_action) {
        return false;
    }
    if ($min_score > 0.0 && isset($data['score']) && (float)$data['score'] < $min_score) {
        return false;
    }

    return true;
}

add_action('init', function () {
    if (!isset($_POST['sscfp_submit'])) {
        return;
    }

    $nonce = isset($_POST['sscfp_nonce']) ? sanitize_text_field(wp_unslash($_POST['sscfp_nonce'])) : '';
    if ($nonce === '' || !wp_verify_nonce($nonce, 'sscfp_submit')) {
        wp_safe_redirect(add_query_arg('sscfp_error', 1, wp_get_referer()));
        exit;
    }

    $captcha_type = get_option('sscfp_captcha_type', 'internal');

    if ($captcha_type === 'internal') {
        $key = sanitize_text_field(isset($_POST['sscfp_key']) ? wp_unslash($_POST['sscfp_key']) : '');
        $answer = get_transient('sscfp_captcha_' . $key);
        $posted = isset( $_POST['sscfp_captcha'] ) ? absint( wp_unslash( $_POST['sscfp_captcha'] ) ) : -1;

        if ($answer === false || $posted !== (int)$answer) {
            wp_safe_redirect(add_query_arg('sscfp_error', 1, wp_get_referer()));
            exit;
        }

        delete_transient('sscfp_captcha_' . $key);
    } elseif ($captcha_type === 'recaptcha_v2') {
        $token = sanitize_text_field(isset($_POST['g-recaptcha-response']) ? wp_unslash($_POST['g-recaptcha-response']) : '');
        if (!sscfp_verify_recaptcha_request($token)) {
            wp_safe_redirect(add_query_arg('sscfp_error', 1, wp_get_referer()));
            exit;
        }
    } elseif ($captcha_type === 'recaptcha_v3') {
        $token = sanitize_text_field(isset($_POST['sscfp_recaptcha_token']) ? wp_unslash($_POST['sscfp_recaptcha_token']) : '');
        // Require action=submit and score >= 0.5 for v3.
        if (!sscfp_verify_recaptcha_request($token, 'submit', 0.5)) {
            wp_safe_redirect(add_query_arg('sscfp_error', 1, wp_get_referer()));
            exit;
        }
    }

    $name    = sanitize_text_field(isset($_POST['sscfp_name']) ? wp_unslash($_POST['sscfp_name']) : '');
    $email   = sanitize_email(isset($_POST['sscfp_email']) ? wp_unslash($_POST['sscfp_email']) : '');
    $subject = sanitize_text_field(isset($_POST['sscfp_subject']) ? wp_unslash($_POST['sscfp_subject']) : '');
    $message = sanitize_textarea_field(isset($_POST['sscfp_message']) ? wp_unslash($_POST['sscfp_message']) : '');

    if (!is_email($email)) {
        wp_safe_redirect(add_query_arg('sscfp_error', 1, wp_get_referer()));
        exit;
    }

    // Build safe default From.
    $domain = wp_parse_url(home_url(), PHP_URL_HOST);
    $fallback_from = $domain ? ('no-reply@' . $domain) : get_option('admin_email');

    $from_email = sscfp_sanitize_email_or_empty(get_option('sscfp_sender', '')) ?: $fallback_from;
    $reply_to   = sscfp_sanitize_email_or_empty(get_option('sscfp_reply_to', '')) ?: $email;

    $headers = [
        'From: ' . wp_strip_all_tags(get_bloginfo('name')) . ' <' . $from_email . '>',
    ];
    if ($reply_to) {
        $headers[] = 'Reply-To: ' . $reply_to;
    }

    $body  = "New contact form submission:\n\n";
    $body .= "Name: {$name}\n";
    $body .= "Email: {$email}\n";
    $body .= "Subject: {$subject}\n";
    $body .= "Page: " . wp_get_referer() . "\n";
    $body .= "IP: " . (isset($_SERVER['REMOTE_ADDR']) ? sanitize_text_field(wp_unslash($_SERVER['REMOTE_ADDR'])) : '') . "\n\n";
    $body .= "Message:\n{$message}\n";

    wp_mail(get_option('admin_email'), $subject, $body, $headers);

    wp_safe_redirect(add_query_arg('sscfp_success', 1, wp_get_referer()));
    exit;
});