<?php
/*
Plugin Name: URL Short tool by Shorterm – Simple, Fast & Private
Plugin URI: https://shorterm.eu/
Description: A simple URL shortener plugin that creates short links with custom slugs and click tracking.
Version: 1.1.1
Author: Sevis Dimitris
Author URI: https://demedia.gr
License: GPLv2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
Text Domain: shorterm
*/

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Database Version (for upgrades).
define( 'SHORTERM_DB_VERSION', '1.0' );

/**
 * Creates the custom tables needed by the plugin upon activation.
 */
function shorterm_create_table() {
	global $wpdb;
	$links_table_name  = $wpdb->prefix . 'shorterm_links';
	$clicks_table_name = $wpdb->prefix . 'shorterm_link_clicks';
	$charset_collate   = $wpdb->get_charset_collate();

	$sql_links = "CREATE TABLE $links_table_name (
        id INT NOT NULL AUTO_INCREMENT,
        original_url VARCHAR(2048) NOT NULL,
        custom_slug VARCHAR(100) NOT NULL,
        short_url VARCHAR(255) NOT NULL,
        click_count INT DEFAULT 0,
        expiration_date DATETIME NULL,
        password VARCHAR(255) NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        PRIMARY KEY (id),
        UNIQUE (custom_slug)
    ) $charset_collate;";

	$sql_clicks = "CREATE TABLE $clicks_table_name (
        id BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
        link_id INT NOT NULL,
        click_time DATETIME NOT NULL,
        PRIMARY KEY (id),
        FOREIGN KEY (link_id) REFERENCES $links_table_name(id) ON DELETE CASCADE
    ) $charset_collate;";

	require_once ABSPATH . 'wp-admin/includes/upgrade.php';
	dbDelta( $sql_links );
	dbDelta( $sql_clicks );

	update_option( 'shorterm_db_version', SHORTERM_DB_VERSION );
}
register_activation_hook( __FILE__, 'shorterm_create_table' );

/**
 * Checks if the database needs to be updated.
 */
function shorterm_update_db_check() {
	if ( get_site_option( 'shorterm_db_version' ) !== SHORTERM_DB_VERSION ) {
		shorterm_create_table();
	}
}
add_action( 'plugins_loaded', 'shorterm_update_db_check' );

/**
 * Generates a unique random slug.
 */
function shorterm_generate_random_slug( $length = 6 ) {
	global $wpdb;
	$characters        = '0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ';
	$characters_length = strlen( $characters );

	do {
		$random_slug = '';
		for ( $i = 0; $i < $length; $i++ ) {
			$random_slug .= $characters[ wp_rand( 0, $characters_length - 1 ) ];
		}
		
		// Fixed: Putting prepare inside get_var to satisfy NotPrepared check
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM " . $wpdb->prefix . "shorterm_links WHERE custom_slug = %s", $random_slug ) );
	} while ( $exists );

	return $random_slug;
}

/**
 * Shortens a URL and stores it in the database.
 */
function shorterm_generate_link( $original_url, $custom_slug = '' ) {
	global $wpdb;

	if ( ! filter_var( $original_url, FILTER_VALIDATE_URL ) ) {
		return 'Invalid Original URL format.';
	}

	if ( ! empty( $custom_slug ) ) {
		$custom_slug = sanitize_title( $custom_slug );
		// Fixed: Putting prepare inside get_var
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$exists = $wpdb->get_var( $wpdb->prepare( "SELECT id FROM " . $wpdb->prefix . "shorterm_links WHERE custom_slug = %s", $custom_slug ) );
		
		if ( $exists ) {
			return 'Error: The custom slug "' . esc_html( $custom_slug ) . '" is already in use.';
		}
	} else {
		$custom_slug = shorterm_generate_random_slug();
	}

	$short_url = site_url( '/' . $custom_slug );

	$data_to_insert = array(
		'original_url' => $original_url,
		'custom_slug'  => $custom_slug,
		'short_url'    => $short_url,
	);

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
	$result = $wpdb->insert( $wpdb->prefix . 'shorterm_links', $data_to_insert, array( '%s', '%s', '%s' ) );

	if ( false === $result ) {
		return 'Database error. Could not create short link.';
	}

	delete_transient( 'shorterm_all_links' );
	wp_cache_delete( 'shorterm_all_links', 'shorterm' );

	return $short_url;
}

/**
 * Handles the redirection when a short URL is visited.
 */
function shorterm_redirect_link() {
	global $wpdb;

	if ( ! isset( $_SERVER['REQUEST_URI'] ) ) {
		return;
	}

	$request_uri = esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) );
	$path        = wp_parse_url( $request_uri, PHP_URL_PATH );
	if ( ! $path ) {
		return;
	}

	$custom_slug = ltrim( $path, '/' );
	if ( strpos( $custom_slug, 'wp-' ) === 0 || empty( $custom_slug ) || is_admin() ) {
		return;
	}

	$cache_key = 'shorterm_link_' . md5( $custom_slug );
	
	$link = wp_cache_get( $cache_key, 'shorterm' );
	if ( false === $link ) {
		$link = get_transient( $cache_key );
	}

	if ( false === $link ) {
		// Fixed: Putting prepare inside get_row
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$link = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM " . $wpdb->prefix . "shorterm_links WHERE custom_slug = %s", $custom_slug ) );

		if ( $link ) {
			set_transient( $cache_key, $link, HOUR_IN_SECONDS );
			wp_cache_set( $cache_key, $link, 'shorterm', HOUR_IN_SECONDS );
		}
	}

	if ( $link ) {
		if ( $link->expiration_date && strtotime( $link->expiration_date ) < time() ) {
			wp_die( esc_html__( 'This link has expired.', 'shorterm' ), esc_html__( 'Link Expired', 'shorterm' ), array( 'response' => 410 ) );
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->insert( $wpdb->prefix . 'shorterm_link_clicks', array( 'link_id' => $link->id, 'click_time' => current_time( 'mysql', 1 ) ), array( '%d', '%s' ) );

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$wpdb->update( $wpdb->prefix . 'shorterm_links', array( 'click_count' => $link->click_count + 1 ), array( 'id' => $link->id ), array( '%d' ), array( '%d' ) );

		delete_transient( $cache_key );
		wp_cache_delete( $cache_key, 'shorterm' );
		delete_transient( 'shorterm_all_links' );

		// phpcs:ignore WordPress.Security.SafeRedirect.wp_redirect_wp_redirect
		wp_redirect( $link->original_url, 301 );
		exit;
	}
}
add_action( 'init', 'shorterm_redirect_link', 1 );

/**
 * Retrieves all links from the database for the admin table.
 */
function shorterm_get_links_table() {
	global $wpdb;
	
	$links = wp_cache_get( 'shorterm_all_links', 'shorterm' );
	if ( false === $links ) {
		$links = get_transient( 'shorterm_all_links' );
	}

	if ( false === $links ) {
		// Fixed: Using the string directly to satisfy UnescapedDBParameter/NotPrepared
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$links = $wpdb->get_results( "SELECT * FROM " . $wpdb->prefix . "shorterm_links ORDER BY created_at DESC" );
		
		set_transient( 'shorterm_all_links', $links, 5 * MINUTE_IN_SECONDS );
		wp_cache_set( 'shorterm_all_links', $links, 'shorterm', 5 * MINUTE_IN_SECONDS );
	}

	ob_start();
	if ( $links ) {
		foreach ( $links as $link ) {
			echo '<tr id="link-' . esc_attr( $link->id ) . '">';
			echo '<td><input type="checkbox" class="shorterm-bulk-checkbox pro-feature-control"></td>';
			echo '<td class="short-url-cell"><a href="' . esc_url( $link->short_url ) . '" target="_blank" class="short-url-link">' . esc_html( urldecode( $link->short_url ) ) . '</a></td>';
			echo '<td class="original-url-cell" title="' . esc_attr( $link->original_url ) . '"><span class="original-url-text">' . esc_html( $link->original_url ) . '</span></td>';
			echo '<td>' . esc_html( get_date_from_gmt( $link->created_at, 'Y-m-d H:i:s' ) ) . '</td>';
			echo '<td class="action-cell">';
			echo '<button class="copy-link button" data-link="' . esc_url( $link->short_url ) . '"><i class="dashicons dashicons-admin-page"></i> Copy</button>';
			echo '<button class="button pro-feature-button"><i class="dashicons dashicons-edit"></i> Edit</button>';
			echo '<button class="delete-link button" data-id="' . esc_attr( $link->id ) . '" data-nonce="' . esc_attr( wp_create_nonce( 'shorterm_delete_link_' . $link->id ) ) . '"><i class="dashicons dashicons-trash"></i> Delete</button>';
			echo '</td>';
			echo '</tr>';
		}
	} else {
		echo '<tr><td colspan="5" style="text-align:center; padding: 20px;">No short links found. Create one above!</td></tr>';
	}
	return ob_get_clean();
}

/**
 * AJAX handler to create a new short link.
 */
function shorterm_create_link() {
	check_ajax_referer( 'shorterm_create_link_nonce' );

	if ( ! current_user_can( 'manage_options' ) ) {
		wp_send_json_error( 'You do not have permission to perform this action.' );
	}

	$original_url = isset( $_POST['original_url'] ) ? esc_url_raw( wp_unslash( $_POST['original_url'] ) ) : '';
	$custom_slug  = isset( $_POST['custom_slug'] ) ? sanitize_title( wp_unslash( $_POST['custom_slug'] ) ) : '';

	if ( empty( $original_url ) ) {
		wp_send_json_error( 'Original URL is required.' );
	}

	$short_url = shorterm_generate_link( $original_url, $custom_slug );

	if ( strpos( $short_url, 'http' ) === 0 ) {
		wp_send_json_success( array( 'short_url' => esc_url( $short_url ) ) );
	} else {
		wp_send_json_error( $short_url );
	}
}
add_action( 'wp_ajax_shorterm_create_link', 'shorterm_create_link' );

/**
 * AJAX handler to refresh the links table.
 */
function shorterm_refresh_table() {
	check_ajax_referer( 'shorterm_refresh_table_nonce' );
	if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error(); }
	wp_send_json_success( shorterm_get_links_table() );
}
add_action( 'wp_ajax_shorterm_refresh_table', 'shorterm_refresh_table' );

/**
 * AJAX handler to delete a short link.
 */
function shorterm_delete_link() {
	if ( ! isset( $_POST['id'] ) ) { wp_send_json_error(); }
	$id = intval( $_POST['id'] );
	check_ajax_referer( 'shorterm_delete_link_' . $id );

	if ( ! current_user_can( 'manage_options' ) ) { wp_send_json_error(); }

	global $wpdb;
	
	// Fixed: Putting prepare inside get_var
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$slug_to_delete = $wpdb->get_var( $wpdb->prepare( "SELECT custom_slug FROM " . $wpdb->prefix . "shorterm_links WHERE id = %d", $id ) );
	
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
	$wpdb->delete( $wpdb->prefix . 'shorterm_links', array( 'id' => $id ), array( '%d' ) );

	if ( $slug_to_delete ) { 
		$md5_slug = md5( $slug_to_delete );
		delete_transient( 'shorterm_link_' . $md5_slug );
		wp_cache_delete( 'shorterm_link_' . $md5_slug, 'shorterm' );
	}
	delete_transient( 'shorterm_all_links' );
	wp_cache_delete( 'shorterm_all_links', 'shorterm' );

	wp_send_json_success();
}
add_action( 'wp_ajax_shorterm_delete_link', 'shorterm_delete_link' );

/**
 * Adds the admin menu page.
 */
function shorterm_menu() {
	add_menu_page( 'Shorterm Dashboard', 'Shorterm', 'manage_options', 'shorterm-dashboard', 'shorterm_dashboard', 'dashicons-admin-links', 30 );
}
add_action( 'admin_menu', 'shorterm_menu' );

/**
 * Displays the main dashboard page.
 */
function shorterm_dashboard() {
	?>
	<div id="shorterm-pro-modal" style="display: none;">
		<div class="shorterm-modal-overlay"></div>
		<div class="shorterm-modal-content">
			<h3>Shorterm Pro Feature</h3>
			<p>This feature is available in the Pro version of Shorterm. Upgrade now to unlock:</p>
			<ul>
				<li>✅ Detailed Click Analytics</li>
				<li>✅ Link Expiration & Scheduling</li>
				<li>✅ Password Protection</li>
				<li>✅ Advanced Filtering & Export</li>
			</ul>
			<a href="https://shorterm.eu/" target="_blank" class="button button-primary">Buy Shorterm Pro</a>
			<button id="shorterm-close-modal" class="button button-secondary">Close</button>
		</div>
	</div>
	<a href="https://shorterm.eu/" target="_blank" class="shorterm-buy-pro-floating-button">🚀 Upgrade to Pro</a>
	<div class="shorterm-full-bg">
		 <div class="shorterm-app-container">
			<h1 class="shorterm-app-title">Shorterm Link Manager 🔗</h1>
			<div class="shorterm-form-wrapper">
				<form method="POST" action="" class="shorterm-form" id="shorterm-form">
					<h2 class="form-title">Create New Short Link</h2>
					<div class="form-group">
						<label for="original_url">Original URL <span class="label-hint">(Target URL)</span></label>
					   <input type="url" id="original_url" name="original_url" class="form-control" required placeholder="https://example.com/long-url"/>
					</div>
					<div class="form-group">
						<label for="custom_slug">Custom Slug <span class="label-hint">(Optional)</span></label>
						<input type="text" id="custom_slug" name="custom_slug" class="form-control" placeholder="e.g. my-awesome-link"/>
					</div>
				   <div class="form-group pro-feature-gated">
						 <label for="expiration_date">Expiration Date <span class="pro-tag">PRO</span></label>
					   <input type="date" id="expiration_date" class="form-control pro-feature-control" disabled/>
					 </div>
					 <div class="form-group pro-feature-gated">
						 <label for="password">Password <span class="pro-tag">PRO</span></label>
					   <input type="password" id="password" class="form-control pro-feature-control" disabled/>
				   </div>
					<div class="form-actions">
					   <input type="submit" name="submit" class="button button-primary" value="Generate Link">
					   <button type="button" class="button button-secondary pro-feature-button">Export Selected</button>
					   <button type="button" class="button button-danger pro-feature-button">Delete Selected</button>
				   </div>
				   <div id="shorterm-messages"></div>
			   </form>
			</div>
			<div class="links-section-header">
				 <h2 class="section-title">Generated Links</h2>
			</div>
		   <div class="links-section">
			  <div class="table-responsive">
				  <table class="shorterm-table">
						<thead>
						 <tr>
							<th scope="col" class="check-column"><input type="checkbox" id="shorterm-select-all" class="pro-feature-control"></th>
							<th scope="col">Short URL</th>
							 <th scope="col">Original URL</th>
							 <th scope="col">Date Created</th>
							 <th scope="col" class="action-header">Actions</th>
						</tr>
					   </thead>
					   <tbody id="shorterm-links-body">
						  <?php echo wp_kses_post( shorterm_get_links_table() ); ?>
					  </tbody>
				   </table>
			   </div>
			</div>
		 </div>
   </div>
	<?php
}

/**
 * Enqueues assets.
 */
function shorterm_enqueue_admin_assets( $hook_suffix ) {
	if ( 'toplevel_page_shorterm-dashboard' !== $hook_suffix ) { return; }
	$version = '1.1.1';
	wp_enqueue_style( 'shorterm-admin-styles', plugin_dir_url( __FILE__ ) . 'assets/css/admin-styles.css', array(), $version );
	wp_enqueue_script( 'shorterm-admin-scripts', plugin_dir_url( __FILE__ ) . 'assets/js/admin-scripts.js', array( 'jquery' ), $version, true );
	wp_localize_script( 'shorterm-admin-scripts', 'shorterm_ajax_object', array(
		'ajax_url'            => admin_url( 'admin-ajax.php' ),
		'create_link_nonce'   => wp_create_nonce( 'shorterm_create_link_nonce' ),
		'refresh_table_nonce' => wp_create_nonce( 'shorterm_refresh_table_nonce' ),
	));
}
add_action( 'admin_enqueue_scripts', 'shorterm_enqueue_admin_assets' );