<?php
/**
 * Admin Settings Class
 *
 * Handles the admin settings page for Shortcode Arcade Crypto Idle Game.
 * Manages cloud saves, leaderboard configuration, and admin UI.
 *
 * @package Shortcode_Arcade_Crypto_Idle_Game
 * @since 0.4.6
 */

// Exit if accessed directly
if (!defined('ABSPATH')) {
    exit;
}

class SACIG_Admin {
    
    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));
    }
    
    /**
     * Add admin menu page
     */
    public function add_admin_menu() {
        add_options_page(
            __( 'Shortcode Arcade Crypto Idle Game Settings', 'shortcodearcade-crypto-idle-game' ),
            __( 'Shortcode Arcade Crypto Idle Game', 'shortcodearcade-crypto-idle-game' ),
            'manage_options',
            'shortcodearcade-crypto-idle-game',
            array($this, 'render_settings_page')
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        // Register settings group
        register_setting('sacig_settings_group', 'sacig_enable_cloud_saves', array(
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => array($this, 'sanitize_checkbox')
        ));
        
        register_setting('sacig_settings_group', 'sacig_enable_leaderboard', array(
            'type' => 'boolean',
            'default' => false,
            'sanitize_callback' => array($this, 'sanitize_checkbox')
        ));
        
        register_setting('sacig_settings_group', 'sacig_leaderboard_limit', array(
            'type' => 'integer',
            'default' => 10,
            'sanitize_callback' => array($this, 'sanitize_leaderboard_limit')
        ));
        
        // Add settings section
        add_settings_section(
            'sacig_main_section',
            'Game Settings',
            array($this, 'render_section_description'),
            'shortcodearcade-crypto-idle-game'
        );
        
        // Add settings fields
        add_settings_field(
            'sacig_enable_cloud_saves',
            'Enable Cloud Saves',
            array($this, 'render_cloud_saves_field'),
            'shortcodearcade-crypto-idle-game',
            'sacig_main_section'
        );
        
        add_settings_field(
            'sacig_enable_leaderboard',
            'Enable Leaderboard',
            array($this, 'render_leaderboard_field'),
            'shortcodearcade-crypto-idle-game',
            'sacig_main_section'
        );
        
        add_settings_field(
            'sacig_leaderboard_limit',
            'Leaderboard Size',
            array($this, 'render_leaderboard_limit_field'),
            'shortcodearcade-crypto-idle-game',
            'sacig_main_section'
        );
    }
    
    /**
     * Sanitize checkbox
     */
    public function sanitize_checkbox($input) {
        return (bool) $input;
    }
    
    /**
     * Sanitize leaderboard limit
     */
    public function sanitize_leaderboard_limit($input) {
        $value = intval($input);
        return max(5, min(100, $value)); // Between 5 and 100
    }
    
    /**
     * Render section description
     */
    public function render_section_description() {
        echo '<p>' . esc_html__( 'Configure cloud saves and leaderboard features for Shortcode Arcade Crypto Idle Game.', 'shortcodearcade-crypto-idle-game' ) . '</p>';
    }
    
    /**
     * Render cloud saves field
     */
    public function render_cloud_saves_field() {
        $value = get_option('sacig_enable_cloud_saves', false);
        ?>
        <label>
            <input type="checkbox" name="sacig_enable_cloud_saves" value="1" <?php checked($value, true); ?>>
            Save game progress to WordPress user accounts
        </label>
        <p class="description">
            <strong>Requires:</strong> Users must be logged in to play. Game saves will be stored in your WordPress database.
        </p>
        <?php
    }
    
    /**
     * Render leaderboard field
     */
    public function render_leaderboard_field() {
        $cloud_enabled = get_option('sacig_enable_cloud_saves', false);
        $value = get_option('sacig_enable_leaderboard', false);
        $disabled = !$cloud_enabled;
        ?>
        <label>
            <input type="checkbox" name="sacig_enable_leaderboard" value="1" 
                <?php checked($value, true); ?> 
                <?php disabled($disabled); ?>>
            Display leaderboard on your site
        </label>
        <p class="description">
            <?php if ($disabled): ?>
                <span class="sacig-warning">⚠️ Cloud Saves must be enabled first</span><br>
            <?php endif; ?>
            Use shortcode: <code>[sacig_crypto_idle_leaderboard]</code>
        </p>
        <?php
    }
    
    /**
     * Render leaderboard limit field
     */
    public function render_leaderboard_limit_field() {
        $value = get_option('sacig_leaderboard_limit', 10);
        ?>
        <input type="number" name="sacig_leaderboard_limit" value="<?php echo esc_attr($value); ?>" 
            min="5" max="100" step="1">
        <p class="description">Number of top players to display (5-100)</p>
        <?php
    }
    
    /**
     * Render settings page
     */
    public function render_settings_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // Get active tab
        $active_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'general'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
        
        // Check if settings were saved
        if (isset($_GET['settings-updated'])) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
            // Check if database table needs to be created
            $cloud_enabled = get_option('sacig_enable_cloud_saves', false);
            if ($cloud_enabled) {
                $this->maybe_create_table();
            }
            
            add_settings_error(
                'sacig_messages',
                'sacig_message',
                __( 'Settings Saved', 'shortcodearcade-crypto-idle-game' ),
                'updated'
            );
        }
        
        settings_errors('sacig_messages');
        ?>
        <div class="wrap sacig-admin-wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            
            <!-- Tab Navigation -->
            <nav class="nav-tab-wrapper sacig-tab-wrapper">
                <a href="?page=shortcodearcade-crypto-idle-game&tab=general" 
                   class="nav-tab <?php echo $active_tab === 'general' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-admin-generic"></span> General
                </a>
                <a href="?page=shortcodearcade-crypto-idle-game&tab=branding" 
                   class="nav-tab <?php echo $active_tab === 'branding' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-art"></span> Branding <span class="sacig-tab-badge">PRO</span>
                </a>
                <a href="?page=shortcodearcade-crypto-idle-game&tab=contests" 
                   class="nav-tab <?php echo $active_tab === 'contests' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-awards"></span> Contests <span class="sacig-tab-badge">PRO</span>
                </a>
                <a href="?page=shortcodearcade-crypto-idle-game&tab=login" 
                   class="nav-tab <?php echo $active_tab === 'login' ? 'nav-tab-active' : ''; ?>">
                    <span class="dashicons dashicons-lock"></span> Login Pages <span class="sacig-tab-badge">PRO</span>
                </a>
            </nav>
            
            <!-- Tab Content -->
            <div class="sacig-tab-content">
                <?php
                switch ($active_tab) {
                    case 'general':
                        $this->render_general_tab();
                        break;
                    case 'branding':
                        $this->render_branding_tab_locked();
                        break;
                    case 'contests':
                        $this->render_contests_tab_locked();
                        break;
                    case 'login':
                        $this->render_login_tab_locked();
                        break;
                    default:
                        $this->render_general_tab();
                }
                ?>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render General tab (FREE - Fully Functional)
     */
    private function render_general_tab() {
        ?>
        <div class="sacig-admin-container">
            <div class="sacig-admin-main">
                <form action="options.php" method="post">
                    <?php
                    settings_fields('sacig_settings_group');
                    do_settings_sections('shortcodearcade-crypto-idle-game');
                    submit_button('Save Settings');
                    ?>
                </form>
                
                <div class="sacig-info-box">
                    <h3>📋 Shortcodes</h3>
                    <p><strong>Game:</strong> <code>[sacig_crypto_idle_game]</code></p>
                    <?php if (get_option('sacig_enable_leaderboard')): ?>
                        <p><strong>Leaderboard:</strong> <code>[sacig_crypto_idle_leaderboard]</code></p>
                    <?php endif; ?>
                </div>
                
                <?php if ( get_option( 'sacig_enable_cloud_saves' ) ) : ?>
    <div class="sacig-info-box">
        <h3>☁️ Cloud Saves Status</h3>
        <?php
        global $wpdb;

$table_name = esc_sql( $wpdb->prefix . 'sacig_saves' );

// Direct query required: aggregate COUNT on custom table; no WP core API exists for custom table statistics.
// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
$count = (int) $wpdb->get_var(
    // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
    "SELECT COUNT(*) FROM {$table_name}"
);
        ?>
        <p><strong>Total Saved Games:</strong> <?php echo esc_html( $count ); ?></p>
    </div>
<?php endif; ?>
            </div>
            
            <div class="sacig-admin-sidebar">
                <div class="sacig-sidebar-box">
                    <h3>ℹ️ About</h3>
                    <p><strong>Shortcode Arcade Crypto Idle Game</strong></p>
                    <p>Version: <?php echo esc_html(SACIG_VERSION); ?></p>
                    <p>An idle clicker game with Elo-balanced progression.</p>
                </div>
                
                <div class="sacig-sidebar-box">
                    <h3>📚 Documentation</h3>
                    <ul>
                        <li><strong>Local Saves:</strong> Uses browser localStorage (default)</li>
                        <li><strong>Cloud Saves:</strong> Requires user login, stores in WordPress DB</li>
                        <li><strong>Leaderboard:</strong> Shows top players with prestige-weighted scoring</li>
                    </ul>
                </div>
                
                <div class="sacig-sidebar-box">
                    <h3>⚠️ Important Notes</h3>
                    <ul>
                        <li>The free version uses standard WordPress user accounts for login.</li>
<li>Cloud saves require users to be logged in</li>
<li>Leaderboards require cloud saves to be enabled</li>
<li>All data is stored in your WordPress database</li>
<li>Player data is private to your site only</li>
                    </ul>
                </div>
                
                <!-- Upgrade CTA Box -->
                <div class="sacig-sidebar-box sacig-upgrade-box">
                    <h3>🚀 Upgrade to Pro</h3>
                    <p>Unlock powerful features:</p>
                    <ul>
                        <li>✨ Custom Branding</li>
                        <li>🏆 Timed Contests</li>
                        <li>🔐 Custom Login Pages</li>
                        <li>🎨 White Label Options</li>
                    </ul>
                    <a href="https://shortcodearcade.com/shortcodearcade-crypto-idle-game-pro" target="_blank" class="button button-primary button-large sacig-upgrade-button">
                        Get Pro - Only $10
                    </a>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Branding tab (PRO - Locked)
     */
    private function render_branding_tab_locked() {
        ?>
        <div class="sacig-admin-container">
            <div class="sacig-admin-main">
                <div class="sacig-pro-placeholder">
                    <div class="sacig-pro-icon">🎨</div>
                    <h2>Custom Branding</h2>
                    <p>Make the game truly yours with full white-label customization.</p>
                    
                    <div class="sacig-feature-list">
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Custom Coin/Token Image:</strong> Replace Bitcoin with your own logo or token
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Color Theme Customization:</strong> Set primary, secondary, and accent colors
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Custom Game Title & Subtitle:</strong> Personalize the game's branding
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Currency Name Override:</strong> Change "Satoshis" to your own currency name
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Footer Customization:</strong> Add your own branding to the game footer
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Dynamic CSS Theming:</strong> All changes apply instantly without code
                            </div>
                        </div>
                    </div>
                    
                    <a href="https://shortcodearcade.com/shortcodearcade-crypto-idle-game-pro" target="_blank" class="button button-primary button-hero sacig-upgrade-cta">
                        <span class="dashicons dashicons-unlock"></span> Unlock Branding Features - Get Pro
                    </a>
                </div>
            </div>
            
            <div class="sacig-admin-sidebar">
                <div class="sacig-sidebar-box sacig-pro-box">
                    <h3>🎨 Why Custom Branding?</h3>
                    <p>Perfect for:</p>
                    <ul>
                        <li>Crypto projects launching tokens</li>
                        <li>Agencies building for clients</li>
                        <li>Community engagement tools</li>
                        <li>Educational platforms</li>
                        <li>White-label solutions</li>
                    </ul>
                </div>
                
                <div class="sacig-sidebar-box">
                    <h3>💡 Pro Features Include</h3>
                    <ul>
                        <li>✅ Everything in Free</li>
                        <li>✅ Custom Branding</li>
                        <li>✅ Timed Contests</li>
                        <li>✅ Custom Login Pages</li>
                        <li>✅ Priority Support</li>
                    </ul>
                    <p style="margin-top: 15px; font-weight: 600;">Only $10 - Lifetime License</p>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Contests tab (PRO - Locked)
     */
    private function render_contests_tab_locked() {
        ?>
        <div class="sacig-admin-container">
            <div class="sacig-admin-main">
                <div class="sacig-pro-placeholder">
                    <div class="sacig-pro-icon">🏆</div>
                    <h2>Timed Contests</h2>
                    <p>Run competitions with leaderboards, prizes, and automatic winner selection.</p>
                    
                    <div class="sacig-feature-list">
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Monthly/Weekly Contests:</strong> Create recurring competitions automatically
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Custom Contest Periods:</strong> Set your own start and end dates
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Prize Management:</strong> Define rewards for 1st, 2nd, 3rd place and beyond
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Automatic Winner Selection:</strong> System ranks players when contest ends
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Email Notifications:</strong> Winners get notified automatically
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Contest Archive:</strong> View historical results and past winners
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Achievement Badges:</strong> Reward players with unlockable achievements
                            </div>
                        </div>
                    </div>
                    
                    <a href="https://shortcodearcade.com/shortcodearcade-crypto-idle-game-pro" target="_blank" class="button button-primary button-hero sacig-upgrade-cta">
                        <span class="dashicons dashicons-unlock"></span> Unlock Contest Features - Get Pro
                    </a>
                </div>
            </div>
            
            <div class="sacig-admin-sidebar">
                <div class="sacig-sidebar-box sacig-pro-box">
                    <h3>🎮 Boost Engagement</h3>
                    <p>Contests dramatically increase:</p>
                    <ul>
                        <li>Player retention</li>
                        <li>Daily active users</li>
                        <li>Community interaction</li>
                        <li>Time spent on site</li>
                        <li>Social sharing</li>
                    </ul>
                </div>
                
                <div class="sacig-sidebar-box">
                    <h3>💰 Use Cases</h3>
                    <ul>
                        <li>Monthly tournaments</li>
                        <li>Holiday competitions</li>
                        <li>Community challenges</li>
                        <li>Prize giveaways</li>
                        <li>Marketing campaigns</li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Login Pages tab (PRO - Locked)
     */
    private function render_login_tab_locked() {
        ?>
        <div class="sacig-admin-container">
            <div class="sacig-admin-main">
                <div class="sacig-pro-placeholder">
                    <div class="sacig-pro-icon">🔐</div>
                    <h2>Custom Login Pages</h2>
                    <p>Create branded login and registration pages that match your game's aesthetic.</p>
                    
                    <div class="sacig-feature-list">
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Custom Login Page:</strong> Replace default WordPress login with branded version
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Branded Registration:</strong> Custom signup page matching game theme
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Email Verification:</strong> Optional double opt-in for security
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Social Login:</strong> Google, Facebook, Twitter integration
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Custom Redirects:</strong> Send users to game after login
                            </div>
                        </div>
                        <div class="sacig-feature-item">
                            <span class="dashicons dashicons-yes-alt"></span>
                            <div>
                                <strong>Password Recovery:</strong> Branded password reset flow
                            </div>
                        </div>
                    </div>
                    
                    <a href="https://shortcodearcade.com/shortcodearcade-crypto-idle-game-pro" target="_blank" class="button button-primary button-hero sacig-upgrade-cta">
                        <span class="dashicons dashicons-unlock"></span> Unlock Login Features - Get Pro
                    </a>
                </div>
            </div>
            
            <div class="sacig-admin-sidebar">
                <div class="sacig-sidebar-box sacig-pro-box">
                    <h3>🎨 Seamless Experience</h3>
                    <p>Benefits of custom login:</p>
                    <ul>
                        <li>Professional appearance</li>
                        <li>Brand consistency</li>
                        <li>Higher conversion rates</li>
                        <li>Better user experience</li>
                        <li>Reduced bounce rates</li>
                    </ul>
                </div>
                
                <div class="sacig-sidebar-box">
                    <h3>🔒 Security First</h3>
                    <p>Custom pages include:</p>
                    <ul>
                        <li>reCAPTCHA support</li>
                        <li>Email verification</li>
                        <li>Rate limiting</li>
                        <li>Secure authentication</li>
                    </ul>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Maybe create database table for cloud saves
     */
    private function maybe_create_table() {
        global $wpdb;
        $table_name = $wpdb->prefix . 'sacig_saves';
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            user_id bigint(20) UNSIGNED NOT NULL,
            save_data longtext NOT NULL,
            base_click_power decimal(20,6) DEFAULT 1,
            base_passive_income decimal(20,6) DEFAULT 0,
            prestige_level int DEFAULT 0,
            total_satoshis decimal(30,6) DEFAULT 0,
            rank_score decimal(30,6) DEFAULT 0,
            last_updated datetime DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (user_id),
            KEY rank_score (rank_score DESC),
            KEY last_updated (last_updated)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on our settings page
        if ($hook !== 'settings_page_shortcodearcade-crypto-idle-game') {
            return;
        }
        
        wp_enqueue_style(
            'sacig-admin-css',
            SACIG_PLUGIN_URL . 'assets/css/sacig-admin.css',
            array(),
            SACIG_VERSION
        );

        wp_enqueue_script(
            'sacig-admin-js',
            SACIG_PLUGIN_URL . 'assets/js/sacig-admin.js',
            array('jquery'),
            SACIG_VERSION,
            true
        );

        // Localize strings for JavaScript
        wp_localize_script(
            'sacig-admin-js',
            'sacigAdminStrings',
            array(
                'confirmDisableCloudTitle' => __( 'Warning: Disabling Cloud Saves', 'shortcodearcade-crypto-idle-game' ),
                'confirmDisableCloudBody' => __( 'Disabling cloud saves will prevent users from saving game progress to your WordPress database. Players will only be able to use local browser storage.', 'shortcodearcade-crypto-idle-game' ),
                'confirmDisableCloudQuestion' => __( 'Are you sure you want to disable cloud saves?', 'shortcodearcade-crypto-idle-game' ),
                'unsavedChangesWarning' => __( 'You have unsaved changes. Are you sure you want to leave?', 'shortcodearcade-crypto-idle-game' ),
                'copiedLabel' => __( 'Copied!', 'shortcodearcade-crypto-idle-game' ),
                'tooltipCloudSaves' => __( 'Saves game data to WordPress database. Requires users to be logged in.', 'shortcodearcade-crypto-idle-game' ),
                'tooltipLeaderboard' => __( 'Display top players using the [sacig_crypto_idle_leaderboard] shortcode.', 'shortcodearcade-crypto-idle-game' ),
            )
        );
    }
}
