/**
 * Shortcode Arcade Crypto Idle Game - Game Logic
 * Version: 0.4.6
 *
 * Public window.* globals (intentional for gameplay):
 * - window.sacigMine() - Mining click handler
 * - window.sacigBuyUpgrade(id) - Upgrade purchase handler
 * - window.sacigPrestige() - Prestige/Hard Fork handler
 * - window.sacigShowModal() - Info modal display
 * - window.sacigHideModal() - Info modal close
 *
 * LocalStorage usage:
 * - sacigCryptoMinerSave - Game state persistence
 * - sacigLastSaveTime - Offline progress calculation
 * - sacigCryptoMinerVisited - First-time user detection
 */

(function() {
    'use strict';
    
    // Game State
    let gameState = {
        satoshis: 0,
        clickPower: 1,
        passiveIncome: 0,
        rating: 1000,
        prestigeLevel: 0,
        prestigeMultiplier: 1,
        upgrades: {},
        version: '0.4.6'
    };
    
    // Cloud save settings (passed from WordPress)
    const cloudSavesEnabled = typeof sacigSettings !== 'undefined' && sacigSettings.cloudSavesEnabled;
    const isUserLoggedIn = typeof sacigSettings !== 'undefined' && sacigSettings.isUserLoggedIn;
    const useCloudSaves = cloudSavesEnabled && isUserLoggedIn;

    // Upgrade Definitions with Elo-based balancing
    // NOTE: Prestige multiplier is applied at EARN-TIME, not purchase-time
    // This ensures idempotent progression and prevents balance issues
    const upgradeDefinitions = [
        {
            id: 'betterClicker',
            name: 'Better Pickaxe',
            baseEffect: 1,
            baseDescription: 'Increases click power',
            baseCost: 10,
            rating: 1000,
            type: 'click',
            costMultiplier: 1.15
        },
        {
            id: 'cpuMiner',
            name: 'CPU Miner',
            baseEffect: 0.1,
            baseDescription: 'Generates satoshis/sec',
            baseCost: 50,
            rating: 1050,
            type: 'passive',
            costMultiplier: 1.2
        },
        {
            id: 'powerfulClicker',
            name: 'Diamond Pickaxe',
            baseEffect: 5,
            baseDescription: 'Increases click power',
            baseCost: 100,
            rating: 1100,
            type: 'click',
            costMultiplier: 1.15
        },
        {
            id: 'gpuRig',
            name: 'GPU Mining Rig',
            baseEffect: 1,
            baseDescription: 'Generates satoshis/sec',
            baseCost: 500,
            rating: 1200,
            type: 'passive',
            costMultiplier: 1.25
        },
        {
            id: 'megaClicker',
            name: 'Quantum Pickaxe',
            baseEffect: 25,
            baseDescription: 'Increases click power',
            baseCost: 1000,
            rating: 1300,
            type: 'click',
            costMultiplier: 1.15
        },
        {
            id: 'asicMiner',
            name: 'ASIC Miner',
            baseEffect: 10,
            baseDescription: 'Generates satoshis/sec',
            baseCost: 5000,
            rating: 1400,
            type: 'passive',
            costMultiplier: 1.3
        },
        {
            id: 'ultraClicker',
            name: 'Neutron Star Drill',
            baseEffect: 100,
            baseDescription: 'Increases click power',
            baseCost: 10000,
            rating: 1500,
            type: 'click',
            costMultiplier: 1.15
        },
        {
            id: 'miningFarm',
            name: 'Mining Farm',
            baseEffect: 50,
            baseDescription: 'Generates satoshis/sec',
            baseCost: 50000,
            rating: 1600,
            type: 'passive',
            costMultiplier: 1.35
        },
        {
            id: 'godClicker',
            name: 'Black Hole Extractor',
            baseEffect: 500,
            baseDescription: 'Increases click power',
            baseCost: 100000,
            rating: 1700,
            type: 'click',
            costMultiplier: 1.15
        },
        {
            id: 'datacenter',
            name: 'Data Center',
            baseEffect: 250,
            baseDescription: 'Generates satoshis/sec',
            baseCost: 500000,
            rating: 1800,
            type: 'passive',
            costMultiplier: 1.4
        }
    ];

    /**
     * Calculate diminishing returns multiplier
     * 1st purchase: 100% (1.0)
     * 2nd purchase: 80% (0.8)
     * 3rd purchase: 60% (0.6)
     * 4th purchase: 40% (0.4)
     * 5th+ purchase: 20% (0.2)
     */
    function getDiminishingReturnsMultiplier(ownedCount) {
        if (ownedCount === 0) return 1.0;  // First purchase: 100%
        if (ownedCount === 1) return 0.8;  // Second purchase: 80%
        if (ownedCount === 2) return 0.6;  // Third purchase: 60%
        if (ownedCount === 3) return 0.4;  // Fourth purchase: 40%
        return 0.2;                         // Fifth+ purchase: 20%
    }

    /**
     * Calculate prestige cost with exponential scaling
     * Level 0→1: 1,000,000
     * Level 1→2: 5,000,000 (5x)
     * Level 2→3: 25,000,000 (5x)
     * Level 3→4: 125,000,000 (5x)
     */
    function getPrestigeCost() {
        const basePrestigeCost = 1000000;
        const prestigeMultiplier = 5;
        
        if (gameState.prestigeLevel === 0) {
            return basePrestigeCost;
        }
        
        return basePrestigeCost * Math.pow(prestigeMultiplier, gameState.prestigeLevel);
    }

    /**
     * Calculate total effect for an upgrade with diminishing returns
     */
    function calculateTotalEffect(upgrade) {
        const owned = gameState.upgrades[upgrade.id] || 0;
        if (owned === 0) return 0;
        
        let totalEffect = 0;
        
        // Calculate effect for each owned upgrade with diminishing returns
        for (let i = 0; i < owned; i++) {
            const multiplier = getDiminishingReturnsMultiplier(i);
            totalEffect += upgrade.baseEffect * multiplier;
        }
        
        return totalEffect;
    }

    /**
     * Recalculate production stats with diminishing returns
     */
    function recalculateProduction() {
        let totalClickPower = 1; // Base click power
        let totalPassiveIncome = 0;
        
        upgradeDefinitions.forEach(upgrade => {
            const effect = calculateTotalEffect(upgrade);
            
            if (upgrade.type === 'click') {
                totalClickPower += effect;
            } else if (upgrade.type === 'passive') {
                totalPassiveIncome += effect;
            }
        });
        
        gameState.clickPower = totalClickPower;
        gameState.passiveIncome = totalPassiveIncome;
    }

    /**
     * Calculate upgrade cost using Elo-based formula
     */
    function getUpgradeCost(upgrade) {
        const owned = gameState.upgrades[upgrade.id] || 0;
        const ratingDiff = upgrade.rating - gameState.rating;
        const eloMultiplier = Math.max(0.5, 1 + (ratingDiff / 400));
        const ownedMultiplier = Math.pow(upgrade.costMultiplier, owned);
        return Math.ceil(upgrade.baseCost * eloMultiplier * ownedMultiplier);
    }

    /**
     * Mining click function
     * Prestige multiplier is applied HERE at earn-time, not at upgrade purchase time
     */
    window.sacigMine = function() {
        const earnedAmount = gameState.clickPower * gameState.prestigeMultiplier;
        gameState.satoshis += earnedAmount;
        gameState.satoshis = Number(gameState.satoshis.toFixed(6)); // Prevent floating point drift
        
        // Create floating particle effect
        const button = document.getElementById('sacig-mineButton');
        if (button) {
            const rect = button.getBoundingClientRect();
            const particle = document.createElement('div');
            particle.className = 'sacig-click-particle';
            particle.textContent = '+' + formatNumber(earnedAmount);
            particle.style.left = (rect.left + rect.width / 2 - 30) + 'px';
            particle.style.top = (rect.top + rect.height / 2) + 'px';
            document.body.appendChild(particle);
            
            setTimeout(() => particle.remove(), 1000);
        }
        
        updateUI();
    };

    /**
     * Buy upgrade function
     */
    window.sacigBuyUpgrade = function(upgradeId) {
        const upgrade = upgradeDefinitions.find(u => u.id === upgradeId);
        if (!upgrade) return;
        
        const cost = getUpgradeCost(upgrade);
        
        if (gameState.satoshis >= cost) {
            gameState.satoshis -= cost;
            
            // Track owned count
            gameState.upgrades[upgradeId] = (gameState.upgrades[upgradeId] || 0) + 1;
            
            // Recalculate production with diminishing returns
            recalculateProduction();
            
            // Increase rating based on upgrade tier
            gameState.rating += 10;
            
            updateUI();
            saveGame();
        }
    };

    /**
     * Prestige function with exponential cost scaling
     */
    window.sacigPrestige = function() {
        const prestigeCost = getPrestigeCost();
        
        if (gameState.satoshis < prestigeCost) {
            alert(`You need ${formatNumber(prestigeCost)} satoshis to perform a Hard Fork!`);
            return;
        }
        
        const currentPrestige = gameState.prestigeLevel;
        const nextPrestigeCost = prestigeCost * 5;
        
        const confirmed = confirm(
            `Perform Hard Fork?\n\n` +
            `Current Level: ${currentPrestige}\n` +
            `New Level: ${currentPrestige + 1}\n` +
            `Current Bonus: +${currentPrestige * 10}%\n` +
            `New Bonus: +${(currentPrestige + 1) * 10}%\n\n` +
            `Cost: ${formatNumber(prestigeCost)} Satoshis\n` +
            `Next Fork Cost: ${formatNumber(nextPrestigeCost)} Satoshis\n\n` +
            `This will reset your progress but give you a permanent +10% production bonus!\n` +
            `Diminishing returns will also reset.`
        );
        
        if (!confirmed) return;
        
        gameState.prestigeLevel++;
        gameState.prestigeMultiplier = 1 + (gameState.prestigeLevel * 0.1);
        gameState.satoshis = 0;
        gameState.clickPower = 1;
        gameState.passiveIncome = 0;
        gameState.rating = 1000;
        gameState.upgrades = {};
        
        updateUI();
        saveGame();
    };

    /**
     * Format large numbers
     */
    function formatNumber(num) {
        if (num >= 1000000000) return (num / 1000000000).toFixed(2) + 'B';
        if (num >= 1000000) return (num / 1000000).toFixed(2) + 'M';
        if (num >= 1000) return (num / 1000).toFixed(2) + 'K';
        return num.toFixed(2);
    }

    /**
     * Update UI
     */
    function updateUI() {
        // Calculate effective values (base * prestige multiplier)
        const effectiveClickPower = gameState.clickPower * gameState.prestigeMultiplier;
        const effectivePassiveIncome = gameState.passiveIncome * gameState.prestigeMultiplier;
        
        // Update stats
        const satoshisEl = document.getElementById('sacig-satoshis');
        const clickPowerEl = document.getElementById('sacig-clickPower');
        const passiveIncomeEl = document.getElementById('sacig-passiveIncome');
        const ratingEl = document.getElementById('sacig-rating');
        
        if (satoshisEl) satoshisEl.textContent = formatNumber(gameState.satoshis);
        if (clickPowerEl) clickPowerEl.textContent = formatNumber(effectiveClickPower);
        if (passiveIncomeEl) passiveIncomeEl.textContent = formatNumber(effectivePassiveIncome);
        if (ratingEl) ratingEl.textContent = Math.floor(gameState.rating);
        
        // Update upgrades list
        const upgradesList = document.getElementById('sacig-upgradesList');
        if (upgradesList) {
            upgradesList.innerHTML = '';
            
            upgradeDefinitions.forEach(upgrade => {
                const cost = getUpgradeCost(upgrade);
                const owned = gameState.upgrades[upgrade.id] || 0;
                const canAfford = gameState.satoshis >= cost;
                const totalEffect = calculateTotalEffect(upgrade);
                
                // Calculate next purchase effect with diminishing returns
                const nextMultiplier = getDiminishingReturnsMultiplier(owned);
                const nextEffect = upgrade.baseEffect * nextMultiplier;
                
                const upgradeDiv = document.createElement('div');
                upgradeDiv.className = 'sacig-upgrade-item' + (canAfford ? '' : ' sacig-disabled');
                upgradeDiv.onclick = () => window.sacigBuyUpgrade(upgrade.id);
                
                // Build effect text
                let effectText = '';
                if (upgrade.type === 'click') {
                    effectText = `+${formatNumber(nextEffect)} per click`;
                } else {
                    effectText = `+${formatNumber(nextEffect)}/sec`;
                }
                
                // Show diminishing returns info
                let diminishingText = '';
                if (owned > 0 && nextMultiplier < 1.0) {
                    diminishingText = ` (${Math.round(nextMultiplier * 100)}% effectiveness)`;
                }
                
                upgradeDiv.innerHTML = `
                    <div class="sacig-upgrade-header">
                        <div class="sacig-upgrade-name">${upgrade.name}</div>
                        <div class="sacig-upgrade-cost">${formatNumber(cost)}</div>
                    </div>
                    <div class="sacig-upgrade-description">${effectText}${diminishingText}</div>
                    <div class="sacig-upgrade-owned">Owned: ${owned}${owned > 0 ? ` | Total: ${formatNumber(totalEffect * gameState.prestigeMultiplier)}` : ''}</div>
                `;
                
                upgradesList.appendChild(upgradeDiv);
            });
        }
        
        // Update prestige button
        const prestigeCost = getPrestigeCost();
        const prestigeButton = document.getElementById('sacig-prestigeButton');
        if (prestigeButton) {
            prestigeButton.disabled = gameState.satoshis < prestigeCost;
            
            // Update button text with current level
            if (gameState.prestigeLevel > 0) {
                prestigeButton.textContent = `HARD FORK (Level ${gameState.prestigeLevel})`;
            }
        }
        
        // Update prestige info
        const prestigeInfo = document.querySelector('.sacig-prestige-info');
        if (prestigeInfo) {
            const nextPrestigeCost = getPrestigeCost();
            const currentBonus = gameState.prestigeLevel * 10;
            
            prestigeInfo.innerHTML = `
                Hard Fork available at ${formatNumber(nextPrestigeCost)} satoshis<br>
                <span style="font-size: 0.9rem; opacity: 0.7;">
                    ${gameState.prestigeLevel > 0 ? `Current Level: ${gameState.prestigeLevel} (+${currentBonus}% bonus)<br>` : ''}
                    Reset with permanent +10% bonus to all production
                </span>
            `;
        }
    }

    /**
     * Passive income loop
     * Prestige multiplier is applied HERE at earn-time, not at upgrade purchase time
     */
    function passiveIncomeLoop() {
        const earned = (gameState.passiveIncome * gameState.prestigeMultiplier) / 10;
        gameState.satoshis += earned; // Update 10 times per second
        gameState.satoshis = Number(gameState.satoshis.toFixed(6)); // Prevent floating point drift
        updateUI();
    }

    /**
     * Calculate offline progress
     */
    function calculateOfflineProgress() {
        const lastSaveTime = localStorage.getItem('sacigLastSaveTime');
        if (!lastSaveTime) return;
        
        const now = Date.now();
        const secondsAway = Math.min((now - lastSaveTime) / 1000, 86400); // Cap at 24 hours
        
        // Only calculate if away for more than 60 seconds
        if (secondsAway < 60) return;
        
        const offlineEarned = secondsAway * gameState.passiveIncome * gameState.prestigeMultiplier;
        
        if (offlineEarned > 0) {
            gameState.satoshis += offlineEarned;
            gameState.satoshis = Number(gameState.satoshis.toFixed(6));
            
            // Show notification to player
            const hours = Math.floor(secondsAway / 3600);
            const minutes = Math.floor((secondsAway % 3600) / 60);
            let timeAway = '';
            if (hours > 0) timeAway = `${hours}h ${minutes}m`;
            else timeAway = `${minutes}m`;
            
            setTimeout(() => {
                alert(`Welcome back! You were away for ${timeAway} and earned ${formatNumber(offlineEarned)} satoshis!`);
            }, 500);
        }
    }

    /**
     * Save game (local or cloud)
     */
    function saveGame() {
        // Always save timestamp for offline progress
        localStorage.setItem('sacigLastSaveTime', Date.now().toString());
        
        if (useCloudSaves) {
            saveToCloud();
        } else {
            saveToLocalStorage();
        }
    }

    /**
     * Save to localStorage
     */
    function saveToLocalStorage() {
        localStorage.setItem('sacigCryptoMinerSave', JSON.stringify(gameState));
        showSaveIndicator();
    }

    /**
     * Save to cloud
     */
    async function saveToCloud() {
        try {
            const response = await fetch(sacigSettings.restUrl + 'save', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': sacigSettings.nonce
                },
                body: JSON.stringify({
                    save_data: gameState
                })
            });
            
            const data = await response.json();
            
            if (data.success) {
                // Also save to localStorage as backup
                saveToLocalStorage();
                showSaveIndicator('☁️ Cloud Saved');
            } else {
                console.error('Cloud save failed:', data.message);
                // Fallback to localStorage
                saveToLocalStorage();
            }
        } catch (error) {
            console.error('Cloud save error:', error);
            // Fallback to localStorage
            saveToLocalStorage();
        }
    }

    /**
     * Load game (local or cloud)
     */
    async function loadGame() {
        if (useCloudSaves) {
            await loadFromCloud();
        } else {
            loadFromLocalStorage();
        }
        
        // After loading, recalculate production to apply diminishing returns
        recalculateProduction();
    }

    /**
     * Load from localStorage
     */
    function loadFromLocalStorage() {
        const saved = localStorage.getItem('sacigCryptoMinerSave');
        if (saved) {
            try {
                const loadedState = JSON.parse(saved);
                
                // Merge loaded state with defaults (for new fields)
                gameState = Object.assign({}, gameState, loadedState);
                
                // Update version
                gameState.version = '0.4.6';
                
                updateUI();
            } catch (e) {
                console.error('Failed to load saved game:', e);
            }
        }
    }

    /**
     * Load from cloud
     */
    async function loadFromCloud() {
        try {
            const response = await fetch(sacigSettings.restUrl + 'load', {
                method: 'GET',
                headers: {
                    'X-WP-Nonce': sacigSettings.nonce
                }
            });
            
            const data = await response.json();
            
            if (data.success && data.data) {
                // Cloud save exists, use it
                gameState = Object.assign({}, gameState, data.data);
                
                // Update version
                gameState.version = '0.4.6';
                
                updateUI();
                console.log('Loaded from cloud');
            } else {
                // No cloud save, try localStorage
                loadFromLocalStorage();
            }
        } catch (error) {
            console.error('Cloud load error:', error);
            // Fallback to localStorage
            loadFromLocalStorage();
        }
    }

    /**
     * Show save indicator
     */
    function showSaveIndicator(message = 'Game Saved') {
        const indicator = document.getElementById('sacig-saveIndicator');
        if (indicator) {
            indicator.textContent = message;
            indicator.classList.add('sacig-show');
            setTimeout(() => indicator.classList.remove('sacig-show'), 2000);
        }
    }

    /**
     * Modal functions
     */
    window.sacigShowModal = function() {
        const modal = document.getElementById('sacig-infoModal');
        if (modal) {
            modal.classList.add('sacig-show');
        }
    };

    window.sacigHideModal = function() {
        const modal = document.getElementById('sacig-infoModal');
        if (modal) {
            modal.classList.remove('sacig-show');
        }
    };

    /**
     * Apply custom branding theme
     */
    function applyBrandingTheme() {
        if (!sacigSettings || !sacigSettings.branding || !sacigSettings.branding.enabled) {
            return;
        }
        
        const branding = sacigSettings.branding;
        const container = document.querySelector('.sacig-container');
        
        if (!container) return;
        
        // Apply custom colors
        if (branding.colors) {
            container.style.setProperty('--sacig-neon-cyan', branding.colors.primary);
            container.style.setProperty('--sacig-neon-magenta', branding.colors.secondary);
            container.style.setProperty('--sacig-neon-yellow', branding.colors.accent);
        }
    }

    /**
     * Initialize game when DOM is ready
     */
    async function initGame() {
        // Apply custom branding theme
        applyBrandingTheme();
        
        // Load game state
        await loadGame();
        
        // Calculate offline progress
        calculateOfflineProgress();
        
        // Update UI
        updateUI();
        
        // Start passive income loop
        setInterval(passiveIncomeLoop, 100); // 10 times per second
        
        // Auto-save every 10 seconds
        setInterval(saveGame, 10000);
        
        // Show info modal on first visit
        if (!localStorage.getItem('sacigCryptoMinerVisited')) {
            setTimeout(() => window.sacigShowModal(), 500);
            localStorage.setItem('sacigCryptoMinerVisited', 'true');
        }
    }

    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', initGame);
    } else {
        initGame();
    }
})();
