<?php
/**
 * کلاینت API برای سرویس لینک کوتاه m8c
 */

defined('ABSPATH') || exit;

class Slfm8c_Api_Client {

    /** @var string */
    private $api_base_url;

    /** @var string */
    private $api_token;

    /**
     * @param string $api_base_url آدرس پایه API (مثلاً https://m8c.ir/api)
     * @param string $api_token توکن احراز هویت
     */
    public function __construct($api_base_url, $api_token) {
        $this->api_base_url = $api_base_url;
        $this->api_token = $api_token;
    }

    /**
     * ساخت لینک کوتاه
     *
     * @param string $url آدرس بلند
     * @param int $code_length طول کد (۳ تا ۱۶)
     * @param string $password رمز اختیاری
     * @return array success, short_url/code/url یا error
     */
    public function create_short_link($url, $code_length = 3, $password = '') {
        $endpoint = rtrim($this->api_base_url, '/') . '/short-links';
        $body = [
            'url' => $url,
            'code_length' => $code_length,
        ];
        if ($password !== '') {
            $body['password'] = $password;
        }

        $response = wp_remote_post($endpoint, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_token,
                'Content-Type'  => 'application/json',
                'Accept'       => 'application/json',
            ],
            'body'    => wp_json_encode($body),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($response);
        $body_raw = wp_remote_retrieve_body($response);
        $data = json_decode($body_raw, true);

        if ($code >= 200 && $code < 300 && is_array($data) && !empty($data['short_url'])) {
            return [
                'success'   => true,
                'short_url' => $data['short_url'],
                'code'      => $data['code'] ?? '',
                'url'       => $data['url'] ?? $url,
            ];
        }

        $message = isset($data['message']) ? $data['message'] : __('خطای ناشناخته از سرور.', 'short-links-for-m8c');
        if (!empty($data['errors'])) {
            $message = is_array($data['errors']) ? implode(' ', array_map(function ($e) {
                return is_array($e) ? implode(' ', $e) : $e;
            }, $data['errors'])) : (string) $data['errors'];
        }
        return ['success' => false, 'error' => $message];
    }

    /**
     * دریافت لیست لینک‌های کوتاه ایجادشده با همین توکن (وب‌سرویس) — با صفحه‌بندی.
     *
     * @param int $page شماره صفحه
     * @param int $per_page تعداد در هر صفحه
     * @return array{success: bool, data?: array, total?: int, current_page?: int, last_page?: int, error?: string}
     */
    public function get_my_links($page = 1, $per_page = 15) {
        $endpoint = rtrim($this->api_base_url, '/') . '/short-links';
        $url = add_query_arg([
            'page' => max(1, (int) $page),
            'per_page' => max(5, min(50, (int) $per_page)),
        ], $endpoint);

        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_token,
                'Accept'        => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return ['success' => false, 'error' => $response->get_error_message()];
        }

        $code = wp_remote_retrieve_response_code($response);
        $body_raw = wp_remote_retrieve_body($response);
        $data = json_decode($body_raw, true);

        if ($code >= 200 && $code < 300 && is_array($data) && isset($data['data'])) {
            return [
                'success'      => true,
                'data'         => $data['data'],
                'total'        => isset($data['total']) ? (int) $data['total'] : 0,
                'current_page' => isset($data['current_page']) ? (int) $data['current_page'] : 1,
                'last_page'    => isset($data['last_page']) ? (int) $data['last_page'] : 1,
                'per_page'     => isset($data['per_page']) ? (int) $data['per_page'] : $per_page,
                'from'         => isset($data['from']) ? (int) $data['from'] : 0,
                'to'           => isset($data['to']) ? (int) $data['to'] : 0,
            ];
        }

        $message = isset($data['message']) ? $data['message'] : __('خطای ناشناخته از سرور.', 'short-links-for-m8c');
        return ['success' => false, 'error' => $message];
    }

    /**
     * بررسی اینکه تنظیمات (آدرس و توکن) پر شده‌اند یا نه
     */
    /** @return bool */
    public static function is_configured() {
        $base = Slfm8c_Settings::get_api_base_url();
        $token = Slfm8c_Settings::get_api_token();
        return $base !== '' && $token !== '';
    }

    /**
     * نمونهٔ آماده از روی تنظیمات پلاگین
     */
    /** @return self */
    public static function from_settings() {
        return new self(
            Slfm8c_Settings::get_api_base_url(),
            Slfm8c_Settings::get_api_token()
        );
    }
}
