<?php
/**
 * The product content operations class - Lite Version
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Product content operations class
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_Product {

	/**
	 * The API handler.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    SHOPWR_Lite_API
	 */
	protected $api;

	/**
	 * The SEO integration handler.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    SHOPWR_Lite_SEO
	 */
	protected $seo;

	/**
	 * The log handler.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    SHOPWR_Lite_Log
	 */
	protected $log;

	/**
	 * The admin handler.
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    SHOPWR_Lite_Admin
	 */
	protected $admin;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 * @param SHOPWR_Lite_API   $api   The API handler.
	 * @param SHOPWR_Lite_SEO   $seo   The SEO integration handler.
	 * @param SHOPWR_Lite_Log   $log   The log handler.
	 * @param SHOPWR_Lite_Admin $admin The admin handler.
	 */
	public function __construct( $api, $seo, $log, $admin ) {
		$this->api   = $api;
		$this->seo   = $seo;
		$this->log   = $log;
		$this->admin = $admin;
	}

	/**
	 * Add meta box to product edit page
	 *
	 * @since 1.0.0
	 */
	public function add_product_meta_box() {
		add_meta_box(
			'shopwr_lite_product_content_generator',
			esc_html__( 'ShopWriter Lite', 'shopwriter-lite' ),
			array( $this, 'render_meta_box' ),
			'product',
			'normal',
			'high'
		);
	}

	/**
	 * Render meta box content
	 *
	 * @since 1.0.0
	 * @param WP_Post $post The post object.
	 */
	public function render_meta_box( $post ) {
		// Get product data.
		$product = wc_get_product( $post->ID );

		// Output the meta box content.
		include SHOPWR_LITE_PLUGIN_DIR . 'admin/partials/product-meta-box.php';
	}

	/**
	 * Get product data for API request
	 *
	 * @since  1.0.0
	 * @param  int $product_id Product ID.
	 * @return array           Product data.
	 */
	public function get_product_data_for_api( $product_id ) {
		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return array();
		}

		// Get featured image information.
		$featured_image_id  = $product->get_image_id();
		$current_alt_text   = '';
		$featured_image_url = '';
		$image_filename     = '';

		if ( $featured_image_id ) {
			$current_alt_text   = get_post_meta( $featured_image_id, '_wp_attachment_image_alt', true );
			$featured_image_url = wp_get_attachment_url( $featured_image_id );
			$image_filename     = basename( $featured_image_url );
		}

		// Get product data.
		$data = array(
			'id'                 => $product_id,
			'name'               => $product->get_name(),
			'title'              => $product->get_name(),
			'sku'                => $product->get_sku(),
			'type'               => $product->get_type(),
			'description'        => $product->get_description(),
			'short_description'  => $product->get_short_description(),
			'categories'         => array(),
			'attributes'         => array(),
			'tags'               => array(),
			'price'              => $product->get_price(),
			'regular_price'      => $product->get_regular_price(),
			'sale_price'         => $product->get_sale_price(),
			'meta_title'         => '',
			'meta_description'   => '',
			'featured_image'     => array(
				'id'               => $featured_image_id,
				'url'              => $featured_image_url,
				'filename'         => $image_filename,
				'current_alt_text' => $current_alt_text,
				'has_image'        => ! empty( $featured_image_id ),
			),
			'alt_text'           => $current_alt_text,
			'featured_image_url' => $featured_image_url,
		);

		// Get categories.
		$categories = get_the_terms( $product_id, 'product_cat' );
		if ( $categories && ! is_wp_error( $categories ) ) {
			foreach ( $categories as $category ) {
				$data['categories'][] = $category->name;
			}
		}

		// Get attributes.
		$attributes = $product->get_attributes();
		if ( ! empty( $attributes ) ) {
			foreach ( $attributes as $attribute ) {
				if ( $attribute->is_taxonomy() ) {
					$terms = wp_get_post_terms( $product_id, $attribute->get_name(), array( 'fields' => 'names' ) );
					if ( ! is_wp_error( $terms ) ) {
						$data['attributes'][ $attribute->get_name() ] = $terms;
					}
				} else {
					$data['attributes'][ $attribute->get_name() ] = $attribute->get_options();
				}
			}
		}

		// Get tags.
		$tags = get_the_terms( $product_id, 'product_tag' );
		if ( $tags && ! is_wp_error( $tags ) ) {
			foreach ( $tags as $tag ) {
				$data['tags'][] = $tag->name;
			}
		}

		// Get SEO meta data.
		$meta_data                = $this->seo->get_seo_meta( $product_id );
		$data['meta_title']       = $meta_data['title'];
		$data['meta_description'] = $meta_data['description'];

		return $data;
	}

	/**
	 * Generate product content
	 *
	 * @since  1.0.0
	 * @param  int    $product_id Product ID.
	 * @param  array  $fields     Fields to generate.
	 * @param  string $operation  Operation type (generate, rewrite, translate).
	 * @return array|WP_Error     Generated content or error.
	 */
	public function generate_content( $product_id, $fields, $operation = 'generate' ) {
		// Filter to only allowed Lite fields.
		$fields = $this->api->filter_allowed_fields( $fields );

		if ( empty( $fields ) ) {
			return new WP_Error( 'no_fields', esc_html__( 'No valid fields selected for generation.', 'shopwriter-lite' ) );
		}

		// Get product data.
		$product_data = $this->get_product_data_for_api( $product_id );

		// Prepare parameters.
		$params = array(
			'product_data' => $product_data,
			'fields'       => $fields,
			'operation'    => $operation,
		);

		// Generate content.
		return $this->api->generate_content( $params );
	}

	/**
	 * Save generated content to product
	 *
	 * @since  1.0.0
	 * @param  int   $product_id Product ID.
	 * @param  array $content    Content to save.
	 * @param  int   $user_id    User ID who initiated the process (optional).
	 * @return bool              True on success, false on failure.
	 */
	public function save_generated_content( $product_id, $content, $user_id = null ) {
		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return false;
		}

		// Get featured image ID.
		$featured_image_id = $product->get_image_id();

		// Prepare original content for logging.
		$original = array(
			'description'       => $product->get_description(),
			'short_description' => $product->get_short_description(),
			'meta_title'        => $this->seo->get_seo_meta( $product_id )['title'],
			'meta_description'  => $this->seo->get_seo_meta( $product_id )['description'],
			'alt_text'          => $featured_image_id ? get_post_meta( $featured_image_id, '_wp_attachment_image_alt', true ) : '',
		);

		$updated       = array();
		$content_saved = false;

		// Lite version: Title update is NOT allowed (PRO only).
		// Update description.
		if ( isset( $content['description'] ) && ! empty( trim( $content['description'] ) ) ) {
			$product->set_description( trim( $content['description'] ) );
			$updated['description'] = trim( $content['description'] );
			$content_saved          = true;
		}

		// Update short description.
		if ( isset( $content['short_description'] ) && ! empty( trim( $content['short_description'] ) ) ) {
			$product->set_short_description( trim( $content['short_description'] ) );
			$updated['short_description'] = trim( $content['short_description'] );
			$content_saved                = true;
		}

		// Save product if any content was updated.
		if ( $content_saved ) {
			$save_result = $product->save();
			if ( is_wp_error( $save_result ) ) {
				return false;
			}
		}

		// Update SEO meta.
		if ( ( isset( $content['meta_title'] ) && ! empty( trim( $content['meta_title'] ) ) ) ||
			( isset( $content['meta_description'] ) && ! empty( trim( $content['meta_description'] ) ) ) ) {

			$meta = array();

			if ( isset( $content['meta_title'] ) && ! empty( trim( $content['meta_title'] ) ) ) {
				$meta['title']         = trim( $content['meta_title'] );
				$updated['meta_title'] = trim( $content['meta_title'] );
				$content_saved         = true;
			}

			if ( isset( $content['meta_description'] ) && ! empty( trim( $content['meta_description'] ) ) ) {
				$meta['description']         = trim( $content['meta_description'] );
				$updated['meta_description'] = trim( $content['meta_description'] );
				$content_saved               = true;
			}

			if ( ! empty( $meta ) ) {
				$this->seo->update_seo_meta( $product_id, $meta );
			}
		}

		// Update alt text.
		if ( isset( $content['alt_text'] ) && ! empty( trim( $content['alt_text'] ) ) && $featured_image_id ) {
			update_post_meta( $featured_image_id, '_wp_attachment_image_alt', sanitize_text_field( $content['alt_text'] ) );
			$updated['alt_text'] = sanitize_text_field( $content['alt_text'] );
			$content_saved       = true;
		}

		// Log the update if any content was saved.
		if ( ! empty( $updated ) ) {
			$log_user_id = null !== $user_id ? $user_id : get_current_user_id();
			$this->log->log_content_update( $product_id, $original, $updated, $log_user_id );
		}

		return $content_saved;
	}

	/**
	 * Handle AJAX request to generate product content
	 *
	 * @since 1.0.0
	 */
	public function ajax_generate_product_content() {
		// Check nonce.
		if ( ! check_ajax_referer( 'shopwr_lite_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired. Please refresh the page and try again.', 'shopwriter-lite' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'You do not have permission to edit products.', 'shopwriter-lite' ) ) );
		}

		// Get parameters.
		$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		$fields     = isset( $_POST['fields'] ) ? array_map( 'sanitize_key', (array) $_POST['fields'] ) : array();
		$operation  = isset( $_POST['operation'] ) ? sanitize_text_field( wp_unslash( $_POST['operation'] ) ) : 'generate';

		// Filter to only allowed Lite fields.
		$fields = $this->api->filter_allowed_fields( $fields );

		if ( empty( $product_id ) || empty( $fields ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid request parameters.', 'shopwriter-lite' ) ) );
		}

		try {
			$result = $this->generate_content( $product_id, $fields, $operation );

			if ( is_wp_error( $result ) ) {
				wp_send_json_error( array( 'message' => $result->get_error_message() ) );
			}

			wp_send_json_success( $result );

		} catch ( Exception $e ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An unexpected error occurred during content generation.', 'shopwriter-lite' ) ) );
		}
	}

	/**
	 * Handle AJAX request to save generated content
	 *
	 * @since 1.0.0
	 */
	public function ajax_save_generated_content() {
		// Check nonce.
		if ( ! check_ajax_referer( 'shopwr_lite_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired. Please refresh the page and try again.', 'shopwriter-lite' ) ) );
		}

		// Check permissions.
		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'You do not have permission to edit products.', 'shopwriter-lite' ) ) );
		}

		// Get parameters.
		$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$content_data = isset( $_POST['content'] ) ? (array) $_POST['content'] : array();

		if ( empty( $product_id ) || empty( $content_data ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid request parameters.', 'shopwriter-lite' ) ) );
		}

		// Sanitize content data - only allow Lite fields.
		$allowed_fields    = $this->api->get_allowed_fields();
		$sanitized_content = array();

		foreach ( $content_data as $key => $value ) {
			$key = sanitize_key( $key );

			// Only allow Lite fields.
			if ( ! in_array( $key, $allowed_fields, true ) ) {
				continue;
			}

			if ( in_array( $key, array( 'description', 'short_description' ), true ) ) {
				$sanitized_content[ $key ] = wp_kses_post( wp_unslash( $value ) );
			} else {
				$sanitized_content[ $key ] = sanitize_text_field( wp_unslash( $value ) );
			}
		}

		$user_id = get_current_user_id();

		try {
			$result = $this->save_generated_content( $product_id, $sanitized_content, $user_id );

			if ( ! $result ) {
				wp_send_json_error( array( 'message' => esc_html__( 'Failed to save content.', 'shopwriter-lite' ) ) );
			}

			wp_send_json_success(
				array(
					'message' => esc_html__( 'Content saved successfully.', 'shopwriter-lite' ),
				)
			);

		} catch ( Exception $e ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An unexpected error occurred while saving content.', 'shopwriter-lite' ) ) );
		}
	}
}
