<?php
/**
 * The logging class for product content updates - ShopWriter Lite
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Log class for handling product content update logs and reverts
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_Log {


	/**
	 * Log table name
	 *
	 * @since    1.0.0
	 * @access   protected
	 * @var      string $table_name Log table name
	 */
	protected $table_name;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		global $wpdb;
		$this->table_name = $wpdb->prefix . 'shopwr_lite_content_logs';
	}

	/**
	 * Log content update
	 *
	 * @param int      $product_id Product ID.
	 * @param array    $original Original content.
	 * @param array    $updated Updated content.
	 * @param int|null $user_id User ID (optional).
	 * @return   int|false                Log ID or false on failure
	 * @since    1.0.0
	 */
	public function log_content_update( $product_id, $original, $updated, $user_id = null ) {
		global $wpdb;

		// Sanitize product ID.
		$product_id = absint( $product_id );
		if ( ! $product_id ) {
			return false;
		}

		// If user_id is null, get current user.
		if ( null === $user_id ) {
			$user_id = get_current_user_id();
		}
		$user_id = absint( $user_id );

		// Sanitize arrays.
		$original = is_array( $original ) ? $original : array();
		$updated  = is_array( $updated ) ? $updated : array();

		// Determine which fields were updated.
		$fields_updated = array();
		foreach ( $updated as $field => $value ) {
			if ( ! isset( $original[ $field ] ) || $original[ $field ] !== $value ) {
				$fields_updated[] = sanitize_text_field( $field );
			}
		}

		// Insert log entry.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery
		$result = $wpdb->insert(
			$this->table_name,
			array(
				'product_id'       => $product_id,
				'fields_updated'   => wp_json_encode( $fields_updated ),
				'original_content' => wp_json_encode( $original ),
				'updated_content'  => wp_json_encode( $updated ),
				'user_id'          => $user_id,
				'date_created'     => current_time( 'mysql' ),
			),
			array( '%d', '%s', '%s', '%s', '%d', '%s' )
		);

		if ( false === $result ) {
			return false;
		}

		return $wpdb->insert_id;
	}

	/**
	 * Get all logs with pagination
	 *
	 * @param int $limit Number of logs to retrieve.
	 * @param int $offset Offset for pagination.
	 * @return   array          Log entries
	 * @since    1.0.0
	 */
	public function get_all_logs( $limit = 50, $offset = 0 ) {
		global $wpdb;

		$limit  = absint( $limit );
		$offset = absint( $offset );

 		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
		$logs = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT l.*, p.post_title as product_name
                FROM %s l
                LEFT JOIN %s p ON l.product_id = p.ID
                ORDER BY l.date_created DESC
                LIMIT %d OFFSET %d',
				$this->table_name,
				$wpdb->posts,
				$limit,
				$offset
			),
			ARRAY_A
		);

		if ( empty( $logs ) ) {
			return array();
		}

		// Decode JSON content and add user data.
		foreach ( $logs as &$log ) {
			$log['fields_updated']   = json_decode( $log['fields_updated'], true );
			$log['original_content'] = json_decode( $log['original_content'], true );
			$log['updated_content']  = json_decode( $log['updated_content'], true );
			$log['user']             = get_user_by( 'id', absint( $log['user_id'] ) );
		}

		return $logs;
	}

	/**
	 * Get a single log entry by ID
	 *
	 * @param int $log_id Log ID.
	 * @return   array|false    Log entry or false if not found
	 * @since    1.0.0
	 */
	public function get_log( $log_id ) {
		global $wpdb;

		$log_id = absint( $log_id );
		if ( ! $log_id ) {
			return false;
		}

	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
		$log = $wpdb->get_row(
			$wpdb->prepare(
				'SELECT l.*, p.post_title as product_name
                FROM %s l
                LEFT JOIN %s p ON l.product_id = p.ID
                WHERE l.id = %d',
				$this->table_name,
				$wpdb->posts,
				$log_id
			),
			ARRAY_A
		);

		if ( empty( $log ) ) {
			return false;
		}

		// Decode JSON content and add user data.
		$log['fields_updated']   = json_decode( $log['fields_updated'], true );
		$log['original_content'] = json_decode( $log['original_content'], true );
		$log['updated_content']  = json_decode( $log['updated_content'], true );
		$log['user']             = get_user_by( 'id', absint( $log['user_id'] ) );

		return $log;
	}

	/**
	 * Count total logs
	 *
	 * @param int|null $product_id Product ID (optional, for filtering).
	 * @return   int                     Total number of logs
	 * @since    1.0.0
	 */
	public function count_logs( $product_id = null ) {
		global $wpdb;

		if ( $product_id ) {
			$product_id = absint( $product_id );
			// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
			return (int) $wpdb->get_var(
				$wpdb->prepare(
					'SELECT COUNT(*) FROM %s WHERE product_id = %d',
					$this->table_name,
					$product_id
				)
			);
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
		return (int) $wpdb->get_var( $wpdb->prepare( 'SELECT COUNT(*) FROM %s', $this->table_name ) );
	}

	/**
	 * Revert changes from a log entry
	 *
	 * @param int $log_id Log ID.
	 * @return   boolean        True on success, false on failure
	 * @since    1.0.0
	 */
	public function revert_changes( $log_id ) {
		// Get log entry.
		$log = $this->get_log( $log_id );

		if ( ! $log ) {
			return false;
		}

		// Get product.
		$product = wc_get_product( absint( $log['product_id'] ) );

		if ( ! $product ) {
			return false;
		}

		// Check if product is in trash.
		if ( 'trash' === $product->get_status() ) {
			return false;
		}

		$original_content = is_array( $log['original_content'] ) ? $log['original_content'] : array();

		// Revert description (Lite only fields).
		if ( isset( $original_content['description'] ) ) {
			$product->set_description( wp_kses_post( $original_content['description'] ) );
		}

		// Revert short description.
		if ( isset( $original_content['short_description'] ) ) {
			$product->set_short_description( wp_kses_post( $original_content['short_description'] ) );
		}

		// Save product.
		$product->save();

		// Revert SEO meta.
		$seo  = new SHOPWR_Lite_SEO();
		$meta = array();

		if ( isset( $original_content['meta_title'] ) ) {
			$meta['title'] = $original_content['meta_title'];
		}

		if ( isset( $original_content['meta_description'] ) ) {
			$meta['description'] = $original_content['meta_description'];
		}

		if ( ! empty( $meta ) ) {
			$seo->update_seo_meta( $log['product_id'], $meta );
		}

		// Revert alt text.
		if ( isset( $original_content['alt_text'] ) ) {
			$featured_image_id = $product->get_image_id();
			if ( $featured_image_id ) {
				update_post_meta( $featured_image_id, '_wp_attachment_image_alt', sanitize_text_field( $original_content['alt_text'] ) );
			}
		}

		// Delete the log entry after revert.
		$this->delete_log( $log_id );

		return true;
	}

	/**
	 * Delete a log entry
	 *
	 * @param int $log_id Log ID.
	 * @return   boolean        True on success, false on failure
	 * @since    1.0.0
	 */
	public function delete_log( $log_id ) {
		global $wpdb;

		$log_id = absint( $log_id );
		if ( ! $log_id ) {
			return false;
		}

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
		$result = $wpdb->delete(
			$this->table_name,
			array( 'id' => $log_id ),
			array( '%d' )
		);

		return false !== $result;
	}

	/**
	 * Clear all logs
	 *
	 * @return   boolean    True on success, false on failure
	 * @since    1.0.0
	 */
	public function clear_all_logs() {
		global $wpdb;

		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
		$result = $wpdb->query( $wpdb->prepare( 'TRUNCATE TABLE %s', $this->table_name ) );

		return false !== $result;
	}

	/**
	 * AJAX handler to get update logs
	 *
	 * @since 1.0.0
	 */
	public function ajax_get_update_log() {
		if ( ! check_ajax_referer( 'shopwr_lite_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		$page     = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
		$per_page = isset( $_POST['per_page'] ) ? absint( $_POST['per_page'] ) : 20;
		$offset   = ( $page - 1 ) * $per_page;

		$logs        = $this->get_all_logs( $per_page, $offset );
		$total       = $this->count_logs();
		$total_pages = ceil( $total / $per_page );

		// Format logs for response.
		$formatted_logs = array();
		foreach ( $logs as $log ) {
			$formatted_logs[] = array(
				'id'             => absint( $log['id'] ),
				'product_id'     => absint( $log['product_id'] ),
				'product_name'   => esc_html( $log['product_name'] ),
				'fields_updated' => $log['fields_updated'],
				'original'       => $log['original_content'],
				'updated'        => $log['updated_content'],
				'user_name'      => $log['user'] ? $log['user']->display_name : esc_html__( 'Unknown', 'shopwriter-lite' ),
				'date'           => $log['date_created'],
				'date_formatted' => human_time_diff( strtotime( $log['date_created'] ), strtotime( current_time( 'mysql' ) ) ) . ' ' . esc_html__( 'ago', 'shopwriter-lite' ),
			);
		}

		wp_send_json_success(
			array(
				'logs'        => $formatted_logs,
				'total'       => $total,
				'total_pages' => $total_pages,
				'page'        => $page,
			)
		);
	}

	/**
	 * AJAX handler to revert changes
	 *
	 * @since 1.0.0
	 */
	public function ajax_revert_changes() {
		if ( ! check_ajax_referer( 'shopwr_lite_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		$log_id = isset( $_POST['log_id'] ) ? absint( $_POST['log_id'] ) : 0;

		if ( empty( $log_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid log ID.', 'shopwriter-lite' ) ) );
			return;
		}

		$result = $this->revert_changes( $log_id );

		if ( $result ) {
			wp_send_json_success(
				array(
					'message' => esc_html__( 'Changes reverted successfully.', 'shopwriter-lite' ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Failed to revert changes.', 'shopwriter-lite' ),
				)
			);
		}
	}

	/**
	 * AJAX handler to clear all logs
	 *
	 * @since 1.0.0
	 */
	public function ajax_clear_all_logs() {
		if ( ! check_ajax_referer( 'shopwr_lite_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		$result = $this->clear_all_logs();

		if ( $result ) {
			wp_send_json_success(
				array(
					'message' => esc_html__( 'All logs cleared successfully.', 'shopwriter-lite' ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Failed to clear logs.', 'shopwriter-lite' ),
				)
			);
		}
	}
}
