<?php
/**
 * The API class for communicating with Shop Writer service - Lite Version
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * API class for handling communication with Shop Writer API service
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_API {


	/**
	 * Initialize the class.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// No license handler needed in Lite version.
	}

	/**
	 * Get plugin settings
	 *
	 * @return array Plugin settings.
	 * @since  1.0.0
	 */
	public function get_settings() {
		$defaults = array(
			'api_key'                      => '',
			'model'                        => 'gpt-3.5-turbo',
			'tone'                         => 'professional',
			'source_language'              => 'en',
			'website_language'             => 'en',
			'short_description_min_length' => 20,
			'short_description_max_length' => 50,
			'description_min_length'       => 100,
			'description_max_length'       => 300,
			'meta_title_min_length'        => 50,
			'meta_title_max_length'        => 60,
			'meta_description_min_length'  => 150,
			'meta_description_max_length'  => 160,
		);

		$settings = get_option( 'shopwr_lite_settings', array() );
		return wp_parse_args( $settings, $defaults );
	}

	/**
	 * Check if API key is valid
	 *
	 * @return bool True if API key is set.
	 * @since  1.0.0
	 */
	public function has_api_key() {
		$settings = $this->get_settings();
		return ! empty( $settings['api_key'] );
	}

	/**
	 * Get allowed fields for Lite version
	 *
	 * @return array Allowed field names.
	 * @since  1.0.0
	 */
	public function get_allowed_fields() {
		return SHOPWR_LITE_ALLOWED_FIELDS;
	}

	/**
	 * Filter fields to only include allowed Lite fields
	 *
	 * @param array $fields Fields to filter.
	 * @return array Filtered fields.
	 * @since  1.0.0
	 */
	public function filter_allowed_fields( $fields ) {
		$allowed = $this->get_allowed_fields();
		return array_intersect( $fields, $allowed );
	}

	/**
	 * Generate content via API
	 *
	 * @param array $params Parameters for content generation.
	 * @return array|WP_Error Generated content or error.
	 * @since  1.0.0
	 */
	public function generate_content( $params ) {
		$settings    = $this->get_settings();
		$license_key = $settings['api_key']; // In Lite, api_key setting holds the license key.

		// Check license key.
		if ( empty( $license_key ) ) {
			return new WP_Error( 'missing_api_key', esc_html__( 'License key is required. Please add your license key in Settings.', 'shopwriter-lite' ) );
		}

		$product_data = isset( $params['product_data'] ) ? $params['product_data'] : array();
		$fields       = isset( $params['fields'] ) ? $params['fields'] : array();

		// Filter to only allowed Lite fields.
		$fields = $this->filter_allowed_fields( $fields );

		if ( empty( $fields ) ) {
			return new WP_Error( 'no_fields', esc_html__( 'No valid fields selected for generation.', 'shopwriter-lite' ) );
		}

		$api_data = array(
			'license_key'         => sanitize_text_field( $license_key ),
			'model'               => sanitize_text_field( $settings['model'] ),
			'tone'                => sanitize_text_field( $settings['tone'] ),
			'source_language'     => sanitize_text_field( $settings['source_language'] ),
			'target_language'     => sanitize_text_field( $settings['website_language'] ),
			'operation'           => isset( $params['operation'] ) ? sanitize_text_field( $params['operation'] ) : 'generate',
			'product_data'        => $product_data,
			'fields'              => array_map( 'sanitize_key', $fields ),
			'length_requirements' => $this->build_length_requirements( $settings, $fields ),
			'is_lite_version'     => true,
		);

		$response = wp_remote_post(
			SHOPWR_LITE_API_SERVER . '/generate',
			array(
				'timeout'   => 60,
				'sslverify' => true,
				'headers'   => array( 'Content-Type' => 'application/json' ),
				'body'      => wp_json_encode( $api_data ),
			)
		);

		// Process response.
		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $response_body, true );

		// Check for non-200 response codes.
		if ( 200 !== $response_code ) {
			$error_message = isset( $response_data['error'] ) ? $response_data['error'] : sprintf(
			/* translators: %d: HTTP response code */
				__( 'API returned error code: %d', 'shopwriter-lite' ),
				$response_code
			);
			return new WP_Error( 'api_error', $error_message );
		}

		// Check for empty content.
		if ( empty( $response_data['content'] ) ) {
			$error_message = isset( $response_data['error'] ) ? $response_data['error'] : __( 'API returned empty content', 'shopwriter-lite' );
			return new WP_Error( 'empty_content', $error_message );
		}

		return $response_data;
	}

	/**
	 * Build length requirements for content fields
	 *
	 * @param array $settings Plugin settings.
	 * @param array $fields Fields to generate.
	 * @return array           Length requirements.
	 * @since  1.0.0
	 */
	private function build_length_requirements( $settings, $fields ) {
		$length_requirements = array();

		foreach ( $fields as $field ) {
			switch ( $field ) {
				case 'short_description':
					$length_requirements['short_description'] = array(
						'min'         => absint( $settings['short_description_min_length'] ),
						'max'         => absint( $settings['short_description_max_length'] ),
						'unit'        => 'words',
						'instruction' => sprintf(
						/* translators: 1: Minimum words, 2: Maximum words */
							esc_html__( 'Write a short description between %1$d and %2$d words', 'shopwriter-lite' ),
							$settings['short_description_min_length'],
							$settings['short_description_max_length']
						),
					);
					break;

				case 'description':
					$length_requirements['description'] = array(
						'min'         => absint( $settings['description_min_length'] ),
						'max'         => absint( $settings['description_max_length'] ),
						'unit'        => 'words',
						'instruction' => sprintf(
						/* translators: 1: Minimum words, 2: Maximum words */
							esc_html__( 'Write a detailed product description between %1$d and %2$d words', 'shopwriter-lite' ),
							$settings['description_min_length'],
							$settings['description_max_length']
						),
					);
					break;

				case 'meta_title':
					$length_requirements['meta_title'] = array(
						'min'         => absint( $settings['meta_title_min_length'] ),
						'max'         => absint( $settings['meta_title_max_length'] ),
						'unit'        => 'characters',
						'instruction' => sprintf(
						/* translators: 1: Minimum characters, 2: Maximum characters */
							esc_html__( 'Write an SEO meta title between %1$d and %2$d characters', 'shopwriter-lite' ),
							$settings['meta_title_min_length'],
							$settings['meta_title_max_length']
						),
					);
					break;

				case 'meta_description':
					$length_requirements['meta_description'] = array(
						'min'         => absint( $settings['meta_description_min_length'] ),
						'max'         => absint( $settings['meta_description_max_length'] ),
						'unit'        => 'characters',
						'instruction' => sprintf(
						/* translators: 1: Minimum characters, 2: Maximum characters */
							esc_html__( 'Write an SEO meta description between %1$d and %2$d characters', 'shopwriter-lite' ),
							$settings['meta_description_min_length'],
							$settings['meta_description_max_length']
						),
					);
					break;

				case 'alt_text':
					$length_requirements['alt_text'] = array(
						'min'         => 5,
						'max'         => 15,
						'unit'        => 'words',
						'instruction' => esc_html__( 'Write a descriptive alt text between 5 and 15 words', 'shopwriter-lite' ),
					);
					break;
			}
		}

		return $length_requirements;
	}

	/**
	 * Get length summary for fields
	 *
	 * @param array $fields Fields to get summary for.
	 * @return array         Length summary.
	 * @since  1.0.0
	 */
	public function get_length_summary( $fields ) {
		$settings = $this->get_settings();
		$summary  = array();

		// Filter to only allowed fields.
		$fields = $this->filter_allowed_fields( $fields );

		foreach ( $fields as $field ) {
			switch ( $field ) {
				case 'short_description':
					$summary[ $field ] = sprintf(
					/* translators: 1: Minimum words, 2: Maximum words */
						esc_html__( '%1$d-%2$d words', 'shopwriter-lite' ),
						$settings['short_description_min_length'],
						$settings['short_description_max_length']
					);
					break;

				case 'description':
					$summary[ $field ] = sprintf(
					/* translators: 1: Minimum words, 2: Maximum words */
						esc_html__( '%1$d-%2$d words', 'shopwriter-lite' ),
						$settings['description_min_length'],
						$settings['description_max_length']
					);
					break;

				case 'meta_title':
					$summary[ $field ] = sprintf(
					/* translators: 1: Minimum characters, 2: Maximum characters */
						esc_html__( '%1$d-%2$d chars', 'shopwriter-lite' ),
						$settings['meta_title_min_length'],
						$settings['meta_title_max_length']
					);
					break;

				case 'meta_description':
					$summary[ $field ] = sprintf(
					/* translators: 1: Minimum characters, 2: Maximum characters */
						esc_html__( '%1$d-%2$d chars', 'shopwriter-lite' ),
						$settings['meta_description_min_length'],
						$settings['meta_description_max_length']
					);
					break;

				case 'alt_text':
					$summary[ $field ] = esc_html__( '5-15 words', 'shopwriter-lite' );
					break;

				default:
					$summary[ $field ] = esc_html__( 'Auto', 'shopwriter-lite' );
					break;
			}
		}

		return $summary;
	}

	/**
	 * Calculate credits needed for an operation
	 *
	 * @param array $params Parameters for credit calculation.
	 * @return array|WP_Error Credit information or error.
	 * @since  1.0.0
	 */
	public function calculate_credits( $params ) {
		$settings    = $this->get_settings();
		$license_key = $settings['api_key']; // In Lite, api_key setting holds the license key.

		if ( empty( $license_key ) ) {
			return new WP_Error( 'missing_api_key', esc_html__( 'License key is required.', 'shopwriter-lite' ) );
		}

		// Filter to only allowed fields.
		$fields = isset( $params['fields'] ) ? $this->filter_allowed_fields( $params['fields'] ) : array();

		$api_params = array(
			'license_key'     => sanitize_text_field( $license_key ),
			'model'           => sanitize_text_field( $settings['model'] ),
			'products_count'  => isset( $params['products_count'] ) ? absint( $params['products_count'] ) : 0,
			'fields'          => array_map( 'sanitize_key', $fields ),
			'is_lite_version' => true,
		);

		$response = wp_remote_post(
			SHOPWR_LITE_API_SERVER . '/calculate-credits',
			array(
				'timeout'   => 60,
				'sslverify' => true,
				'headers'   => array( 'Content-Type' => 'application/json' ),
				'body'      => wp_json_encode( $api_params ),
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $response_body, true );

		if ( 200 !== $response_code ) {
			$error_message = isset( $response_data['message'] ) ? sanitize_text_field( $response_data['message'] ) : esc_html__( 'Unknown error occurred', 'shopwriter-lite' );
			return new WP_Error( 'api_error', $error_message );
		}

		return $response_data;
	}

	/**
	 * Consumer key for license API
	 *
	 * @var string
	 */
	private $consumer_key = 'ck_60290b2dc0c17668e46d8865d7af10f3ff7b32db';

	/**
	 * Consumer secret for license API
	 *
	 * @var string
	 */
	private $consumer_secret = 'cs_0bcdeea6dfc936488220b095d5054321667dab14';

	/**
	 * Get available credits from API
	 *
	 * @return array|WP_Error Credit balance information or error.
	 * @since  1.0.0
	 */
	public function get_credits_balance() {
		$settings    = $this->get_settings();
		$license_key = $settings['api_key']; // In Lite, api_key is the license key.

		if ( empty( $license_key ) ) {
			return new WP_Error( 'missing_api_key', esc_html__( 'API key is required.', 'shopwriter-lite' ) );
		}

		// Use the same license validation endpoint as the Pro version.
		$validation_url = 'https://shop-writer.com/wp-json/lmfwc/v2/licenses/validate/' . sanitize_text_field( $license_key );

		$response = wp_remote_get(
			$validation_url,
			array(
				'timeout'   => 30,
				'sslverify' => true,
				'headers'   => array(
					'Content-Type'  => 'application/json',
					// This is intentionally using base64_encode() for Authorization header.
					// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode
					'Authorization' => 'Basic ' . base64_encode( $this->consumer_key . ':' . $this->consumer_secret ),
				),
			)
		);

		if ( is_wp_error( $response ) ) {
			return $response;
		}

		$response_code = wp_remote_retrieve_response_code( $response );
		$response_body = wp_remote_retrieve_body( $response );
		$response_data = json_decode( $response_body, true );

		if ( 200 !== $response_code ) {
			$error_message = isset( $response_data['message'] ) ? sanitize_text_field( $response_data['message'] ) : esc_html__( 'Could not retrieve credit balance', 'shopwriter-lite' );
			return new WP_Error( 'api_error', $error_message );
		}

		// Check if license validation was successful.
		$success = isset( $response_data['success'] ) ? $response_data['success'] : false;

		if ( ! $success || ! isset( $response_data['data'] ) ) {
			return new WP_Error( 'invalid_license', esc_html__( 'Invalid license key', 'shopwriter-lite' ) );
		}

		$api_data = $response_data['data'];

		// Get credits from license data.
		$credits_remaining = isset( $api_data['credit_left'] ) ? absint( $api_data['credit_left'] ) : 0;
		$credits_total     = isset( $api_data['credit_total'] ) ? absint( $api_data['credit_total'] ) : 0;

		return array(
			'credits_available' => $credits_remaining,
			'credits_total'     => $credits_total,
			'license_status'    => isset( $api_data['status'] ) ? absint( $api_data['status'] ) : 0,
		);
	}
}
