<?php
/**
 * Fired during plugin activation
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Fired during plugin activation.
 *
 * This class defines all code necessary to run during the plugin's activation.
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_Activator {


	/**
	 * Activate the plugin.
	 *
	 * Creates necessary database tables and sets default options.
	 *
	 * @since 1.0.0
	 */
	public static function activate() {
		self::create_tables();
		self::set_default_options();

		// Set activation flag for welcome notice.
		set_transient( 'shopwr_lite_activated', true, 30 );

		// Flush rewrite rules.
		flush_rewrite_rules();
	}

	/**
	 * Create database tables
	 *
	 * @since 1.0.0
	 */
	private static function create_tables() {
		global $wpdb;

		$charset_collate = $wpdb->get_charset_collate();

		// Content logs table.
		$table_name = $wpdb->prefix . 'shopwr_lite_content_logs';

		$sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            product_id bigint(20) unsigned NOT NULL,
            fields_updated longtext,
            original_content longtext,
            updated_content longtext,
            user_id bigint(20) unsigned DEFAULT 0,
            date_created datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY product_id (product_id),
            KEY user_id (user_id),
            KEY date_created (date_created)
        ) $charset_collate;";

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		dbDelta( $sql );
	}

	/**
	 * Set default options
	 *
	 * @since 1.0.0
	 */
	private static function set_default_options() {
		// Set default settings if not already set.
		if ( ! get_option( 'shopwr_lite_settings' ) ) {
			$default_settings = array(
				'api_key'                      => '',
				'model'                        => 'gpt-3.5-turbo',
				'tone'                         => 'professional',
				'source_language'              => 'en',
				'website_language'             => 'en',
				'short_description_min_length' => 20,
				'short_description_max_length' => 50,
				'description_min_length'       => 100,
				'description_max_length'       => 300,
				'meta_title_min_length'        => 50,
				'meta_title_max_length'        => 60,
				'meta_description_min_length'  => 150,
				'meta_description_max_length'  => 160,
			);
			update_option( 'shopwr_lite_settings', $default_settings );
		}

		// Set plugin version.
		update_option( 'shopwr_lite_version', SHOPWR_LITE_VERSION );
	}
}
