<?php
/**
 * ShopWriter Lite
 *
 * @package     SHOPWR_Lite
 *
 * Plugin Name: ShopWriter Lite
 * Plugin URI: https://shop-writer.com
 * Description: ShopWriter Lite helps you save time by generating product descriptions, SEO titles, and meta descriptions for your WooCommerce store. Upgrade to Pro for advanced features.
 * Version: 1.0.3
 * Author: Webshoptiger B.V.
 * Author URI: https://webshoptiger.com
 * Text Domain: shopwriter-lite
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 5.0
 * WC tested up to: 10.4
 * License: GPLv2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Define plugin constants.
define( 'SHOPWR_LITE_VERSION', '1.0.3' );
define( 'SHOPWR_LITE_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );
define( 'SHOPWR_LITE_PLUGIN_URL', plugin_dir_url( __FILE__ ) );
define( 'SHOPWR_LITE_PLUGIN_BASENAME', plugin_basename( __FILE__ ) );
define( 'SHOPWR_LITE_API_SERVER', 'https://shop-writer.com/wp-json/woowriter/v1' );

// Define Lite version allowed fields.
define(
	'SHOPWR_LITE_ALLOWED_FIELDS',
	array(
		'short_description',
		'description',
		'meta_title',
		'meta_description',
		'alt_text',
	)
);

/**
 * Declare WooCommerce HPOS compatibility.
 */
add_action(
	'before_woocommerce_init',
	function () {
		if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
			\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
		}
	}
);

/**
 * Check if WooCommerce is active
 *
 * @return bool True if WooCommerce is active.
 * @since 1.0.0
 */
function shopwr_lite_is_woocommerce_active() {
	$active_plugins = (array) get_option( 'active_plugins', array() );

	if ( is_multisite() ) {
		$active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );
	}

	return in_array( 'woocommerce/woocommerce.php', $active_plugins, true ) || array_key_exists( 'woocommerce/woocommerce.php', $active_plugins );
}

/**
 * Check if ShopWriter Pro is active
 *
 * @return bool True if Pro version is active.
 * @since 1.0.0
 */
function shopwr_lite_is_pro_active() {
	$active_plugins = (array) get_option( 'active_plugins', array() );

	if ( is_multisite() ) {
		$active_plugins = array_merge( $active_plugins, get_site_option( 'active_sitewide_plugins', array() ) );
	}

	return in_array( 'shop-writer/shopwriter.php', $active_plugins, true ) ||
		array_key_exists( 'shop-writer/shopwriter.php', $active_plugins ) ||
		in_array( 'shopwriter/shopwriter.php', $active_plugins, true ) ||
		array_key_exists( 'shopwriter/shopwriter.php', $active_plugins );
}

/**
 * Display admin notice if WooCommerce is not active
 *
 * @since 1.0.0
 */
function shopwr_lite_admin_notice_missing_woocommerce() {
	?>
	<div class="notice notice-error">
		<p><?php esc_html_e( 'ShopWriter Lite requires WooCommerce to be activated.', 'shopwriter-lite' ); ?></p>
	</div>
	<?php
}

/**
 * Display admin notice if Pro version is active
 *
 * @since 1.0.0
 */
function shopwr_lite_admin_notice_pro_active() {
	?>
	<div class="notice notice-warning">
		<p><?php esc_html_e( 'ShopWriter Pro is already active. ShopWriter Lite has been disabled to prevent conflicts.', 'shopwriter-lite' ); ?></p>
	</div>
	<?php
}

/**
 * Initialize plugin if WooCommerce is active, otherwise show admin notice
 *
 * @since 1.0.0
 */
function shopwr_lite_init() {
	// Check if Pro version is active.
	if ( shopwr_lite_is_pro_active() ) {
		add_action( 'admin_notices', 'shopwr_lite_admin_notice_pro_active' );
		return;
	}

	if ( shopwr_lite_is_woocommerce_active() ) {
		// Declare compatibility with HPOS.
		if ( function_exists( 'woocommerce_declare_compatibility' ) ) {
			woocommerce_declare_compatibility( 'high-performance-order-storage', __FILE__, true );
		}

		// Include necessary files.
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-core.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-api.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-admin.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-product.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-bulk.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-log.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-seo.php';
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-language-helper.php';

		// Initialize the main plugin class.
		$plugin = new SHOPWR_Lite_Core();
		$plugin->run();

	} else {
		add_action( 'admin_notices', 'shopwr_lite_admin_notice_missing_woocommerce' );
	}
}

add_action( 'plugins_loaded', 'shopwr_lite_init' );

/**
 * Plugin activation hook
 *
 * @since 1.0.0
 */
function shopwr_lite_activate() {
	require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-activator.php';
	SHOPWR_Lite_Activator::activate();
}

register_activation_hook( __FILE__, 'shopwr_lite_activate' );

/**
 * Check and create database tables if they don't exist
 *
 * This ensures tables exist even after plugin updates without reactivation
 *
 * @since 1.0.0
 */
function shopwr_lite_check_db_tables() {
	// Only run in admin and if not doing AJAX.
	if ( ! is_admin() ) {
		return;
	}

	global $wpdb;
	$table_name = $wpdb->prefix . 'shopwr_lite_content_logs';

	// Check if table exists.
	// phpcs:ignore Generic.Commenting.Todo.TaskFound
	// TODO: Fix only check for tables when activating the plugin.
	// phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery
	$table_exists = $wpdb->get_var( $wpdb->prepare( 'SHOW TABLES LIKE %s', $table_name ) );

	if ( $table_exists !== $table_name ) {
		require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-activator.php';
		SHOPWR_Lite_Activator::activate();
	}
}

add_action( 'admin_init', 'shopwr_lite_check_db_tables' );

/**
 * Plugin deactivation hook
 *
 * @since 1.0.0
 */
function shopwr_lite_deactivate() {
	require_once SHOPWR_LITE_PLUGIN_DIR . 'includes/class-shopwr-lite-deactivator.php';
	SHOPWR_Lite_Deactivator::deactivate();
}

register_deactivation_hook( __FILE__, 'shopwr_lite_deactivate' );

/**
 * Add settings link on plugin page
 *
 * @param array $links Plugin action links.
 * @return array Modified plugin action links.
 * @since 1.0.0
 */
function shopwr_lite_settings_link( $links ) {
	$settings_link = '<a href="admin.php?page=shopwriter-lite-settings">' . __( 'Settings', 'shopwriter-lite' ) . '</a>';
	$upgrade_link  = '<a href="https://shop-writer.com/pricing/?utm_source=plugin&utm_medium=link&utm_campaign=lite" target="_blank" style="color:#FF5722;font-weight:bold;">' . __( 'Upgrade to Pro', 'shopwriter-lite' ) . '</a>';
	array_unshift( $links, $settings_link );
	array_push( $links, $upgrade_link );
	return $links;
}

add_filter( 'plugin_action_links_' . SHOPWR_LITE_PLUGIN_BASENAME, 'shopwr_lite_settings_link' );
