<?php
/**
 * The SEO integration class for ShopWriter Lite
 * No Internal Linking - removed from Lite version
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * SEO integration class for Yoast SEO and Rank Math support
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_SEO {

	/**
	 * Flag for Yoast SEO plugin
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    bool
	 */
	protected $has_yoast;

	/**
	 * Flag for Rank Math plugin
	 *
	 * @since  1.0.0
	 * @access protected
	 * @var    bool
	 */
	protected $has_rank_math;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		$this->has_yoast     = defined( 'WPSEO_VERSION' );
		$this->has_rank_math = defined( 'RANK_MATH_VERSION' );
	}

	/**
	 * Check if any supported SEO plugin is active
	 *
	 * @since  1.0.0
	 * @return bool True if a supported SEO plugin is active.
	 */
	public function has_seo_plugin() {
		return $this->has_yoast || $this->has_rank_math;
	}

	/**
	 * Check if Yoast SEO is active
	 *
	 * @since  1.0.0
	 * @return bool True if Yoast SEO is active.
	 */
	public function is_yoast_active() {
		return $this->has_yoast;
	}

	/**
	 * Check if Rank Math is active
	 *
	 * @since  1.0.0
	 * @return bool True if Rank Math is active.
	 */
	public function is_rank_math_active() {
		return $this->has_rank_math;
	}

	/**
	 * Get active SEO plugin name
	 *
	 * @since  1.0.0
	 * @return string Name of the active SEO plugin or empty string.
	 */
	public function get_active_seo_plugin() {
		if ( $this->has_yoast ) {
			return 'Yoast SEO';
		} elseif ( $this->has_rank_math ) {
			return 'Rank Math';
		}

		return '';
	}

	/**
	 * Get SEO meta data for a product
	 *
	 * @since  1.0.0
	 * @param  int $product_id Product ID.
	 * @return array           SEO meta data.
	 */
	public function get_seo_meta( $product_id ) {
		$meta = array(
			'title'       => '',
			'description' => '',
		);

		$product_id = absint( $product_id );

		if ( $this->has_yoast ) {
			$meta['title']       = get_post_meta( $product_id, '_yoast_wpseo_title', true );
			$meta['description'] = get_post_meta( $product_id, '_yoast_wpseo_metadesc', true );
		} elseif ( $this->has_rank_math ) {
			$meta['title']       = get_post_meta( $product_id, 'rank_math_title', true );
			$meta['description'] = get_post_meta( $product_id, 'rank_math_description', true );
		}

		return array(
			'title'       => sanitize_text_field( $meta['title'] ),
			'description' => sanitize_text_field( $meta['description'] ),
		);
	}

	/**
	 * Update SEO meta data for a product
	 *
	 * @since  1.0.0
	 * @param  int   $product_id Product ID.
	 * @param  array $meta       SEO meta data.
	 * @return bool              True on success, false on failure.
	 */
	public function update_seo_meta( $product_id, $meta ) {
		if ( ! $this->has_seo_plugin() ) {
			return false;
		}

		$product_id = absint( $product_id );

		if ( $this->has_yoast ) {
			if ( isset( $meta['title'] ) ) {
				update_post_meta( $product_id, '_yoast_wpseo_title', sanitize_text_field( $meta['title'] ) );
			}

			if ( isset( $meta['description'] ) ) {
				update_post_meta( $product_id, '_yoast_wpseo_metadesc', sanitize_text_field( $meta['description'] ) );
			}

			return true;
		} elseif ( $this->has_rank_math ) {
			if ( isset( $meta['title'] ) ) {
				update_post_meta( $product_id, 'rank_math_title', sanitize_text_field( $meta['title'] ) );
			}

			if ( isset( $meta['description'] ) ) {
				update_post_meta( $product_id, 'rank_math_description', sanitize_text_field( $meta['description'] ) );
			}

			return true;
		}

		return false;
	}
}
