<?php
/**
 * The core plugin class for ShopWriter Lite
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Main plugin class that initializes all components
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_Core {

	/**
	 * Admin handler instance
	 *
	 * @var SHOPWR_Lite_Admin
	 */
	protected $admin;

	/**
	 * API handler instance
	 *
	 * @var SHOPWR_Lite_API
	 */
	protected $api;

	/**
	 * Product handler instance
	 *
	 * @var SHOPWR_Lite_Product
	 */
	protected $product;

	/**
	 * Bulk operations handler instance
	 *
	 * @var SHOPWR_Lite_Bulk
	 */
	protected $bulk;

	/**
	 * Log handler instance
	 *
	 * @var SHOPWR_Lite_Log
	 */
	protected $log;

	/**
	 * SEO integration handler instance
	 *
	 * @var SHOPWR_Lite_SEO
	 */
	protected $seo;

	/**
	 * Language helper instance
	 *
	 * @var SHOPWR_Lite_Language_Helper
	 */
	protected $language_helper;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		// Initialize language helper first.
		$this->language_helper = new SHOPWR_Lite_Language_Helper();

		// Initialize basic components.
		$this->api = new SHOPWR_Lite_API();
		$this->seo = new SHOPWR_Lite_SEO();
		$this->log = new SHOPWR_Lite_Log();

		// Initialize admin.
		$this->admin = new SHOPWR_Lite_Admin( $this->api, null, null, $this->log, $this->seo );

		// Initialize product handler.
		$this->product = new SHOPWR_Lite_Product( $this->api, $this->seo, $this->log, $this->admin );

		// Initialize bulk handler.
		$this->bulk = new SHOPWR_Lite_Bulk( $this->api, $this->product, $this->log, $this->admin );

		// Update admin references.
		$this->admin->set_product( $this->product );
		$this->admin->set_bulk( $this->bulk );

		// Clear language cache when translation plugins are activated/deactivated.
		add_action( 'activated_plugin', array( $this, 'clear_language_cache' ) );
		add_action( 'deactivated_plugin', array( $this, 'clear_language_cache' ) );
	}

	/**
	 * Clear language cache
	 *
	 * @since 1.0.0
	 */
	public function clear_language_cache() {
		SHOPWR_Lite_Language_Helper::clear_cache();
	}

	/**
	 * Run the plugin by registering all hooks
	 *
	 * @since 1.0.0
	 */
	public function run() {
		// Admin hooks.
		add_action( 'admin_menu', array( $this->admin, 'add_plugin_admin_menu' ) );
		add_action( 'admin_init', array( $this->admin, 'register_assets' ) );
		add_action( 'admin_enqueue_scripts', array( $this->admin, 'enqueue_styles' ) );
		add_action( 'admin_enqueue_scripts', array( $this->admin, 'enqueue_scripts' ) );

		// Product page hooks.
		add_action( 'add_meta_boxes', array( $this->product, 'add_product_meta_box' ) );
		add_action( 'wp_ajax_shopwr_lite_generate_product_content', array( $this->product, 'ajax_generate_product_content' ) );
		add_action( 'wp_ajax_shopwr_lite_save_generated_content', array( $this->product, 'ajax_save_generated_content' ) );

		// Bulk operations hooks.
		add_action( 'wp_ajax_shopwr_lite_bulk_generate_preview', array( $this->bulk, 'ajax_bulk_generate_preview' ) );
		add_action( 'wp_ajax_shopwr_lite_start_bulk_process', array( $this->bulk, 'ajax_start_bulk_process' ) );
		add_action( 'wp_ajax_shopwr_lite_process_next_batch', array( $this->bulk, 'ajax_process_next_batch' ) );
		add_action( 'wp_ajax_shopwr_lite_apply_preview_changes', array( $this->bulk, 'ajax_apply_preview_changes' ) );
		add_action( 'wp_ajax_shopwr_lite_get_bulk_product_ids', array( $this->bulk, 'ajax_get_bulk_product_ids' ) );
		add_action( 'wp_ajax_shopwr_lite_generate_page_previews', array( $this->bulk, 'ajax_generate_page_previews' ) );
		add_action( 'wp_ajax_shopwr_lite_bulk_approve_previews', array( $this->bulk, 'ajax_bulk_approve_previews' ) );
		add_action( 'wp_ajax_shopwr_lite_generate_single_preview', array( $this->bulk, 'ajax_generate_single_preview' ) );

		// Log hooks.
		add_action( 'wp_ajax_shopwr_lite_get_update_log', array( $this->log, 'ajax_get_update_log' ) );
		add_action( 'wp_ajax_shopwr_lite_revert_changes', array( $this->log, 'ajax_revert_changes' ) );
		add_action( 'wp_ajax_shopwr_lite_clear_all_logs', array( $this->log, 'ajax_clear_all_logs' ) );

		// Settings hooks.
		add_action( 'wp_ajax_shopwr_lite_get_length_settings', array( $this->admin, 'ajax_get_length_settings' ) );
	}
}
