<?php
/**
 * Bulk operations class for ShopWriter Lite
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Bulk operations class
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_Bulk {


	/**
	 * The API handler.
	 *
	 * @var SHOPWR_Lite_API
	 */
	protected $api;

	/**
	 * The product handler.
	 *
	 * @var SHOPWR_Lite_Product
	 */
	protected $product;

	/**
	 * The log handler.
	 *
	 * @var SHOPWR_Lite_Log
	 */
	protected $log;

	/**
	 * The admin handler.
	 *
	 * @var SHOPWR_Lite_Admin
	 */
	protected $admin;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param SHOPWR_Lite_API     $api The API handler.
	 * @param SHOPWR_Lite_Product $product The product handler.
	 * @param SHOPWR_Lite_Log     $log The log handler.
	 * @param SHOPWR_Lite_Admin   $admin The admin handler.
	 * @since 1.0.0
	 */
	public function __construct( $api, $product, $log, $admin ) {
		$this->api     = $api;
		$this->product = $product;
		$this->log     = $log;
		$this->admin   = $admin;
	}

	/**
	 * Get filtered products based on filters
	 *
	 * @param array $filters Filter parameters.
	 * @return array          Filtered products with stats.
	 * @since  1.0.0
	 */
	public function get_filtered_products( $filters ) {
		$args = array(
			'post_type'      => 'product',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'fields'         => 'ids',
			'no_found_rows'  => true,
		);

		$tax_query = array();

		if ( ! empty( $filters['categories'] ) ) {
			$tax_query[] = array(
				'taxonomy' => 'product_cat',
				'field'    => 'term_id',
				'terms'    => array_map( 'absint', $filters['categories'] ),
				'operator' => 'IN',
			);
		}

		if ( ! empty( $filters['attributes'] ) ) {
			foreach ( $filters['attributes'] as $taxonomy => $terms ) {
				if ( ! empty( $terms ) ) {
					$tax_query[] = array(
						'taxonomy' => sanitize_key( $taxonomy ),
						'field'    => 'term_id',
						'terms'    => array_map( 'absint', $terms ),
						'operator' => 'IN',
					);
				}
			}
		}

		if ( ! empty( $tax_query ) ) {
			// Only selecting product ids, so performance impact is minimal.
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			$args['tax_query'] = $tax_query;
		}

		$query       = new WP_Query( $args );
		$product_ids = $query->posts;

		$stats = array(
			'total_matched' => count( $product_ids ),
			'final_count'   => count( $product_ids ),
		);

		return array(
			'products' => $product_ids,
			'stats'    => $stats,
		);
	}

	/**
	 * Sanitize and normalize incoming filters array from a POST request.
	 *
	 * Ensures categories and attributes are arrays of integers and that
	 * taxonomy keys are sanitized.
	 *
	 * @param mixed $raw_filters Raw filters data (usually from wp_unslash( $_POST['filters'] ) ).
	 * @return array Sanitized filters array.
	 */
	private function sanitize_filters( $raw_filters ) {
		$filters = array();

		if ( empty( $raw_filters ) || ! is_array( $raw_filters ) ) {
			return $filters;
		}

		// Categories: expect an array of term IDs.
		if ( isset( $raw_filters['categories'] ) ) {
			$filters['categories'] = array_map( 'absint', (array) $raw_filters['categories'] );
		} else {
			$filters['categories'] = array();
		}

		// Attributes: expect an associative array taxonomy => array(term_ids).
		$filters['attributes'] = array();
		if ( isset( $raw_filters['attributes'] ) && is_array( $raw_filters['attributes'] ) ) {
			foreach ( $raw_filters['attributes'] as $taxonomy => $terms ) {
				$taxonomy = sanitize_key( $taxonomy );
				if ( empty( $taxonomy ) ) {
					continue;
				}

				$filters['attributes'][ $taxonomy ] = array_map( 'absint', (array) $terms );
			}
		}

		return $filters;
	}

	/**
	 * Get field credits
	 *
	 * @param string $field Field name.
	 * @return int           Credits for field.
	 * @since  1.0.0
	 */
	private function get_field_credits( $field ) {
		$field_credits = array(
			'description'       => 3,
			'short_description' => 2,
			'meta_title'        => 1,
			'meta_description'  => 1,
			'alt_text'          => 1,
		);

		return isset( $field_credits[ $field ] ) ? $field_credits[ $field ] : 1;
	}

	/**
	 * AJAX handler for bulk generate preview
	 *
	 * @since 1.0.0
	 */
	public function ajax_bulk_generate_preview() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired. Please refresh the page.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$filters = isset( $_POST['filters'] ) ? (array) $_POST['filters'] : array();
		$fields  = isset( $_POST['fields'] ) ? array_map( 'sanitize_key', (array) $_POST['fields'] ) : array();

		// Filter to only allowed Lite fields.
		$fields = $this->api->filter_allowed_fields( $fields );

		if ( empty( $fields ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'No valid fields selected for generation.', 'shopwriter-lite' ) ) );
			return;
		}

		try {
			$filter_array = array(
				'categories' => isset( $filters['categories'] ) ? array_map( 'absint', (array) $filters['categories'] ) : array(),
				'attributes' => isset( $filters['attributes'] ) ? (array) $filters['attributes'] : array(),
			);

			$filter_result = $this->get_filtered_products( $filter_array );
			$product_ids   = $filter_result['products'];
			$filter_stats  = $filter_result['stats'];

			if ( empty( $product_ids ) ) {
				wp_send_json_error(
					array(
						'message'      => esc_html__( 'No products match the selected filters.', 'shopwriter-lite' ),
						'filter_stats' => $filter_stats,
					)
				);
				return;
			}

			$settings            = $this->api->get_settings();
			$credits_per_product = 0;

			foreach ( $fields as $field ) {
				$credits_per_product += $this->get_field_credits( $field );
			}

			$model_multiplier       = ( 'gpt-4' === $settings['model'] ) ? 5 : 1;
			$credits_per_product    = $credits_per_product * $model_multiplier;
			$total_credits_required = count( $product_ids ) * $credits_per_product;

			// Get available credits from API.
			$credits_balance   = $this->api->get_credits_balance();
			$credits_available = 0;

			if ( ! is_wp_error( $credits_balance ) && isset( $credits_balance['credits_available'] ) ) {
				$credits_available = $credits_balance['credits_available'];
			}

			// Get sample products for preview.
			$sample_product_ids = array_slice( $product_ids, 0, 5 );
			$product_previews   = array();

			foreach ( $sample_product_ids as $product_id ) {
				$product_id = absint( $product_id );
				if ( $product_id <= 0 ) {
					continue;
				}

				$product = wc_get_product( $product_id );
				if ( $product ) {
					$product_previews[] = array(
						'id'   => $product_id,
						'name' => $product->get_name(),
						'sku'  => $product->get_sku(),
					);
				}
			}

			wp_send_json_success(
				array(
					'products_count'    => count( $product_ids ),
					'credits_required'  => $total_credits_required,
					'credits_available' => $credits_available,
					'product_previews'  => $product_previews,
					'filter_stats'      => $filter_stats,
				)
			);

		} catch ( Exception $e ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred. Please try again.', 'shopwriter-lite' ) ) );
		}
	}

	/**
	 * AJAX handler for starting bulk process
	 *
	 * @since 1.0.0
	 */
	public function ajax_start_bulk_process() {
		try {
			$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
			if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'Session expired. Please refresh the page.', 'shopwriter-lite' ) ) );
				return;
			}

			if ( ! current_user_can( 'edit_products' ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
				return;
			}

			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$filters = isset( $_POST['filters'] ) ? (array) $_POST['filters'] : array();
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$fields = isset( $_POST['fields'] ) ? array_map( 'sanitize_key', (array) $_POST['fields'] ) : array();
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
			$operation = isset( $_POST['operation'] ) ? sanitize_text_field( $_POST['operation'] ) : 'generate';

			// Filter to only allowed Lite fields.
			$fields = $this->api->filter_allowed_fields( $fields );

			if ( empty( $fields ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'No valid fields selected for generation.', 'shopwriter-lite' ) ) );
				return;
			}

			$filter_result = $this->get_filtered_products( $filters );
			$product_ids   = $filter_result['products'];

			if ( empty( $product_ids ) ) {
				wp_send_json_error( array( 'message' => esc_html__( 'No products match the selected filters.', 'shopwriter-lite' ) ) );
				return;
			}

			// Create batch process data.
			$batch_id = 'shopwr_lite_batch_' . time() . '_' . wp_rand( 1000, 9999 );

			$batch_data = array(
				'batch_id'    => $batch_id,
				'product_ids' => $product_ids,
				'fields'      => $fields,
				'operation'   => $operation,
				'total'       => count( $product_ids ),
				'processed'   => 0,
				'success'     => 0,
				'failed'      => 0,
				'started_at'  => current_time( 'mysql' ),
				'user_id'     => get_current_user_id(),
			);

			set_transient( $batch_id, $batch_data, HOUR_IN_SECONDS );

			wp_send_json_success(
				array(
					'batch_id'       => $batch_id,
					'total_products' => count( $product_ids ),
					'fields'         => $fields,
					'message'        => esc_html__( 'Bulk process started successfully.', 'shopwriter-lite' ),
				)
			);

		} catch ( Exception $e ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Failed to start bulk process.', 'shopwriter-lite' ) ) );
		}
	}

	/**
	 * AJAX handler for processing next batch
	 *
	 * @since 1.0.0
	 */
	public function ajax_process_next_batch() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired. Please refresh the page.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$batch_id = isset( $_POST['batch_id'] ) ? sanitize_text_field( $_POST['batch_id'] ) : '';

		if ( empty( $batch_id ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid batch ID.', 'shopwriter-lite' ) ) );
			return;
		}

		$batch_data = get_transient( $batch_id );

		if ( ! $batch_data ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Batch process not found or expired.', 'shopwriter-lite' ) ) );
			return;
		}

		$product_ids = $batch_data['product_ids'];
		$fields      = $batch_data['fields'];
		$operation   = $batch_data['operation'];
		$processed   = $batch_data['processed'];

		// Process one product at a time.
		if ( $processed >= count( $product_ids ) ) {
			delete_transient( $batch_id );
			wp_send_json_success(
				array(
					'completed' => true,
					'message'   => esc_html__( 'Bulk process completed.', 'shopwriter-lite' ),
					'stats'     => array(
						'total'   => $batch_data['total'],
						'success' => $batch_data['success'],
						'failed'  => $batch_data['failed'],
					),
				)
			);
			return;
		}

		$product_id = $product_ids[ $processed ];
		$product    = wc_get_product( $product_id );

		if ( ! $product ) {
			++$batch_data['processed'];
			++$batch_data['failed'];
			set_transient( $batch_id, $batch_data, HOUR_IN_SECONDS );

			wp_send_json_success(
				array(
					'completed' => false,
					'processed' => $batch_data['processed'],
					'total'     => $batch_data['total'],
					'success'   => $batch_data['success'],
					'failed'    => $batch_data['failed'],
					'current'   => array(
						'product_id' => $product_id,
						'name'       => 'Unknown',
						'status'     => 'failed',
						'message'    => esc_html__( 'Product not found.', 'shopwriter-lite' ),
					),
				)
			);
			return;
		}

		try {
			$result = $this->product->generate_content( $product_id, $fields, $operation );

			if ( is_wp_error( $result ) ) {
				++$batch_data['processed'];
				++$batch_data['failed'];
				set_transient( $batch_id, $batch_data, HOUR_IN_SECONDS );

				wp_send_json_success(
					array(
						'completed' => false,
						'processed' => $batch_data['processed'],
						'total'     => $batch_data['total'],
						'success'   => $batch_data['success'],
						'failed'    => $batch_data['failed'],
						'current'   => array(
							'product_id' => $product_id,
							'name'       => $product->get_name(),
							'status'     => 'failed',
							'message'    => $result->get_error_message(),
						),
					)
				);
				return;
			}

			// Save the generated content.
			if ( isset( $result['content'] ) ) {
				$this->product->save_generated_content( $product_id, $result['content'], $batch_data['user_id'] );
			}

			++$batch_data['processed'];
			++$batch_data['success'];
			set_transient( $batch_id, $batch_data, HOUR_IN_SECONDS );

			wp_send_json_success(
				array(
					'completed' => false,
					'processed' => $batch_data['processed'],
					'total'     => $batch_data['total'],
					'success'   => $batch_data['success'],
					'failed'    => $batch_data['failed'],
					'current'   => array(
						'product_id' => $product_id,
						'name'       => $product->get_name(),
						'status'     => 'success',
						'message'    => esc_html__( 'Content generated successfully.', 'shopwriter-lite' ),
					),
				)
			);

		} catch ( Exception $e ) {
			++$batch_data['processed'];
			++$batch_data['failed'];
			set_transient( $batch_id, $batch_data, HOUR_IN_SECONDS );

			wp_send_json_success(
				array(
					'completed' => false,
					'processed' => $batch_data['processed'],
					'total'     => $batch_data['total'],
					'success'   => $batch_data['success'],
					'failed'    => $batch_data['failed'],
					'current'   => array(
						'product_id' => $product_id,
						'name'       => $product->get_name(),
						'status'     => 'failed',
						'message'    => esc_html__( 'An unexpected error occurred.', 'shopwriter-lite' ),
					),
				)
			);
		}
	}

	/**
	 * AJAX handler to get bulk product IDs
	 *
	 * @since 1.0.0
	 */
	public function ajax_get_bulk_product_ids() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$filters = isset( $_POST['filters'] ) ? (array) $_POST['filters'] : array();
		$result  = $this->get_filtered_products( $filters );

		wp_send_json_success(
			array(
				'product_ids' => $result['products'],
				'count'       => count( $result['products'] ),
			)
		);
	}

	/**
	 * AJAX handler to generate page previews
	 *
	 * @since 1.0.0
	 */
	public function ajax_generate_page_previews() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		$product_ids = isset( $_POST['product_ids'] ) ? array_map( 'absint', (array) $_POST['product_ids'] ) : array();
		$fields      = isset( $_POST['fields'] ) ? array_map( 'sanitize_key', (array) $_POST['fields'] ) : array();
		$page        = isset( $_POST['page'] ) ? absint( $_POST['page'] ) : 1;
		$per_page    = isset( $_POST['per_page'] ) ? absint( $_POST['per_page'] ) : 10;

		// Filter to only allowed Lite fields.
		$fields = $this->api->filter_allowed_fields( $fields );

		$offset   = ( $page - 1 ) * $per_page;
		$page_ids = array_slice( $product_ids, $offset, $per_page );
		$previews = array();

		foreach ( $page_ids as $product_id ) {
			$product = wc_get_product( $product_id );
			if ( ! $product ) {
				continue;
			}

			$original = array();
			foreach ( $fields as $field ) {
				$original[ $field ] = $this->get_field_content( $product, $product_id, $field );
			}

			$previews[] = array(
				'product_id' => $product_id,
				'name'       => $product->get_name(),
				'sku'        => $product->get_sku(),
				'image_url'  => $this->get_product_image_url( $product ),
				'original'   => $original,
				'generated'  => array(),
				'status'     => 'pending',
			);
		}

		wp_send_json_success(
			array(
				'previews'    => $previews,
				'page'        => $page,
				'total'       => count( $product_ids ),
				'total_pages' => ceil( count( $product_ids ) / $per_page ),
			)
		);
	}

	/**
	 * Get field content from product
	 *
	 * @param WC_Product $product Product object.
	 * @param int        $product_id Product ID.
	 * @param string     $field Field name.
	 * @return string                 Field content.
	 * @since  1.0.0
	 */
	private function get_field_content( $product, $product_id, $field ) {
		$seo = new SHOPWR_Lite_SEO();

		switch ( $field ) {
			case 'description':
				return $product->get_description();

			case 'short_description':
				return $product->get_short_description();

			case 'meta_title':
			case 'meta_description':
				$seo_meta   = $seo->get_seo_meta( $product_id );
				$meta_field = str_replace( 'meta_', '', $field );
				return isset( $seo_meta[ $meta_field ] ) ? $seo_meta[ $meta_field ] : '';

			case 'alt_text':
				$featured_image_id = $product->get_image_id();
				if ( $featured_image_id ) {
					return get_post_meta( $featured_image_id, '_wp_attachment_image_alt', true );
				}
				return '';

			default:
				return '';
		}
	}

	/**
	 * Get product image URL
	 *
	 * @param WC_Product $product Product object.
	 * @return string              Image URL.
	 * @since  1.0.0
	 */
	private function get_product_image_url( $product ) {
		$image_id = $product->get_image_id();

		if ( $image_id ) {
			$image_data = wp_get_attachment_image_src( $image_id, 'thumbnail' );
			if ( $image_data && isset( $image_data[0] ) ) {
				return $image_data[0];
			}
		}

		if ( function_exists( 'wc_placeholder_img_src' ) ) {
			return wc_placeholder_img_src();
		}

		return '';
	}

	/**
	 * AJAX handler to bulk approve previews
	 *
	 * @since 1.0.0
	 */
	public function ajax_bulk_approve_previews() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$previews = isset( $_POST['previews'] ) ? (array) $_POST['previews'] : array();
		$user_id  = get_current_user_id();
		$results  = array(
			'success' => 0,
			'failed'  => 0,
		);

		foreach ( $previews as $preview ) {
			$product_id = isset( $preview['product_id'] ) ? absint( $preview['product_id'] ) : 0;
			$content    = isset( $preview['content'] ) ? (array) $preview['content'] : array();

			if ( empty( $product_id ) || empty( $content ) ) {
				++$results['failed'];
				continue;
			}

			// Filter content to only allowed fields.
			$allowed_fields   = $this->api->get_allowed_fields();
			$filtered_content = array();

			foreach ( $content as $key => $value ) {
				$key = sanitize_key( $key );
				if ( in_array( $key, $allowed_fields, true ) ) {
					if ( in_array( $key, array( 'description', 'short_description' ), true ) ) {
						$filtered_content[ $key ] = wp_kses_post( $value );
					} else {
						$filtered_content[ $key ] = sanitize_text_field( $value );
					}
				}
			}

			$saved = $this->product->save_generated_content( $product_id, $filtered_content, $user_id );

			if ( $saved ) {
				++$results['success'];
			} else {
				++$results['failed'];
			}
		}

		wp_send_json_success(
			array(
				'message' => sprintf(
				/* translators: 1: success count, 2: failed count */
					esc_html__( 'Applied changes: %1$d successful, %2$d failed.', 'shopwriter-lite' ),
					$results['success'],
					$results['failed']
				),
				'results' => $results,
			)
		);
	}

	/**
	 * AJAX handler for applying preview changes
	 *
	 * @since 1.0.0
	 */
	public function ajax_apply_preview_changes() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$content = isset( $_POST['content'] ) ? (array) $_POST['content'] : array();
		$user_id = get_current_user_id();

		if ( empty( $product_id ) || empty( $content ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid parameters.', 'shopwriter-lite' ) ) );
			return;
		}

		// Filter content to only allowed fields.
		$allowed_fields   = $this->api->get_allowed_fields();
		$filtered_content = array();

		foreach ( $content as $key => $value ) {
			$key = sanitize_key( $key );
			if ( in_array( $key, $allowed_fields, true ) ) {
				if ( in_array( $key, array( 'description', 'short_description' ), true ) ) {
					$filtered_content[ $key ] = wp_kses_post( $value );
				} else {
					$filtered_content[ $key ] = sanitize_text_field( $value );
				}
			}
		}

		$saved = $this->product->save_generated_content( $product_id, $filtered_content, $user_id );

		if ( $saved ) {
			wp_send_json_success(
				array(
					'message' => esc_html__( 'Changes applied successfully.', 'shopwriter-lite' ),
				)
			);
		} else {
			wp_send_json_error(
				array(
					'message' => esc_html__( 'Failed to apply changes.', 'shopwriter-lite' ),
				)
			);
		}
	}

	/**
	 * AJAX handler to generate single preview
	 *
	 * @since 1.0.0
	 */
	public function ajax_generate_single_preview() {
		$nonce = isset( $_POST['nonce'] ) ? sanitize_text_field( wp_unslash( $_POST['nonce'] ) ) : '';
		if ( ! wp_verify_nonce( $nonce, 'shopwr_lite_nonce' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired.', 'shopwriter-lite' ) ) );
			return;
		}

		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Permission denied.', 'shopwriter-lite' ) ) );
			return;
		}

		$product_id = isset( $_POST['product_id'] ) ? absint( $_POST['product_id'] ) : 0;
		$fields     = isset( $_POST['fields'] ) ? array_map( 'sanitize_key', (array) $_POST['fields'] ) : array();
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$operation = isset( $_POST['operation'] ) ? sanitize_text_field( $_POST['operation'] ) : 'generate';

		// Filter to only allowed Lite fields.
		$fields = $this->api->filter_allowed_fields( $fields );

		if ( empty( $product_id ) || empty( $fields ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Invalid parameters.', 'shopwriter-lite' ) ) );
			return;
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Product not found.', 'shopwriter-lite' ) ) );
			return;
		}

		try {
			$result = $this->product->generate_content( $product_id, $fields, $operation );

			if ( is_wp_error( $result ) ) {
				wp_send_json_error( array( 'message' => $result->get_error_message() ) );
				return;
			}

			wp_send_json_success(
				array(
					'product_id' => $product_id,
					'name'       => $product->get_name(),
					'generated'  => isset( $result['content'] ) ? $result['content'] : array(),
					'status'     => 'generated',
				)
			);

		} catch ( Exception $e ) {
			wp_send_json_error( array( 'message' => esc_html__( 'An error occurred during generation.', 'shopwriter-lite' ) ) );
		}
	}
}
