<?php
/**
 * The admin-specific functionality of ShopWriter Lite
 *
 * @since      1.0.0
 * @package    ShopWriter_Lite
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Admin class for handling admin-specific functionality
 *
 * @since 1.0.0
 */
class SHOPWR_Lite_Admin {


	/**
	 * The API handler.
	 *
	 * @var SHOPWR_Lite_API
	 */
	protected $api;

	/**
	 * The product handler.
	 *
	 * @var SHOPWR_Lite_Product
	 */
	protected $product;

	/**
	 * The bulk operations handler.
	 *
	 * @var SHOPWR_Lite_Bulk
	 */
	protected $bulk;

	/**
	 * The log handler.
	 *
	 * @var SHOPWR_Lite_Log
	 */
	protected $log;

	/**
	 * The SEO integration handler.
	 *
	 * @var SHOPWR_Lite_SEO
	 */
	protected $seo;

	/**
	 * Initialize the class and set its properties.
	 *
	 * @param SHOPWR_Lite_API          $api The API handler.
	 * @param SHOPWR_Lite_Product|null $product The product handler (can be null).
	 * @param SHOPWR_Lite_Bulk|null    $bulk The bulk operations handler (can be null).
	 * @param SHOPWR_Lite_Log          $log The log handler.
	 * @param SHOPWR_Lite_SEO          $seo The SEO integration handler.
	 * @since 1.0.0
	 */
	public function __construct( $api, $product, $bulk, $log, $seo ) {
		$this->api     = $api;
		$this->product = $product;
		$this->bulk    = $bulk;
		$this->log     = $log;
		$this->seo     = $seo;
	}

	/**
	 * Set product handler after initialization
	 *
	 * @param SHOPWR_Lite_Product $product The product handler.
	 * @since 1.0.0
	 */
	public function set_product( $product ) {
		$this->product = $product;
	}

	/**
	 * Set bulk operations handler after initialization
	 *
	 * @param SHOPWR_Lite_Bulk $bulk The bulk operations handler.
	 * @since 1.0.0
	 */
	public function set_bulk( $bulk ) {
		$this->bulk = $bulk;
	}

	/**
	 * Register the assets for the admin area.
	 *
	 * @since 1.0.0
	 */
	public function register_assets() {
		wp_register_style(
			'shopwr-lite-admin-global',
			SHOPWR_LITE_PLUGIN_URL . 'admin/css/shopwr-lite-admin-global.css',
			array(),
			SHOPWR_LITE_VERSION,
			'all'
		);

		wp_register_style(
			'shopwr-lite-admin',
			SHOPWR_LITE_PLUGIN_URL . 'admin/css/shopwr-lite-admin.css',
			array(),
			SHOPWR_LITE_VERSION,
			'all'
		);

		wp_register_style(
			'shopwr-lite-bulk',
			SHOPWR_LITE_PLUGIN_URL . 'admin/css/bulk.css',
			array( 'shopwr-lite-admin' ),
			SHOPWR_LITE_VERSION,
			'all'
		);

		wp_register_style(
			'shopwr-lite-settings',
			SHOPWR_LITE_PLUGIN_URL . 'admin/css/settings.css',
			array( 'shopwr-lite-admin' ),
			SHOPWR_LITE_VERSION,
			'all'
		);

		wp_register_script(
			'shopwr-lite-admin',
			SHOPWR_LITE_PLUGIN_URL . 'admin/js/shopwr-lite-admin.js',
			array( 'jquery' ),
			SHOPWR_LITE_VERSION,
			true
		);

		wp_register_script(
			'shopwr-lite-bulk',
			SHOPWR_LITE_PLUGIN_URL . 'admin/js/bulk.js',
			array( 'jquery', 'shopwr-lite-admin' ),
			SHOPWR_LITE_VERSION,
			true
		);

		wp_register_script(
			'shopwr-lite-settings',
			SHOPWR_LITE_PLUGIN_URL . 'admin/js/settings.js',
			array( 'jquery', 'shopwr-lite-admin' ),
			SHOPWR_LITE_VERSION,
			true
		);

		wp_register_script(
			'shopwr-lite-logs',
			SHOPWR_LITE_PLUGIN_URL . 'admin/js/logs.js',
			array( 'jquery', 'wp-util', 'shopwr-lite-admin' ),
			SHOPWR_LITE_VERSION,
			true
		);
	}

	/**
	 * Enqueue the stylesheets for the admin area.
	 *
	 * @param string $hook The current admin page hook.
	 * @since 1.0.0
	 */
	public function enqueue_styles( $hook ) {
		// Enqueue global admin styles.
		wp_enqueue_style( 'shopwr-lite-admin-global' );

		// Only load on ShopWriter Lite pages or product edit pages.
		if ( ! $hook || ! is_string( $hook ) ) {
			return;
		}

		$is_shopwriter_page = strpos( $hook, 'shopwriter-lite' ) !== false;
		$is_product_page    = 'product' === $hook;

		if ( ! $is_shopwriter_page && ! $is_product_page ) {
			return;
		}

		wp_enqueue_style( 'shopwr-lite-admin' );

		// Enqueue bulk generator styles on bulk page.
		if ( strpos( $hook, 'shopwr-lite-bulk-generator' ) !== false ) {
			wp_enqueue_style( 'shopwr-lite-bulk' );
		}

		// Enqueue settings styles on settings page.
		if ( strpos( $hook, 'shopwr-lite-settings' ) !== false ) {
			wp_enqueue_style( 'shopwr-lite-settings' );
		}
	}

	/**
	 * Register the JavaScript for the admin area.
	 *
	 * @param string $hook The current admin page hook.
	 * @since 1.0.0
	 */
	public function enqueue_scripts( $hook ) {
		// Only load on ShopWriter Lite pages or product edit pages.
		if ( ! $hook || ! is_string( $hook ) ) {
			return;
		}

		$is_shopwriter_page = strpos( $hook, 'shopwriter-lite' ) !== false;
		$is_product_page    = 'product' === $hook;

		if ( ! $is_shopwriter_page && ! $is_product_page ) {
			return;
		}

		wp_enqueue_script( 'shopwr-lite-admin' );

		// Add localized script data.
		wp_localize_script(
			'shopwr-lite-admin',
			'shopwr_lite_data',
			array(
				'ajax_url'       => admin_url( 'admin-ajax.php' ),
				'nonce'          => wp_create_nonce( 'shopwr_lite_nonce' ),
				'admin_url'      => admin_url(),
				'upgrade_url'    => 'https://shop-writer.com/pricing/?utm_source=plugin&utm_medium=link&utm_campaign=lite',
				'is_lite'        => true,
				'allowed_fields' => SHOPWR_LITE_ALLOWED_FIELDS,
				'i18n'           => array(
					'generating'                   => esc_html__( 'Generating content...', 'shopwriter-lite' ),
					'saving'                       => esc_html__( 'Saving content...', 'shopwriter-lite' ),
					'confirm_revert'               => esc_html__( 'Are you sure you want to revert these changes? This action cannot be undone.', 'shopwriter-lite' ),
					'confirm_bulk'                 => esc_html__( 'Are you sure you want to start this bulk operation? This will update content for all selected products.', 'shopwriter-lite' ),
					'processing'                   => esc_html__( 'Processing products...', 'shopwriter-lite' ),
					'refreshing'                   => esc_html__( 'Refreshing...', 'shopwriter-lite' ),
					'reverting'                    => esc_html__( 'Reverting...', 'shopwriter-lite' ),
					'revert_changes'               => esc_html__( 'Revert Changes', 'shopwriter-lite' ),
					'view_details'                 => esc_html__( 'View Details', 'shopwriter-lite' ),
					'description'                  => esc_html__( 'Description', 'shopwriter-lite' ),
					'short_description'            => esc_html__( 'Short Description', 'shopwriter-lite' ),
					'meta_title'                   => esc_html__( 'Meta Title', 'shopwriter-lite' ),
					'meta_description'             => esc_html__( 'Meta Description', 'shopwriter-lite' ),
					'no_products'                  => esc_html__( 'No products to preview', 'shopwriter-lite' ),
					'ajax_error'                   => esc_html__( 'An error occurred while processing your request. Please try again.', 'shopwriter-lite' ),
					'load_details_error'           => esc_html__( 'Failed to load log details. Please try again.', 'shopwriter-lite' ),
					'items'                        => esc_html__( 'items', 'shopwriter-lite' ),
					'processing_complete'          => esc_html__( 'Processing Complete', 'shopwriter-lite' ),
					'alt_text'                     => esc_html__( 'Alt Text', 'shopwriter-lite' ),
					'featured_image_alt_text'      => esc_html__( 'Featured Image Alt Text', 'shopwriter-lite' ),
					'preview_mode'                 => esc_html__( 'Preview Mode', 'shopwriter-lite' ),
					'preview_changes'              => esc_html__( 'Preview Changes', 'shopwriter-lite' ),
					'apply_changes'                => esc_html__( 'Apply Changes', 'shopwriter-lite' ),
					'regenerate'                   => esc_html__( 'Regenerate', 'shopwriter-lite' ),
					'approve'                      => esc_html__( 'Approve', 'shopwriter-lite' ),
					'back_to_setup'                => esc_html__( 'Back to Setup', 'shopwriter-lite' ),
					'select_all'                   => esc_html__( 'Select All', 'shopwriter-lite' ),
					'approved'                     => esc_html__( 'Approved', 'shopwriter-lite' ),
					'modified'                     => esc_html__( 'Modified', 'shopwriter-lite' ),
					'products'                     => esc_html__( 'Products', 'shopwriter-lite' ),
					'original_content'             => esc_html__( 'Original Content', 'shopwriter-lite' ),
					'generated_content'            => esc_html__( 'Generated Content', 'shopwriter-lite' ),
					'apply_approved_changes'       => esc_html__( 'Apply Approved Changes', 'shopwriter-lite' ),
					'regenerate_selected'          => esc_html__( 'Regenerate Selected', 'shopwriter-lite' ),
					'generating_previews'          => esc_html__( 'Generating previews...', 'shopwriter-lite' ),
					'no_changes_to_apply'          => esc_html__( 'No changes to apply', 'shopwriter-lite' ),
					'changes_applied_successfully' => esc_html__( 'Changes applied successfully', 'shopwriter-lite' ),
					'error_applying_changes'       => esc_html__( 'Error applying changes', 'shopwriter-lite' ),
					'words'                        => esc_html__( 'words', 'shopwriter-lite' ),
					'characters'                   => esc_html__( 'characters', 'shopwriter-lite' ),
					'chars'                        => esc_html__( 'chars', 'shopwriter-lite' ),
					'pro_feature'                  => esc_html__( 'This feature is available in ShopWriter Pro', 'shopwriter-lite' ),
					'upgrade_to_pro'               => esc_html__( 'Upgrade to Pro', 'shopwriter-lite' ),
				),
				'settings'       => $this->api->get_settings(),
			)
		);

		// Enqueue bulk generator scripts on bulk page.
		if ( strpos( $hook, 'shopwr-lite-bulk-generator' ) !== false ) {
			wp_enqueue_script( 'shopwr-lite-bulk' );

			// Add bulk-specific localized data.
			wp_localize_script(
				'shopwr-lite-bulk',
				'shopwr_lite_bulk_data',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'nonce'    => wp_create_nonce( 'shopwr_lite_nonce' ),
					'i18n'     => array(
						'loading'              => esc_html__( 'Loading...', 'shopwriter-lite' ),
						'no_categories'        => esc_html__( 'Please select at least one category.', 'shopwriter-lite' ),
						'no_fields'            => esc_html__( 'Please select at least one field to generate.', 'shopwriter-lite' ),
						'no_products'          => esc_html__( 'No products found matching your criteria.', 'shopwriter-lite' ),
						'confirm_generation'   => esc_html__( 'Confirm Generation', 'shopwriter-lite' ),
						'products'             => esc_html__( 'Products', 'shopwriter-lite' ),
						'credits_required'     => esc_html__( 'Credits Required', 'shopwriter-lite' ),
						'credits_available'    => esc_html__( 'Credits Available', 'shopwriter-lite' ),
						/* translators: 1: number of required credits, 2: number of available credits. */
						'insufficient_credits' => esc_html__( 'Insufficient credits. Required: %1$d, Available: %2$d', 'shopwriter-lite' ),
						'generation_warning'   => esc_html__( 'This will generate content for the selected products. This action cannot be undone.', 'shopwriter-lite' ),
						'cancel'               => esc_html__( 'Cancel', 'shopwriter-lite' ),
						'start_generation'     => esc_html__( 'Start Generation', 'shopwriter-lite' ),
						/* translators: %d: number of products. */
						'generation_complete'  => esc_html__( 'Generation complete! %d products processed successfully.', 'shopwriter-lite' ),
						'ajax_error'           => esc_html__( 'An error occurred. Please try again.', 'shopwriter-lite' ),
					),
				)
			);
		}

		// Enqueue settings scripts on settings page.
		if ( strpos( $hook, 'shopwr-lite-settings' ) !== false ) {
			wp_enqueue_script( 'shopwr-lite-settings' );
		}

		// Enqueue logs scripts on update logs page.
		if ( strpos( $hook, 'shopwr-lite-update-logs' ) !== false ) {
			wp_enqueue_script( 'shopwr-lite-logs' );

			// Add logs-specific localized data.
			wp_localize_script(
				'shopwr-lite-logs',
				'shopwr_lite_logs_data',
				array(
					'ajax_url' => admin_url( 'admin-ajax.php' ),
					'nonce'    => wp_create_nonce( 'shopwr_lite_nonce' ),
					'edit_url' => admin_url( 'post.php?post=__ID__&action=edit' ),
					'i18n'     => array(
						'loading'           => esc_html__( 'Loading logs...', 'shopwriter-lite' ),
						'no_logs'           => esc_html__( 'No logs found.', 'shopwriter-lite' ),
						'error'             => esc_html__( 'An error occurred.', 'shopwriter-lite' ),
						'confirm_clear'     => esc_html__( 'Are you sure you want to clear all logs? This cannot be undone.', 'shopwriter-lite' ),
						'confirm_revert'    => esc_html__( 'Are you sure you want to revert these changes?', 'shopwriter-lite' ),
						'description'       => esc_html__( 'Description', 'shopwriter-lite' ),
						'short_description' => esc_html__( 'Short Description', 'shopwriter-lite' ),
						'meta_title'        => esc_html__( 'Meta Title', 'shopwriter-lite' ),
						'meta_description'  => esc_html__( 'Meta Description', 'shopwriter-lite' ),
						'alt_text'          => esc_html__( 'Alt Text', 'shopwriter-lite' ),
						'original'          => esc_html__( 'Original', 'shopwriter-lite' ),
						'updated'           => esc_html__( 'Updated', 'shopwriter-lite' ),
						'no_details'        => esc_html__( 'No details available.', 'shopwriter-lite' ),
						'page'              => esc_html__( 'Page', 'shopwriter-lite' ),
						'of'                => esc_html__( 'of', 'shopwriter-lite' ),
						'prev'              => esc_html__( 'Previous', 'shopwriter-lite' ),
						'next'              => esc_html__( 'Next', 'shopwriter-lite' ),
					),
				)
			);
		}
	}

	/**
	 * Add plugin admin menu items.
	 *
	 * @since 1.0.0
	 */
	public function add_plugin_admin_menu() {
		// Main menu.
		add_menu_page(
			esc_html__( 'ShopWriter Lite', 'shopwriter-lite' ),
			esc_html__( 'ShopWriter Lite', 'shopwriter-lite' ),
			'manage_woocommerce',
			'shopwriter-lite',
			array( $this, 'display_plugin_admin_page' ),
			SHOPWR_LITE_PLUGIN_URL . 'assets/images/menu-logo.png',
			58
		);

		// Bulk operations submenu (available in Lite).
		add_submenu_page(
			'shopwriter-lite',
			esc_html__( 'Bulk Content Generator', 'shopwriter-lite' ),
			esc_html__( 'Bulk Generator', 'shopwriter-lite' ),
			'edit_products',
			'shopwr-lite-bulk-generator',
			array( $this, 'display_bulk_generator_page' )
		);

		// Update logs submenu (available in Lite).
		add_submenu_page(
			'shopwriter-lite',
			esc_html__( 'Update Logs', 'shopwriter-lite' ),
			esc_html__( 'Update Logs', 'shopwriter-lite' ),
			'manage_woocommerce',
			'shopwr-lite-update-logs',
			array( $this, 'display_update_logs_page' )
		);

		// PRO-only menu items (disabled with PRO badge).
		// Category Generator - PRO only.
		add_submenu_page(
			'shopwriter-lite',
			esc_html__( 'Category Generator', 'shopwriter-lite' ),
			$this->get_pro_menu_label( esc_html__( 'Category Generator', 'shopwriter-lite' ) ),
			'edit_products',
			'shopwr-lite-category-generator-pro',
			array( $this, 'display_pro_upgrade_page' )
		);

		// 301 Redirects - PRO only.
		add_submenu_page(
			'shopwriter-lite',
			esc_html__( '301 Redirects', 'shopwriter-lite' ),
			$this->get_pro_menu_label( esc_html__( '301 Redirects', 'shopwriter-lite' ) ),
			'manage_woocommerce',
			'shopwr-lite-redirects-pro',
			array( $this, 'display_pro_upgrade_page' )
		);

		// Queue - PRO only.
		add_submenu_page(
			'shopwriter-lite',
			esc_html__( 'Queue', 'shopwriter-lite' ),
			$this->get_pro_menu_label( esc_html__( 'Queue', 'shopwriter-lite' ) ),
			'manage_woocommerce',
			'shopwr-lite-queue-pro',
			array( $this, 'display_pro_upgrade_page' )
		);

		// Settings submenu (available in Lite with limited options).
		add_submenu_page(
			'shopwriter-lite',
			esc_html__( 'Settings', 'shopwriter-lite' ),
			esc_html__( 'Settings', 'shopwriter-lite' ),
			'manage_woocommerce',
			'shopwriter-lite-settings',
			array( $this, 'display_settings_page' )
		);
	}

	/**
	 * Get PRO menu label with badge
	 *
	 * @param string $label Menu label.
	 * @return string Menu label with PRO badge.
	 * @since 1.0.0
	 */
	private function get_pro_menu_label( $label ) {
		return $label . ' <span class="shopwr-lite-pro-badge">PRO</span>';
	}

	/**
	 * Display the main plugin admin page.
	 *
	 * @since 1.0.0
	 */
	public function display_plugin_admin_page() {
		include_once SHOPWR_LITE_PLUGIN_DIR . 'admin/partials/admin-display.php';
	}

	/**
	 * Display the bulk generator page.
	 *
	 * @since 1.0.0
	 */
	public function display_bulk_generator_page() {
		include_once SHOPWR_LITE_PLUGIN_DIR . 'admin/partials/bulk-generator.php';
	}

	/**
	 * Display the update logs page.
	 *
	 * @since 1.0.0
	 */
	public function display_update_logs_page() {
		include_once SHOPWR_LITE_PLUGIN_DIR . 'admin/partials/update-logs.php';
	}

	/**
	 * Display the PRO upgrade page for locked features
	 *
	 * @since 1.0.0
	 */
	public function display_pro_upgrade_page() {
		include_once SHOPWR_LITE_PLUGIN_DIR . 'admin/partials/pro-upgrade.php';
	}

	/**
	 * AJAX handler to get length settings
	 *
	 * @since 1.0.0
	 */
	public function ajax_get_length_settings() {
		// Check nonce.
		if ( ! check_ajax_referer( 'shopwr_lite_nonce', 'nonce', false ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'Session expired. Please refresh the page and try again.', 'shopwriter-lite' ) ) );
			return;
		}

		// Check permissions.
		if ( ! current_user_can( 'edit_products' ) ) {
			wp_send_json_error( array( 'message' => esc_html__( 'You do not have permission to access settings.', 'shopwriter-lite' ) ) );
			return;
		}

		$settings = $this->api->get_settings();

		// Only return length settings for Lite allowed fields.
		$length_settings = array(
			'short_description'  => array(
				'min'  => isset( $settings['short_description_min_length'] ) ? $settings['short_description_min_length'] : 20,
				'max'  => isset( $settings['short_description_max_length'] ) ? $settings['short_description_max_length'] : 50,
				'unit' => 'words',
			),
			'description'        => array(
				'min'  => isset( $settings['description_min_length'] ) ? $settings['description_min_length'] : 100,
				'max'  => isset( $settings['description_max_length'] ) ? $settings['description_max_length'] : 300,
				'unit' => 'words',
			),
			'meta_title'         => array(
				'min'  => isset( $settings['meta_title_min_length'] ) ? $settings['meta_title_min_length'] : 50,
				'max'  => isset( $settings['meta_title_max_length'] ) ? $settings['meta_title_max_length'] : 60,
				'unit' => 'characters',
			),
			'meta_description'   => array(
				'min'  => isset( $settings['meta_description_min_length'] ) ? $settings['meta_description_min_length'] : 150,
				'max'  => isset( $settings['meta_description_max_length'] ) ? $settings['meta_description_max_length'] : 160,
				'unit' => 'characters',
			),
			'alt_text'           => array(
				'min'  => 5,
				'max'  => 15,
				'unit' => 'words',
			),
			'validation_enabled' => false, // Lite version does not have length validation.
		);

		wp_send_json_success( $length_settings );
	}

	/**
	 * Display the settings page.
	 *
	 * @since 1.0.0
	 */
	public function display_settings_page() {
		// Process form submission.
		// phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['shopwr_lite_save_settings'] ) ) {
			$this->process_settings_form();
		}

		include_once SHOPWR_LITE_PLUGIN_DIR . 'admin/partials/settings.php';
	}

	/**
	 * Process settings form submission
	 *
	 * @since 1.0.0
	 */
	private function process_settings_form() {
		// Verify nonce.
		if ( ! isset( $_POST['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['_wpnonce'] ), 'shopwr_lite_settings_nonce' ) ) ) {
			add_settings_error(
				'shopwr_lite_settings',
				'nonce_failed',
				esc_html__( 'Security check failed. Please try again.', 'shopwriter-lite' ),
				'error'
			);
			return;
		}

		// Lite version only saves basic settings.
		$settings = array(
			'api_key'          => isset( $_POST['api_key'] ) ? sanitize_text_field( wp_unslash( $_POST['api_key'] ) ) : '',
			'model'            => isset( $_POST['model'] ) ? sanitize_text_field( wp_unslash( $_POST['model'] ) ) : 'gpt-3.5-turbo',
			'tone'             => isset( $_POST['tone'] ) ? sanitize_text_field( wp_unslash( $_POST['tone'] ) ) : 'professional',
			'source_language'  => isset( $_POST['source_language'] ) ? sanitize_text_field( wp_unslash( $_POST['source_language'] ) ) : 'en',
			'website_language' => isset( $_POST['website_language'] ) ? sanitize_text_field( wp_unslash( $_POST['website_language'] ) ) : 'en',
		);

		// Update settings.
		update_option( 'shopwr_lite_settings', $settings );

		add_settings_error(
			'shopwr_lite_settings',
			'settings_updated',
			esc_html__( 'Settings saved successfully.', 'shopwriter-lite' ),
			'updated'
		);
	}
}
