/**
 * ShopWriter Lite Admin JavaScript
 *
 * @since 1.0.0
 * @package ShopWriter_Lite
 */

(function ($) {
	'use strict';

	// Namespace.
	var ShopWriterLite = {
		data: window.shopwr_lite_data || {},

		init: function () {
			this.bindEvents();
			this.initProductMetaBox();
		},

		bindEvents: function () {
			// Product meta box generate button.
			$( document ).on( 'click', '#shopwr-lite-generate', this.handleProductGenerate.bind( this ) );
			$( document ).on( 'click', '#shopwr-lite-apply', this.handleApplyContent.bind( this ) );
			$( document ).on( 'click', '#shopwr-lite-regenerate', this.handleRegenerate.bind( this ) );
			$( document ).on( 'click', '#shopwr-lite-cancel', this.handleCancel.bind( this ) );
		},

		initProductMetaBox: function () {
			// Initialize product meta box functionality.
		},

		handleProductGenerate: function (e) {
			e.preventDefault();

			var $button    = $( e.currentTarget );
			var productId  = $button.data( 'product-id' );
			var $container = $button.closest( '.shopwr-lite-product-meta-box' );
			var $spinner   = $container.find( '.spinner' );
			var $message   = $container.find( '#shopwr-lite-message' );
			var $preview   = $container.find( '#shopwr-lite-preview' );

			// Get selected fields.
			var fields = [];
			$container.find( 'input[name="shopwr_fields[]"]:checked' ).each(
				function () {
					fields.push( $( this ).val() );
				}
			);

			if (fields.length === 0) {
				this.showMessage( $message, this.data.i18n.no_fields || 'Please select at least one field.', 'error' );
				return;
			}

			// Get operation.
			var operation = $container.find( 'input[name="shopwr_operation"]:checked' ).val() || 'generate';

			// Show loading state.
			$button.prop( 'disabled', true );
			$spinner.addClass( 'is-active' );
			$message.hide();
			$preview.hide();

			// Make AJAX request.
			$.ajax(
				{
					url: this.data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_generate_product_content',
						nonce: this.data.nonce,
						product_id: productId,
						fields: fields,
						operation: operation
					},
					success: function (response) {
						$button.prop( 'disabled', false );
						$spinner.removeClass( 'is-active' );

						if (response.success) {
							ShopWriterLite.showPreview( $preview, response.data );
						} else {
							ShopWriterLite.showMessage( $message, response.data.message || 'An error occurred.', 'error' );
						}
					},
					error: function () {
						$button.prop( 'disabled', false );
						$spinner.removeClass( 'is-active' );
						ShopWriterLite.showMessage( $message, ShopWriterLite.data.i18n.ajax_error || 'An error occurred. Please try again.', 'error' );
					}
				}
			);
		},

		showPreview: function ($preview, data) {
			var $content = $preview.find( '.shopwr-lite-preview-content' );
			$content.empty();

			if (data.content) {
				var fieldLabels = {
					'short_description': 'Short Description',
					'description': 'Product Description',
					'meta_title': 'Meta Title',
					'meta_description': 'Meta Description',
					'alt_text': 'Featured Image Alt Text'
				};

				for (var field in data.content) {
					if (data.content.hasOwnProperty( field )) {
						var label = fieldLabels[field] || field;
						var value = data.content[field];

						var $field = $(
							'<div class="preview-field">' +
							'<label>' + label + '</label>' +
							'<div class="preview-value"></div>' +
							'</div>'
						);

						$field.find( '.preview-value' ).text( value );
						$content.append( $field );
					}
				}
			}

			$preview.data( 'content', data.content );
			$preview.show();
		},

		handleApplyContent: function (e) {
			e.preventDefault();

			var $button    = $( e.currentTarget );
			var $container = $button.closest( '.shopwr-lite-product-meta-box' );
			var $preview   = $container.find( '#shopwr-lite-preview' );
			var $message   = $container.find( '#shopwr-lite-message' );
			var content    = $preview.data( 'content' );
			var productId  = $container.find( '#shopwr-lite-generate' ).data( 'product-id' );

			if ( ! content) {
				return;
			}

			$button.prop( 'disabled', true );

			$.ajax(
				{
					url: this.data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_save_generated_content',
						nonce: this.data.nonce,
						product_id: productId,
						content: content
					},
					success: function (response) {
						$button.prop( 'disabled', false );

						if (response.success) {
							$preview.hide();
							ShopWriterLite.showMessage( $message, response.data.message || 'Content saved successfully.', 'success' );

							// Reload the page after a short delay to show updated content.
							setTimeout(
								function () {
									window.location.reload();
								},
								1500
							);
						} else {
							ShopWriterLite.showMessage( $message, response.data.message || 'Failed to save content.', 'error' );
						}
					},
					error: function () {
						$button.prop( 'disabled', false );
						ShopWriterLite.showMessage( $message, ShopWriterLite.data.i18n.ajax_error || 'An error occurred.', 'error' );
					}
				}
			);
		},

		handleRegenerate: function (e) {
			e.preventDefault();
			var $container = $( e.currentTarget ).closest( '.shopwr-lite-product-meta-box' );
			$container.find( '#shopwr-lite-generate' ).trigger( 'click' );
		},

		handleCancel: function (e) {
			e.preventDefault();
			var $container = $( e.currentTarget ).closest( '.shopwr-lite-product-meta-box' );
			$container.find( '#shopwr-lite-preview' ).hide();
		},

		showMessage: function ($element, message, type) {
			$element
				.removeClass( 'success error' )
				.addClass( type )
				.html( message )
				.show();
		}
	};

	// Initialize on document ready.
	$( document ).ready(
		function () {
			ShopWriterLite.init();
		}
	);

	// Expose to window for debugging.
	window.ShopWriterLite = ShopWriterLite;

})( jQuery );
