/**
 * ShopWriter Lite Settings JavaScript
 *
 * @since 1.0.0
 * @package ShopWriter_Lite
 */

jQuery( document ).ready(
	function ($) {
		'use strict';

		// ================================
		// LENGTH VALIDATION
		// ================================

		/**
		 * Validate length range (min must be less than max)
		 */
		function validateLengthRange(minField, maxField) {
			var minVal = parseInt( $( minField ).val() );
			var maxVal = parseInt( $( maxField ).val() );

			if (minVal >= maxVal) {
				$( maxField ).val( minVal + 1 );
			}
		}

		// Add validation listeners for length fields.
		$( '#short_description_min_length, #short_description_max_length' ).on(
			'change',
			function () {
				validateLengthRange( '#short_description_min_length', '#short_description_max_length' );
			}
		);

		$( '#description_min_length, #description_max_length' ).on(
			'change',
			function () {
				validateLengthRange( '#description_min_length', '#description_max_length' );
			}
		);

		$( '#meta_title_min_length, #meta_title_max_length' ).on(
			'change',
			function () {
				validateLengthRange( '#meta_title_min_length', '#meta_title_max_length' );
			}
		);

		$( '#meta_description_min_length, #meta_description_max_length' ).on(
			'change',
			function () {
				validateLengthRange( '#meta_description_min_length', '#meta_description_max_length' );
			}
		);

		// ================================
		// PRO SECTION HANDLING
		// ================================

		/**
		 * Disable all inputs in PRO sections
		 */
		function disableProSections() {
			$( '.shopwr-lite-pro-section.disabled' ).each(
				function () {
					$( this ).find( 'input, select, textarea, button' ).prop( 'disabled', true );
				}
			);
		}

		// Initialize PRO section handling.
		disableProSections();

		// Prevent clicks on disabled sections.
		$( document ).on(
			'click',
			'.shopwr-lite-pro-section.disabled',
			function (e) {
				if ( ! $( e.target ).is( 'a' )) {
					e.preventDefault();
					e.stopPropagation();
				}
			}
		);

		// ================================
		// API KEY VALIDATION
		// ================================

		/**
		 * Visual feedback for API key field
		 */
		$( '#api_key' ).on(
			'input',
			function () {
				var $field = $( this );
				var value  = $field.val().trim();

				if (value.length > 0) {
					$field.removeClass( 'error' ).addClass( 'valid' );
				} else {
					$field.removeClass( 'valid error' );
				}
			}
		);

		// ================================
		// SETTINGS FORM HANDLING
		// ================================

		/**
		 * Show unsaved changes warning
		 */
		var formChanged = false;

		$( '.shopwr-lite-settings-form input, .shopwr-lite-settings-form select, .shopwr-lite-settings-form textarea' ).on(
			'change input',
			function () {
				formChanged = true;
			}
		);

		$( window ).on(
			'beforeunload',
			function () {
				if (formChanged) {
					return 'You have unsaved changes. Are you sure you want to leave?';
				}
			}
		);

		// Clear warning on form submit.
		$( '.shopwr-lite-settings-form' ).on(
			'submit',
			function () {
				formChanged = false;
			}
		);

		// ================================
		// TABS HANDLING
		// ================================

		/**
		 * Initialize tabs if present
		 */
		function initTabs() {
			var $tabs = $( '.shopwr-lite-settings-tabs' );

			if ($tabs.length === 0) {
				return;
			}

			var $tabLinks    = $tabs.find( '.shopwr-lite-tab-link' );
			var $tabContents = $( '.shopwr-lite-tab-content' );

			// Get active tab from URL or default to first.
			var activeTab = getUrlParameter( 'tab' ) || $tabLinks.first().data( 'tab' );

			// Show active tab.
			showTab( activeTab );

			// Bind tab clicks.
			$tabLinks.on(
				'click',
				function (e) {
					e.preventDefault();
					var tab = $( this ).data( 'tab' );
					showTab( tab );

					// Update URL without reload.
					var newUrl = updateUrlParameter( window.location.href, 'tab', tab );
					window.history.pushState( {}, '', newUrl );
				}
			);

			function showTab(tab) {
				$tabLinks.removeClass( 'active' );
				$tabContents.removeClass( 'active' ).hide();

				$tabLinks.filter( '[data-tab="' + tab + '"]' ).addClass( 'active' );
				$( '#tab-' + tab ).addClass( 'active' ).show();
			}
		}

		// Initialize tabs.
		initTabs();

		// ================================
		// UTILITY FUNCTIONS
		// ================================

		/**
		 * Get URL parameter
		 */
		function getUrlParameter(name) {
			var urlParams = new URLSearchParams( window.location.search );
			return urlParams.get( name );
		}

		/**
		 * Update URL parameter
		 */
		function updateUrlParameter(url, key, value) {
			var urlObj = new URL( url );
			urlObj.searchParams.set( key, value );
			return urlObj.toString();
		}

		// ================================
		// PRESET BUTTONS
		// ================================

		var presets = {
			'brief-short-desc': { min: 20, max: 40 },
			'standard-short-desc': { min: 40, max: 60 },
			'brief-desc': { min: 50, max: 100 },
			'standard-desc': { min: 100, max: 200 },
			'detailed-desc': { min: 200, max: 400 }
		};

		// Apply preset on button click.
		$( document ).on(
			'click',
			'.shopwr-lite-preset-btn',
			function () {
				var preset = $( this ).data( 'preset' );
				var target = $( this ).data( 'target' );

				if (presets[preset]) {
					$( '#' + target + '_min_length' ).val( presets[preset].min );
					$( '#' + target + '_max_length' ).val( presets[preset].max );
					formChanged = true;
				}
			}
		);

		// ================================
		// UPGRADE PROMPTS
		// ================================

		/**
		 * Handle upgrade link clicks
		 */
		$( document ).on(
			'click',
			'.shopwr-lite-upgrade-link',
			function (e) {
				// Allow normal link behavior but track the click.
				if (typeof gtag === 'function') {
					gtag(
						'event',
						'upgrade_click',
						{
							'event_category': 'ShopWriter Lite',
							'event_label': $( this ).data( 'feature' ) || 'Unknown'
						}
					);
				}
			}
		);

		// ================================
		// RESPONSIVE HANDLING
		// ================================

		/**
		 * Adjust layout for mobile
		 */
		function handleResponsive() {
			var windowWidth = $( window ).width();

			if (windowWidth < 782) {
				$( '.shopwr-lite-settings-form' ).addClass( 'mobile-view' );
			} else {
				$( '.shopwr-lite-settings-form' ).removeClass( 'mobile-view' );
			}
		}

		// Initial check.
		handleResponsive();

		// Check on resize.
		$( window ).on( 'resize', handleResponsive );

	}
);
