/**
 * ShopWriter Lite Update Logs JavaScript
 *
 * @since 1.0.0
 * @package ShopWriter_Lite
 */

(function ($) {
	'use strict';

	var ShopWriterLiteLogs = {
		data: window.shopwr_lite_logs_data || {},
		currentPage: 1,
		perPage: 20,
		currentLogId: null,
		logsCache: {},

		init: function () {
			this.bindEvents();
			this.loadLogs();
		},

		bindEvents: function () {
			// Refresh button.
			$( '#refresh-logs' ).on( 'click', this.handleRefresh.bind( this ) );

			// Clear all logs.
			$( '#clear-all-logs' ).on( 'click', this.handleClearAll.bind( this ) );

			// View details button (delegated).
			$( document ).on( 'click', '.view-details', this.handleViewDetails.bind( this ) );

			// Revert button (delegated).
			$( document ).on( 'click', '.revert-btn', this.handleRevert.bind( this ) );

			// Panel revert button.
			$( '#panel-revert-btn' ).on( 'click', this.handlePanelRevert.bind( this ) );

			// Panel close buttons.
			$( '#close-panel, #close-panel-btn, #log-panel-overlay' ).on( 'click', this.closePanel.bind( this ) );

			// ESC key to close panel.
			$( document ).on(
				'keydown',
				function (e) {
					if (e.keyCode === 27 && $( '#log-details-panel' ).hasClass( 'open' )) {
						ShopWriterLiteLogs.closePanel();
					}
				}
			);

			// Pagination (delegated).
			$( document ).on( 'click', '.shopwr-lite-logs-pagination a', this.handlePagination.bind( this ) );
		},

		loadLogs: function (page) {
			var self         = this;
			page             = page || 1;
			this.currentPage = page;

			var $tbody = $( '#logs-body' );
			$tbody.html( '<tr class="no-items"><td colspan="5">' + (this.data.i18n.loading || 'Loading logs...') + '</td></tr>' );

			$.ajax(
				{
					url: this.data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_get_update_log',
						nonce: this.data.nonce,
						page: page,
						per_page: this.perPage
					},
					success: function (response) {
						if (response.success && response.data.logs) {
							self.renderLogs( response.data.logs );
							self.renderPagination( response.data );

							// Cache logs for quick detail view.
							response.data.logs.forEach(
								function (log) {
									self.logsCache[log.id] = log;
								}
							);
						} else {
							$tbody.html( '<tr class="no-items"><td colspan="5">' + (self.data.i18n.no_logs || 'No logs found.') + '</td></tr>' );
						}
					},
					error: function () {
						$tbody.html( '<tr class="no-items"><td colspan="5">' + (self.data.i18n.error || 'Error loading logs.') + '</td></tr>' );
					}
				}
			);
		},

		renderLogs: function (logs) {
			var self     = this;
			var $tbody   = $( '#logs-body' );
			var template = wp.template( 'log-row' );

			if (logs.length === 0) {
				$tbody.html( '<tr class="no-items"><td colspan="5">' + (this.data.i18n.no_logs || 'No logs found.') + '</td></tr>' );
				return;
			}

			$tbody.empty();

			logs.forEach(
				function (log) {
					log.edit_link = self.data.edit_url.replace( '__ID__', log.product_id );
					$tbody.append( template( log ) );
				}
			);
		},

		renderPagination: function (data) {
			var $pagination = $( '#logs-pagination' );
			$pagination.empty();

			if (data.total_pages <= 1) {
				return;
			}

			var html = '<span class="pagination-info">' +
						(this.data.i18n.page || 'Page') + ' ' + data.page + ' ' +
						(this.data.i18n.of || 'of') + ' ' + data.total_pages +
						'</span>';

			if (data.page > 1) {
				html += '<a href="#" class="button" data-page="' + (data.page - 1) + '">&laquo; ' + (this.data.i18n.prev || 'Previous') + '</a> ';
			}

			if (data.page < data.total_pages) {
				html += '<a href="#" class="button" data-page="' + (data.page + 1) + '">' + (this.data.i18n.next || 'Next') + ' &raquo;</a>';
			}

			$pagination.html( html );
		},

		handleRefresh: function (e) {
			e.preventDefault();
			this.logsCache = {};
			this.loadLogs( this.currentPage );
		},

		handleClearAll: function (e) {
			e.preventDefault();

			if ( ! confirm( this.data.i18n.confirm_clear || 'Are you sure you want to clear all logs? This cannot be undone.' )) {
				return;
			}

			var self    = this;
			var $button = $( e.currentTarget );
			$button.prop( 'disabled', true );

			$.ajax(
				{
					url: this.data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_clear_all_logs',
						nonce: this.data.nonce
					},
					success: function (response) {
						$button.prop( 'disabled', false );

						if (response.success) {
							self.logsCache = {};
							self.loadLogs( 1 );
							self.showNotice( 'success', response.data.message || 'All logs cleared.' );
						} else {
							self.showNotice( 'error', response.data.message || 'Failed to clear logs.' );
						}
					},
					error: function () {
						$button.prop( 'disabled', false );
						self.showNotice( 'error', self.data.i18n.error || 'An error occurred.' );
					}
				}
			);
		},

		handleViewDetails: function (e) {
			e.preventDefault();

			var logId         = $( e.currentTarget ).data( 'log-id' );
			this.currentLogId = logId;

			var log = this.logsCache[logId];
			if (log) {
				this.showLogDetails( log );
			}
		},

		showLogDetails: function (log) {
			var self     = this;
			var $content = $( '#log-details-content' );
			var html     = '';

			var fieldLabels = {
				'description': this.data.i18n.description || 'Description',
				'short_description': this.data.i18n.short_description || 'Short Description',
				'meta_title': this.data.i18n.meta_title || 'Meta Title',
				'meta_description': this.data.i18n.meta_description || 'Meta Description',
				'alt_text': this.data.i18n.alt_text || 'Alt Text'
			};

			// Add product info header.
			html += '<div class="log-product-info" style="margin-bottom: 20px; padding-bottom: 15px; border-bottom: 1px solid #dcdcde;">';
			html += '<strong style="font-size: 14px;">' + self.escapeHtml( log.product_name || 'Product #' + log.product_id ) + '</strong>';
			html += '<br><small style="color: #646970;">' + (log.date_formatted || log.date) + '</small>';
			html += '</div>';

			if (log.fields_updated && log.fields_updated.length > 0) {
				log.fields_updated.forEach(
					function (field) {
						var label    = fieldLabels[field] || field.replace( /_/g, ' ' );
						var original = log.original && log.original[field] ? log.original[field] : '';
						var updated  = log.updated && log.updated[field] ? log.updated[field] : '';

						html += '<div class="log-detail-field">';
						html += '<h4>' + self.escapeHtml( label ) + '</h4>';
						html += '<div class="log-detail-comparison">';
						html += '<div class="log-detail-original">';
						html += '<strong>' + (self.data.i18n.original || 'Original') + '</strong>';
						html += '<div class="log-detail-value">' + self.escapeHtml( original ) + '</div>';
						html += '</div>';
						html += '<div class="log-detail-updated">';
						html += '<strong>' + (self.data.i18n.updated || 'Updated') + '</strong>';
						html += '<div class="log-detail-value">' + self.escapeHtml( updated ) + '</div>';
						html += '</div>';
						html += '</div>';
						html += '</div>';
					}
				);
			} else {
				html += '<p style="color: #646970;">' + (this.data.i18n.no_details || 'No details available.') + '</p>';
			}

			$content.html( html );
			this.openPanel();
		},

		openPanel: function () {
			$( '#log-panel-overlay' ).addClass( 'open' );
			$( '#log-details-panel' ).addClass( 'open' );
			$( 'body' ).css( 'overflow', 'hidden' );
		},

		closePanel: function () {
			$( '#log-panel-overlay' ).removeClass( 'open' );
			$( '#log-details-panel' ).removeClass( 'open' );
			$( 'body' ).css( 'overflow', '' );
			this.currentLogId = null;
		},

		escapeHtml: function (text) {
			if ( ! text) {
				return '';
			}
			var div         = document.createElement( 'div' );
			div.textContent = text;
			return div.innerHTML;
		},

		handleRevert: function (e) {
			e.preventDefault();

			var logId = $( e.currentTarget ).data( 'log-id' );

			if ( ! confirm( this.data.i18n.confirm_revert || 'Are you sure you want to revert these changes?' )) {
				return;
			}

			this.revertLog( logId );
		},

		handlePanelRevert: function (e) {
			e.preventDefault();

			if ( ! this.currentLogId) {
				return;
			}

			if ( ! confirm( this.data.i18n.confirm_revert || 'Are you sure you want to revert these changes?' )) {
				return;
			}

			this.revertLog( this.currentLogId );
		},

		revertLog: function (logId) {
			var self = this;

			$.ajax(
				{
					url: this.data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_revert_changes',
						nonce: this.data.nonce,
						log_id: logId
					},
					success: function (response) {
						if (response.success) {
							self.closePanel();
							delete self.logsCache[logId];
							self.loadLogs( self.currentPage );
							self.showNotice( 'success', response.data.message || 'Changes reverted successfully.' );
						} else {
							self.showNotice( 'error', response.data.message || 'Failed to revert changes.' );
						}
					},
					error: function () {
						self.showNotice( 'error', self.data.i18n.error || 'An error occurred.' );
					}
				}
			);
		},

		handlePagination: function (e) {
			e.preventDefault();
			var page = $( e.currentTarget ).data( 'page' );
			if (page) {
				this.loadLogs( page );
			}
		},

		showNotice: function (type, message) {
			var $notice = $( '<div class="notice notice-' + type + ' is-dismissible"><p>' + message + '</p></div>' );
			$( '.shopwr-lite-update-logs h1' ).after( $notice );

			// Auto dismiss after 5 seconds.
			setTimeout(
				function () {
					$notice.fadeOut(
						function () {
							$( this ).remove();
						}
					);
				},
				5000
			);
		}
	};

	// Initialize on document ready.
	$( document ).ready(
		function () {
			if ($( '.shopwr-lite-update-logs' ).length > 0) {
				ShopWriterLiteLogs.init();
			}
		}
	);

})( jQuery );
