/**
 * ShopWriter Lite Bulk Generator JavaScript
 *
 * @since 1.0.0
 * @package ShopWriter_Lite
 */

jQuery( document ).ready(
	function ($) {
		'use strict';

		// ================================
		// GLOBAL VARIABLES
		// ================================
		var previewData         = {};
		var isProcessingRequest = false;
		var currentBatchId      = null;
		var processedCount      = 0;
		var totalProducts       = 0;
		var successCount        = 0;
		var failedCount         = 0;

		// ================================
		// UTILITY FUNCTIONS
		// ================================

		/**
		 * Show notification messages
		 */
		function showNotification(type, message, duration) {
			duration = duration || 4000;

			var colorMap = {
				'success': { bg: '#d1ecf1', border: '#bee5eb', color: '#0c5460' },
				'error': { bg: '#f8d7da', border: '#f5c6cb', color: '#721c24' },
				'warning': { bg: '#fff3cd', border: '#ffeaa7', color: '#856404' },
				'info': { bg: '#d1ecf1', border: '#bee5eb', color: '#0c5460' }
			};

			var iconMap = {
				'info': 'dashicons-info',
				'success': 'dashicons-yes-alt',
				'warning': 'dashicons-warning',
				'error': 'dashicons-dismiss'
			};

			// Remove existing notifications.
			$( '.shopwr-lite-notification' ).remove();

			var $notification = $( '<div class="shopwr-lite-notification">' )
			.css(
				{
					'position': 'fixed',
					'top': '32px',
					'right': '20px',
					'background': colorMap[type].bg,
					'border': '1px solid ' + colorMap[type].border,
					'color': colorMap[type].color,
					'padding': '15px 20px',
					'border-radius': '6px',
					'box-shadow': '0 4px 12px rgba(0,0,0,0.15)',
					'z-index': '999999',
					'max-width': '400px',
					'display': 'flex',
					'align-items': 'center',
					'gap': '10px'
				}
			)
				.html(
					'<span class="dashicons ' + iconMap[type] + '" style="font-size: 18px;"></span>' +
					'<span>' + message + '</span>' +
					'<button type="button" class="shopwr-lite-notification-close" style="background: none; border: none; color: inherit; cursor: pointer; margin-left: 10px; padding: 2px;">' +
					'<span class="dashicons dashicons-no-alt"></span>' +
					'</button>'
				);

			$( 'body' ).append( $notification );

			// Close button.
			$notification.find( '.shopwr-lite-notification-close' ).on(
				'click',
				function () {
					$notification.fadeOut(
						300,
						function () {
							$( this ).remove(); }
					);
				}
			);

			// Auto remove.
			if (duration > 0) {
				setTimeout(
					function () {
						if ($notification.is( ':visible' )) {
							$notification.fadeOut(
								300,
								function () {
									$( this ).remove(); }
							);
						}
					},
					duration
				);
			}
		}

		/**
		 * Get form data for bulk operations
		 */
		function getBulkFormData() {
			var formData = {
				categories: [],
				attributes: {},
				fields: [],
				operation: $( 'input[name="operation"]:checked' ).val() || 'generate'
			};

			// Get selected categories.
			$( 'input[name="categories[]"]:checked' ).each(
				function () {
					formData.categories.push( $( this ).val() );
				}
			);

			// Get selected attributes.
			$( 'input[name^="attributes["]' ).each(
				function () {
					if ($( this ).is( ':checked' )) {
						var match = $( this ).attr( 'name' ).match( /attributes\[(.*?)\]\[\]/ );
						if (match) {
							var attrName = match[1];
							if ( ! formData.attributes[attrName]) {
								formData.attributes[attrName] = [];
							}
							formData.attributes[attrName].push( $( this ).val() );
						}
					}
				}
			);

			// Get selected fields (only enabled Lite fields).
			$( 'input[name="fields[]"]:checked:not(:disabled)' ).each(
				function () {
					formData.fields.push( $( this ).val() );
				}
			);

			return formData;
		}

		/**
		 * Validate form before submission
		 */
		function validateForm() {
			var formData = getBulkFormData();

			// Check if at least one category is selected.
			if (formData.categories.length === 0) {
				showNotification( 'error', shopwr_lite_bulk_data.i18n.no_categories || 'Please select at least one category.' );
				return false;
			}

			if (formData.fields.length === 0) {
				showNotification( 'error', shopwr_lite_bulk_data.i18n.no_fields || 'Please select at least one field to generate.' );
				return false;
			}

			return true;
		}

		// ================================
		// SELECT ALL / DESELECT ALL
		// ================================

		$( '#select-all-categories' ).on(
			'click',
			function () {
				$( 'input[name="categories[]"]' ).prop( 'checked', true );
			}
		);

		$( '#deselect-all-categories' ).on(
			'click',
			function () {
				$( 'input[name="categories[]"]' ).prop( 'checked', false );
			}
		);

		// ================================
		// PREVIEW PRODUCTS
		// ================================

		$( '#start-bulk-preview' ).on(
			'click',
			function () {
				if (isProcessingRequest) {
					return;
				}

				if ( ! validateForm()) {
					return;
				}

				isProcessingRequest = true;
				var $button         = $( this );
				var originalText    = $button.text();

				$button.prop( 'disabled', true ).text( shopwr_lite_bulk_data.i18n.loading || 'Loading...' );

				var formData = getBulkFormData();

				$.ajax(
					{
						url: shopwr_lite_bulk_data.ajax_url,
						type: 'POST',
						data: {
							action: 'shopwr_lite_bulk_generate_preview',
							nonce: shopwr_lite_bulk_data.nonce,
							filters: {
								categories: formData.categories,
								attributes: formData.attributes
							},
							fields: formData.fields,
							operation: formData.operation
						},
						success: function (response) {
							isProcessingRequest = false;
							$button.prop( 'disabled', false ).text( originalText );

							if (response.success) {
								previewData = response.data;
								showPreviewModal( response.data );
							} else {
								showNotification( 'error', response.data.message || 'An error occurred.' );
							}
						},
						error: function () {
							isProcessingRequest = false;
							$button.prop( 'disabled', false ).text( originalText );
							showNotification( 'error', shopwr_lite_bulk_data.i18n.ajax_error || 'An error occurred. Please try again.' );
						}
					}
				);
			}
		);

		/**
		 * Show preview modal
		 */
		function showPreviewModal(data) {
			var productsCount    = data.products_count || 0;
			var creditsRequired  = data.credits_required || 0;
			var creditsAvailable = data.credits_available;
			var hasCreditsInfo   = typeof creditsAvailable === 'number';

			if (productsCount === 0) {
				showNotification( 'warning', shopwr_lite_bulk_data.i18n.no_products || 'No products found matching your criteria.' );
				return;
			}

			// Check if we have credit info from API.
			if ( ! hasCreditsInfo || creditsAvailable === null || creditsAvailable === undefined) {
				showNotification( 'error', 'Could not retrieve credit balance. Please check your API key in Settings.' );
				return;
			}

			// Check if enough credits.
			if (creditsRequired > creditsAvailable) {
				showNotification(
					'error',
					(shopwr_lite_bulk_data.i18n.insufficient_credits || 'Insufficient credits. Required: %1$d, Available: %2$d')
					.replace( '%1$d', creditsRequired )
					.replace( '%2$d', creditsAvailable )
				);
				return;
			}

			// Format credits display.
			var creditsAvailableDisplay = creditsAvailable;

			// Create confirmation modal.
			var modalHtml =
			'<div class="shopwr-lite-modal-overlay" id="bulk-preview-modal">' +
				'<div class="shopwr-lite-modal">' +
					'<div class="shopwr-lite-modal-header">' +
						'<h2>' + (shopwr_lite_bulk_data.i18n.confirm_generation || 'Confirm Generation') + '</h2>' +
						'<button type="button" class="shopwr-lite-modal-close">&times;</button>' +
					'</div>' +
					'<div class="shopwr-lite-modal-body">' +
						'<div class="shopwr-lite-preview-stats">' +
							'<div class="stat-item">' +
								'<span class="stat-value">' + productsCount + '</span>' +
								'<span class="stat-label">' + (shopwr_lite_bulk_data.i18n.products || 'Products') + '</span>' +
							'</div>' +
							'<div class="stat-item">' +
								'<span class="stat-value">' + creditsRequired + '</span>' +
								'<span class="stat-label">' + (shopwr_lite_bulk_data.i18n.credits_required || 'Credits Required') + '</span>' +
							'</div>' +
							'<div class="stat-item">' +
								'<span class="stat-value">' + creditsAvailableDisplay + '</span>' +
								'<span class="stat-label">' + (shopwr_lite_bulk_data.i18n.credits_available || 'Credits Available') + '</span>' +
							'</div>' +
						'</div>' +
						'<p class="shopwr-lite-preview-warning">' +
							(shopwr_lite_bulk_data.i18n.generation_warning || 'This will generate content for the selected products. This action cannot be undone.') +
						'</p>' +
					'</div>' +
					'<div class="shopwr-lite-modal-footer">' +
						'<button type="button" class="button" id="cancel-bulk-generation">' +
							(shopwr_lite_bulk_data.i18n.cancel || 'Cancel') +
						'</button>' +
						'<button type="button" class="button button-primary" id="confirm-bulk-generation">' +
							(shopwr_lite_bulk_data.i18n.start_generation || 'Start Generation') +
						'</button>' +
					'</div>' +
				'</div>' +
			'</div>';

			$( 'body' ).append( modalHtml );

			// Bind modal events.
			$( '#bulk-preview-modal .shopwr-lite-modal-close, #cancel-bulk-generation' ).on(
				'click',
				function () {
					$( '#bulk-preview-modal' ).remove();
				}
			);

			$( '#confirm-bulk-generation' ).on(
				'click',
				function () {
					$( '#bulk-preview-modal' ).remove();
					startBulkGeneration();
				}
			);
		}

		// ================================
		// BULK GENERATION
		// ================================

		/**
		 * Start bulk generation
		 */
		function startBulkGeneration() {
			var formData = getBulkFormData();

			// Reset counters.
			processedCount = 0;
			successCount   = 0;
			failedCount    = 0;
			totalProducts  = previewData.products_count || 0;

			// Show progress section.
			$( '#step-filter, #step-fields, #step-operation, .shopwr-lite-bulk-actions' ).hide();
			$( '#bulk-progress' ).show();
			$( '#bulk-results' ).hide();

			updateProgress( 0, totalProducts, '' );

			// Start processing.
			$.ajax(
				{
					url: shopwr_lite_bulk_data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_start_bulk_process',
						nonce: shopwr_lite_bulk_data.nonce,
						filters: {
							categories: formData.categories,
							attributes: formData.attributes
						},
						fields: formData.fields,
						operation: formData.operation
					},
					success: function (response) {
						if (response.success) {
							currentBatchId = response.data.batch_id;
							processBatch();
						} else {
							showGenerationError( response.data.message || 'Failed to start generation.' );
						}
					},
					error: function () {
						showGenerationError( shopwr_lite_bulk_data.i18n.ajax_error || 'An error occurred. Please try again.' );
					}
				}
			);
		}

		/**
		 * Process batch
		 */
		function processBatch() {
			if ( ! currentBatchId) {
				showGenerationComplete();
				return;
			}

			$.ajax(
				{
					url: shopwr_lite_bulk_data.ajax_url,
					type: 'POST',
					data: {
						action: 'shopwr_lite_process_next_batch',
						nonce: shopwr_lite_bulk_data.nonce,
						batch_id: currentBatchId
					},
					success: function (response) {
						if (response.success) {
							var data = response.data;

							// Check if this is a completion response with stats object.
							if (data.completed && data.stats) {
								// Completion response has stats in data.stats.
								successCount   = data.stats.success || 0;
								failedCount    = data.stats.failed || 0;
								totalProducts  = data.stats.total || totalProducts;
								processedCount = totalProducts;
							} else {
								// Progress response has values directly in data.
								processedCount = data.processed || 0;
								successCount   = data.success || 0;
								failedCount    = data.failed || 0;
								totalProducts  = data.total || totalProducts;
							}

							var currentProductName = data.current && data.current.name ? data.current.name : '';
							updateProgress( processedCount, totalProducts, currentProductName );

							if (data.completed) {
								showGenerationComplete();
							} else {
								// Continue processing.
								setTimeout( processBatch, 500 );
							}
						} else {
							showGenerationError( response.data.message || 'An error occurred during processing.' );
						}
					},
					error: function () {
						showGenerationError( shopwr_lite_bulk_data.i18n.ajax_error || 'An error occurred. Please try again.' );
					}
				}
			);
		}

		/**
		 * Update progress display
		 */
		function updateProgress(processed, total, currentProduct) {
			var percentage = total > 0 ? Math.round( (processed / total) * 100 ) : 0;

			$( '#bulk-progress .shopwr-lite-progress-fill' ).css( 'width', percentage + '%' );
			$( '#bulk-progress .processed' ).text( processed );
			$( '#bulk-progress .total' ).text( total );
			$( '#bulk-progress .current-product' ).text( currentProduct );
		}

		/**
		 * Show generation complete
		 */
		function showGenerationComplete() {
			currentBatchId = null;

			$( '#bulk-progress' ).hide();
			$( '#bulk-results' ).show();

			$( '#bulk-results .result-item.success .count' ).text( successCount );
			$( '#bulk-results .result-item.failed .count' ).text( failedCount );

			if (successCount > 0) {
				showNotification(
					'success',
					(shopwr_lite_bulk_data.i18n.generation_complete || 'Generation complete! %d products processed successfully.')
					.replace( '%d', successCount )
				);
			}
		}

		/**
		 * Show generation error
		 */
		function showGenerationError(message) {
			currentBatchId = null;

			$( '#bulk-progress' ).hide();
			$( '#step-filter, #step-fields, #step-operation, .shopwr-lite-bulk-actions' ).show();

			showNotification( 'error', message );
		}

		// ================================
		// START NEW BATCH
		// ================================

		$( '#start-new-batch' ).on(
			'click',
			function () {
				// Reset and show form.
				previewData    = {};
				processedCount = 0;
				successCount   = 0;
				failedCount    = 0;
				totalProducts  = 0;

				$( '#bulk-progress, #bulk-results' ).hide();
				$( '#step-filter, #step-fields, #step-operation, .shopwr-lite-bulk-actions' ).show();
			}
		);

		// ================================
		// CREDIT CALCULATION
		// ================================

		/**
		 * Update credit estimation display
		 */
		function updateCreditEstimation() {
			var selectedFields        = $( 'input[name="fields[]"]:checked:not(:disabled)' );
			var baseCreditsPerProduct = 0;

			selectedFields.each(
				function () {
					var fieldValue = $( this ).val();

					switch (fieldValue) {
						case 'meta_title':
						case 'meta_description':
						case 'alt_text':
							baseCreditsPerProduct += 1;
							break;
						case 'short_description':
							baseCreditsPerProduct += 2;
							break;
						case 'description':
							baseCreditsPerProduct += 3;
							break;
						default:
							baseCreditsPerProduct += 1;
							break;
					}
				}
			);

			// Update display if element exists.
			if ($( '.shopwr-lite-credit-estimation' ).length) {
				$( '.shopwr-lite-credit-estimation' ).text( baseCreditsPerProduct + ' credits per product' );
			}
		}

		// Update credit estimation when fields change.
		$( 'input[name="fields[]"]' ).on( 'change', updateCreditEstimation );

		// Initial credit estimation.
		updateCreditEstimation();

	}
);
