<?php
if (!defined('ABSPATH')) { exit; }

class Shopcatflow_Category_Manager {

    public function __construct() {
        add_action('init', array($this, 'init'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));

        add_action('wp_ajax_shopc_update_category_order', array($this, 'ajax_update_category_order'));
        add_action('wp_ajax_shopc_update_category_orders', array($this, 'ajax_update_category_orders'));
        add_action('wp_ajax_shopc_update_category_parent', array($this, 'ajax_update_category_parent'));
        add_action('wp_ajax_shopc_get_categories', array($this, 'ajax_get_categories'));
        add_action('wp_ajax_shopc_delete_category', array($this, 'ajax_delete_category'));
        add_action('wp_ajax_shopc_delete_categories', array($this, 'ajax_delete_categories'));
    }

    public function init() {}

    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'ShopCatFlow', 'shopcatflow' ),
            __( 'ShopCatFlow', 'shopcatflow' ),
            'manage_woocommerce',
            'shopcatflow',
            array($this, 'admin_page')
        );
    }

    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'woocommerce_page_shopcatflow') { return; }

        wp_enqueue_script('jquery-ui-sortable');

        // Cache-bust on live via file modification time (falls back to plugin version)
        $ver_admin_js  = @filemtime(SHOPC_PLUGIN_PATH . 'assets/js/admin.js') ?: SHOPC_VERSION;
        $ver_admin_css = @filemtime(SHOPC_PLUGIN_PATH . 'assets/css/admin.css') ?: SHOPC_VERSION;
        $ver_graph_js  = @filemtime(SHOPC_PLUGIN_PATH . 'assets/js/graph.js') ?: SHOPC_VERSION;
        $ver_graph_css = @filemtime(SHOPC_PLUGIN_PATH . 'assets/css/graph.css') ?: SHOPC_VERSION;

        wp_enqueue_script('shopc-admin-script', SHOPC_PLUGIN_URL . 'assets/js/admin.js', array('jquery','jquery-ui-sortable'), $ver_admin_js, true);
        wp_enqueue_style('shopc-admin-style', SHOPC_PLUGIN_URL . 'assets/css/admin.css', array(), $ver_admin_css);

        wp_enqueue_script('shopc-graph-script', SHOPC_PLUGIN_URL . 'assets/js/graph.js', array('jquery'), $ver_graph_js, true);
        wp_enqueue_style('shopc-graph-style', SHOPC_PLUGIN_URL . 'assets/css/graph.css', array(), $ver_graph_css);

        $shopc_loc = array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce'    => wp_create_nonce('shopc_nonce'),
            'strings'  => array(
                'saving' => __( 'Saving...', 'shopcatflow' ),
                'saved'  => __( 'Saved!', 'shopcatflow' ),
                'error'  => __( 'Error occurred!', 'shopcatflow' ),
                'nonce_failed' => __( 'Security check failed (nonce). Please refresh the page.', 'shopcatflow' ),
                'no_permission' => __( 'You do not have permission for this action.', 'shopcatflow' ),
                'unknown_error' => __( 'Unknown error occurred.', 'shopcatflow' ),
                'cannot_make_default_child' => __( "Default product category can't be a child.", 'shopcatflow' ),
                'drag_to_reorder' => __( 'Drag to reorder / change level', 'shopcatflow' ),
                'toggle_subcategories' => __( 'Toggle subcategories', 'shopcatflow' ),
                'level' => __( 'Level', 'shopcatflow' ),
                'items' => __( 'items', 'shopcatflow' ),
                'order_updated' => __( 'Order updated', 'shopcatflow' ),
                'parent_updated' => __( 'Parent category updated', 'shopcatflow' ),
                'will_be_parent' => __( 'Will become parent', 'shopcatflow' ),
                'invalid_target' => __( 'Invalid target', 'shopcatflow' ),
                'drop_to_root' => __( 'Drop to make root', 'shopcatflow' ),
                'delete_category' => __( 'Delete category', 'shopcatflow' ),
                // translators: %s: category name to be deleted
                'confirm_delete' => __( 'Delete "%s"?', 'shopcatflow' ),
                'confirm_delete_children' => __( 'Also delete all subcategories?', 'shopcatflow' ),
                'deleted' => __( 'Deleted', 'shopcatflow' ),
            ),
            'default_category_id' => $this->get_default_product_category_id()
        );
        wp_localize_script('shopc-admin-script', 'shopc_ajax', $shopc_loc);
        wp_localize_script('shopc-graph-script', 'shopc_ajax', $shopc_loc);

        $ver_debug_js = @filemtime(SHOPC_PLUGIN_PATH . 'assets/js/debug.js') ?: SHOPC_VERSION;
        wp_enqueue_script('shopc-debug-script', SHOPC_PLUGIN_URL . 'assets/js/debug.js', array('jquery','shopc-admin-script','shopc-graph-script'), $ver_debug_js, true);
    }

    // AJAX: delete category (optionally with children)
    public function ajax_delete_category() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if (!wp_verify_nonce($nonce, 'shopc_nonce')) {
            wp_send_json_error(array('message' => __( 'Security check failed (nonce).', 'shopcatflow' )));
        }
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __( 'You do not have permission.', 'shopcatflow' )));
        }
        $category_id = isset($_POST['category_id']) ? absint($_POST['category_id']) : 0;
        $delete_children = !empty($_POST['delete_children']);
        if ($category_id <= 0) {
            wp_send_json_error(array('message' => __( 'Invalid request', 'shopcatflow' )));
        }
        // Protect default product category
        $default_id = $this->get_default_product_category_id();
        if ($default_id && $category_id === $default_id) {
            wp_send_json_error(array('message' => __( 'Default product category cannot be deleted.', 'shopcatflow' )));
        }

        // Helper: get direct children
        $get_children = function($pid){
            return get_terms(array('taxonomy'=>'product_cat','hide_empty'=>false,'parent'=>$pid, 'fields'=>'ids'));
        };

        $deleted = 0; $touched = 0;
        if ($delete_children) {
            // Post-order traversal: delete descendants first
            $stack = array($category_id);
            $to_delete = array();
            while ($stack) {
                $id = array_pop($stack);
                $children = $get_children($id);
                foreach ($children as $cid) { $stack[] = $cid; }
                $to_delete[] = $id;
            }
            // Delete all gathered ids
            foreach ($to_delete as $tid) {
                // Safeguard default cat once more
                if ($default_id && $tid === $default_id) { continue; }
                $res = wp_delete_term($tid, 'product_cat');
                if (!is_wp_error($res)) { $deleted++; }
            }
        } else {
            // Reparent direct children to root, then delete the node
            $children = $get_children($category_id);
            foreach ($children as $cid) {
                $res = wp_update_term($cid, 'product_cat', array('parent'=>0));
                if (!is_wp_error($res)) { $touched++; }
            }
            $res = wp_delete_term($category_id, 'product_cat');
            if (!is_wp_error($res)) { $deleted++; }
        }

        if ($deleted>0 || $touched>0) {
            wp_send_json_success(array('message'=>__( 'Deleted', 'shopcatflow' ), 'deleted'=>$deleted, 'reparented'=>$touched));
        }
        wp_send_json_error(array('message'=>__( 'Nothing deleted', 'shopcatflow' )));
    }

    // AJAX: bulk delete categories
    public function ajax_delete_categories() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if (!wp_verify_nonce($nonce, 'shopc_nonce')) {
            wp_send_json_error(array('message' => __( 'Security check failed (nonce).', 'shopcatflow' )));
        }
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __( 'You do not have permission.', 'shopcatflow' )));
        }
        $ids_source = isset($_POST['ids']) ? sanitize_text_field(wp_unslash($_POST['ids'])) : array();
        if (is_string($ids_source)) {
            $ids = json_decode($ids_source, true);
            if (json_last_error() !== JSON_ERROR_NONE) {
                wp_send_json_error(array('message' => __('Invalid data format', 'shopcatflow')));
                return;
            }
        } else {
            $ids = $ids_source;
        }
        if (!is_array($ids)) { wp_send_json_error(array('message'=>__( 'Invalid data', 'shopcatflow' ))); }
        $delete_children = !empty($_POST['delete_children']);

        $default_id = $this->get_default_product_category_id();
        $get_children = function($pid){ return get_terms(array('taxonomy'=>'product_cat','hide_empty'=>false,'parent'=>$pid,'fields'=>'ids')); };
        $deleted=0; $reparented=0;
        foreach ($ids as $category_id) {
            $category_id = absint($category_id);
            if ($category_id <= 0) continue;
            if ($default_id && $category_id === $default_id) continue;
            
            if ($delete_children) {
                $stack = array($category_id);
                $to_delete = array();
                while ($stack) {
                    $id = array_pop($stack);
                    $children = $get_children($id);
                    foreach ($children as $cid) {
                        $stack[] = $cid;
                    }
                    $to_delete[] = $id;
                }
                foreach ($to_delete as $tid) {
                    if ($default_id && $tid === $default_id) continue;
                    $res = wp_delete_term($tid, 'product_cat');
                    if (!is_wp_error($res)) $deleted++;
                }
            } else {
                $children = $get_children($category_id);
                foreach ($children as $cid) {
                    $res = wp_update_term($cid, 'product_cat', array('parent' => 0));
                    if (!is_wp_error($res)) $reparented++;
                }
                $res = wp_delete_term($category_id, 'product_cat');
                if (!is_wp_error($res)) $deleted++;
            }
        }
        wp_send_json_success(array('message'=>__( 'Deleted', 'shopcatflow' ), 'deleted'=>$deleted, 'reparented'=>$reparented));
    }

    public function admin_page() {
        $categories = $this->get_categories_hierarchical();
        include SHOPC_PLUGIN_PATH . 'templates/admin-page.php';
    }

    public function get_categories_hierarchical($parent_id = 0, $level = 0) {
        $categories = get_terms(array(
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'parent'     => $parent_id,
            'orderby'    => 'name',
            'order'      => 'ASC',
        ));

        $result = array();
        if (!empty($categories) && !is_wp_error($categories)) {
            $index = 1;
            foreach ($categories as $category) {
                $category_order = get_term_meta($category->term_id, 'shopc_category_order', true);
                if ($category_order === '' || $category_order === false) {
                    $legacy = get_term_meta($category->term_id, 'wcm_category_order', true);
                    if ($legacy !== '' && $legacy !== false) {
                        $category_order = (int) $legacy;
                        update_term_meta($category->term_id, 'shopc_category_order', $category_order);
                    } else {
                        update_term_meta($category->term_id, 'shopc_category_order', $index);
                        $category_order = $index;
                    }
                }
                $category->shopc_order = absint($category_order);
                $index++;
            }
            usort($categories, function($a,$b){
                if ($a->shopc_order === $b->shopc_order) { return strcmp($a->name, $b->name); }
                return ($a->shopc_order < $b->shopc_order) ? -1 : 1;
            });
            foreach ($categories as $category) {
                $category->level = $level;
                $result[] = $category;
                $children = $this->get_categories_hierarchical($category->term_id, $level + 1);
                $result = array_merge($result, $children);
            }
        }
        return $result;
    }

    public function has_children($category_id) {
        $children = get_terms(array(
            'taxonomy'   => 'product_cat',
            'hide_empty' => false,
            'parent'     => $category_id,
        ));
        return !empty($children) && !is_wp_error($children);
    }

    public function is_descendant($ancestor_id, $descendant_id) {
        $category = get_term($descendant_id, 'product_cat');
        while ($category && $category->parent != 0) {
            if ((int) $category->parent === (int) $ancestor_id) { return true; }
            $category = get_term($category->parent, 'product_cat');
        }
        return false;
    }

    private function get_default_product_category_id() {
        $default_id = absint(get_option('default_product_cat'));
        if ($default_id > 0) { return $default_id; }
        $term = get_term_by('slug', 'uncategorized', 'product_cat');
        return $term && !is_wp_error($term) ? absint($term->term_id) : 0;
    }

    // AJAX: single order update
    public function ajax_update_category_order() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if (!wp_verify_nonce($nonce, 'shopc_nonce')) {
            wp_send_json_error(array('message' => __( 'Security check failed (nonce).', 'shopcatflow' )));
        }
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __( 'You do not have permission.', 'shopcatflow' )));
        }
        $category_id = isset($_POST['category_id']) ? absint($_POST['category_id']) : 0;
        $new_order   = isset($_POST['new_order']) ? absint($_POST['new_order']) : 0;
        if ($category_id > 0 && $new_order > 0) {
            update_term_meta($category_id, 'shopc_category_order', $new_order);
            wp_send_json_success(array('message' => __( 'Order updated', 'shopcatflow' )));
        }
        wp_send_json_error(array('message' => __( 'Invalid request', 'shopcatflow' )));
    }

    // AJAX: bulk orders
    public function ajax_update_category_orders() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if (!wp_verify_nonce($nonce, 'shopc_nonce')) {
            wp_send_json_error(array('message' => __( 'Security check failed (nonce).', 'shopcatflow' )));
        }
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __( 'You do not have permission.', 'shopcatflow' )));
        }
        $orders_source = isset($_POST['orders']) ? sanitize_text_field(wp_unslash($_POST['orders'])) : array();
        if (is_string($orders_source)) {
            $decoded = json_decode($orders_source, true);
            if (json_last_error() === JSON_ERROR_NONE) { 
                $orders = $decoded; 
            } else { 
                wp_send_json_error(array('message' => __('Invalid data format', 'shopcatflow')));
                return;
            }
        } else {
            $orders = $orders_source;
        }
        if (!is_array($orders)) {
            wp_send_json_error(array('message' => __( 'Invalid data', 'shopcatflow' )));
        }
        $updated = 0;
        foreach ($orders as $item) {
            if (!is_array($item)) { continue; }
            $id    = isset($item['id']) ? absint($item['id']) : 0;
            $order = isset($item['order']) ? absint($item['order']) : 0;
            if ($id > 0 && $order > 0) { 
                update_term_meta($id, 'shopc_category_order', $order); 
                $updated++; 
            }
        }
        wp_send_json_success(array('message' => __( 'Order updated', 'shopcatflow' ), 'updated' => $updated));
    }

    // AJAX: update parent
    public function ajax_update_category_parent() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if (!wp_verify_nonce($nonce, 'shopc_nonce')) {
            wp_send_json_error(array('message' => __( 'Security check failed (nonce).', 'shopcatflow' )));
        }
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __( 'You do not have permission.', 'shopcatflow' )));
        }
        $category_id = isset($_POST['category_id']) ? absint($_POST['category_id']) : 0;
        $new_parent  = isset($_POST['new_parent']) ? absint($_POST['new_parent']) : 0;
        if ($category_id <= 0) { wp_send_json_error(array('message' => __( 'Invalid request', 'shopcatflow' ))); }

        // Default category cannot be a child
        $default_id = $this->get_default_product_category_id();
        if ($default_id && $category_id === $default_id && $new_parent !== 0) {
            wp_send_json_error(array('message' => __( "Default product category can't be a child.", 'shopcatflow' )));
        }
        // Self/descendant protection
        if ($new_parent === $category_id) {
            wp_send_json_error(array('message' => __( 'A category cannot be its own parent.', 'shopcatflow' )));
        }
        if ($new_parent !== 0 && $this->is_descendant($category_id, $new_parent)) {
            wp_send_json_error(array('message' => __( 'A category cannot be moved under its own descendant.', 'shopcatflow' )));
        }

        $result = wp_update_term($category_id, 'product_cat', array('parent' => $new_parent));
        if (!is_wp_error($result)) {
            wp_send_json_success(array('message' => __( 'Parent category updated', 'shopcatflow' )));
        }
        wp_send_json_error(array('message' => __( 'Error: ', 'shopcatflow' ) . $result->get_error_message()));
    }

    // AJAX: list
    public function ajax_get_categories() {
        $nonce = isset($_POST['nonce']) ? sanitize_text_field( wp_unslash($_POST['nonce']) ) : '';
        if (!wp_verify_nonce($nonce, 'shopc_nonce')) {
            wp_send_json_error(array('message' => __( 'Security check failed (nonce).', 'shopcatflow' )));
        }
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error(array('message' => __( 'You do not have permission.', 'shopcatflow' )));
        }
        $cats = $this->get_categories_hierarchical();
        $items = array();
        foreach ($cats as $cat) {
            $order = get_term_meta($cat->term_id, 'shopc_category_order', true);
            $items[] = array(
                'id'           => absint($cat->term_id),
                'name'         => sanitize_text_field($cat->name),
                'level'        => isset($cat->level) ? absint($cat->level) : 0,
                'parent'       => absint($cat->parent),
                'order'        => absint($order),
                'count'        => absint($cat->count),
                'has_children' => $this->has_children($cat->term_id),
            );
        }
        wp_send_json_success(array('items' => $items));
    }
}
