
(function($){
    function buildTree(items) {
        var byId = {}, roots = [];
        items.forEach(function(it){ byId[it.id] = Object.assign({children:[], parentId: it.parent || 0}, it); });
        items.forEach(function(it){
            if (it.parent && byId[it.parent]) {
                var child = byId[it.id];
                var parent = byId[it.parent];
                child.parent = parent;
                parent.children.push(child);
            } else {
                roots.push(byId[it.id]);
            }
        });
        function sortNode(n){
            n.children.sort(function(a,b){ if (a.order===b.order) return a.name.localeCompare(b.name); return a.order-b.order; });
            n.children.forEach(sortNode);
        }
        roots.forEach(sortNode);
        return { roots: roots, map: byId };
    }

    // Vertical layout: depth goes downward (Y); siblings spread along X
    function layoutTree(roots, cfg) {
        var depthY = cfg.depthX || 220; // reuse provided depth step
        var gapX = cfg.gapY || 120;     // horizontal gap widened to reduce label overlap
        var nodeX = 0;
        function firstWalk(n, depth) {
            n.y = depth * depthY;
            if (!n.children || n.children.length===0) {
                n.x = nodeX * gapX; nodeX++;
            } else {
                n.children.forEach(function(c){ firstWalk(c, depth+1); });
                var minX = n.children[0].x, maxX = n.children[n.children.length-1].x;
                n.x = (minX + maxX)/2;
            }
        }
        roots.forEach(function(r){ firstWalk(r, 0); });
    }

    function renderGraph($svg, items) {
        var built = buildTree(items);
        var roots = built.roots; var map = built.map;
        layoutTree(roots, { gapY: 120, depthX: 220 });
        var svg = $svg[0];
        while (svg.firstChild) svg.removeChild(svg.firstChild);
        var ns = 'http://www.w3.org/2000/svg';

        // Container group for pan/zoom/rotate
        var container = document.createElementNS(ns, 'g');
        container.setAttribute('class', 'wcm-graph-content');
        svg.appendChild(container);

        function add(el){ container.appendChild(el); return el; }
        function make(tag, attrs){ var el = document.createElementNS(ns, tag); for (var k in attrs) el.setAttribute(k, attrs[k]); return el; }

        // Collect nodes list
        var nodes = [];
        function walk(n) { nodes.push(n); (n.children||[]).forEach(walk); }
        roots.forEach(walk);

        // Depth guides (horizontal lines per level)
        var maxDepth = 0; nodes.forEach(function(n){ var d = Math.round(n.y/220); if (d>maxDepth) maxDepth = d; });
        for (var d=0; d<=maxDepth; d++) {
            var y = d*220 + 40;
            add(make('line', { x1:0, y1:y, x2:10000, y2:y, class:'wcm-depth' }));
        }

        // Color palette per root and shade by level
        var hues = [200, 340, 20, 120, 260, 40, 170, 300, 0, 210]; // extend as needed
        var rootHue = {};
        roots.forEach(function(r, idx){ rootHue[r.id] = hues[idx % hues.length]; });

        function getRoot(node){ var n=node; while (n && n.parent) n = n.parent; return n; }
        function hslToHex(h,s,l){
            h/=360; s/=100; l/=100;
            function f(n){
                var k=(n+h*12)%12; var a=s*Math.min(l,1-l);
                var c=l - a*Math.max(-1, Math.min(k-3, Math.min(9-k,1)));
                return Math.round(255*c).toString(16).padStart(2,'0');
            }
            return '#'+f(0)+f(8)+f(4);
        }
        function nodeFillColor(n){
            var r = getRoot(n); var hue = rootHue[r.id] || 200;
            var baseL = 56; var delta = Math.min(4+n.level*6, 24);
            var l = Math.max(24, Math.min(84, baseL + (n.level>0 ? 12 : 0) - delta));
            return hslToHex(hue, 60, l);
        }
        function nodeStrokeColor(n){
            var r = getRoot(n); var hue = rootHue[r.id] || 200;
            var l = Math.max(18, 48 - n.level*4);
            return hslToHex(hue, 70, l);
        }

        // Links (vertical curve)
        nodes.forEach(function(n){
            (n.children||[]).forEach(function(c){
                var p = make('path', { d: bezier(n.x, n.y, c.x, c.y), class: 'wcm-link' });
                // tint link with parent color
                p.setAttribute('stroke', nodeStrokeColor(c));
                add(p);
            });
        });

        // Nodes
        nodes.forEach(function(n){
            var g = make('g', { class: 'wcm-node' });
            g.setAttribute('transform', 'translate('+ (n.x+40) +','+ (n.y+40) +')');
            g.dataset.id = n.id;
            var radius = 18 - Math.min(10, (n.level||0));
            var circle = make('circle', { r: Math.max(10, radius), filter:'url(#wcm-shadow)' });
            // apply dynamic colors (inline for fill so it wins over CSS defaults)
            circle.setAttribute('style', 'fill:'+nodeFillColor(n));
            circle.setAttribute('stroke', nodeStrokeColor(n));
            var label = make('text', { x: 0, y: 0, 'text-anchor':'start' });
            var full = n.name || '';
            var maxLen = 24;
            var display = full.length > maxLen ? (full.slice(0, maxLen) + '…') : full;
            label.textContent = display;
            // rotate label diagonally to reduce overlap
            label.setAttribute('transform', 'translate(22, 10) rotate(-28)');
            label.setAttribute('class', 'wcm-label');
            label.setAttribute('data-base-transform', 'translate(22, 10) rotate(-28)');
            // title tooltip with full text
            try { var ttl = make('title', {}); ttl.textContent = full; label.appendChild(ttl); } catch(e){}
            if (!n.parent) g.classList.add('root');
            g.appendChild(circle); g.appendChild(label);
            add(g);
        });

        // Drag to reparent with preview & constraints
        // Clear previous handlers to avoid duplicates when re-rendering
        $(document).off('.wcmGraph');
        $svg.off('mousedown', '.wcm-node');
        var dragging = null, ghostLink = null, $hint = null;
        var defaultId = parseInt((window.shopc_ajax && shopc_ajax.default_category_id) || 0, 10) || 0;
        var draggingOrigParentId = null;

        function showHint(text, x, y){
            if (!$hint) { $hint = $('<div class="wcm-graph-hint"/>').appendTo($svg.parent()); }
            $hint.text(text).css({ left: x+12, top: y+12 }).addClass('visible');
        }
        function hideHint(){ if ($hint) { $hint.removeClass('visible'); $hint.remove(); $hint=null; } }

        var spacePan = false;
        $(document).on('keydown.wcmGraph', function(e){
            if (e.target && (/input|textarea|select/i).test(e.target.tagName)) return;
            var step = 40; // px
            if (e.key === ' ') { spacePan = true; e.preventDefault(); }
            else if (e.key === 'ArrowLeft') { state.tx += step; applyTransform(); e.preventDefault(); }
            else if (e.key === 'ArrowRight') { state.tx -= step; applyTransform(); e.preventDefault(); }
            else if (e.key === 'ArrowUp') { state.ty += step; applyTransform(); e.preventDefault(); }
            else if (e.key === 'ArrowDown') { state.ty -= step; applyTransform(); e.preventDefault(); }
        });
        $(document).on('keyup.wcmGraph', function(e){ if (e.key === ' ') { spacePan = false; } });

        var lastSelectedNode = null;
        var selectedSet = new Set();
        $svg.on('mousedown', '.wcm-node', function(e){
            if (spacePan) { // space+drag pans canvas instead of node
                panning = true; sx = e.clientX; sy = e.clientY; stx = state.tx; sty = state.ty; e.preventDefault(); return;
            }
            dragging = this; lastSelectedNode = this; this.classList.add('dragging'); highlightDroppables(this); e.preventDefault();
            try { var did=parseInt(this.dataset.id,10)||0; var node = map[did]; draggingOrigParentId = (node && node.parent) ? (node.parent.id || node.parentId || 0) : (node ? node.parentId||0 : 0); } catch(_){ draggingOrigParentId = 0; }
        });
        // Select on click (for Delete key usage)
        $svg.on('click', '.wcm-node', function(e){
            lastSelectedNode = this;
            var id=parseInt(this.dataset.id,10)||0; if(!id) return;
            if (e.ctrlKey || e.metaKey) {
                if (selectedSet.has(id)) { selectedSet.delete(id); this.classList.remove('selected'); }
                else { selectedSet.add(id); this.classList.add('selected'); }
            } else {
                // single select
                $('.wcm-node.selected', $svg).removeClass('selected'); selectedSet.clear(); selectedSet.add(id); this.classList.add('selected');
            }
        });
        // Context menu delete
        $svg.on('contextmenu', '.wcm-node', function(e){ e.preventDefault(); var g=this; var id=parseInt(g.dataset.id,10)||0; var name=$(g).find('text').text(); if(!id) return; var msg=(shopc_ajax.strings&&shopc_ajax.strings.confirm_delete)? shopc_ajax.strings.confirm_delete.replace('%s', name):('Delete "'+name+'"?'); if(!window.confirm(msg)) return; var cascade=window.confirm((shopc_ajax.strings&&shopc_ajax.strings.confirm_delete_children)||'Also delete all subcategories?'); $.post(shopc_ajax.ajax_url,{action:'shopc_delete_category',category_id:id,delete_children:cascade?1:0,nonce:shopc_ajax.nonce},function(res){ if(res&&res.success){ if(window.showMessage) window.showMessage((shopc_ajax.strings&&shopc_ajax.strings.deleted)||'Deleted','success'); refreshGraph($svg); if(window.refreshCategoryList) window.refreshCategoryList(); } else { if(window.showMessage) window.showMessage(res&&res.data&&res.data.message?res.data.message:(shopc_ajax.strings&&shopc_ajax.strings.error)||'Error','error'); }}); });
        // Delete key handler
        $(document).on('keydown.wcmGraphDelete', function(e){
            if ((e.key==='Delete'||e.key==='Backspace') && !dragging) {
                var ids = selectedSet.size ? Array.from(selectedSet) : (function(){ var g=lastSelectedNode; return g?[parseInt(g.dataset.id,10)||0]:[]; })();
                ids = ids.filter(function(x){return x>0;}); if(!ids.length) return;
                var msg=(shopc_ajax.strings&&shopc_ajax.strings.delete_selected)||'Delete selected'; if(!window.confirm(msg+' ('+ids.length+')?')) return;
                var cascade=window.confirm((shopc_ajax.strings&&shopc_ajax.strings.confirm_delete_children)||'Also delete all subcategories?');
                $.post(shopc_ajax.ajax_url,{action:'shopc_delete_categories',ids:JSON.stringify(ids),delete_children:cascade?1:0,nonce:shopc_ajax.nonce},function(res){
                    if(res&&res.success){ if(window.showMessage) window.showMessage((shopc_ajax.strings&&shopc_ajax.strings.deleted)||'Deleted','success'); selectedSet.clear(); $('.wcm-node.selected',$svg).removeClass('selected'); refreshGraph($svg); if(window.refreshCategoryList) window.refreshCategoryList(); }
                    else { if(window.showMessage) window.showMessage(res&&res.data&&res.data.message?res.data.message:(shopc_ajax.strings&&shopc_ajax.strings.error)||'Error','error'); }
                });
                e.preventDefault();
            }
        });
        $(document).on('mousemove.wcmGraph', function(e){
            if (!dragging) return;
            // Auto-pan when cursor nears edges to enable long drags
            try {
                var rect = $svg[0].getBoundingClientRect();
                var edge = 60, step = 16;
                var dx = 0, dy = 0;
                if (e.clientX - rect.left < edge) dx = step; else if (rect.right - e.clientX < edge) dx = -step;
                if (e.clientY - rect.top < edge) dy = step; else if (rect.bottom - e.clientY < edge) dy = -step;
                if (dx || dy) { state.tx += dx; state.ty += dy; applyTransform(); }
            } catch(_){ }
            var pt = (typeof container !== 'undefined' && container && container.getScreenCTM) ? (function(){
                var svgEl = container.ownerSVGElement || svg; var p = svgEl.createSVGPoint(); p.x=e.clientX; p.y=e.clientY; return p.matrixTransform(container.getScreenCTM().inverse());
            })() : clientToSvg(svg, e.clientX, e.clientY);
            dragging.setAttribute('transform', 'translate('+pt.x+','+pt.y+')');
            if (!ghostLink) ghostLink = add(make('path', { class:'wcm-ghost-link', d:'M0 0 L0 0' }));
            var origin = nodeCenter(dragging);
            ghostLink.setAttribute('d', 'M '+origin.x+' '+origin.y+' L '+pt.x+' '+pt.y);
            var target = findClosestNode($svg[0], dragging);
            $('.wcm-node.target', $svg).removeClass('target');
            if (target && target!==dragging) {
                var tid = parseInt(target.dataset.id,10)||0; var did = parseInt(dragging.dataset.id,10)||0;
                var ok = isDroppable(did, tid, map, defaultId);
                $(target).toggleClass('target', true).toggleClass('nodrop', !ok).toggleClass('droppable', ok);
                var hint = ok ? ((shopc_ajax.strings && shopc_ajax.strings.will_be_parent) || 'Will become parent')
                               : ((shopc_ajax.strings && shopc_ajax.strings.invalid_target) || 'Invalid target');
                showHint(hint, e.pageX, e.pageY);
            } else {
                // No node target under cursor: suggest root drop
                showHint((shopc_ajax.strings && shopc_ajax.strings.drop_to_root) || 'Drop to make root', e.pageX, e.pageY);
            }
        });
        $(document).on('mouseup.wcmGraph', function(e){
            if (!dragging) return;
            var $drag = $(dragging);
            var id = parseInt(dragging.dataset.id, 10);
            var target = findClosestNode($svg[0], dragging);
            $drag.removeClass('dragging');
            if (ghostLink) {
                if (ghostLink.parentNode) { ghostLink.parentNode.removeChild(ghostLink); }
                ghostLink=null;
            }
            $('.wcm-node.droppable, .wcm-node.nodrop', $svg).removeClass('droppable nodrop');
            hideHint();
            if (target && target!==dragging) {
                var newParent = parseInt(target.dataset.id, 10) || 0;
                if (isDroppable(id, newParent, map, defaultId)) {
                    attemptReparent(id, newParent, draggingOrigParentId);
                } else {
                    refreshGraph($svg);
                }
            } else {
                // Drop on background: make it a root category
                attemptReparent(id, 0, draggingOrigParentId);
            }
            dragging = null; draggingOrigParentId = null; $('.wcm-node.target', $svg).removeClass('target');
        });

        function highlightDroppables(dragEl){
            var did = parseInt(dragEl.dataset.id,10)||0;
            nodes.forEach(function(n){
                if (n.id === did) return;
                var ok = isDroppable(did, n.id, map, defaultId);
                var g = $svg[0].querySelector('.wcm-node[data-id="'+n.id+'"]');
                if (g) g.classList.add(ok? 'droppable' : 'nodrop');
            });
        }

        function isDroppable(dragId, targetId, map, defaultId){
            if (dragId === targetId) return false;
            if (defaultId && dragId === defaultId && targetId !== 0) return false;
            // target is descendant of drag? If yes, invalid
            var t = map[targetId];
            while (t && t.parent) { if (t.parent.id === dragId) return false; t = t.parent; }
            return true;
        }

        function bezier(x1,y1,x2,y2){
            var my = (y1+y2)/2; return 'M '+(x1+40)+' '+(y1+58)+' C '+(x1+40)+' '+(my)+' '+(x2+40)+' '+(my)+' '+(x2+40)+' '+(y2+22);
        }
        function clientToSvg(svg, cx, cy){
            var pt = svg.createSVGPoint(); pt.x = cx; pt.y = cy; return pt.matrixTransform(svg.getScreenCTM().inverse());
        }
        function nodeCenter(g){
            var m = g.transform.baseVal.consolidate().matrix; return { x: m.e, y: m.f };
        }
        function findClosestNode(svgEl, excludeEl){
            var nodes = Array.prototype.slice.call(svgEl.querySelectorAll('.wcm-node'));
            var minD = Infinity, best = null;
            nodes.forEach(function(n){ if (n===excludeEl) return; var bb = n.getBoundingClientRect(); var cx = (bb.left+bb.right)/2; var cy=(bb.top+bb.bottom)/2; var d = Math.hypot(cx-lastClientX, cy-lastClientY); if (d<minD){minD=d; best=n;} });
            return (minD<80)? best : null;
        }

        var lastClientX=0,lastClientY=0,lastMouseX=0,lastMouseY=0; // track for hit-test
        $svg.on('mousemove', function(e){ lastClientX=e.clientX; lastClientY=e.clientY; var pt = clientToSvg(svg, e.clientX, e.clientY); lastMouseX=pt.x; lastMouseY=pt.y; });

        // Shadow filter defs
        var defs = make('defs', {});
        var filter = make('filter', { id:'wcm-shadow', x:'-20%', y:'-20%', width:'140%', height:'140%' });
        var fe = document.createElementNS(ns, 'feDropShadow');
        fe.setAttribute('dx','0'); fe.setAttribute('dy','1'); fe.setAttribute('stdDeviation','2'); fe.setAttribute('flood-opacity','.25');
        filter.appendChild(fe); defs.appendChild(filter); svg.insertBefore(defs, svg.firstChild);

        // Pan / Zoom / Rotate controls with persistent state
        var state = $svg.data('viewState') || { scale: 1, tx: 0, ty: 0, rot: 0, initialized: false };
        function applyTransform(){
            container.setAttribute('transform', 'translate('+state.tx+','+state.ty+') scale('+state.scale+') rotate('+state.rot+')');
            $svg.data('viewState', state);
            updateLabelScaling();
        }

        function updateLabelScaling(){
            try {
                var s = state.scale || 1;
                // Keep labels readable when zoomed out: inversely scale up to a cap
                var k = (s < 1) ? (1 / s) : 1;
                if (k > 2.4) k = 2.4; // cap to avoid giant labels
                var texts = container.querySelectorAll('.wcm-label');
                texts.forEach(function(t){
                    var base = t.getAttribute('data-base-transform') || '';
                    t.setAttribute('transform', base + (k!==1 ? (' scale(' + k + ')') : ''));
                });
            } catch(e){}
        }

        function fitToScreen(){
            var minX=Infinity, minY=Infinity, maxX=-Infinity, maxY=-Infinity;
            nodes.forEach(function(n){ var x=n.x+40, y=n.y+40; var r=Math.max(10, 18 - Math.min(10,(n.level||0))); minX=Math.min(minX, x-r-40); minY=Math.min(minY, y-r-40); maxX=Math.max(maxX, x+r+120); maxY=Math.max(maxY, y+r+40); });
            var vbW = $svg[0].clientWidth, vbH = $svg[0].clientHeight; if (!vbW) vbW = 1; if (!vbH) vbH = 1;
            var contentW = (maxX - minX) || 1; var contentH = (maxY - minY) || 1; var margin = 20;
            var s = Math.min((vbW - margin*2)/contentW, (vbH - margin*2)/contentH); s = Math.max(0.2, Math.min(3, s));
            state.scale = s; state.rot = state.rot || 0;
            state.tx = (vbW - contentW*s)/2 - minX*s; state.ty = (vbH - contentH*s)/2 - minY*s;
            applyTransform();
        }

        function animateZoomTo(ns, cx, cy, ms){
            var old = state.scale; if (ns === old) return;
            var dur = Math.max(80, Math.min(220, ms||140));
            var start = performance.now();
            var pt = clientToSvg(svg, cx, cy);
            var startTx = state.tx, startTy = state.ty;
            function step(t){
                var k = Math.min(1, (t - start) / dur);
                // easeOutCubic
                var e = 1 - Math.pow(1-k, 3);
                var cur = old + (ns - old) * e;
                // adjust translation to keep focus point stable
                state.tx = pt.x - (pt.x - startTx) * (cur/old);
                state.ty = pt.y - (pt.y - startTy) * (cur/old);
                state.scale = cur; applyTransform();
                if (k < 1) requestAnimationFrame(step);
            }
            requestAnimationFrame(step);
        }

        function zoomAt(delta, cx, cy){
            // delta>0: zoom out; <0: zoom in. Use gentle factor and animate
            var old = state.scale;
            var ns = Math.max(0.2, Math.min(3, old * (delta>0 ? 0.96 : 1.04)));
            if (Math.abs(ns-old) < 0.0001) return;
            animateZoomTo(ns, cx, cy, 140);
        }

        function zoomSteps(steps, sign, cx, cy){
            var factor = Math.pow(sign>0 ? 0.96 : 1.04, Math.max(1, steps));
            var old = state.scale; var ns = Math.max(0.2, Math.min(3, old * factor));
            animateZoomTo(ns, cx, cy, 140 + Math.min(200, steps*20));
        }

        // Wheel zoom
        $svg.on('wheel.wcmGraph', function(e){
            e.preventDefault();
            var ev = e.originalEvent || e; var dy = ev.deltaY || 0; var mode = ev.deltaMode || 0; // 0: pixel, 1: line, 2: page
            // Normalize rough steps; larger wheels or trackpads produce varied deltas
            var magnitude = Math.abs(dy);
            if (mode === 1) magnitude *= 16; // lines -> approx pixels
            else if (mode === 2) magnitude *= 240; // pages -> large
            var steps = Math.max(1, Math.round(magnitude / 120));
            var sign = dy>0 ? 1 : -1;
            zoomSteps(steps, sign, ev.clientX, ev.clientY);
        });

        // Drag pan when clicking background
        var panning=false, sx=0, sy=0, stx=0, sty=0;
        $svg.on('mousedown.wcmGraph', function(e){
            if ($(e.target).closest('.wcm-node').length) return; // ignore node drags
            panning = true; sx = e.clientX; sy = e.clientY; stx = state.tx; sty = state.ty; e.preventDefault();
        });
        $(document).on('mousemove.wcmGraph', function(e){ if (!panning || dragging) return; state.tx = stx + (e.clientX - sx); state.ty = sty + (e.clientY - sy); applyTransform(); });
        $(document).on('mouseup.wcmGraph', function(){ panning=false; });

        // Toolbar buttons
        $('.wcm-graph-zoom-in').off('click').on('click', function(){ var rect = $svg[0].getBoundingClientRect(); zoomSteps(1, -1, rect.left+rect.width/2, rect.top+rect.height/2); });
        $('.wcm-graph-zoom-out').off('click').on('click', function(){ var rect = $svg[0].getBoundingClientRect(); zoomSteps(1, 1, rect.left+rect.width/2, rect.top+rect.height/2); });
        $('.wcm-graph-fit').off('click').on('click', function(){ fitToScreen(); state.initialized = true; $svg.data('viewState', state); });
        $('.wcm-graph-reset').off('click').on('click', function(){ state={scale:1,tx:0,ty:0,rot:0,initialized:true}; applyTransform(); });
        $('.wcm-graph-rotate').off('click').on('click', function(){ state.rot = (state.rot + 90)%360; applyTransform(); });
        $('.wcm-graph-fullscreen').off('click').on('click', function(){
            toggleFullscreen($svg);
        });
        $('.wcm-graph-undo').off('click').on('click', function(){ try { if (window.wcmDoUndo) window.wcmDoUndo(); } catch(e){} });
        $('.wcm-graph-redo').off('click').on('click', function(){ try { if (window.wcmDoRedo) window.wcmDoRedo(); } catch(e){} });

        // Make canvas fill viewport and apply state
        $(window).off('resize.wcmGraphCanvas').on('resize.wcmGraphCanvas', function(){ sizeGraphCanvas($svg); });
        sizeGraphCanvas($svg);
        if (!state.initialized) { fitToScreen(); state.initialized = true; $svg.data('viewState', state); }
        else { applyTransform(); }
        // Ensure labels sized correctly after initial render
        updateLabelScaling();

        // Allow external refit requests (e.g., after toggling from list)
        $svg.off('wcm-refit').on('wcm-refit', function(){ fitToScreen(); state.initialized = true; $svg.data('viewState', state); });
    }

    function sizeGraphCanvas($svg, force){
        var rect = $svg[0].getBoundingClientRect();
        var top = rect.top;
        var $container = $('#wcm-graph-view');
        
        if ($container.hasClass('wcm-fullscreen')) {
            // Fullscreen mode
            $svg.attr('height', window.innerHeight - 60);
            $svg.attr('width', '100%');
            return;
        }
        
        try { var cont = document.getElementById('wcm-graph-view'); if (cont && cont.classList.contains('wcm-fullscreen')) { top = 40; } } catch(e){}
        var available = window.innerHeight - top - 20;
        if (available < 400) available = 400;
        $svg.attr('height', available);
    }

    function ensureFullscreenButton(){
        var $toolbar = $('#wcm-graph-toolbar .right');
        if (!$toolbar.length) return;
        if (!$toolbar.find('.wcm-graph-fullscreen').length) {
            $('<button type="button" class="button wcm-graph-fullscreen" title="Fullscreen">Full</button>').appendTo($toolbar);
        }
    }

    function toggleFullscreen($svg){
        var $container = $('#wcm-graph-view');
        var isFullscreen = $container.hasClass('wcm-fullscreen');
        
        if (!isFullscreen) {
            // Enter fullscreen
            $container.addClass('wcm-fullscreen');
            $('body').addClass('wcm-fullscreen-active');
            
            // Force immediate resize and visibility
            setTimeout(function(){ 
                try { 
                    var $s = $('#wcm-graph-svg');
                    $s.attr('height', window.innerHeight - 60);
                    $s.attr('width', window.innerWidth);
                    $s.css({
                        'display': 'block',
                        'visibility': 'visible',
                        'opacity': '1'
                    });
                    sizeGraphCanvas($s, true);
                    $s.trigger('wcm-refit'); 
                } catch(e){} 
            }, 50);
            
            // Force again after render
            setTimeout(function(){ 
                try { 
                    var $s = $('#wcm-graph-svg');
                    $s.attr('height', window.innerHeight - 60);
                    $s.attr('width', window.innerWidth);
                    $s.css({
                        'display': 'block',
                        'visibility': 'visible',
                        'opacity': '1'
                    });
                    sizeGraphCanvas($s, true);
                    $s.trigger('wcm-refit'); 
                } catch(e){} 
            }, 200);
            
        } else {
            // Exit fullscreen
            $container.removeClass('wcm-fullscreen');
            $('body').removeClass('wcm-fullscreen-active');
            
            setTimeout(function(){ 
                try { 
                    var $s = $('#wcm-graph-svg');
                    sizeGraphCanvas($s, true);
                    $s.trigger('wcm-refit'); 
                } catch(e){} 
            }, 50);
        }
    }

    function fetchCategories(cb){
        $.post(shopc_ajax.ajax_url, { action:'shopc_get_categories', nonce:shopc_ajax.nonce }, function(res){
            if (res === -1 || res === "-1") {
                if (window.showMessage) window.showMessage((shopc_ajax.strings && shopc_ajax.strings.nonce_failed) || 'Nonce failed','error');
                return cb(new Error('nonce failed'));
            }
            if (res && res.success && res.data && Array.isArray(res.data.items)) return cb(null, res.data.items);
            if (window.showMessage) {
                var msg = (res && res.data && res.data.message) ? res.data.message : ((shopc_ajax.strings && shopc_ajax.strings.error) || 'Error');
                window.showMessage(msg,'error');
            }
            cb(new Error('fetch failed'));
        }).fail(function(xhr){
            var body = (xhr && xhr.responseText) ? xhr.responseText.trim() : '';
            if (window.showMessage) {
                if (body === '-1') window.showMessage((shopc_ajax.strings && shopc_ajax.strings.nonce_failed) || 'Nonce failed','error');
                else if (body === '0') window.showMessage((shopc_ajax.strings && shopc_ajax.strings.unknown_error) || 'Unknown error','error');
                else window.showMessage((shopc_ajax.strings && shopc_ajax.strings.error) || 'Error','error');
            }
            if (window.console) console.error('WCM AJAX error (graph fetch):', xhr);
            cb(new Error('fetch failed'));
        });
    }

    function refreshGraph($svg){
        fetchCategories(function(err, items){ if (err) return; renderGraph($svg, items); });
    }
    // Expose refreshGraph globally for admin.js to call after undo/redo
    window.refreshGraph = function($svg){ var $s = ($svg && $svg.length) ? $svg : $('#wcm-graph-svg'); refreshGraph($s); };

    function attemptReparent(categoryId, newParentId, fromParentId){
        if (shopc_ajax.default_category_id && categoryId === parseInt(shopc_ajax.default_category_id,10) && newParentId!==0) {
            return window.showMessage ? window.showMessage(shopc_ajax.strings.cannot_make_default_child,'error') : null;
        }
        $.post(shopc_ajax.ajax_url, { action:'shopc_update_category_parent', category_id:categoryId, new_parent:newParentId, nonce:shopc_ajax.nonce }, function(res){
            if (res && res.success) {
                if (window.showMessage) window.showMessage((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.parent_updated) || 'Parent category updated','success');
                // push to shared history stack for undo/redo
                try { if (window.wcmPushHistory) window.wcmPushHistory({ type:'reparent', id:categoryId, from: (typeof fromParentId==='number'?fromParentId:0), to: newParentId }); } catch(e){}
                refreshGraph($('#wcm-graph-svg'));
                if (window.refreshCategoryList) window.refreshCategoryList();
            } else {
                if (window.showMessage) window.showMessage(res && res.data && res.data.message ? res.data.message : 'Hata','error');
                refreshGraph($('#wcm-graph-svg'));
            }
        });
    }

    function initToggle(){
        $(document).on('click', '.wcm-view-toggle .button', function(){
            $('.wcm-view-toggle .button').removeClass('is-active'); $(this).addClass('is-active');
            var target = $(this).data('target');
            if (target==='graph') {
                $('#wcm-list-view').hide(); $('#wcm-graph-view').show();
                var $svg = $('#wcm-graph-svg');
                ensureFullscreenButton();
                // dblclick to toggle fullscreen
                $('#wcm-graph-view').off('dblclick.wcmFull').on('dblclick.wcmFull', function(e){ if ($(e.target).closest('.wcm-node,.button').length) return; toggleFullscreen($svg); });
                sizeGraphCanvas($svg);
                // Reset view state so first render fits to screen
                $svg.data('viewState', { scale:1, tx:0, ty:0, rot:0, initialized:false });
                refreshGraph($svg);
                // After render, request a refit
                setTimeout(function(){ try { $svg.trigger('wcm-refit'); } catch(e){} }, 80);
            } else {
                $('#wcm-graph-view').hide(); $('#wcm-list-view').show();
            }
        });

        // Global: press "f" to toggle fullscreen when graph is visible
        $(document).on('keydown.wcmGraphFull', function(e){
            if (e.key && (e.key.toLowerCase() === 'f')) {
                if ($('#wcm-graph-view:visible').length) { toggleFullscreen($('#wcm-graph-svg')); e.preventDefault(); }
            }
        });
    }

    $(function(){
        if ($('#wcm-graph-view').length) initToggle();
    });
})(jQuery);


(function($){
    function buildTree(items) {
        var byId = {}, roots = [];
        items.forEach(function(it){ byId[it.id] = Object.assign({children:[], parentId: it.parent || 0}, it); });
        items.forEach(function(it){
            if (it.parent && byId[it.parent]) {
                var child = byId[it.id];
                var parent = byId[it.parent];
                child.parent = parent;
                parent.children.push(child);
            } else {
                roots.push(byId[it.id]);
            }
        });
        function sortNode(n){
            n.children.sort(function(a,b){ if (a.order===b.order) return a.name.localeCompare(b.name); return a.order-b.order; });
            n.children.forEach(sortNode);
        }
        roots.forEach(sortNode);
        return { roots: roots, map: byId };
    }

    // Vertical layout: depth goes downward (Y); siblings spread along X
    function layoutTree(roots, cfg) {
        var depthY = cfg.depthX || 220; // reuse provided depth step
        var gapX = cfg.gapY || 120;     // horizontal gap widened to reduce label overlap
        var nodeX = 0;
        function firstWalk(n, depth) {
            n.y = depth * depthY;
            if (!n.children || n.children.length===0) {
                n.x = nodeX * gapX; nodeX++;
            } else {
                n.children.forEach(function(c){ firstWalk(c, depth+1); });
                var minX = n.children[0].x, maxX = n.children[n.children.length-1].x;
                n.x = (minX + maxX)/2;
            }
        }
        roots.forEach(function(r){ firstWalk(r, 0); });
    }

    function renderGraph($svg, items) {
        var built = buildTree(items);
        var roots = built.roots; var map = built.map;
        layoutTree(roots, { gapY: 120, depthX: 220 });
        var svg = $svg[0];
        while (svg.firstChild) svg.removeChild(svg.firstChild);
        var ns = 'http://www.w3.org/2000/svg';

        // Container group for pan/zoom/rotate
        var container = document.createElementNS(ns, 'g');
        container.setAttribute('class', 'wcm-graph-content');
        svg.appendChild(container);

        function add(el){ container.appendChild(el); return el; }
        function make(tag, attrs){ var el = document.createElementNS(ns, tag); for (var k in attrs) el.setAttribute(k, attrs[k]); return el; }

        // Collect nodes list
        var nodes = [];
        function walk(n) { nodes.push(n); (n.children||[]).forEach(walk); }
        roots.forEach(walk);

        // Depth guides (horizontal lines per level)
        var maxDepth = 0; nodes.forEach(function(n){ var d = Math.round(n.y/220); if (d>maxDepth) maxDepth = d; });
        for (var d=0; d<=maxDepth; d++) {
            var y = d*220 + 40;
            add(make('line', { x1:0, y1:y, x2:10000, y2:y, class:'wcm-depth' }));
        }

        // Color palette per root and shade by level
        var hues = [200, 340, 20, 120, 260, 40, 170, 300, 0, 210]; // extend as needed
        var rootHue = {};
        roots.forEach(function(r, idx){ rootHue[r.id] = hues[idx % hues.length]; });

        function getRoot(node){ var n=node; while (n && n.parent) n = n.parent; return n; }
        function hslToHex(h,s,l){
            h/=360; s/=100; l/=100;
            function f(n){
                var k=(n+h*12)%12; var a=s*Math.min(l,1-l);
                var c=l - a*Math.max(-1, Math.min(k-3, Math.min(9-k,1)));
                return Math.round(255*c).toString(16).padStart(2,'0');
            }
            return '#'+f(0)+f(8)+f(4);
        }
        function nodeFillColor(n){
            var r = getRoot(n); var hue = rootHue[r.id] || 200;
            var baseL = 56; var delta = Math.min(4+n.level*6, 24);
            var l = Math.max(24, Math.min(84, baseL + (n.level>0 ? 12 : 0) - delta));
            return hslToHex(hue, 60, l);
        }
        function nodeStrokeColor(n){
            var r = getRoot(n); var hue = rootHue[r.id] || 200;
            var l = Math.max(18, 48 - n.level*4);
            return hslToHex(hue, 70, l);
        }

        // Links (vertical curve)
        nodes.forEach(function(n){
            (n.children||[]).forEach(function(c){
                var p = make('path', { d: bezier(n.x, n.y, c.x, c.y), class: 'wcm-link' });
                // tint link with parent color
                p.setAttribute('stroke', nodeStrokeColor(c));
                add(p);
            });
        });

        // Nodes
        nodes.forEach(function(n){
            var g = make('g', { class: 'wcm-node' });
            g.setAttribute('transform', 'translate('+ (n.x+40) +','+ (n.y+40) +')');
            g.dataset.id = n.id;
            var radius = 18 - Math.min(10, (n.level||0));
            var circle = make('circle', { r: Math.max(10, radius), filter:'url(#wcm-shadow)' });
            // apply dynamic colors (inline for fill so it wins over CSS defaults)
            circle.setAttribute('style', 'fill:'+nodeFillColor(n));
            circle.setAttribute('stroke', nodeStrokeColor(n));
            var label = make('text', { x: 0, y: 0, 'text-anchor':'start' });
            var full = n.name || '';
            var maxLen = 24;
            var display = full.length > maxLen ? (full.slice(0, maxLen) + '…') : full;
            label.textContent = display;
            // rotate label diagonally to reduce overlap
            label.setAttribute('transform', 'translate(22, 10) rotate(-28)');
            label.setAttribute('class', 'wcm-label');
            label.setAttribute('data-base-transform', 'translate(22, 10) rotate(-28)');
            // title tooltip with full text
            try { var ttl = make('title', {}); ttl.textContent = full; label.appendChild(ttl); } catch(e){}
            if (!n.parent) g.classList.add('root');
            g.appendChild(circle); g.appendChild(label);
            add(g);
        });

        // Drag to reparent with preview & constraints
        // Clear previous handlers to avoid duplicates when re-rendering
        $(document).off('.wcmGraph');
        $svg.off('mousedown', '.wcm-node');
        var dragging = null, ghostLink = null, $hint = null;
        var defaultId = parseInt(shopc_ajax.default_category_id || 0, 10) || 0;
        var draggingOrigParentId = null;

        function showHint(text, x, y){
            if (!$hint) { $hint = $('<div class="wcm-graph-hint"/>').appendTo($svg.parent()); }
            $hint.text(text).css({ left: x+12, top: y+12 }).addClass('visible');
        }
        function hideHint(){ if ($hint) { $hint.removeClass('visible'); $hint.remove(); $hint=null; } }

        var spacePan = false;
        $(document).on('keydown.wcmGraph', function(e){
            if (e.target && (/input|textarea|select/i).test(e.target.tagName)) return;
            var step = 40; // px
            if (e.key === ' ') { spacePan = true; e.preventDefault(); }
            else if (e.key === 'ArrowLeft') { state.tx += step; applyTransform(); e.preventDefault(); }
            else if (e.key === 'ArrowRight') { state.tx -= step; applyTransform(); e.preventDefault(); }
            else if (e.key === 'ArrowUp') { state.ty += step; applyTransform(); e.preventDefault(); }
            else if (e.key === 'ArrowDown') { state.ty -= step; applyTransform(); e.preventDefault(); }
        });
        $(document).on('keyup.wcmGraph', function(e){ if (e.key === ' ') { spacePan = false; } });

        var lastSelectedNode = null;
        var selectedSet = new Set();
        $svg.on('mousedown', '.wcm-node', function(e){
            if (spacePan) { // space+drag pans canvas instead of node
                panning = true; sx = e.clientX; sy = e.clientY; stx = state.tx; sty = state.ty; e.preventDefault(); return;
            }
            dragging = this; lastSelectedNode = this; this.classList.add('dragging'); highlightDroppables(this); e.preventDefault();
            try { var did=parseInt(this.dataset.id,10)||0; var node = map[did]; draggingOrigParentId = (node && node.parent) ? (node.parent.id || node.parentId || 0) : (node ? node.parentId||0 : 0); } catch(_){ draggingOrigParentId = 0; }
        });
        // Select on click (for Delete key usage)
        $svg.on('click', '.wcm-node', function(e){
            lastSelectedNode = this;
            var id=parseInt(this.dataset.id,10)||0; if(!id) return;
            if (e.ctrlKey || e.metaKey) {
                if (selectedSet.has(id)) { selectedSet.delete(id); this.classList.remove('selected'); }
                else { selectedSet.add(id); this.classList.add('selected'); }
            } else {
                // single select
                $('.wcm-node.selected', $svg).removeClass('selected'); selectedSet.clear(); selectedSet.add(id); this.classList.add('selected');
            }
        });
        // Context menu delete
        $svg.on('contextmenu', '.wcm-node', function(e){ e.preventDefault(); var g=this; var id=parseInt(g.dataset.id,10)||0; var name=$(g).find('text').text(); if(!id) return; var msg=(shopc_ajax.strings&&shopc_ajax.strings.confirm_delete)? shopc_ajax.strings.confirm_delete.replace('%s', name):('Delete "'+name+'"?'); if(!window.confirm(msg)) return; var cascade=window.confirm((shopc_ajax.strings&&shopc_ajax.strings.confirm_delete_children)||'Also delete all subcategories?'); $.post(shopc_ajax.ajax_url,{action:'shopc_delete_category',category_id:id,delete_children:cascade?1:0,nonce:shopc_ajax.nonce},function(res){ if(res&&res.success){ if(window.showMessage) window.showMessage((shopc_ajax.strings&&shopc_ajax.strings.deleted)||'Deleted','success'); refreshGraph($svg); if(window.refreshCategoryList) window.refreshCategoryList(); } else { if(window.showMessage) window.showMessage(res&&res.data&&res.data.message?res.data.message:(shopc_ajax.strings&&shopc_ajax.strings.error)||'Error','error'); }}); });
        // Delete key handler
        $(document).on('keydown.wcmGraphDelete', function(e){
            if ((e.key==='Delete'||e.key==='Backspace') && !dragging) {
                var ids = selectedSet.size ? Array.from(selectedSet) : (function(){ var g=lastSelectedNode; return g?[parseInt(g.dataset.id,10)||0]:[]; })();
                ids = ids.filter(function(x){return x>0;}); if(!ids.length) return;
                var msg=(shopc_ajax.strings&&shopc_ajax.strings.delete_selected)||'Delete selected'; if(!window.confirm(msg+' ('+ids.length+')?')) return;
                var cascade=window.confirm((shopc_ajax.strings&&shopc_ajax.strings.confirm_delete_children)||'Also delete all subcategories?');
                $.post(shopc_ajax.ajax_url,{action:'shopc_delete_categories',ids:JSON.stringify(ids),delete_children:cascade?1:0,nonce:shopc_ajax.nonce},function(res){
                    if(res&&res.success){ if(window.showMessage) window.showMessage((shopc_ajax.strings&&shopc_ajax.strings.deleted)||'Deleted','success'); selectedSet.clear(); $('.wcm-node.selected',$svg).removeClass('selected'); refreshGraph($svg); if(window.refreshCategoryList) window.refreshCategoryList(); }
                    else { if(window.showMessage) window.showMessage(res&&res.data&&res.data.message?res.data.message:(shopc_ajax.strings&&shopc_ajax.strings.error)||'Error','error'); }
                });
                e.preventDefault();
            }
        });
        $(document).on('mousemove.wcmGraph', function(e){
            if (!dragging) return;
            // Auto-pan when cursor nears edges to enable long drags
            try {
                var rect = $svg[0].getBoundingClientRect();
                var edge = 60, step = 16;
                var dx = 0, dy = 0;
                if (e.clientX - rect.left < edge) dx = step; else if (rect.right - e.clientX < edge) dx = -step;
                if (e.clientY - rect.top < edge) dy = step; else if (rect.bottom - e.clientY < edge) dy = -step;
                if (dx || dy) { state.tx += dx; state.ty += dy; applyTransform(); }
            } catch(_){ }
            var pt = (typeof container !== 'undefined' && container && container.getScreenCTM) ? (function(){
                var svgEl = container.ownerSVGElement || svg; var p = svgEl.createSVGPoint(); p.x=e.clientX; p.y=e.clientY; return p.matrixTransform(container.getScreenCTM().inverse());
            })() : clientToSvg(svg, e.clientX, e.clientY);
            dragging.setAttribute('transform', 'translate('+pt.x+','+pt.y+')');
            if (!ghostLink) ghostLink = add(make('path', { class:'wcm-ghost-link', d:'M0 0 L0 0' }));
            var origin = nodeCenter(dragging);
            ghostLink.setAttribute('d', 'M '+origin.x+' '+origin.y+' L '+pt.x+' '+pt.y);
            var target = findClosestNode($svg[0], dragging);
            $('.wcm-node.target', $svg).removeClass('target');
            if (target && target!==dragging) {
                var tid = parseInt(target.dataset.id,10)||0; var did = parseInt(dragging.dataset.id,10)||0;
                var ok = isDroppable(did, tid, map, defaultId);
                $(target).toggleClass('target', true).toggleClass('nodrop', !ok).toggleClass('droppable', ok);
                var hint = ok ? ((shopc_ajax.strings && shopc_ajax.strings.will_be_parent) || 'Will become parent')
                               : ((shopc_ajax.strings && shopc_ajax.strings.invalid_target) || 'Invalid target');
                showHint(hint, e.pageX, e.pageY);
            } else {
                // No node target under cursor: suggest root drop
                showHint((shopc_ajax.strings && shopc_ajax.strings.drop_to_root) || 'Drop to make root', e.pageX, e.pageY);
            }
        });
        $(document).on('mouseup.wcmGraph', function(e){
            if (!dragging) return;
            var $drag = $(dragging);
            var id = parseInt(dragging.dataset.id, 10);
            var target = findClosestNode($svg[0], dragging);
            $drag.removeClass('dragging');
            if (ghostLink) {
                if (ghostLink.parentNode) { ghostLink.parentNode.removeChild(ghostLink); }
                ghostLink=null;
            }
            $('.wcm-node.droppable, .wcm-node.nodrop', $svg).removeClass('droppable nodrop');
            hideHint();
            if (target && target!==dragging) {
                var newParent = parseInt(target.dataset.id, 10) || 0;
                if (isDroppable(id, newParent, map, defaultId)) {
                    attemptReparent(id, newParent, draggingOrigParentId);
                } else {
                    refreshGraph($svg);
                }
            } else {
                // Drop on background: make it a root category
                attemptReparent(id, 0, draggingOrigParentId);
            }
            dragging = null; draggingOrigParentId = null; $('.wcm-node.target', $svg).removeClass('target');
        });

        function highlightDroppables(dragEl){
            var did = parseInt(dragEl.dataset.id,10)||0;
            nodes.forEach(function(n){
                if (n.id === did) return;
                var ok = isDroppable(did, n.id, map, defaultId);
                var g = $svg[0].querySelector('.wcm-node[data-id="'+n.id+'"]');
                if (g) g.classList.add(ok? 'droppable' : 'nodrop');
            });
        }

        function isDroppable(dragId, targetId, map, defaultId){
            if (dragId === targetId) return false;
            if (defaultId && dragId === defaultId && targetId !== 0) return false;
            // target is descendant of drag? If yes, invalid
            var t = map[targetId];
            while (t && t.parent) { if (t.parent.id === dragId) return false; t = t.parent; }
            return true;
        }

        function bezier(x1,y1,x2,y2){
            var my = (y1+y2)/2; return 'M '+(x1+40)+' '+(y1+58)+' C '+(x1+40)+' '+(my)+' '+(x2+40)+' '+(my)+' '+(x2+40)+' '+(y2+22);
        }
        function clientToSvg(svg, cx, cy){
            var pt = svg.createSVGPoint(); pt.x = cx; pt.y = cy; return pt.matrixTransform(svg.getScreenCTM().inverse());
        }
        function nodeCenter(g){
            var m = g.transform.baseVal.consolidate().matrix; return { x: m.e, y: m.f };
        }
        function findClosestNode(svgEl, excludeEl){
            var nodes = Array.prototype.slice.call(svgEl.querySelectorAll('.wcm-node'));
            var minD = Infinity, best = null;
            nodes.forEach(function(n){ if (n===excludeEl) return; var bb = n.getBoundingClientRect(); var cx = (bb.left+bb.right)/2; var cy=(bb.top+bb.bottom)/2; var d = Math.hypot(cx-lastClientX, cy-lastClientY); if (d<minD){minD=d; best=n;} });
            return (minD<80)? best : null;
        }

        var lastClientX=0,lastClientY=0,lastMouseX=0,lastMouseY=0; // track for hit-test
        $svg.on('mousemove', function(e){ lastClientX=e.clientX; lastClientY=e.clientY; var pt = clientToSvg(svg, e.clientX, e.clientY); lastMouseX=pt.x; lastMouseY=pt.y; });

        // Shadow filter defs
        var defs = make('defs', {});
        var filter = make('filter', { id:'wcm-shadow', x:'-20%', y:'-20%', width:'140%', height:'140%' });
        var fe = document.createElementNS(ns, 'feDropShadow');
        fe.setAttribute('dx','0'); fe.setAttribute('dy','1'); fe.setAttribute('stdDeviation','2'); fe.setAttribute('flood-opacity','.25');
        filter.appendChild(fe); defs.appendChild(filter); svg.insertBefore(defs, svg.firstChild);

        // Pan / Zoom / Rotate controls with persistent state
        var state = $svg.data('viewState') || { scale: 1, tx: 0, ty: 0, rot: 0, initialized: false };
        function applyTransform(){
            container.setAttribute('transform', 'translate('+state.tx+','+state.ty+') scale('+state.scale+') rotate('+state.rot+')');
            $svg.data('viewState', state);
            updateLabelScaling();
        }

        function updateLabelScaling(){
            try {
                var s = state.scale || 1;
                // Keep labels readable when zoomed out: inversely scale up to a cap
                var k = (s < 1) ? (1 / s) : 1;
                if (k > 2.4) k = 2.4; // cap to avoid giant labels
                var texts = container.querySelectorAll('.wcm-label');
                texts.forEach(function(t){
                    var base = t.getAttribute('data-base-transform') || '';
                    t.setAttribute('transform', base + (k!==1 ? (' scale(' + k + ')') : ''));
                });
            } catch(e){}
        }

        function fitToScreen(){
            var minX=Infinity, minY=Infinity, maxX=-Infinity, maxY=-Infinity;
            nodes.forEach(function(n){ var x=n.x+40, y=n.y+40; var r=Math.max(10, 18 - Math.min(10,(n.level||0))); minX=Math.min(minX, x-r-40); minY=Math.min(minY, y-r-40); maxX=Math.max(maxX, x+r+120); maxY=Math.max(maxY, y+r+40); });
            var vbW = $svg[0].clientWidth, vbH = $svg[0].clientHeight; if (!vbW) vbW = 1; if (!vbH) vbH = 1;
            var contentW = (maxX - minX) || 1; var contentH = (maxY - minY) || 1; var margin = 20;
            var s = Math.min((vbW - margin*2)/contentW, (vbH - margin*2)/contentH); s = Math.max(0.2, Math.min(3, s));
            state.scale = s; state.rot = state.rot || 0;
            state.tx = (vbW - contentW*s)/2 - minX*s; state.ty = (vbH - contentH*s)/2 - minY*s;
            applyTransform();
        }

        function animateZoomTo(ns, cx, cy, ms){
            var old = state.scale; if (ns === old) return;
            var dur = Math.max(80, Math.min(220, ms||140));
            var start = performance.now();
            var pt = clientToSvg(svg, cx, cy);
            var startTx = state.tx, startTy = state.ty;
            function step(t){
                var k = Math.min(1, (t - start) / dur);
                // easeOutCubic
                var e = 1 - Math.pow(1-k, 3);
                var cur = old + (ns - old) * e;
                // adjust translation to keep focus point stable
                state.tx = pt.x - (pt.x - startTx) * (cur/old);
                state.ty = pt.y - (pt.y - startTy) * (cur/old);
                state.scale = cur; applyTransform();
                if (k < 1) requestAnimationFrame(step);
            }
            requestAnimationFrame(step);
        }

        function zoomAt(delta, cx, cy){
            // delta>0: zoom out; <0: zoom in. Use gentle factor and animate
            var old = state.scale;
            var ns = Math.max(0.2, Math.min(3, old * (delta>0 ? 0.96 : 1.04)));
            if (Math.abs(ns-old) < 0.0001) return;
            animateZoomTo(ns, cx, cy, 140);
        }

        function zoomSteps(steps, sign, cx, cy){
            var factor = Math.pow(sign>0 ? 0.96 : 1.04, Math.max(1, steps));
            var old = state.scale; var ns = Math.max(0.2, Math.min(3, old * factor));
            animateZoomTo(ns, cx, cy, 140 + Math.min(200, steps*20));
        }

        // Wheel zoom
        $svg.on('wheel.wcmGraph', function(e){
            e.preventDefault();
            var ev = e.originalEvent || e; var dy = ev.deltaY || 0; var mode = ev.deltaMode || 0; // 0: pixel, 1: line, 2: page
            // Normalize rough steps; larger wheels or trackpads produce varied deltas
            var magnitude = Math.abs(dy);
            if (mode === 1) magnitude *= 16; // lines -> approx pixels
            else if (mode === 2) magnitude *= 240; // pages -> large
            var steps = Math.max(1, Math.round(magnitude / 120));
            var sign = dy>0 ? 1 : -1;
            zoomSteps(steps, sign, ev.clientX, ev.clientY);
        });

        // Drag pan when clicking background
        var panning=false, sx=0, sy=0, stx=0, sty=0;
        $svg.on('mousedown.wcmGraph', function(e){
            if ($(e.target).closest('.wcm-node').length) return; // ignore node drags
            panning = true; sx = e.clientX; sy = e.clientY; stx = state.tx; sty = state.ty; e.preventDefault();
        });
        $(document).on('mousemove.wcmGraph', function(e){ if (!panning || dragging) return; state.tx = stx + (e.clientX - sx); state.ty = sty + (e.clientY - sy); applyTransform(); });
        $(document).on('mouseup.wcmGraph', function(){ panning=false; });

        // Toolbar buttons
        $('.wcm-graph-zoom-in').off('click').on('click', function(){ var rect = $svg[0].getBoundingClientRect(); zoomSteps(1, -1, rect.left+rect.width/2, rect.top+rect.height/2); });
        $('.wcm-graph-zoom-out').off('click').on('click', function(){ var rect = $svg[0].getBoundingClientRect(); zoomSteps(1, 1, rect.left+rect.width/2, rect.top+rect.height/2); });
        $('.wcm-graph-fit').off('click').on('click', function(){ fitToScreen(); state.initialized = true; $svg.data('viewState', state); });
        $('.wcm-graph-reset').off('click').on('click', function(){ state={scale:1,tx:0,ty:0,rot:0,initialized:true}; applyTransform(); });
        $('.wcm-graph-rotate').off('click').on('click', function(){ state.rot = (state.rot + 90)%360; applyTransform(); });
        $('.wcm-graph-fullscreen').off('click').on('click', function(){
            toggleFullscreen($svg);
        });
        $('.wcm-graph-undo').off('click').on('click', function(){ try { if (window.wcmDoUndo) window.wcmDoUndo(); } catch(e){} });
        $('.wcm-graph-redo').off('click').on('click', function(){ try { if (window.wcmDoRedo) window.wcmDoRedo(); } catch(e){} });

        // Make canvas fill viewport and apply state
        $(window).off('resize.wcmGraphCanvas').on('resize.wcmGraphCanvas', function(){ sizeGraphCanvas($svg); });
        sizeGraphCanvas($svg);
        if (!state.initialized) { fitToScreen(); state.initialized = true; $svg.data('viewState', state); }
        else { applyTransform(); }
        // Ensure labels sized correctly after initial render
        updateLabelScaling();

        // Allow external refit requests (e.g., after toggling from list)
        $svg.off('wcm-refit').on('wcm-refit', function(){ fitToScreen(); state.initialized = true; $svg.data('viewState', state); });
    }

    function sizeGraphCanvas($svg, force){
        var rect = $svg[0].getBoundingClientRect();
        var top = rect.top;
        var $container = $('#wcm-graph-view');
        
        if ($container.hasClass('wcm-fullscreen')) {
            // Fullscreen mode
            $svg.attr('height', window.innerHeight - 60);
            $svg.attr('width', '100%');
            return;
        }
        
        try { var cont = document.getElementById('wcm-graph-view'); if (cont && cont.classList.contains('wcm-fullscreen')) { top = 40; } } catch(e){}
        var available = window.innerHeight - top - 20;
        if (available < 400) available = 400;
        $svg.attr('height', available);
    }

    function ensureFullscreenButton(){
        var $toolbar = $('#wcm-graph-toolbar .right');
        if (!$toolbar.length) return;
        if (!$toolbar.find('.wcm-graph-fullscreen').length) {
            $('<button type="button" class="button wcm-graph-fullscreen" title="Fullscreen">Full</button>').appendTo($toolbar);
        }
    }

    function toggleFullscreen($svg){
        var $container = $('#wcm-graph-view');
        var isFullscreen = $container.hasClass('wcm-fullscreen');
        
        if (!isFullscreen) {
            // Enter fullscreen
            $container.addClass('wcm-fullscreen');
            $('body').addClass('wcm-fullscreen-active');
            
            // Force immediate resize and visibility
            setTimeout(function(){ 
                try { 
                    var $s = $('#wcm-graph-svg');
                    $s.attr('height', window.innerHeight - 60);
                    $s.attr('width', window.innerWidth);
                    $s.css({
                        'display': 'block',
                        'visibility': 'visible',
                        'opacity': '1'
                    });
                    sizeGraphCanvas($s, true);
                    $s.trigger('wcm-refit'); 
                } catch(e){} 
            }, 50);
            
            // Force again after render
            setTimeout(function(){ 
                try { 
                    var $s = $('#wcm-graph-svg');
                    $s.attr('height', window.innerHeight - 60);
                    $s.attr('width', window.innerWidth);
                    $s.css({
                        'display': 'block',
                        'visibility': 'visible',
                        'opacity': '1'
                    });
                    sizeGraphCanvas($s, true);
                    $s.trigger('wcm-refit'); 
                } catch(e){} 
            }, 200);
            
        } else {
            // Exit fullscreen
            $container.removeClass('wcm-fullscreen');
            $('body').removeClass('wcm-fullscreen-active');
            
            setTimeout(function(){ 
                try { 
                    var $s = $('#wcm-graph-svg');
                    sizeGraphCanvas($s, true);
                    $s.trigger('wcm-refit'); 
                } catch(e){} 
            }, 50);
        }
    }

    function fetchCategories(cb){
        $.post(shopc_ajax.ajax_url, { action:'shopc_get_categories', nonce:shopc_ajax.nonce }, function(res){
            if (res === -1 || res === "-1") {
                if (window.showMessage) window.showMessage((shopc_ajax.strings && shopc_ajax.strings.nonce_failed) || 'Nonce failed','error');
                return cb(new Error('nonce failed'));
            }
            if (res && res.success && res.data && Array.isArray(res.data.items)) return cb(null, res.data.items);
            if (window.showMessage) {
                var msg = (res && res.data && res.data.message) ? res.data.message : ((shopc_ajax.strings && shopc_ajax.strings.error) || 'Error');
                window.showMessage(msg,'error');
            }
            cb(new Error('fetch failed'));
        }).fail(function(xhr){
            var body = (xhr && xhr.responseText) ? xhr.responseText.trim() : '';
            if (window.showMessage) {
                if (body === '-1') window.showMessage((shopc_ajax.strings && shopc_ajax.strings.nonce_failed) || 'Nonce failed','error');
                else if (body === '0') window.showMessage((shopc_ajax.strings && shopc_ajax.strings.unknown_error) || 'Unknown error','error');
                else window.showMessage((shopc_ajax.strings && shopc_ajax.strings.error) || 'Error','error');
            }
            if (window.console) console.error('WCM AJAX error (graph fetch):', xhr);
            cb(new Error('fetch failed'));
        });
    }

    function refreshGraph($svg){
        fetchCategories(function(err, items){ if (err) return; renderGraph($svg, items); });
    }
    // Expose refreshGraph globally for admin.js to call after undo/redo
    window.refreshGraph = function($svg){ var $s = ($svg && $svg.length) ? $svg : $('#wcm-graph-svg'); refreshGraph($s); };

    function attemptReparent(categoryId, newParentId, fromParentId){
        if (shopc_ajax.default_category_id && categoryId === parseInt(shopc_ajax.default_category_id,10) && newParentId!==0) {
            return window.showMessage ? window.showMessage(shopc_ajax.strings.cannot_make_default_child,'error') : null;
        }
        $.post(shopc_ajax.ajax_url, { action:'shopc_update_category_parent', category_id:categoryId, new_parent:newParentId, nonce:shopc_ajax.nonce }, function(res){
            if (res && res.success) {
                if (window.showMessage) window.showMessage((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.parent_updated) || 'Parent category updated','success');
                // push to shared history stack for undo/redo
                try { if (window.wcmPushHistory) window.wcmPushHistory({ type:'reparent', id:categoryId, from: (typeof fromParentId==='number'?fromParentId:0), to: newParentId }); } catch(e){}
                refreshGraph($('#wcm-graph-svg'));
                if (window.refreshCategoryList) window.refreshCategoryList();
            } else {
                if (window.showMessage) window.showMessage(res && res.data && res.data.message ? res.data.message : 'Hata','error');
                refreshGraph($('#wcm-graph-svg'));
            }
        });
    }

    function initToggle(){
        $(document).on('click', '.wcm-view-toggle .button', function(){
            $('.wcm-view-toggle .button').removeClass('is-active'); $(this).addClass('is-active');
            var target = $(this).data('target');
            if (target==='graph') {
                $('#wcm-list-view').hide(); $('#wcm-graph-view').show();
                var $svg = $('#wcm-graph-svg');
                ensureFullscreenButton();
                // dblclick to toggle fullscreen
                $('#wcm-graph-view').off('dblclick.wcmFull').on('dblclick.wcmFull', function(e){ if ($(e.target).closest('.wcm-node,.button').length) return; toggleFullscreen($svg); });
                sizeGraphCanvas($svg);
                // Reset view state so first render fits to screen
                $svg.data('viewState', { scale:1, tx:0, ty:0, rot:0, initialized:false });
                refreshGraph($svg);
                // After render, request a refit
                setTimeout(function(){ try { $svg.trigger('wcm-refit'); } catch(e){} }, 80);
            } else {
                $('#wcm-graph-view').hide(); $('#wcm-list-view').show();
            }
        });

        // Global: press "f" to toggle fullscreen when graph is visible
        $(document).on('keydown.wcmGraphFull', function(e){
            if (e.key && (e.key.toLowerCase() === 'f')) {
                if ($('#wcm-graph-view:visible').length) { toggleFullscreen($('#wcm-graph-svg')); e.preventDefault(); }
            }
        });
    }

    $(function(){
        if ($('#wcm-graph-view').length) initToggle();
    });
})(jQuery);
