jQuery(document).ready(function($) {
    // Debug helper (toggle with window.wcmDebug = true/false)
    window.wcmDebug = (window.wcmDebug !== undefined) ? window.wcmDebug : true;
    function wcmLog(){ try { if (window.wcmDebug && window.console && console.log) { console.log.apply(console, ['[WCM]'].concat([].slice.call(arguments))); } } catch(e){} }
    var $categoryList = $("#wcm-category-list");
    var isUpdating = false;
    var $dragPreview = null; // simplified mode: not used
    var $highlighted = null;
    var dragState = { mode: 'sibling', parentId: 0, level: 0, startX: 0, startLevel: 0, children: [], detached: false, anchorId: 0, anchorBefore: false, anchorEl: null, cursorOffsetX: 0, cursorOffsetY: 0, anchorLockUntil: 0 };
    var INDENT_STEP = 16; // reference step (px) per hierarchy level
    // Persisted selection across refreshes
    var SEL_KEY = 'wcm_selected_ids_v1';
    function loadSel(){ try { return new Set(JSON.parse(sessionStorage.getItem(SEL_KEY)||'[]')); } catch(e){ return new Set(); } }
    function saveSel(s){ try { sessionStorage.setItem(SEL_KEY, JSON.stringify(Array.from(s||[]))); } catch(e){} }
    var wcmSelSet = loadSel();

    // Coalesced colorize scheduler to reduce jank
    var _clrTimer = null;
    function scheduleColorize(){
        try { if (_clrTimer) { clearTimeout(_clrTimer); _clrTimer = null; } } catch(_){ }
        var run = function(){ try { if (typeof colorizeList === 'function') colorizeList(); } catch(e){} };
        try {
            if ('requestIdleCallback' in window) {
                _clrTimer = setTimeout(function(){ window.requestIdleCallback(run, { timeout: 180 }); }, 0);
            } else {
                _clrTimer = setTimeout(run, 90);
            }
        } catch(_){ _clrTimer = setTimeout(run, 120); }
    }

    // Overlay loader for undo/redo and quick operations
    function wcmEnsureOverlay(){
        if (!document.getElementById('wcm-overlay')) {
            var $ov = $('<div id="wcm-overlay" aria-hidden="true" />').css({ position:'fixed', inset:0, background:'rgba(255,255,255,0.7)', zIndex: 100000, display:'none' });
            var $sp = $('<div class="wcm-overlay-spinner" />').css({ position:'absolute', left:'50%', top:'40%', transform:'translate(-50%,-50%)', width:'44px', height:'44px', borderRadius:'50%', border:'4px solid rgba(0,0,0,0.15)', borderTopColor:'#2271b1', animation:'wcm-spin 0.8s linear infinite' });
            $ov.append($sp);
            $('body').append($ov);
            // spinner keyframes
            var css = '@keyframes wcm-spin{from{transform:translate(-50%,-50%) rotate(0)}to{transform:translate(-50%,-50%) rotate(360deg)}}';
            try { var style=document.createElement('style'); style.appendChild(document.createTextNode(css)); document.head.appendChild(style); } catch(_){ }
        }
    }
    function wcmShowOverlay(){ try { wcmEnsureOverlay(); $('#wcm-overlay').stop(true,true).fadeIn(100); } catch(_){}}
    function wcmHideOverlay(){ try { $('#wcm-overlay').stop(true,true).fadeOut(120); } catch(_){}}

    // --- Undo/Redo (max 10) ---
    window.wcmHistory = window.wcmHistory || [];
    window.wcmRedo = window.wcmRedo || [];
    function wcmPushHistory(action){ try { window.wcmRedo.length = 0; window.wcmHistory.push(action); if (window.wcmHistory.length > 10) window.wcmHistory.shift(); } catch(e){} }
    function wcmDoUndo(){ if (!window.wcmHistory.length) return; var a = window.wcmHistory.pop(); if (!a) return; if (a.type === 'reparent'){ wcmApplyReparent(a.id, a.from, function(){ try { window.wcmRedo.push({ type:'reparent', id:a.id, from:a.to, to:a.from }); if (window.wcmRedo.length>10) window.wcmRedo.shift(); } catch(e){} try { if (window.showMessage) window.showMessage('Undo: Parent change','success'); } catch(_){} }); } else if (a.type==='reorder'){ wcmApplyOrders(a.before, function(){ try { window.wcmRedo.push({ type:'reorder', before:a.after, after:a.before }); if (window.wcmRedo.length>10) window.wcmRedo.shift(); } catch(e){} try { if (window.showMessage) window.showMessage('Undo: Reorder','success'); } catch(_){} }); } }
    function wcmDoRedo(){ if (!window.wcmRedo.length) return; var a = window.wcmRedo.pop(); if (!a) return; if (a.type === 'reparent'){ wcmApplyReparent(a.id, a.from, function(){ try { window.wcmHistory.push({ type:'reparent', id:a.id, from:a.from, to:a.to }); if (window.wcmHistory.length>10) window.wcmHistory.shift(); } catch(e){} try { if (window.showMessage) window.showMessage('Redo: Parent change','success'); } catch(_){} }); } else if (a.type==='reorder'){ wcmApplyOrders(a.before, function(){ try { window.wcmHistory.push({ type:'reorder', before:a.before, after:a.after }); if (window.wcmHistory.length>10) window.wcmHistory.shift(); } catch(e){} try { if (window.showMessage) window.showMessage('Redo: Reorder','success'); } catch(_){} }); } }
    function wcmApplyReparent(id, newParent, done){ wcmShowOverlay(); updateCategoryParent(id, newParent, function(){ try { refreshCategoryList(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg')); } catch(e){} wcmHideOverlay(); if (typeof done==='function') done(); }); }
    function wcmApplyOrders(orders, done){ wcmShowOverlay(); bulkUpdateCategoryOrders(orders, function(){ try { refreshCategoryList(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg')); } catch(e){} wcmHideOverlay(); if (typeof done==='function') done(); }, ''); }
    // Expose to window for Graph toolbar buttons
    window.wcmPushHistory = wcmPushHistory;
    window.wcmDoUndo = wcmDoUndo;
    window.wcmDoRedo = wcmDoRedo;

    // Keyboard shortcuts (Ctrl/Cmd+Z, Ctrl+Y or Ctrl+Shift+Z)
    $(document).on('keydown.wcmUndoRedo', function(e){ try { var tag = (e.target && e.target.tagName||'').toLowerCase(); if (tag==='input'||tag==='textarea') return; var ctrl = e.ctrlKey||e.metaKey; if (!ctrl) return; if (e.key.toLowerCase()==='z' && !e.shiftKey){ e.preventDefault(); wcmDoUndo(); } else if ((e.key.toLowerCase()==='y') || (e.key.toLowerCase()==='z' && e.shiftKey)){ e.preventDefault(); wcmDoRedo(); } } catch(_){}});
    // Button bindings (List View toolbar)
    $(document).on('click', '#wcm-undo', function(){ try { wcmDoUndo(); } catch(e){} });
    $(document).on('click', '#wcm-redo', function(){ try { wcmDoRedo(); } catch(e){} });

    var CHILD_ZONE_PX = 56; // child-drop zone width (px)
    // Use a small delay to stabilize hover target
    var HOVER_DELAY_MS = 170;
    var _hover = { lastId: null, since: 0, stableId: null, $stable: $() };

    // Pointer smoothing state
    var _smooth = { y: null, lastTs: 0, vy: 0 };

    // Drag-drop markers (drag and target); placeholder activates only when they overlap
    var _markers = { $drag: null, $target: null, $root: null, allow: false, allowRoot: false, wasAllow: false, wasAllowRoot: false };
    function _mkEnsure(){
        if (!_markers.$drag || !_markers.$drag.length) {
            _markers.$drag = $('<div id="wcm-drag-marker" />').css({ position:'absolute', zIndex: 99999, width:'14px', height:'14px', marginLeft:'-7px', marginTop:'-7px', borderRadius:'50%', border:'2px solid rgba(0,0,0,0.25)', boxShadow:'0 0 0 2px rgba(255,255,255,0.9)', pointerEvents:'none', background:'#e67e22', transition:'transform 140ms ease, background 120ms ease, box-shadow 140ms ease' }).appendTo('body');
        }
        if (!_markers.$target || !_markers.$target.length) {
            _markers.$target = $('<div id="wcm-target-marker" />').css({ position:'absolute', zIndex: 99998, width:'16px', height:'16px', marginLeft:'-8px', marginTop:'-8px', borderRadius:'50%', border:'2px dashed rgba(0,0,0,0.25)', boxShadow:'0 0 0 2px rgba(255,255,255,0.9)', pointerEvents:'none', background:'#95a5a6', transition:'transform 160ms ease, border-color 120ms ease, background 160ms ease, box-shadow 160ms ease' }).appendTo('body');
        }
        if (!_markers.$root || !_markers.$root.length) {
            _markers.$root = $('<div id="wcm-root-marker">Root</div>').css({ position:'absolute', zIndex: 99997, padding:'2px 6px', fontSize:'12px', lineHeight:'16px', borderRadius:'10px', color:'#1f2d3d', background:'rgba(149,165,166,0.9)', border:'1px solid rgba(0,0,0,0.15)', pointerEvents:'none', transition:'transform 160ms ease, background 160ms ease, box-shadow 160ms ease, color 160ms ease' }).appendTo('body');
        }
    }
    function _mkHide(){ try { if (_markers.$drag) _markers.$drag.hide(); if (_markers.$target) _markers.$target.hide(); if (_markers.$root) _markers.$root.hide(); } catch(e){} }
    function _mkShow(){ try { if (_markers.$drag) _markers.$drag.show(); if (_markers.$target) _markers.$target.show(); if (_markers.$root) _markers.$root.show(); } catch(e){} }
    function _mkCleanup(){ try { if (_markers.$drag) { _markers.$drag.remove(); _markers.$drag=null; } if (_markers.$target) { _markers.$target.remove(); _markers.$target=null; } if (_markers.$root) { _markers.$root.remove(); _markers.$root=null; } } catch(e){} }


    // Light throttling to reduce jitter while sorting
    var _wcmLastSortTs = 0;
    // Vertical hysteresis to avoid rapid before/after flips
    var VERTICAL_HYSTERESIS_PX = 10;
    // Persist drop parent computed during sort to avoid stale state at update
    window.wcmDropParentId = null;
    $categoryList.sortable({
        handle: ".wcm-drag-handle",
        placeholder: "wcm-placeholder",
        tolerance: "intersect",
        cursor: "move",
        helper: 'clone',
        appendTo: 'body',
        forcePlaceholderSize: true,
        // Restrict to vertical axis (horizontal level changes are computed manually)
        axis: 'y',
        distance: 8,
        refreshPositions: true,
        scroll: true,
        scrollSensitivity: 80,
        scrollSpeed: 28,
        start: function(event, ui) {
            ui.item.addClass("wcm-dragging");
            $categoryList.addClass('wcm-dragging-mode');
            if (ui && ui.helper) {
                ui.helper
                    .addClass('wcm-drag-helper')
                    .css({
                        opacity: 0.92,
                        boxShadow: '0 6px 18px rgba(0,0,0,0.18)',
                        pointerEvents: 'none',
                        width: ui.item.outerWidth(),
                        height: ui.item.outerHeight()
                    });
            }
            _hover = { lastId: null, since: 0, stableId: null, $stable: $() };
            // simplified: no floating preview tooltip

            // Collect immediate subtree to preserve visual continuity during drag
            var baseLevel = parseInt(ui.item.data('level'), 10) || 0;
            dragState.children = [];
            dragState.detached = false;
            dragState.anchorId = 0;
            dragState.anchorBefore = false;
            dragState.anchorEl = null;
            // Keep helper close to cursor with small fixed offsets
            dragState.cursorOffsetX = 12;
            dragState.cursorOffsetY = 10;
            if (ui && ui.helper) {
                ui.helper.css({
                    left: (event.pageX + dragState.cursorOffsetX) + 'px',
                    top: (event.pageY + dragState.cursorOffsetY) + 'px'
                });
            }
            var $scan = ui.item.next();
            while ($scan.length) {
                var lvl = parseInt($scan.data('level'), 10) || 0;
                if (lvl <= baseLevel) break;
                var $next = $scan.next();
                // Detach subtree rows to move as a block visually
                dragState.children.push($scan.detach());
                $scan = $next;
            }

            // Initialize for horizontal drag (level change)
            dragState.startX = event.pageX;
            dragState.startLevel = baseLevel;
            // Snapshot pre-drag order for undo (within same parent)
            try { var curParent = parseInt(ui.item.data('parent-id'),10)||0; dragState.beforeOrders = collectOrdersForParent(curParent).map(function(o){ return { id:o.id, order:o.order }; }); dragState.beforeParent = curParent; } catch(e){}

            // ensure markers
            try { _mkEnsure(); _mkShow(); } catch(e){}

            // Drag diagnostics
            try { window._wcmDidUpdate = false; wcmLog('drag start', { itemId: parseInt(ui.item.data('category-id'),10)||0, startLevel: baseLevel }); } catch(e){}
        },
        sort: function(event, ui) {
            // Throttle recalculation to avoid excessive work
            try {
                var now = Date.now();
                if (now - _wcmLastSortTs < 30) { return; }
                _wcmLastSortTs = now;
            } catch(e) {}
            // Smooth Y to reduce micro jitter
            try {
                var t = Date.now();
                var y = event.pageY;
                if (_smooth.y === null) { _smooth.y = y; _smooth.lastTs = t; _smooth.vy = 0; }
                var dt = Math.max(1, t - _smooth.lastTs);
                var alpha = Math.min(0.7, Math.max(0.25, dt / 60 / 4)); // adapt alpha ~0.25..0.7
                _smooth.y = _smooth.y + alpha * (y - _smooth.y);
                _smooth.vy = (y - _smooth.y) / dt; // px per ms (smoothed delta)
                _smooth.lastTs = t;
            } catch(e){}
            // Auto-scroll window when dragging near top/bottom edges for long lists
            try {
                var edge = 80;
                var vy = 0;
                if (event.clientY < edge) vy = -24; else if (window.innerHeight - event.clientY < edge) vy = 24;
                if (vy !== 0) {
                    window.scrollBy({ top: vy, behavior: 'auto' });
                }
            } catch(e){}
            var $item = ui.item;

            if (ui && ui.helper) {
                var helperLeft = event.pageX + (dragState.cursorOffsetX || 0);
                var helperTop = event.pageY + (dragState.cursorOffsetY || 0);
                ui.helper.css({ left: helperLeft + 'px', top: helperTop + 'px' });
            }


            // Find target row (element under the cursor), ignoring the helper
            var el = document.elementFromPoint(event.clientX, event.clientY);
            var $el = $(el);
            if ($el.closest('.ui-sortable-helper').length) {
                var adjustedY = event.clientY + 2;
                el = document.elementFromPoint(event.clientX, adjustedY);
                $el = $(el);
            }
            var $under = $el.closest('.wcm-category-item');
            if (!$under.length || $under.is($item) || $under.hasClass('wcm-placeholder')) {
                $under = $item.prev('.wcm-category-item'); // fallback: previous row
            }

            // Update markers positions and decide allowance
            try {
                _mkEnsure();
                if (_markers.$drag) { _markers.$drag.css({ left: event.pageX + 'px', top: event.pageY + 'px' }); }
                if ($under && $under.length) {
                    var uOff = $under.offset();
                    var uH = $under.outerHeight() || 32;
                    var uW = $under.outerWidth() || 120;
                    var tx = (uOff.left || 0) + Math.min(CHILD_ZONE_PX, Math.max(24, uW*0.25));
                    var ty = (uOff.top || 0) + (uH/2);
                    if (_markers.$target) { _markers.$target.css({ left: tx+'px', top: ty+'px' }).show(); }
                    var dx = (event.pageX - tx);
                    var dy = (event.pageY - ty);
                    var dist = Math.sqrt(dx*dx + dy*dy);
                    _markers.allow = (dist <= 18);
                    if (_markers.allow !== _markers.wasAllow) {
                        // visual state transition
                        if (_markers.allow) {
                            if (_markers.$drag) { _markers.$drag.css({ background:'#2ecc71', boxShadow:'0 0 0 2px rgba(255,255,255,0.9), 0 0 0 6px rgba(46,204,113,0.25)', transform:'scale(1.12)' }); }
                            if (_markers.$target) { _markers.$target.css({ border:'2px solid #2ecc71', background:'rgba(46,204,113,0.18)', boxShadow:'0 0 0 2px rgba(255,255,255,0.9), 0 0 0 10px rgba(46,204,113,0.18)', transform:'scale(1.18)' }); }
                        } else {
                            if (_markers.$drag) { _markers.$drag.css({ background:'#e67e22', boxShadow:'0 0 0 2px rgba(255,255,255,0.9)', transform:'scale(1.0)' }); }
                            if (_markers.$target) { _markers.$target.css({ border:'2px dashed rgba(0,0,0,0.25)', background:'#95a5a6', boxShadow:'0 0 0 2px rgba(255,255,255,0.9)', transform:'scale(1.0)' }); }
                        }
                        _markers.wasAllow = _markers.allow;
                    }
                } else {
                    _markers.allow = false; if (_markers.$target) _markers.$target.hide();
                    _markers.wasAllow = false;
                }
                // root marker follows list left edge / pointer y
                var listOff = $categoryList.offset() || { left: 0, top: 0 };
                var listH = $categoryList.outerHeight() || 200;
                var ry = Math.max(listOff.top + 8, Math.min(listOff.top + listH - 8, event.pageY));
                var rx = listOff.left + 12;
                if (_markers.$root) { _markers.$root.css({ left: rx+'px', top: ry+'px' }).show(); }
                var rdx = (event.pageX - rx);
                var rdy = (event.pageY - ry);
                var rdist = Math.sqrt(rdx*rdx + rdy*rdy);
                _markers.allowRoot = (rdist <= 18);
                if (_markers.allowRoot !== _markers.wasAllowRoot) {
                    if (_markers.allowRoot) {
                        if (_markers.$root) { _markers.$root.css({ background:'rgba(46,204,113,0.9)', color:'#0b2817', boxShadow:'0 0 0 6px rgba(46,204,113,0.18)', transform:'scale(1.04)' }).text('Root ✓'); }
                    } else {
                        if (_markers.$root) { _markers.$root.css({ background:'rgba(149,165,166,0.9)', color:'#1f2d3d', boxShadow:'', transform:'scale(1.0)' }).text('Root'); }
                    }
                    _markers.wasAllowRoot = _markers.allowRoot;
                }
            } catch(e){}

            // Protection against moving a node under its own descendant (quick check)
            var itemLevel = parseInt($item.data('level'), 10) || 0;
            var draggingId = parseInt($item.data('category-id'), 10);
            var isInvalidDescendant = false;
            if ($under && $under.length) {
                var $scan = $item.next();
                while ($scan.length) {
                    var scanLevel = parseInt($scan.data('level'), 10) || 0;
                    if (scanLevel <= itemLevel) break;
                    if (parseInt($scan.data('category-id'), 10) === parseInt($under.data('category-id'), 10)) { isInvalidDescendant = true; break; }
                    $scan = $scan.next();
                }
            }

            // Desired position and previous anchor (vertical placement)
            var desiredBefore = false;
            var desiredAnchor = null;
            if ($under && $under.length) {
                var underTop = $under.offset().top;
                var midY = underTop + ($under.outerHeight() / 2);
                desiredAnchor = $under;
                // Dynamic hysteresis based on vertical velocity
                var dynBand = VERTICAL_HYSTERESIS_PX + Math.min(12, Math.abs(_smooth.vy || 0) * 12);
                var yUse = (_smooth.y == null) ? event.pageY : _smooth.y;
                if (Math.abs(yUse - midY) < dynBand || (Date.now() < (dragState.anchorLockUntil || 0))) {
                    desiredBefore = dragState.anchorBefore; // keep last decision or anchor lock
                } else {
                    desiredBefore = (yUse < midY);
                }
            }
            // Stabilize target with a short delay window
            var isStable = false;
            try {
                var _anchId = (desiredAnchor && desiredAnchor.length) ? parseInt(desiredAnchor.data('category-id'), 10) : 0;
                var _t = Date.now();
                if (_anchId !== _hover.lastId) { _hover.lastId = _anchId; _hover.since = _t; }
                isStable = (_hover.lastId && (_t - _hover.since) >= HOVER_DELAY_MS);
            } catch(e){}

            // Compute prospective level by horizontal drag delta
            var newAnchorId = (desiredAnchor && desiredAnchor.length) ? parseInt(desiredAnchor.data('category-id'), 10) : 0;
            var desiredBeforeBool = !!desiredBefore;

            if (desiredAnchor && desiredAnchor.length) {
                var anchorStillInDom = dragState.anchorEl && $.contains(document, dragState.anchorEl);
                var samePosition = (dragState.anchorId === newAnchorId) && (dragState.anchorBefore === desiredBeforeBool);
                if (!dragState.anchorEl || !anchorStillInDom) {
                    dragState.anchorEl = desiredAnchor[0];
                    dragState.anchorId = newAnchorId;
                    dragState.anchorBefore = desiredBeforeBool;
                } else if (!samePosition) {
                    dragState.anchorEl = desiredAnchor[0];
                    dragState.anchorId = newAnchorId;
                    dragState.anchorBefore = desiredBeforeBool;
                }
            } else {
                if (dragState.anchorEl && !$.contains(document, dragState.anchorEl)) {
                    dragState.anchorEl = null;
                }
                dragState.anchorId = 0;
                dragState.anchorBefore = false;
            }

            if (isStable && newAnchorId) {
                _hover.stableId = newAnchorId;
            }

            desiredBefore = desiredBeforeBool;
            dragState.anchorBefore = desiredBeforeBool;
            var deltaX = event.pageX - dragState.startX;
            var levelOffset = Math.round(deltaX / (INDENT_STEP + 10));
            if (Math.abs(deltaX) < ((INDENT_STEP + 10) * 0.8)) {
                levelOffset = 0;
            }
            var previewLevel = Math.max(0, dragState.startLevel + levelOffset);
            // If pointer is within the child-drop horizontal zone on the target row, force at least one deeper level
            try {
                if ($under && $under.length) {
                    var _uLv = parseInt($under.data('level'), 10) || 0;
                    var _left = $under.offset().left || 0;
                    var _cz = CHILD_ZONE_PX;
                    var beforePL = previewLevel;
                    if (event.pageX > (_left + _cz)) {
                        previewLevel = Math.max(previewLevel, _uLv + 1);
                    }
                    // If markers are aligned, also treat as child intention
                    if (_markers && _markers.allow) {
                        previewLevel = Math.max(previewLevel, _uLv + 1);
                    }
                    try { wcmLog('child-zone check', { pageX: event.pageX, left: _left, childZone: _cz, underLevel: _uLv, before: beforePL, after: previewLevel, allow: !!(_markers&&_markers.allow) }); } catch(e){}
                }
            } catch(e){}

            // If root alignment is active, force root drop (level 0)
            if (_markers.allowRoot) {
                previewLevel = 0;
            }

            // Maximum level allowed relative to previous row: prev level + 1
            var $prevForParent = null;
            if (desiredAnchor) {
                $prevForParent = desiredBefore ? desiredAnchor.prev('.wcm-category-item') : desiredAnchor;
            } else {
                $prevForParent = $item.prev('.wcm-category-item');
            }
            var prevLevel = $prevForParent && $prevForParent.length ? (parseInt($prevForParent.data('level'), 10) || 0) : -1;
            var maxLevel = prevLevel + 1;
            previewLevel = Math.min(previewLevel, Math.max(0, maxLevel));

            // Resolve parent id by scanning backwards to a row with level == previewLevel - 1
            var parentId = 0;
            if (previewLevel > 0) {
                var $p = $prevForParent;
                while ($p && $p.length) {
                    var lv = parseInt($p.data('level'), 10) || 0;
                    if (lv === previewLevel - 1) { parentId = parseInt($p.data('category-id'), 10) || 0; break; }
                    $p = $p.prev('.wcm-category-item');
                }
            }
            // Compute drop mode BEFORE using it to adjust parent
            var mode = (previewLevel > (parseInt($under && $under.length ? $under.data('level') : -1, 10) || 0)) ? 'child' : 'sibling';
            if (_markers.allowRoot) { mode = 'sibling'; }

            // If explicitly aiming as child over a hovered row, prefer that row as parent (hard-set)
            try {
                if (mode === 'child' && $under && $under.length) {
                    parentId = parseInt($under.data('category-id'), 10) || 0;
                }
            } catch(e){}
            if (_markers.allowRoot) { parentId = 0; }

            // Debug: preview info
            try {
                var underIdDbg = ($under && $under.length) ? (parseInt($under.data('category-id'), 10) || 0) : 0;
                var underLvDbg = ($under && $under.length) ? (parseInt($under.data('level'), 10) || 0) : -1;
                wcmLog('sort preview', {
                    itemId: parseInt($item.data('category-id'), 10) || 0,
                    previewLevel: previewLevel,
                    mode: mode,
                    underId: underIdDbg,
                    underLevel: underLvDbg,
                    computedParentId: parentId,
                    allow: !!_markers.allow,
                    allowRoot: !!_markers.allowRoot
                });
            } catch(e){}

            // When dropping as child, place placeholder immediately after the target row
            if (mode === 'child' && $under && $under.length) {
                desiredBefore = false;
                desiredAnchor = $under;
                // If target is collapsed, expand one level when hover is stable to allow easy dropping into children
                try {
                    if (isStable && $under.hasClass('wcm-collapsed')) {
                        if (typeof wcmExpandOneLevel === 'function') { wcmExpandOneLevel($under); }
                    }
                } catch(e){}
            }
            var previewParentName = parentId ? (((($prevForParent && $prevForParent.length) ? $prevForParent.find('.wcm-category-name').text() : 'Root Category') || '').trim()) : 'Root Category';

            var $ph = (ui && ui.placeholder) ? ui.placeholder : $('.wcm-placeholder');
            var indentPx = Math.max(0, previewLevel) * 20;

            // Only move the placeholder when the destination actually changes (reduces jitter)
            if (_markers.allowRoot) {
                // For root drop, position placeholder relative to the nearest visible item under cursor but keep level 0
                if ($ph.length && $under && $under.length) {
                    var shouldMoveRoot = true;
                    if (desiredBefore && $ph.next()[0] === $under[0]) { shouldMoveRoot = false; }
                    if (!desiredBefore && $ph.prev()[0] === $under[0]) { shouldMoveRoot = false; }
                    if (shouldMoveRoot) {
                        if (desiredBefore) { $ph.insertBefore($under); } else { $ph.insertAfter($under); }
                        try { var currentH2=$ph.outerHeight()||0; var targetH2=$under.outerHeight()||32; $ph.stop(true,true).css('height',currentH2).animate({height:Math.max(28,targetH2)},220,'swing',function(){ $ph.css('height',''); }); } catch(e){}
                    }
                }
            } else if ($ph.length && desiredAnchor && desiredAnchor.length && _markers.allow) {
                var shouldMove = true;
                if (desiredBefore && $ph.next()[0] === desiredAnchor[0]) { shouldMove = false; }
                if (!desiredBefore && $ph.prev()[0] === desiredAnchor[0]) { shouldMove = false; }
                if (shouldMove) {
                    var inList = desiredAnchor.closest('#wcm-category-list').length > 0;
                    var phInList = $ph.closest('#wcm-category-list').length > 0;
                    if (inList && phInList) {
                        // Smooth height animation for placeholder move
                        try {
                            var currentH = $ph.outerHeight() || 0;
                            var targetH = desiredAnchor.outerHeight() || 32;
                            if (desiredBefore) { $ph.insertBefore(desiredAnchor); } else { $ph.insertAfter(desiredAnchor); }
                            $ph.stop(true, true).css('height', currentH).animate({ height: Math.max(28, targetH) }, 220, 'swing', function(){ $ph.css('height',''); });
                        } catch(e) {
                            if (desiredBefore) { $ph.insertBefore(desiredAnchor); } else { $ph.insertAfter(desiredAnchor); }
                        }
                    }
                }
            }

            // Only update visuals when something actually changed
            if (dragState.level !== previewLevel || dragState.mode !== mode) {
                $ph.css({ paddingLeft: (20 + indentPx) + 'px', transition:'background-color 140ms ease, border-color 140ms ease' })
                   .toggleClass('as-child', mode === 'child')
                   .toggleClass('as-sibling', mode === 'sibling');

                // Render a minimal ghost row inside the placeholder
                try {
                    var ghostName = ($item.find('.wcm-category-name').text()||'').trim();
                    var ghostHtml = '';
                    ghostHtml += '  <div class="wcm-category-info">';
                    ghostHtml += '    <span class="wcm-tree-gutter" style="--wcm-level: '+previewLevel+';"></span>';
                    ghostHtml += '    <span class="wcm-category-name">'+$('<div>').text(ghostName).html()+'</span>';
                    ghostHtml += '  </div>';
                    var phOk = (_markers.allow || _markers.allowRoot);
                    var phLabel = _markers.allowRoot ? 'Release to drop at root' : (phOk ? 'Release to drop' : 'Align markers to drop');
                    $ph.html(ghostHtml).addClass('wcm-placeholder-ghost')
                       .attr('data-label', phLabel)
                       .css(phOk ? { backgroundColor:'rgba(46,204,113,0.08)', borderColor:'#2ecc71' } : { backgroundColor:'', borderColor:'' });
                } catch(e) {}

                if (ui && ui.helper && !ui.helper.hasClass('wcm-no-ghost')) {
                    var $gutter = ui.helper.find('.wcm-tree-gutter');
                    if (!$gutter.length) {
                        var $name = ui.helper.find('.wcm-category-name');
                        $('<span class="wcm-tree-gutter" />').insertBefore($name);
                        $gutter = ui.helper.find('.wcm-tree-gutter');
                    }
                    $gutter.css('width', indentPx + 'px').css('flex-basis', indentPx + 'px');
                }
            }

            // Minimal mode: rely only on placeholder for feedback
            if ($highlighted && $highlighted.length) {
                try { $highlighted.get(0).style.removeProperty('--wcm-next-indent'); } catch(e) {}
                $highlighted.removeClass('wcm-opening-child');
                try { $highlighted.css('padding-bottom','').css('transition','').css('box-shadow',''); } catch(e){}
                $highlighted = null;
            }
            if ($under && $under.length && _markers.allow && !_markers.allowRoot) {
                if (mode === 'child') {
                    $under.addClass('wcm-opening-child');
                    try { $under.css('transition','padding-bottom 220ms cubic-bezier(0.22,0.61,0.36,1), box-shadow 140ms ease').css('padding-bottom','14px').css('box-shadow','inset 0 -6px 0 rgba(46,204,113,0.15)'); } catch(e){}
                    $highlighted = $under;
                }
            }
            // Info balloon (ASCII-only)
            var info = (previewLevel > 0 ? 'As child of ' : 'At root ') + previewParentName + ' - Level: ' + previewLevel;
            if (shopc_ajax.default_category_id && draggingId === parseInt(shopc_ajax.default_category_id, 10) && mode === 'child') {
                info += ' (not allowed: default category)';
            }
            if (isInvalidDescendant) {
                info += ' (not allowed: cannot move under own descendant)';
            }
            // simplified: no floating preview tooltip

            // Persist state
            dragState.mode = mode;
            dragState.parentId = parentId;
            dragState.level = previewLevel;
            try { window.wcmDropParentId = parentId; } catch(_){ }
        },
        stop: function(event, ui) {
            ui.item.removeClass("wcm-dragging");
            $categoryList.removeClass('wcm-dragging-mode');
            // simplified: no floating preview tooltip
            if ($highlighted && $highlighted.length) {
                try { $highlighted.get(0).style.removeProperty('--wcm-next-indent'); } catch(e) {}
                $highlighted.removeClass('wcm-drop-target wcm-drop-sibling wcm-opening-child wcm-parent-target wcm-sibling-target');
                try { $highlighted.css('padding-bottom','').css('transition',''); } catch(e){}
                $highlighted = null;
            }
            try {
                var did = !!window._wcmDidUpdate;
                var it = parseInt(ui.item.data('category-id'),10)||0;
                wcmLog('drag stop', { itemId: it, didUpdate: did, finalLevelPreview: dragState.level, finalParentPreview: dragState.parentId, mode: dragState.mode });
                // Fallback: persist if update didn't fire
                if (!did) {
                    var $item = ui.item;
                    var itemId = parseInt($item.data('category-id'), 10);
                    var currentParent = parseInt($item.data('parent-id'), 10) || 0;
                    var targetLevel = dragState.level || 0;
                    var newParentId = 0;
                    if (targetLevel > 0) {
                        var $p = $item.prev('.wcm-category-item');
                        while ($p && $p.length) {
                            var lv = parseInt($p.data('level'), 10) || 0;
                            if (lv === targetLevel - 1) { newParentId = parseInt($p.data('category-id'), 10) || 0; break; }
                            $p = $p.prev('.wcm-category-item');
                        }
                        if (!newParentId && dragState && dragState.parentId) { newParentId = dragState.parentId; }
                    }
                    wcmLog('stop fallback apply', { itemId: itemId, targetLevel: targetLevel, currentParent: currentParent, newParentId: newParentId });
                    if (newParentId !== currentParent) {
                        updateCategoryParent(itemId, newParentId, function(){
                            try {
                                // Reattach subtree and relevel
                                if (Array.isArray(dragState.children) && dragState.children.length) {
                                    var $anchorFb = $item; dragState.children.forEach(function($c){ $anchorFb.after($c); $anchorFb = $c; }); dragState.children = [];
                                }
                                relevelSubtree($item, targetLevel);
                                // Normalize orders for both affected parents
                                bulkUpdateCategoryOrders(collectOrdersForParent(newParentId), function(){
                                    if (currentParent !== newParentId) {
                                        bulkUpdateCategoryOrders(collectOrdersForParent(currentParent), function(){ refreshCategoryList(); scheduleColorize(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg')); }, '');
                                    } else {
                                        refreshCategoryList(); scheduleColorize(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg'));
                                    }
                                }, '');
                            } catch(e) { try { refreshCategoryList(); } catch(_){} }
                        });
                    }
                }
            } catch(e){}
            dragState.anchorId = 0;
            dragState.anchorBefore = false;
            dragState.anchorEl = null;
            dragState.cursorOffsetX = 0;
            dragState.cursorOffsetY = 0;
            if (ui && ui.helper) {
                ui.helper.css('pointer-events', '');
            }

            // Clean markers
            try { _mkCleanup(); _markers.allow=false; _markers.wasAllow=false; _markers.allowRoot=false; _markers.wasAllowRoot=false; } catch(e){}

            // If dropped without update (cancel), reattach subtree after item
            if (Array.isArray(dragState.children) && dragState.children.length) {
                var $anchor = ui.item;
                dragState.children.forEach(function($c){ $anchor.after($c); $anchor = $c; });
                dragState.children = [];
            }
        },
        update: function(event, ui) {
            if (isUpdating) return;

            try { window._wcmDidUpdate = true; } catch(e){}
            var $item = ui.item;
            var itemId = parseInt($item.data('category-id'), 10);
            var itemName = ($item.find('.wcm-category-name').text() || '').trim();
            var currentParent = parseInt($item.data("parent-id"), 10) || 0;
            var targetLevel = dragState.level || 0;
            // Recompute parent from final DOM position for reliability
            var newParentId = 0;
            if (targetLevel > 0) {
                var $p = $item.prev('.wcm-category-item');
                while ($p && $p.length) {
                    var lv = parseInt($p.data('level'), 10) || 0;
                    if (lv === targetLevel - 1) { newParentId = parseInt($p.data('category-id'), 10) || 0; break; }
                    $p = $p.prev('.wcm-category-item');
                }
                // Fallback: prefer dragState.parentId if scan failed
                if (!newParentId && dragState && dragState.parentId) {
                    newParentId = dragState.parentId;
                }
            }
            wcmLog('update drop', { itemId: itemId, targetLevel: targetLevel, currentParent: currentParent, newParentId: newParentId, dragStateParent: dragState.parentId, dragStateMode: dragState.mode });
            try { window.wcmDropParentId = null; } catch(_){ }

            // Note: markers guide visually; do not hard-block drops based on marker alignment

            // Block moving under own descendant (extra safety)
            try {
                if (dragState.mode === 'child') {
                    var isInvalidDesc = false; var baseLevel = parseInt($item.data('level'), 10) || 0; var $scan = $item.next();
                    while ($scan.length){ var lv=parseInt($scan.data('level'),10)||0; if (lv<=baseLevel) break; if (parseInt($scan.data('category-id'),10)===newParentId){ isInvalidDesc = true; break; } $scan=$scan.next(); }
                    if (isInvalidDesc) { wcmLog('blocked: under own descendant', { itemId: itemId, attemptedParentId: newParentId }); showMessage('Cannot move under own descendant', 'error'); $categoryList.sortable('cancel'); try { refreshCategoryList(); } catch(e){} return; }
                }
            } catch(e){}

            // Default product category cannot become a child
            if (shopc_ajax.default_category_id && itemId === parseInt(shopc_ajax.default_category_id, 10) && newParentId !== 0) {
                wcmLog('blocked: default category cannot be child', { itemId: itemId, attemptedParentId: newParentId });
                showMessage(shopc_ajax.strings.cannot_make_default_child || 'Default product category cannot be a child.', 'error');
                $categoryList.sortable('cancel');
                // Ensure UI re-syncs with server after cancel
                try { if (typeof refreshCategoryList === 'function') refreshCategoryList(); } catch(e){}
                // Reinsert children back to original positions (if needed)
                if (Array.isArray(dragState.children) && dragState.children.length) {
                    var $anchorCancel = $item;
                    dragState.children.forEach(function($c){ $anchorCancel.after($c); $anchorCancel = $c; });
                    dragState.children = [];
                }
                return;
            }

            if (newParentId !== currentParent) {
                // Update UI attributes and persist parent change
                $item.attr('data-parent-id', newParentId);
                var oldParent = currentParent;
                wcmLog('updateCategoryParent call', { itemId: itemId, from: oldParent, to: newParentId });
                updateCategoryParent(itemId, newParentId, function() {
                    wcmLog('updateCategoryParent done', { itemId: itemId, to: newParentId });
                    // If children were detached earlier, attach them under the item at the new location
                    if (Array.isArray(dragState.children) && dragState.children.length) {
                        var $anchor = $item;
                        dragState.children.forEach(function($c){ $anchor.after($c); $anchor = $c; });
                        dragState.children = [];
                    }
                    // Update level and structure (including subtree)
                    relevelSubtree($item, targetLevel);
                    // Normalize sibling order and then refresh list from server
                    var parentLabel = 'root';
                    if (newParentId) {
                        try { var $pl = $categoryList.find('.wcm-category-item[data-category-id="'+newParentId+'"]'); var t = ($pl.find('.wcm-category-name').text()||'').trim(); if (t) parentLabel = t; } catch(e){}
                    }
                    var okMove = newParentId ? ('Moved "'+itemName+'" under "'+parentLabel+'"') : ('Moved "'+itemName+'" to root');
                    // Push history for undo (reparent)
                    try { wcmPushHistory({ type:'reparent', id:itemId, from:oldParent, to:newParentId }); } catch(e){}
                    wcmLog('bulkUpdateCategoryOrders start for newParent', { parentId: newParentId });
                    bulkUpdateCategoryOrders(collectOrdersForParent(newParentId), function() {
                        wcmLog('bulkUpdateCategoryOrders done for newParent', { parentId: newParentId });
                        if (oldParent !== newParentId) {
                            wcmLog('bulkUpdateCategoryOrders start for oldParent', { parentId: oldParent });
                            bulkUpdateCategoryOrders(collectOrdersForParent(oldParent), function(){
                                refreshCategoryList(); scheduleColorize(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg'));
                            }, '');
                        } else {
                            refreshCategoryList(); scheduleColorize(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg'));
                        }
                    }, okMove);
                });
                return;
            }

            // If parent did not change, insert subtree in correct position and keep levels
            if (Array.isArray(dragState.children) && dragState.children.length) {
                var $anchor2 = $item;
                dragState.children.forEach(function($c){ $anchor2.after($c); $anchor2 = $c; });
                // Adjust subtree levels based on new level
                relevelSubtree($item, targetLevel);
                dragState.children = [];
            }

            // Capture after-orders for undo/redo, using parent before drag
            var beforeParent = dragState.beforeParent || currentParent;
            var beforeOrders = Array.isArray(dragState.beforeOrders) ? dragState.beforeOrders.slice() : [];
            var afterOrders = collectOrdersForParent(currentParent).map(function(o){ return { id:o.id, order:o.order }; });
            // Push history for undo (reorder within same parent)
            try { if (beforeOrders && beforeOrders.length) { wcmPushHistory({ type:'reorder', before: beforeOrders, after: afterOrders }); } } catch(e){}

            var orders = collectOrdersForParent(currentParent);
            var okReorder = 'Reordered "'+itemName+'"';
            bulkUpdateCategoryOrders(orders, function(){
                refreshCategoryList(); scheduleColorize(); if (window.refreshGraph) window.refreshGraph(jQuery('#wcm-graph-svg'));
            }, okReorder);
        }
    });

    // Parent selection dropdown removed; no change handler

    function bulkUpdateCategoryOrders(orders, done, okMessage) {
        if (!orders || !orders.length) { if (typeof done === 'function') try{ done(); } catch(_){ } return; }

        isUpdating = true;
        $.ajax({
            url: shopc_ajax.ajax_url,
            type: "POST",
            data: {
                action: "shopc_update_category_orders",
                orders: JSON.stringify(orders),
                nonce: shopc_ajax.nonce
            },
            success: function(response) {
                if (response && response.success) {
                    if (okMessage !== '') {
                        var ok = okMessage || ((shopc_ajax && shopc_ajax.strings && (shopc_ajax.strings.order_updated || shopc_ajax.strings.saved)) || 'Saved!');
                        showMessage(ok, "success");
                    }
                } else {
                    var msg = response && response.data && response.data.message ? response.data.message : shopc_ajax.strings.error;
                    showMessage(msg, "error");
                    // Fallback: refresh list to re-sync UI
                    try { if (typeof refreshCategoryList === 'function') refreshCategoryList(); } catch(e){}
                }
            },
            error: function() {
                showMessage(shopc_ajax.strings.error, "error");
                // Fallback: refresh list to re-sync UI
                try { if (typeof refreshCategoryList === 'function') refreshCategoryList(); } catch(e){}
            },
            complete: function() {
                isUpdating = false;
                if (typeof done === 'function') done();
            }
        });
    }

    function updateCategoryParent(categoryId, newParent, onSuccess) {
        isUpdating = true;
        $.ajax({
            url: shopc_ajax.ajax_url,
            type: "POST",
            data: {
                action: "shopc_update_category_parent",
                category_id: categoryId,
                new_parent: newParent,
                nonce: shopc_ajax.nonce
            },
            success: function(response) {
                if (response && response.success) {
                    if (typeof onSuccess === 'function') onSuccess();
                } else {
                    var msg = response && response.data && response.data.message ? response.data.message : shopc_ajax.strings.error;
                    showMessage(msg, "error");
                    // Fallback: refresh list to re-sync UI
                    try { if (typeof refreshCategoryList === 'function') refreshCategoryList(); } catch(e){}
                }
            },
            error: function() {
                showMessage(shopc_ajax.strings.error, "error");
                // Fallback: refresh list to re-sync UI
                try { if (typeof refreshCategoryList === 'function') refreshCategoryList(); } catch(e){}
            },
            complete: function() {
                isUpdating = false;
            }
        });
    }

    function refreshCategoryList() {
        isUpdating = true;
        var $container = $categoryList.closest('.wcm-category-container');
        $container.addClass('wcm-loading');
        $.ajax({
            url: shopc_ajax.ajax_url,
            type: 'POST',
            data: { action: 'shopc_get_categories', nonce: shopc_ajax.nonce },
            success: function(res){
                if (res === -1 || res === "-1") {
                    if (typeof showMessage === 'function') showMessage((shopc_ajax.strings && shopc_ajax.strings.nonce_failed) || 'Nonce failed', "error");
                    return;
                }
                if (!res || !res.success || !res.data || !Array.isArray(res.data.items)) {
                    if (typeof showMessage === 'function') showMessage((shopc_ajax.strings && shopc_ajax.strings.error) || 'Error', 'error');
                    return;
                }
                var html = '';
                res.data.items.forEach(function(cat){
                    var indentPx = Math.max(0, cat.level) * 20;
                    html += '\n<li class="wcm-category-item'+(wcmSelSet.has(cat.id)?' wcm-selected':'')+'" data-category-id="'+cat.id+'" data-order="'+cat.order+'" data-level="'+cat.level+'" data-parent-id="'+cat.parent+'">';
                    html += '  <input type="checkbox" class="wcm-select" '+(wcmSelSet.has(cat.id)?'checked':'')+' />';
                    html += '  <div class="wcm-drag-handle" title="'+((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.drag_to_reorder) || 'Drag to reorder / change level')+'"></div>';
                    html += '  <div class="wcm-category-info">';
                    html += '    <span class="wcm-tree-gutter" style="--wcm-level: '+cat.level+';"></span>';
                    html += '    <span class="wcm-category-name">'+$('<div>').text(cat.name).html()+'</span>';
                    if (cat.level > 0) { html += '    <span class="wcm-category-level">'+((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.level) || 'Level')+' '+cat.level+'</span>'; }
                    html += '    <span class="wcm-category-count">('+(cat.count||0)+' '+((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.items) || 'items')+')</span>';
                    html += '  </div>';
                    html += '  <div class="wcm-category-actions">';
                    if (cat.has_children) {
                        html += '    <button type="button" class="wcm-toggle-children" title="'+((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.toggle_subcategories) || 'Toggle subcategories')+'" aria-expanded="false">-</button>';
                    }
                    html += '    <button type="button" class="wcm-delete-category" title="'+((shopc_ajax && shopc_ajax.strings && shopc_ajax.strings.delete_category) || 'Delete category')+'" aria-label="Delete category '+$('<div>').text(cat.name).html()+'">&times;</button>';
                    html += '  </div>';
                    html += '</li>';
                });
                // Sanitize any garbled encodings in built HTML
                try {
                    // Normalize count label to ASCII: "(N items)"
                    html = html.replace(/\((\d+)\s*[^)]*\)/g, function(_, n){ return '('+n+' items)'; });
                } catch(e) { if (window.console) console.warn('WCM sanitize html failed', e); }
                $categoryList.html(html);
                refreshToggleButtons();
                // After AJAX refresh: keep only top-level open by default
                // If saved expansion state exists, apply it; otherwise collapse to top level
                try { if (typeof wcmOpenSet !== "undefined" && wcmOpenSet.size) { if (typeof wcmApplyExpansionState==="function") wcmApplyExpansionState(); } else { collapseToTopLevel(); } } catch(e){ collapseToTopLevel(); }
                // Reapply active filter if any
                try { var q = (sessionStorage.getItem('wcm_filter_query')||''); if (q) { if (typeof wcmApplyFilter==='function') wcmApplyFilter(q); jQuery('#wcm-filter-input').val(q); } } catch(e){}
                // Re-apply colors after any rebuild (scheduled to avoid jank)
                scheduleColorize();
            },
            error: function(xhr){
                var body = (xhr && xhr.responseText) ? xhr.responseText.trim() : '';
                if (typeof showMessage === 'function') {
                    if (body === '-1') showMessage((shopc_ajax.strings && shopc_ajax.strings.nonce_failed) || 'Nonce failed', "error");
                    else if (body === '0') showMessage((shopc_ajax.strings && shopc_ajax.strings.unknown_error) || 'Unknown error', "error");
                    else showMessage(shopc_ajax.strings.error, "error");
                }
                if (window.console) console.error('WCM AJAX error (get_categories):', xhr);
            },
            complete: function(){
                $container.removeClass('wcm-loading');
                isUpdating = false;
            }
        });
    }

    function collectOrdersForParent(parentId) {
        var orders = [];
        var order = 1;
        $categoryList.find('.wcm-category-item').each(function(){
            var $li = $(this);
            var liParent = parseInt($li.data('parent-id'), 10) || 0;
            if (liParent === parentId) {
                var id = parseInt($li.data('category-id'), 10);
                $li.attr('data-order', order);
                orders.push({ id: id, order: order });
                order++;
            }
        });
        return orders;
    }

    function setLevel($el, level) {
        $el.attr('data-level', level);
        var gutter = $el.find('.wcm-tree-gutter').get(0);
        if (gutter) {
            gutter.style.setProperty('--wcm-level', level);
        }
    }

    function relevelSubtree($root, newLevel) {
        var oldLevel = parseInt($root.data('level'), 10) || 0;
        var delta = (newLevel || 0) - oldLevel;
        setLevel($root, newLevel || 0);
        var $n = $root.next();
        while ($n.length) {
            var lvl = parseInt($n.data('level'), 10) || 0;
            if (lvl <= oldLevel) break;
            setLevel($n, lvl + delta);
            $n = $n.next();
        }
    }

    function refreshToggleButtons() {
        $categoryList.find('.wcm-category-item').each(function(){
            var $li = $(this);
            var level = parseInt($li.data('level'), 10) || 0;
            var $next = $li.next('.wcm-category-item');
            var hasChild = false;
            if ($next && $next.length) {
                var nextLevel = parseInt($next.data('level'), 10) || 0;
                hasChild = nextLevel > level;
            }
            var $actions = $li.find('.wcm-category-actions');
            var $btn = $actions.find('.wcm-toggle-children');
            if (hasChild) {
                if (!$btn.length) {
                    $btn = $('<button type="button" class="wcm-toggle-children" title="Toggle subcategories" aria-expanded="false">-</button>');
                    $actions.append($btn);
                }
                $btn.text($li.hasClass('wcm-collapsed') ? '+' : '-');
                $btn.attr('aria-expanded', !$li.hasClass('wcm-collapsed'));
            } else {
                if ($btn.length) $btn.remove();
            }
        });
    }

    // Expand one level under a collapsed item (used during drag when hovering stably)
    function wcmExpandOneLevel($item){
        try {
            var baseLevel = parseInt($item.data('level'), 10) || 0;
            $item.removeClass('wcm-collapsed');
            var $btn = $item.find('.wcm-toggle-children'); if ($btn.length){ $btn.text('-').attr('aria-expanded', true); }
            var $next = $item.next();
            while ($next.length) {
                var nextLevel = parseInt($next.data('level'), 10) || 0;
                if (nextLevel <= baseLevel) break;
                if (nextLevel === baseLevel + 1) { $next.show(); } else { $next.hide(); }
                $next = $next.next();
            }
        } catch(e){}
    }

    // Show only top-level categories: mark upper levels collapsed and refresh toggle buttons
    function collapseToTopLevel() {
        $categoryList.find('.wcm-category-item').each(function(){
            var $li = $(this);
            var level = parseInt($li.data('level'), 10) || 0;
            if (level === 0) {
                $li.addClass('wcm-collapsed');
            } else {
                $li.addClass('wcm-collapsed').hide();
            }
        });
        // Update all button labels
        refreshToggleButtons();
    }

    // Colorize list: assign a color per root and lighter tones for descendants
    function colorizeList() {
        var hue = 12; // starting hue
        var GOLDEN_ANGLE = 137.508; // good distribution for hues
        var currentRootHue = null;

        $categoryList.find('.wcm-category-item').each(function(){
            var $li = $(this);
            var level = parseInt($li.data('level'), 10) || 0;
            if (level === 0) {
                // yeni ana kategori: yeni hue ata
                currentRootHue = (hue % 360);
                hue = (hue + GOLDEN_ANGLE) % 360;
            }
            if (currentRootHue === null) return; // safety

            var depth = level; // distance from root
            var sat = Math.max(35, 70 - depth * 12); // reduce saturation as depth increases
            var light = Math.min(98, 92 + depth * 3); // increase lightness as depth increases

            var bg = 'hsl(' + currentRootHue + ', ' + sat + '%, ' + light + '%)';
            var border = 'hsl(' + currentRootHue + ', 60%, ' + (50 - Math.min(20, depth * 6)) + '%)';

            $li.css({
                'background-color': bg,
                'border-left-color': border
            });
        });
    }

    function showMessage(message, type) {
        var t = (type || 'info');
        var $toast = $('<div class="wcm-toast wcm-toast-' + t + '">' + message + '</div>');
        var $stack = jQuery('#wcm-toast-stack');
        if (!$stack.length) { $stack = jQuery('<div id="wcm-toast-stack" class="wcm-toast-stack" aria-live="polite" aria-atomic="true"/>').appendTo('body'); }
        $stack.append($toast);
        // small delay to trigger transition
        requestAnimationFrame(function(){ $toast.addClass('visible'); });
        // auto close after 2.2s
        setTimeout(function(){
            $toast.removeClass('visible');
            var removeFn = function(){ $toast.off('transitionend webkitTransitionEnd', removeFn); $toast.remove(); };
            $toast.on('transitionend webkitTransitionEnd', removeFn);
        }, 2200);
    }

    // Toggle subcategories: manage visibility by level
    $(document).on("click", ".wcm-toggle-children", function() {
        var $button = $(this);
        var $item = $button.closest('.wcm-category-item');
        var baseLevel = parseInt($item.data('level'), 10) || 0;
        var collapse = !$item.hasClass('wcm-collapsed');

        $item.toggleClass('wcm-collapsed', collapse);
        $button.text(collapse ? '+' : '-');
        $button.attr('aria-expanded', !collapse);

        var $next = $item.next();
        while ($next.length) {
            var nextLevel = parseInt($next.data('level'), 10) || 0;
            if (nextLevel <= baseLevel) break;
            if (collapse) {
                // Toggle subcategories
                $next.hide();
            } else {
                // Expanded behavior: show one deeper level; keep deeper descendants hidden
                if (nextLevel === baseLevel + 1) {
                    $next.show().addClass('wcm-collapsed');
                } else {
                    $next.hide().addClass('wcm-collapsed');
                }
            }
            $next = $next.next();
        }
        // Update toggle button labels based on collapsed state
        refreshToggleButtons();
    });

    // Hooks for potential graph-view helpers (reserved)
    // On first load: collapse to top level or apply saved expansion state
    collapseToTopLevel();
    scheduleColorize();
    window.showMessage = showMessage;
    window.refreshCategoryList = refreshCategoryList;

    // --- Persisted expansion state (sessionStorage) ---
    var OPEN_KEY = 'wcm_open_set_v1';
    function wcmLoadOpenSet(){ try { return new Set(JSON.parse(sessionStorage.getItem(OPEN_KEY)||'[]')); } catch(e){ return new Set(); } }
    function wcmSaveOpenSet(s){ try { sessionStorage.setItem(OPEN_KEY, JSON.stringify(Array.from(s||[]))); } catch(e){} }
    var wcmOpenSet = wcmLoadOpenSet();

    function wcmSetCollapsed($item, collapse){
        var base = parseInt($item.data('level'),10)||0;
        $item.toggleClass('wcm-collapsed', collapse);
        var $btn=$item.find('.wcm-toggle-children'); if($btn.length){ $btn.text(collapse?'+':'-'); }
        var $n=$item.next();
        while($n.length){ var lv=parseInt($n.data('level'),10)||0; if(lv<=base) break; if(collapse){ $n.hide(); } else { if(lv===base+1){ $n.show(); } else { $n.hide(); } } $n=$n.next(); }
    }

    function wcmApplyExpansionState(){
        if(!(wcmOpenSet && wcmOpenSet.size)) return;
        jQuery('#wcm-category-list .wcm-category-item').each(function(){ var $li=jQuery(this); var lv=parseInt($li.data('level'),10)||0; if(lv===0){ var id=parseInt($li.data('category-id'),10); wcmSetCollapsed($li, !wcmOpenSet.has(id)); } else { $li.hide(); } });
        jQuery('#wcm-category-list .wcm-category-item').each(function(){ var $li=jQuery(this); var id=parseInt($li.data('category-id'),10); if(wcmOpenSet.has(id)){ wcmSetCollapsed($li,false); } });
        if (typeof refreshToggleButtons === 'function') refreshToggleButtons();
    }

    // Clicks: update open set after default behavior
    jQuery(document).on('click', '.wcm-toggle-children', function(){
        var $li=jQuery(this).closest('.wcm-category-item'); var id=parseInt($li.data('category-id'),10); var base=parseInt($li.data('level'),10)||0;
        var collapsed=$li.hasClass('wcm-collapsed');
        if(!collapsed){ wcmOpenSet.add(id); } else { wcmOpenSet.delete(id); var $n=$li.next(); while($n.length){ var lv=parseInt($n.data('level'),10)||0; if(lv<=base) break; wcmOpenSet.delete(parseInt($n.data('category-id'),10)); $n=$n.next(); } }
        wcmSaveOpenSet(wcmOpenSet);
    });

    // Delete category (with optional cascade)
    jQuery(document).on('click', '.wcm-delete-category', function(){
        var $li=jQuery(this).closest('.wcm-category-item');
        var id=parseInt($li.data('category-id'),10)||0;
        var name=($li.find('.wcm-category-name').text()||'').trim();
        if (!id) return;
        var msg = (shopc_ajax.strings && shopc_ajax.strings.confirm_delete) ? shopc_ajax.strings.confirm_delete.replace('%s', name) : ('Delete "'+name+'"?');
        if (!window.confirm(msg)) return;
        var cascade = window.confirm((shopc_ajax.strings && shopc_ajax.strings.confirm_delete_children) || 'Also delete all subcategories?');
        jQuery.post(shopc_ajax.ajax_url, { action:'shopc_delete_category', category_id:id, delete_children: cascade?1:0, nonce:shopc_ajax.nonce }, function(res){
            if (res && res.success) {
                showMessage((shopc_ajax.strings && shopc_ajax.strings.deleted) || 'Deleted', 'success');
                refreshCategoryList();
            } else {
                var m = res && res.data && res.data.message ? res.data.message : (shopc_ajax.strings && shopc_ajax.strings.error) || 'Error';
                showMessage(m, 'error');
            }
        });
    });

    // Selection toggle per row
    jQuery(document).on('change', '.wcm-select', function(){
        var $li=jQuery(this).closest('.wcm-category-item'); var id=parseInt($li.data('category-id'),10)||0; if(!id) return;
        if (this.checked){ wcmSelSet.add(id); $li.addClass('wcm-selected'); } else { wcmSelSet.delete(id); $li.removeClass('wcm-selected'); }
        saveSel(wcmSelSet);
    });
    // Select visible toggle
    jQuery(document).on('change', '#wcm-select-visible', function(){
        var check=this.checked; jQuery('#wcm-category-list .wcm-category-item:visible').each(function(){ var id=parseInt(this.getAttribute('data-category-id'),10)||0; if(!id) return; if (check){ wcmSelSet.add(id); jQuery(this).addClass('wcm-selected').find('.wcm-select').prop('checked',true); } else { wcmSelSet.delete(id); jQuery(this).removeClass('wcm-selected').find('.wcm-select').prop('checked',false); } }); saveSel(wcmSelSet);
    });
    // Bulk delete selected
    jQuery(document).on('click', '#wcm-bulk-delete', function(){
        var ids=Array.from(wcmSelSet||[]); if(!ids.length){ showMessage((shopc_ajax.strings&&shopc_ajax.strings.error)||'Error', 'error'); return; }
        var msg=(shopc_ajax.strings&&shopc_ajax.strings.delete_selected)||'Delete selected'; if(!window.confirm(msg+' ('+ids.length+')?')) return;
        var cascade=jQuery('#wcm-delete-children').is(':checked');
        jQuery.post(shopc_ajax.ajax_url, { action:'shopc_delete_categories', ids: JSON.stringify(ids), delete_children: cascade?1:0, nonce: shopc_ajax.nonce }, function(res){
            if(res&&res.success){ showMessage((shopc_ajax.strings&&shopc_ajax.strings.deleted)||'Deleted','success'); ids.forEach(function(id){ wcmSelSet.delete(id); }); saveSel(wcmSelSet); refreshCategoryList(); }
            else { var m=res&&res.data&&res.data.message?res.data.message:((shopc_ajax.strings&&shopc_ajax.strings.error)||'Error'); showMessage(m,'error'); }
        });
    });

    // Wrap refresh to re-apply state after AJAX reload
    (function(){ var _orig = window.refreshCategoryList; window.refreshCategoryList = function(){ if(typeof _orig==='function'){ _orig(); } setTimeout(function(){ if(wcmOpenSet&&wcmOpenSet.size){ wcmApplyExpansionState(); } }, 150); }; })();

    // Apply saved state on first load (without reloading page)
    if (wcmOpenSet && wcmOpenSet.size) { setTimeout(wcmApplyExpansionState, 0); }

    // --- Quick filter for long lists ---
    function wcmApplyFilter(query){
        var q = (query||'').toLowerCase().trim();
        var $items = $categoryList.find('.wcm-category-item');
        if (!q) { $items.show(); refreshToggleButtons(); return; }
        var levels = []; var names = []; var match = []; var i=0;
        $items.each(function(){
            var $li=jQuery(this); var lv=parseInt($li.data('level'),10)||0; levels[i]=lv;
            var name=$li.find('.wcm-category-name').text().toLowerCase(); names[i]=name; match[i]=name.indexOf(q)!==-1; i++; });
        var visible=new Array(i).fill(false); var childHasMatch=false; var lastLevel=levels[i-1]||0;
        for (var k=i-1;k>=0;k--){
            var m = match[k] || childHasMatch; visible[k]=m; var curLv=levels[k]||0; if (match[k]){ childHasMatch=true; lastLevel=curLv; } else { if (curLv>=lastLevel) { /* keep */ } else { childHasMatch=false; lastLevel=curLv; } }
        }
        $items.each(function(idx){ jQuery(this).toggle(!!visible[idx]); });
        refreshToggleButtons();
    }
    function wcmSetupFilter(){
        var $input=jQuery('#wcm-filter-input'); if(!$input.length) return; var t;
        // initialize from persisted query
        try { var q0=sessionStorage.getItem('wcm_filter_query'); if(q0){ $input.val(q0); wcmApplyFilter(q0); } } catch(e){}
        $input.on('input', function(){ var v=this.value; clearTimeout(t); t=setTimeout(function(){ try{ sessionStorage.setItem('wcm_filter_query', v||''); }catch(e){} wcmApplyFilter(v); }, 150); });
    }
    wcmSetupFilter();
});









