<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

/*
Plugin Name: Close Up Shop
Description: Redirects visitors from WooCommerce pages to a "Shop Closed" page when the shop is closed. Admin can define multiple closure periods; the live reopening time is shown with a shortcode.
Version: 1.4
Author: Eliyahna
Author URI: https://eliyahna.com
License: GPL2
License URI: https://www.gnu.org/licenses/gpl-2.0.html
*/

define( 'CLOSEUPSHOP_PLUGIN_DIR', plugin_dir_path( __FILE__ ) );

/* ------------------------------------------------------------------------- *
 *  Helpers
 * ------------------------------------------------------------------------- */

function closeupshop_recursive_sanitize( $data ) {
	if ( is_array( $data ) ) {
		foreach ( $data as $k => $v ) {
			$data[ $k ] = closeupshop_recursive_sanitize( $v );
		}
		return $data;
	}
	return sanitize_text_field( $data );
}

function closeupshop_to_ts( $date, $time ) {
	try {
		$tz = wp_timezone();
		$dt = new DateTimeImmutable( "{$date} {$time}", $tz );
		return $dt->getTimestamp();
	} catch ( Exception $e ) {
		return false;
	}
}

function closeupshop_get_page_by_title( $title ) {
	$pages = get_posts( array(
		'post_type'      => 'page',
		'title'          => $title,
		'posts_per_page' => 1,
		'post_status'    => 'publish',
	) );
	return $pages ? $pages[0] : null;
}

function closeupshop_choose_period( $periods ) {
	$now      = current_time( 'timestamp' );
	$active   = array();
	$upcoming = array();

	foreach ( $periods as $p ) {
		$start_ts = closeupshop_to_ts( $p['start_date'], $p['start_time'] );
		$end_ts   = closeupshop_to_ts( $p['end_date'],   $p['end_time']   );
		if ( ! $start_ts || ! $end_ts ) {
			continue;
		}

		if ( $now >= $start_ts && $now < $end_ts ) {
			$p['end_ts'] = $end_ts;
			$active[]    = $p;
		} elseif ( $start_ts > $now ) {
			$p['start_ts'] = $start_ts;
			$upcoming[]    = $p;
		}
	}

	if ( $active ) {
		usort( $active, fn ( $a, $b ) => $b['end_ts'] <=> $a['end_ts'] );
		return $active[0];
	}
	if ( $upcoming ) {
		usort( $upcoming, fn ( $a, $b ) => $a['start_ts'] <=> $b['start_ts'] );
		return $upcoming[0];
	}
	return false;
}

/* ------------------------------------------------------------------------- *
 *  Activation – create “Shop Closed” page
 * ------------------------------------------------------------------------- */

function closeupshop_create_page() {
	if ( ! closeupshop_get_page_by_title( 'Shop Closed' ) ) {
		$content = '<h2>Sorry, the shop is currently closed.</h2>'
		         . '<p>We will reopen at: [closeupshop_reopen_time]</p>';
		wp_insert_post( array(
			'post_title'   => 'Shop Closed',
			'post_content' => $content,
			'post_status'  => 'publish',
			'post_type'    => 'page',
		) );
	}
}
register_activation_hook( __FILE__, 'closeupshop_create_page' );

/* ------------------------------------------------------------------------- *
 *  Short-code – live reopening time
 * ------------------------------------------------------------------------- */

function closeupshop_reopen_time_shortcode() {
	$periods = get_option( 'closeupshop_closure_periods', array() );
	$chosen  = closeupshop_choose_period( $periods );
	if ( ! $chosen ) {
		return esc_html__( 'not set', 'closeupshop' );
	}
	$end_ts = closeupshop_to_ts( $chosen['end_date'], $chosen['end_time'] );
	if ( ! $end_ts ) {
		return esc_html__( 'not set', 'closeupshop' );
	}
	return wp_date(
		get_option( 'date_format' ) . ' ' . get_option( 'time_format' ),
		$end_ts
	);
}
add_shortcode( 'closeupshop_reopen_time', 'closeupshop_reopen_time_shortcode' );

/* ------------------------------------------------------------------------- *
 *  Admin – settings page
 * ------------------------------------------------------------------------- */

if ( is_admin() ) {

	function closeupshop_add_settings_submenu() {
		global $closeupshop_hook;
		$closeupshop_hook = add_options_page(
			'Close Up Shop Settings',
			'Close Up Shop',
			'manage_options',
			'closeupshopsettings',
			'closeupshop_render_settings_page'
		);
	}
	add_action( 'admin_menu', 'closeupshop_add_settings_submenu' );

	function closeupshop_admin_enqueue_scripts( $hook_suffix ) {
		global $closeupshop_hook;
		if ( $hook_suffix === $closeupshop_hook ) {
			wp_enqueue_script(
				'shpclsd-admin-script',
				plugin_dir_url( __FILE__ ) . 'js/admin.js',
				array( 'jquery' ),
				'1.4',
				true
			);
		}
	}
	add_action( 'admin_enqueue_scripts', 'closeupshop_admin_enqueue_scripts' );

	function closeupshop_render_settings_page() {
		$periods = get_option( 'closeupshop_closure_periods', array() );
		settings_errors( 'closeupshop' );
		?>
		<div class="wrap">
			<h1>Close Up Shop Settings</h1>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
				<?php wp_nonce_field( 'closeupshop_save_settings' ); ?>
				<input type="hidden" name="action" value="closeupshop_save_settings">

				<table class="widefat">
					<thead>
					<tr>
						<th>Start Date</th><th>Start Time</th>
						<th>End Date</th><th>End Time</th>
						<th>Remove?</th>
					</tr>
					</thead>
					<tbody id="shpclsd-rows">
					<?php foreach ( $periods as $i => $p ) : ?>
						<tr>
							<td><input type="date" name="periods[<?php echo esc_attr( $i ); ?>][start_date]" value="<?php echo esc_attr( $p['start_date'] ); ?>" required></td>
							<td><input type="time" name="periods[<?php echo esc_attr( $i ); ?>][start_time]" value="<?php echo esc_attr( $p['start_time'] ); ?>" required></td>
							<td><input type="date" name="periods[<?php echo esc_attr( $i ); ?>][end_date]"   value="<?php echo esc_attr( $p['end_date'] ); ?>"   required></td>
							<td><input type="time" name="periods[<?php echo esc_attr( $i ); ?>][end_time]"   value="<?php echo esc_attr( $p['end_time'] ); ?>"   required></td>
							<td><button type="button" class="button shpclsd-remove-row">Remove</button></td>
						</tr>
					<?php endforeach; ?>
					</tbody>
				</table>

				<p><button type="button" class="button" id="shpclsd-add-row">Add Another Period</button></p>
				<?php submit_button( 'Save Settings' ); ?>
			</form>

			<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:20px;">
				<?php wp_nonce_field( 'closeupshop_reset_settings' ); ?>
				<input type="hidden" name="action" value="closeupshop_reset_settings">
				<?php submit_button( 'Reset Settings', 'secondary' ); ?>
			</form>
		</div>
		<?php
	}

	function closeupshop_handle_save_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'Insufficient permissions.', 'closeupshop' ) );
		}
		$nonce = $_POST['_wpnonce'] ?? '';
		if ( ! wp_verify_nonce( $nonce, 'closeupshop_save_settings' ) ) {
			wp_die( __( 'Nonce verification failed.', 'closeupshop' ) );
		}

		$raw  = filter_input( INPUT_POST, 'periods', FILTER_DEFAULT, FILTER_REQUIRE_ARRAY );
		$raw  = $raw ? closeupshop_recursive_sanitize( $raw ) : array();
		$good = array();

		foreach ( $raw as $p ) {
			if ( empty( $p['start_date'] ) || empty( $p['start_time'] ) ||
			     empty( $p['end_date'] )   || empty( $p['end_time']   ) ) {
				continue;
			}
			if ( ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $p['start_date'] ) ||
			     ! preg_match( '/^\d{4}-\d{2}-\d{2}$/', $p['end_date']   ) ||
			     ! preg_match( '/^\d{2}:\d{2}$/',       $p['start_time'] ) ||
			     ! preg_match( '/^\d{2}:\d{2}$/',       $p['end_time']   ) ) {
				continue;
			}
			$start_ts = closeupshop_to_ts( $p['start_date'], $p['start_time'] );
			$end_ts   = closeupshop_to_ts( $p['end_date'],   $p['end_time']   );
			if ( ! $start_ts || ! $end_ts || $start_ts >= $end_ts ) {
				continue;
			}
			$good[] = array(
				'start_date' => $p['start_date'],
				'start_time' => $p['start_time'],
				'end_date'   => $p['end_date'],
				'end_time'   => $p['end_time'],
			);
		}

		update_option( 'closeupshop_closure_periods', $good );
		wp_cache_flush();

		if ( $page = closeupshop_get_page_by_title( 'Shop Closed' ) ) {
			$msg = '<h2>Sorry, the shop is currently closed.</h2>'
			     . '<p>We will reopen at: [closeupshop_reopen_time]</p>';
			wp_update_post( array(
				'ID'           => $page->ID,
				'post_content' => $msg,
				'post_status'  => 'publish',
			) );
			clean_post_cache( $page->ID );
		}

		add_settings_error( 'closeupshop', 'settings_updated', 'Settings saved successfully.', 'updated' );
		wp_redirect( esc_url( admin_url( 'admin.php?page=closeupshopsettings' ) ) );
		exit;
	}
	add_action( 'admin_post_closeupshop_save_settings', 'closeupshop_handle_save_settings' );

	function closeupshop_handle_reset_settings() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( __( 'Insufficient permissions.', 'closeupshop' ) );
		}
		check_admin_referer( 'closeupshop_reset_settings' );

		delete_option( 'closeupshop_closure_periods' );
		wp_cache_flush();

		$content = '<h2>Sorry, the shop is currently closed.</h2>'
		         . '<p>We will reopen at: [closeupshop_reopen_time]</p>';
		if ( $page = closeupshop_get_page_by_title( 'Shop Closed' ) ) {
			wp_update_post( array(
				'ID'           => $page->ID,
				'post_content' => $content,
				'post_status'  => 'publish',
			) );
			clean_post_cache( $page->ID );
		}

		add_settings_error( 'closeupshop', 'settings_reset', 'Settings have been reset.', 'updated' );
		wp_redirect( esc_url( admin_url( 'admin.php?page=closeupshopsettings' ) ) );
		exit;
	}
	add_action( 'admin_post_closeupshop_reset_settings', 'closeupshop_handle_reset_settings' );
}

/* ------------------------------------------------------------------------- *
 *  Front-end – closed check / redirect
 * ------------------------------------------------------------------------- */

function shopcl_is_closed() {
	$periods = get_option( 'closeupshop_closure_periods', array() );
	if ( ! $periods ) {
		return false;
	}
	$now = current_time( 'timestamp' );
	foreach ( $periods as $p ) {
		$s = closeupshop_to_ts( $p['start_date'], $p['start_time'] );
		$e = closeupshop_to_ts( $p['end_date'],   $p['end_time']   );
		if ( $s && $e && $now >= $s && $now < $e ) {
			return $p;
		}
	}
	return false;
}

function shopcl_redirect_if_closed() {
	if ( ! function_exists( 'is_woocommerce' ) ) {
		return;
	}
	if ( is_cart() || is_checkout() || is_account_page() || is_product() || is_shop() ) {
		if ( shopcl_is_closed() ) {
			if ( ! defined( 'DONOTCACHEPAGE' ) ) {
				define( 'DONOTCACHEPAGE', true );
			}
			nocache_headers();
			if ( $page = closeupshop_get_page_by_title( 'Shop Closed' ) ) {
				wp_safe_redirect( get_permalink( $page->ID ) );
				exit;
			}
		}
	}
}
add_action( 'template_redirect', 'shopcl_redirect_if_closed' );

/* ------------------------------------------------------------------------- *
 *  Style for the notice page
 * ------------------------------------------------------------------------- */

function shopcl_enqueue_frontend_style() {
	if ( is_page() ) {
		if ( $page = closeupshop_get_page_by_title( 'Shop Closed' ) ) {
			if ( get_the_ID() == $page->ID ) {
				wp_register_style( 'closeupshop-style', false, array(), '1.4' );
				wp_add_inline_style( 'closeupshop-style', '
					.entry-title,
					.entry-content,
					.page-content { text-align: center; }
				' );
				wp_enqueue_style( 'closeupshop-style' );
				nocache_headers();
			}
		}
	}
}
add_action( 'wp_enqueue_scripts', 'shopcl_enqueue_frontend_style' );
