<?php
/**
 * Plugin Name: Shipping Online Tracker for CTT
 * Description: Management of CTT tracking with History, Email, and Customer Area. Developed for manual shipments without a contract.
 * Version: 1.7.3
 * Author: Cordelima Unipessoal Lda
 * License: GPL2
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 9.5
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * 0. HPOS COMPATIBILITY
 */
add_action( 'before_woocommerce_init', 'ctt_pro_declare_hpos_compatibility' );
function ctt_pro_declare_hpos_compatibility() {
    if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
    }
}

/**
 * 0.1 PROTECTION: Check if WooCommerce is active
 */
if ( ! in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
    return;
}

/**
 * 1. ENQUEUE SCRIPTS & STYLES (The correct WP way)
 */
add_action( 'wp_enqueue_scripts', 'ctt_pro_enqueue_custom_scripts' );
function ctt_pro_enqueue_custom_scripts() {
    if ( is_account_page() ) {
        // Inline Style
        $custom_css = ".button.ctt_tracking { background-color: #d61919 !important; color: white !important; } .button.ctt_tracking:hover { background-color: #000 !important; }";
        wp_register_style( 'ctt-pro-style', false );
        wp_enqueue_style( 'ctt-pro-style' );
        wp_add_inline_style( 'ctt-pro-style', $custom_css );

        // Inline Script
        $custom_js = "document.addEventListener('DOMContentLoaded', function() {
            document.querySelectorAll('.button.ctt_tracking').forEach(function(btn) {
                btn.setAttribute('target', '_blank');
                btn.setAttribute('rel', 'noopener noreferrer');
            });
        });";
        wp_register_script( 'ctt-pro-js', false );
        wp_enqueue_script( 'ctt-pro-js' );
        wp_add_inline_script( 'ctt-pro-js', $custom_js );
    }
}

/**
 * 2. ADMIN META BOX
 */
add_action( 'add_meta_boxes', 'ctt_pro_add_tracking_box' );
function ctt_pro_add_tracking_box() {
    $screens = array( 'shop_order', 'woocommerce_page_wc-orders' );
    foreach ( $screens as $screen ) {
        add_meta_box( 'ctt_tracking_id', 'CTT Shipping (No Contract)', 'ctt_pro_content', $screen, 'side', 'high' );
    }
}

function ctt_pro_content( $post_or_order ) {
    $order_id = ( $post_or_order instanceof WP_Post ) ? $post_or_order->ID : $post_or_order->get_id();
    
    $tracking = get_post_meta( $order_id, '_ctt_tracking_number', true );
    $history  = get_post_meta( $order_id, '_ctt_tracking_history', true ) ?: array();
    
    echo '<p style="margin-bottom:5px;"><strong>1. Prepare:</strong></p>';
    echo '<a href="https://www.ctt.pt/particulares/enviar/envios-online" target="_blank" style="display:block; text-align:center; background:#d61919; color:white; padding:10px; text-decoration:none; border-radius:4px; font-weight:bold; margin-bottom:15px;">Open CTT Envios Online</a>';
    echo '<hr>';
    echo '<p style="margin-top:10px;"><label><strong>2. Tracking Code:</strong></label></p>';
    echo '<input type="text" name="ctt_tracking_field" value="' . esc_attr( $tracking ) . '" style="width:100%; height:35px; border-radius:4px;" placeholder="Ex: RF123456789PT" />';
    
    if ( ! empty( $history ) ) {
        echo '<div style="margin-top:15px; background:#f0f0f0; padding:10px; border-radius:4px; border:1px solid #ccc; font-size:11px;">';
        echo '<strong style="display:block; margin-bottom:5px; color:#555;">Shipping History:</strong>';
        echo '<ul style="margin:0; padding:0; list-style:none; max-height:80px; overflow-y:auto;">';
        foreach ( array_reverse( $history ) as $entry ) {
            echo '<li style="margin-bottom:3px; border-bottom:1px solid #e0e0e0; padding-bottom:2px;">';
            echo '<span style="color:#888;">' . esc_html( $entry['date'] ) . ':</span> <strong>' . esc_html( $entry['code'] ) . '</strong>';
            echo '</li>';
        }
        echo '</ul></div>';
    }
}

/**
 * 3. SAVE DATA & HISTORY
 */
add_action( 'woocommerce_process_shop_order_meta', 'ctt_pro_save_with_history' );
function ctt_pro_save_with_history( $order_id ) {
    if ( isset( $_POST['ctt_tracking_field'] ) ) {
        $old_tracking = get_post_meta( $order_id, '_ctt_tracking_number', true );
        $new_tracking = sanitize_text_field( $_POST['ctt_tracking_field'] );
        
        if ( $new_tracking !== $old_tracking && ! empty( $new_tracking ) ) {
            update_post_meta( $order_id, '_ctt_tracking_number', $new_tracking );
            $history = get_post_meta( $order_id, '_ctt_tracking_history', true ) ?: array();
            $history[] = array(
                'date' => current_time( 'd/m/Y H:i' ), 
                'code' => $new_tracking
            );
            update_post_meta( $order_id, '_ctt_tracking_history', $history );
        } elseif ( empty( $new_tracking ) ) {
            update_post_meta( $order_id, '_ctt_tracking_number', '' );
        }
    }
}

/**
 * 4. MY ACCOUNT ORDER LIST ACTIONS
 */
add_filter( 'woocommerce_my_account_my_orders_actions', 'ctt_pro_add_tracking_button_to_list', 10, 2 );
function ctt_pro_add_tracking_button_to_list( $actions, $order ) {
    $tracking = get_post_meta( $order->get_id(), '_ctt_tracking_number', true );
    if ( ! empty( $tracking ) ) {
        $actions['ctt_tracking'] = array(
            'url'  => 'https://www.ctt.pt/feapl/app/open/objectSearch/objectSearch.jspx?objects=' . esc_attr( $tracking ),
            'name' => 'Track CTT',
        );
    }
    return $actions;
}

/**
 * 5. CUSTOMER ORDER DETAILS PAGE
 */
add_action( 'woocommerce_order_details_after_order_table', 'ctt_pro_display_customer', 10 );
function ctt_pro_display_customer( $order ) {
    $tracking = get_post_meta( $order->get_id(), '_ctt_tracking_number', true );
    if ( ! empty( $tracking ) ) {
        echo '<div style="background: #fdfdfd; border: 2px solid #d61919; padding: 20px; border-radius: 8px; margin: 20px 0; clear: both;">';
        echo '<h3 style="color: #d61919; margin-top: 0;">Order Tracking</h3>';
        echo '<p style="font-size: 1.1em;"><strong>Code:</strong> <code style="background:#eee; padding:4px;">' . esc_html( $tracking ) . '</code></p>';
        echo '<a href="https://www.ctt.pt/feapl/app/open/objectSearch/objectSearch.jspx?objects=' . esc_attr( $tracking ) . '" target="_blank" style="background-color: #d61919; color: white; padding: 10px 20px; text-decoration: none; border-radius: 4px; display: inline-block; font-weight: bold;">Track on CTT website</a>';
        echo '</div>';
    }
}

/**
 * 6. EMAIL CONTENT
 */
add_action( 'woocommerce_email_before_order_table', 'ctt_pro_email_content', 10, 4 );
function ctt_pro_email_content( $order, $sent_to_admin, $plain_text, $email ) {
    if ( $email->id == 'customer_completed_order' ) {
        $tracking = get_post_meta( $order->get_id(), '_ctt_tracking_number', true );
        if ( $tracking ) {
            echo '<div style="padding: 20px; border: 2px solid #d61919; text-align: center; border-radius: 10px; margin-bottom: 25px;">';
            echo '<h2 style="color:#d61919; margin-top:0;">Your order has been shipped!</h2>';
            echo '<p>Tracking Code: <strong>' . esc_html( $tracking ) . '</strong></p>';
            echo '<p style="margin-top:20px;"><a href="https://www.ctt.pt/feapl/app/open/objectSearch/objectSearch.jspx?objects=' . esc_attr( $tracking ) . '" style="background:#d61919; color:white; padding:12px 25px; text-decoration:none; border-radius:5px; font-weight:bold;">Track on CTT</a></p>';
            echo '</div>';
        }
    }
}

/**
 * 7. ACTIVATION NOTICE
 */
add_action( 'admin_notices', 'ctt_pro_activation_notice' );
function ctt_pro_activation_notice() {
    if ( get_transient( 'ctt_pro_activated' ) ) {
        ?>
        <div class="notice notice-success is-dismissible" style="border-left-color: #d61919;">
            <p style="font-size: 15px;">
                <strong>📦 Shipping Online Tracker for CTT:</strong> Thank you for installing! 
                The tracking field is now available in the sidebar of your orders.
            </p>
        </div>
        <?php
        delete_transient( 'ctt_pro_activated' );
    }
}

register_activation_hook( __FILE__, 'ctt_pro_register_activation' );
function ctt_pro_register_activation() {
    set_transient( 'ctt_pro_activated', true, 5 );
}