<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://shipper360.io
 * @since      1.0
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0
 * @package    Shipper360
 * @subpackage Shipper360/includes
 * @author     Shipper <info@shipper360.io>
 */
class Shipper360 {

	/**
	 * The loader that's responsible for maintaining and registering all hooks that power
	 * the plugin.
	 *
	 * @since    1.0
	 * @access   protected
	 * @var      Shipper360_Loader    $loader    Maintains and registers all hooks for the plugin.
	 */
	protected $loader;

	/**
	 * The unique identifier of this plugin.
	 *
	 * @since    1.0
	 * @access   protected
	 * @var      string    $plugin_name    The string used to uniquely identify this plugin.
	 */
	protected $plugin_name;

	/**
	 * The current version of the plugin.
	 *
	 * @since    1.0
	 * @access   protected
	 * @var      string    $version    The current version of the plugin.
	 */
	protected $version;

	/**
	 * Define the core functionality of the plugin.
	 *
	 * Set the plugin name and the plugin version that can be used throughout the plugin.
	 * Load the dependencies, define the locale, and set the hooks for the admin area and
	 * the public-facing side of the site.
	 *
	 * @since    1.0
	 */
	public function __construct() {
		if ( defined( 'SHIPPER360_VERSION' ) ) {
			$this->version = SHIPPER360_VERSION;
		} else {
			$this->version = '1.0';
		}
		$this->plugin_name = 'shipper';

		$this->load_dependencies();
		//$this->define_admin_hooks();
		$this->define_public_hooks();
		$this->define_webhook_hooks();
		$this->define_shipping_hooks();
		$this->define_tracking_hooks();

	}

	/**
	 * Load the required dependencies for this plugin.
	 *
	 * Include the following files that make up the plugin:
	 *
	 * - Shipper_Loader. Orchestrates the hooks of the plugin.
	 * - Shipper_i18n. Defines internationalization functionality.
	 * - Shipper_Admin. Defines all hooks for the admin area.
	 * - Shipper_Public. Defines all hooks for the public side of the site.
	 *
	 * Create an instance of the loader which will be used to register the hooks
	 * with WordPress.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function load_dependencies() {

		/**
		 * The class responsible for orchestrating the actions and filters of the
		 * core plugin.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-shipper360-loader.php';

		/**
		 * The class responsible for defining all actions that occur on the webhook.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-shipper360-webhook.php';

		/**
		 * The class responsible for defining all actions that occur on the shipping rates.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-shipper360-shipping-rates.php';

		/**
         * The class responsible for defining tracking functionality.
         */
        require_once plugin_dir_path( dirname( __FILE__ ) ) . 'includes/class-shipper360-tracking.php';


		/**
		 * The class responsible for defining all actions that occur in the admin area.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/class-shipper360-admin.php';

		/**
		 * The class responsible for defining all actions that occur in the public-facing
		 * side of the site.
		 */
		require_once plugin_dir_path( dirname( __FILE__ ) ) . 'public/class-shipper360-public.php';

		$this->loader = new Shipper360_Loader();

	}

	/**
	 * Register all of the hooks related to the admin area functionality
	 * of the plugin.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function define_admin_hooks() {

		$plugin_admin = new Shipper360_Admin( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_styles' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );
		$this->loader->add_action( 'admin_enqueue_scripts', $plugin_admin, 'enqueue_scripts' );

	}

	/**
	 * Register all of the hooks related to the public-facing functionality
	 * of the plugin.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function define_public_hooks() {

		$plugin_public = new Shipper360_Public( $this->get_plugin_name(), $this->get_version() );

		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_styles' );
		$this->loader->add_action( 'wp_enqueue_scripts', $plugin_public, 'enqueue_scripts' );

	}

	/**
	 * Register all of the hooks related to the webhook functionality
	 * of the plugin.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function define_webhook_hooks() {

		$plugin_webhook = new Shipper360_Webhook( $this->get_plugin_name(), $this->get_version() );
        $this->loader->add_filter('woocommerce_webhook_payload', $plugin_webhook, 'extend_webhook_payload', 10, 4);
    }

	/**
	 * Register all of the hooks related to the shipping functionality
	 * of the plugin.
	 *
	 * @since    1.0
	 * @access   private
	 */
	private function define_shipping_hooks() {
		$this->loader->add_filter('woocommerce_shipping_methods', $this, 'add_shipping_method');
	}

	public function add_shipping_method($methods) {
		$methods['shipper'] = 'Shipper360_Shipping_Rates';
		return $methods;
	}

	/**
     * Register all of the hooks related to the tracking functionality
     * of the plugin.
     */
    private function define_tracking_hooks() {
        $plugin_tracking = new Shipper360_Tracking( $this->get_plugin_name(), $this->get_version() );
    }

	/**
	 * Run the loader to execute all of the hooks with WordPress.
	 *
	 * @since    1.0
	 */
	public function run() {
		$this->loader->run();
	}

	/**
	 * The name of the plugin used to uniquely identify it within the context of
	 * WordPress and to define internationalization functionality.
	 *
	 * @since     1.0
	 * @return    string    The name of the plugin.
	 */
	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * The reference to the class that orchestrates the hooks with the plugin.
	 *
	 * @since     1.0
	 * @return    Shipper_Loader    Orchestrates the hooks of the plugin.
	 */
	public function get_loader() {
		return $this->loader;
	}

	/**
	 * Retrieve the version number of the plugin.
	 *
	 * @since     1.0
	 * @return    string    The version number of the plugin.
	 */
	public function get_version() {
		return $this->version;
	}

}
