<?php

/**
 * The webhook-specific functionality of the plugin.
 *
 * @link       https://shipper360.io
 * @since      1.0
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 */

/**
 * The webhook-specific functionality of the plugin.
 *
 * Defines the plugin name, version, and hooks for webhook modifications.
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 * @author     Shipper <info@shipper360.io>
 */
class Shipper360_Webhook {

    /**
     * The ID of this plugin.
     *
     * @since    1.0
     * @access   private
     * @var      string    $plugin_name    The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     *
     * @since    1.0
     * @access   private
     * @var      string    $version    The current version of this plugin.
     */
    private $version;

    /**
     * Initialize the class and set its properties.
     *
     * @since    1.0
     * @param    string    $plugin_name       The name of this plugin.
     * @param    string    $version    The version of this plugin.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;

        // Hook to check if webhook should be delivered (has access to order ID)
        add_filter('woocommerce_webhook_should_deliver', array($this, 'should_deliver_webhook'), 10, 3);
        
        // Hook to control webhook delivery method
        add_filter('woocommerce_webhook_deliver_async', array($this, 'control_webhook_delivery'), 10, 2);
    }

    /**
     * Extends the WooCommerce webhook payload with additional data
     *
     * @param array $payload The original webhook payload
     * @param string $resource The webhook resource
     * @param int $resource_id The webhook resource ID
     * @param int $webhook_id The webhook ID
     * @return array Modified payload
     */
    public function extend_webhook_payload($payload, $resource, $resource_id, $webhook_id) {
        if (!class_exists('WooCommerce')) {
            return $payload;
        }
    
        if ($resource === 'order' && isset($payload['line_items'])) {
            foreach ($payload['line_items'] as $key => $item) {
                $product_id = $item['variation_id'] ? $item['variation_id'] : $item['product_id'];
                $product = wc_get_product($product_id);
                
                if ($product) {
                    $payload['line_items'][$key]['shipper_data'] = [
                        'dimensions' => [
                            'length' => $product->get_length(),
                            'width'  => $product->get_width(),
                            'height' => $product->get_height(),
                            'unit'   => get_option('woocommerce_dimension_unit')
                        ],
                        'weight' => [
                            'value' => $product->get_weight(),
                            'unit'  => get_option('woocommerce_weight_unit')
                        ],
                        'stock_quantity' => $product->get_stock_quantity(),
                        'stock_status'   => $product->get_stock_status()
                    ];
                }
            }
        }
    
        return $payload;
    }

    /**
     * Determine if webhook should be delivered
     *
     * @param bool $should_deliver Whether the webhook should be delivered
     * @param WC_Webhook $webhook The webhook object
     * @param mixed $arg The resource being delivered (order ID for order webhooks)
     * @return bool Whether to deliver the webhook
     */
    public function should_deliver_webhook($should_deliver, $webhook, $arg) {
        if (!$should_deliver) {
            return $should_deliver;
        }

        if ($webhook instanceof WC_Webhook) {
            $topic = $webhook->get_topic();
            
            // Check for order webhooks
            if (in_array($topic, ['order.created', 'order.updated'])) {
                // $arg is the order ID for order webhooks
                $order = wc_get_order($arg);
                
                // Don't deliver webhook if payment method is yith_pos_cash_gateway
                if ($order && $order->get_payment_method() === 'yith_pos_cash_gateway') {
                    return false;
                }

                // Don't deliver webhook if update came from REST API
                if ($topic === 'order.updated' && $this->is_rest_api_request()) {
                    return false;
                }
            }
        }
        
        return $should_deliver;
    }

    /**
     * Control webhook delivery method (sync vs async)
     *
     * @param bool $deliver_async Whether to deliver webhook asynchronously
     * @param WC_Webhook $webhook The webhook object
     * @return bool Modified delivery method
     */
    public function control_webhook_delivery($deliver_async, $webhook) {
        if ($webhook instanceof WC_Webhook) {
            $topic = $webhook->get_topic();
            
            // Optional: Also check for specific delivery URLs if needed
            $delivery_url = $webhook->get_delivery_url();
            if (strpos($delivery_url, 'shipper360.io') !== false) {
                return false; // Run Shipper webhooks synchronously
            }
        }
        
        return $deliver_async; // Use default behavior for other webhooks
    }

    /**
     * Check if current request is a REST API request
     *
     * @return bool True if REST API request, false otherwise
     */
    private function is_rest_api_request() {
        // Check if WooCommerce REST API request
        if (defined('REST_REQUEST') && REST_REQUEST) {
            return true;
        }
        
        // Check if request URI contains REST API endpoint
        if (isset($_SERVER['REQUEST_URI'])) {
            $request_uri = sanitize_text_field(wp_unslash($_SERVER['REQUEST_URI']));
            if (strpos($request_uri, '/wp-json/wc/') !== false) {
                return true;
            }
        }
        
        return false;
    }
}