<?php

/**
 * The tracking functionality of the plugin.
 *
 * @link       https://shipper360.io
 * @since      1.0
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 */

/**
 * The tracking functionality of the plugin.
 *
 * Defines tracking shortcode, AJAX handlers, and API communication.
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 * @author     Shipper <info@shipper360.io>
 */
class Shipper360_Tracking {

    /**
     * The ID of this plugin.
     */
    private $plugin_name;

    /**
     * The version of this plugin.
     */
    private $version;

    /**
     * The API token for authentication.
     */
    private $api_token;

    /**
     * Initialize the class and set its properties.
     */
    public function __construct($plugin_name, $version) {
        $this->plugin_name = $plugin_name;
        $this->version = $version;
        
        // Get API token from shipping method settings
        $this->load_api_token();
        
        add_shortcode('shipper_tracking', array($this, 'tracking_shortcode'));
        add_action('wp_ajax_shipper_track', array($this, 'ajax_track_shipment'));
        add_action('wp_ajax_nopriv_shipper_track', array($this, 'ajax_track_shipment'));
    }

    /**
     * Load API token from shipping method settings
     */
    private function load_api_token() {
        // Check if WooCommerce is loaded
        if (!class_exists('WooCommerce') || !function_exists('WC')) {
            return;
        }

        // Try to get from WordPress options directly
        $shipper_options = get_option('woocommerce_shipper_settings', array());
        if (!empty($shipper_options['api_token'])) {
            $this->api_token = $shipper_options['api_token'];
            return;
        }

        // Alternative: defer loading until WooCommerce is ready
        add_action('woocommerce_init', array($this, 'load_api_token_deferred'));
    }

    /**
     * Load API token after WooCommerce is initialized
     */
    public function load_api_token_deferred() {
        if (!empty($this->api_token)) {
            return; // Already loaded
        }

        try {
            $shipping_zones = WC_Shipping_Zones::get_zones();
            
            foreach ($shipping_zones as $zone) {
                if (isset($zone['shipping_methods'])) {
                    foreach ($zone['shipping_methods'] as $method) {
                        if ($method instanceof Shipper360_Shipping_Rates) {
                            $this->api_token = $method->get_option('api_token');
                            if (!empty($this->api_token)) {
                                return;
                            }
                        }
                    }
                }
            }
            
            // Fallback: try to get from zone 0 (rest of the world)
            $zone_0 = new WC_Shipping_Zone(0);
            $shipping_methods = $zone_0->get_shipping_methods();
            foreach ($shipping_methods as $method) {
                if ($method instanceof Shipper360_Shipping_Rates) {
                    $this->api_token = $method->get_option('api_token');
                    if (!empty($this->api_token)) {
                        return;
                    }
                }
            }
        } catch (Exception $e) {
            return;
        }
    }

    /**
     * Get API token (load on demand if not available)
     */
    private function get_api_token() {
        if (empty($this->api_token)) {
            // Try direct options lookup first
            $shipper_options = get_option('woocommerce_shipper_settings', array());
            if (!empty($shipper_options['api_token'])) {
                $this->api_token = $shipper_options['api_token'];
            } else {
                // Try loading from shipping zones if WooCommerce is ready
                if (class_exists('WC_Shipping_Zones')) {
                    $this->load_api_token_deferred();
                }
            }
        }
        
        return $this->api_token;
    }

    /**
     * Get tracking information from API
     */
    public function get_tracking_info($tracking_code) {
        if (empty($this->api_token)) {
            return new WP_Error('no_api_token', esc_html__('API token not configured.', 'shipper360'));
        }

        $endpoint = 'https://app.shipper360.io/api/tracking/' . urlencode($tracking_code);
        
        $response = wp_remote_get($endpoint, array(
            'headers' => array(
                'Authorization' => 'Token ' . $this->api_token,
                'Content-Type' => 'application/json',
                'Cache-Control' => 'no-cache, no-store, must-revalidate',
                'Pragma' => 'no-cache',
                'Expires' => '0'
            ),
            'timeout' => 30,
            'sslverify' => false,
            'redirection' => 0,
            'blocking' => true,
            'data_format' => 'body'
        ));

        if (is_wp_error($response)) {
            return $response;
        }

        $response_code = wp_remote_retrieve_response_code($response);
        if ($response_code !== 200) {
            return new WP_Error('api_error', esc_html__('API returned error code: ', 'shipper360') . $response_code);
        }

        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return new WP_Error('invalid_json', esc_html__('Invalid response from tracking service.', 'shipper360'));
        }

        return $body;
    }

    /**
     * Tracking shortcode
     */
    public function tracking_shortcode($atts) {
        $atts = shortcode_atts(array(
            'title' => esc_html__('Track Your Shipment', 'shipper360'),
            'placeholder' => esc_html__('Enter tracking code', 'shipper360'),
            'button_text' => esc_html__('Track', 'shipper360'),
        ), $atts, 'shipper_tracking');

        // Enqueue scripts
        wp_enqueue_script(
            'shipper360-tracking',
            plugin_dir_url(dirname(__FILE__)) . 'public/js/shipper360-tracking.js',
            array('jquery'),
            $this->version,
            true
        );

        // Localize script with translatable strings and AJAX URL
        wp_localize_script('shipper360-tracking', 'shipper360_tracking_localize', array(
            'ajax_url' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('shipper_tracking_nonce'),
            'error_message' => esc_html__('Error occurred while tracking. Please try again.', 'shipper360'),
            'not_found_message' => esc_html__('Tracking code not found.', 'shipper360'),
            'tracking_info_title' => esc_html__('Tracking Information', 'shipper360'),
            'tracking_code_label' => esc_html__('Tracking Code:', 'shipper360'),
            'carrier_label' => esc_html__('Carrier:', 'shipper360'),
            'tracking_events_label' => esc_html__('Tracking Events', 'shipper360')
        ));

        ob_start();
        ?>
        <div class="shipper-tracking-container">
            <h3><?php echo esc_html($atts['title']); ?></h3>
            
            <form id="shipper-tracking-form" class="shipper-tracking-form">
                <div class="tracking-input-group">
                    <input type="text" 
                           id="tracking-code" 
                           name="tracking_code" 
                           placeholder="<?php echo esc_attr($atts['placeholder']); ?>" 
                           required />
                    <button type="submit" class="shipper-track-btn">
                        <?php echo esc_html($atts['button_text']); ?>
                    </button>
                </div>
            </form>

            <div id="shipper-tracking-loading" class="shipper-loading" style="display: none;">
                <p><?php esc_html_e('Searching for tracking information...', 'shipper360'); ?></p>
            </div>

            <div id="shipper-tracking-results" class="shipper-tracking-results"></div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * AJAX handler for tracking
     */
    public function ajax_track_shipment() {
        if (
			!isset($_POST['nonce']) || 
			!wp_verify_nonce(sanitize_text_field(wp_unslash($_POST['nonce'])), 'shipper_tracking_nonce')
		) {
            wp_die(esc_html__('Security check failed', 'shipper360'));
        }

        if(!isset($_POST['tracking_code'])){
            wp_send_json_error(esc_html__('Please enter a tracking code.', 'shipper360'));
        }
        $tracking_code = sanitize_text_field(wp_unslash($_POST['tracking_code']));

        // Get tracking info using our internal method
        $tracking_info = $this->get_tracking_info($tracking_code);
        
        if (is_wp_error($tracking_info)) {
            wp_send_json_error($tracking_info->get_error_message());
        }

        wp_send_json_success($tracking_info);
    }

    /**
     * Get carrier name from carrier code
     */
    public function get_carrier_name($carrier_code) {
        $carriers = array(
            'acs_shipper' => 'ACS Courier',
            'speedex' => 'Speedex',
            'elta' => 'ELTA',
            'dhl' => 'DHL',
            'ups' => 'UPS',
            'courier-center' => 'Courier Center'
        );
        
        return isset($carriers[$carrier_code]) ? $carriers[$carrier_code] : $carrier_code;
    }

    /**
     * Format tracking event date
     */
    public function format_tracking_date($date_string) {
        $date = new DateTime($date_string);
        return $date->format('Y-m-d H:i:s');
    }
}