<?php

/**
 * Extend Woocommerce with shipping rates functionality from Shipper app
 *
 * @link       https://shipper360.io
 * @since      1.0
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 */

/**
  * Extend Woocommerce with shipping rates functionality from Shipper app
 *
 * Defines the plugin name, version, and hooks for webhook modifications.
 *
 * @package    Shipper360
 * @subpackage Shipper360/includes
 * @author     Shipper <info@shipper360.io>
 */
function shipper360_init_shipping_method() {
    if (!class_exists('WC_Shipping_Method')) {
        return;
    }
    
    class Shipper360_Shipping_Rates extends WC_Shipping_Method {

        /**
         * The ID of this plugin.
         *
         * @since    1.0
         * @access   private
         * @var      string    $plugin_name    The ID of this plugin.
         */
        private $plugin_name;

        /**
         * The version of this plugin.
         *
         * @since    1.0
         * @access   private
         * @var      string    $version    The current version of this plugin.
         */
        private $version;

        /**
         * Initialize the class and set its properties.
         *
         * @since    1.0
         * @param    string    $plugin_name       The name of this plugin.
         * @param    string    $version    The version of this plugin.
         */
        public function __construct($instance_id = 0) {
            $this->id = 'shipper';
            $this->instance_id = absint($instance_id);
            $this->method_title = __('Shipper Shipping', 'shipper360');
            $this->method_description = __('Get live shipping rates from multiple carriers', 'shipper360');
            $this->supports = array(
                'shipping-zones',
                'instance-settings',
                'settings',
            );

            $this->init();
        }

    /**
         * Initialize shipping method
         */
        public function init() {
            $this->init_form_fields();
            $this->init_settings();

            $this->enabled = $this->get_option('enabled');
            $this->title = $this->get_option('title', $this->method_title);
            $this->api_token = $this->get_option('api_token');

            add_action('woocommerce_update_options_shipping_' . $this->id, array($this, 'process_admin_options'));

            // Hook to disable COD for specific carriers
            add_filter('woocommerce_available_payment_gateways', array($this, 'disable_cod_for_carriers'));
        }

        /**
         * Initialize form fields
         */
        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => __('Enable', 'shipper360'),
                    'type' => 'checkbox',
                    'label' => __('Enable this shipping method', 'shipper360'),
                    'default' => 'yes',
                ),
                'title' => array(
                    'title' => __('Title', 'shipper360'),
                    'type' => 'text',
                    'description' => __('This controls the title which the user sees during checkout.', 'shipper360'),
                    'default' => __('Shipper Shipping', 'shipper360'),
                    'desc_tip' => true,
                ),
                'api_token' => array(
                    'title' => __('API Token', 'shipper360'),
                    'type' => 'text',
                    'description' => __('Enter your Shipper API token', 'shipper360'),
                    'default' => '',
                    'desc_tip' => true,
                ),
            );
        }

        /**
         * Calculate shipping rates
         */
        public function calculate_shipping($package = array()) {
            if ($this->enabled !== 'yes') {
                return;
            }

            // Prepare request data
            $request_data = $this->prepare_rate_request($package);
            
            // Get rates from API
            $rates = $this->get_rates_from_api($request_data);
            
            if (is_wp_error($rates)) {
                return;
            }

            // Add rates to WooCommerce
            foreach ($rates as $rate) {
                $label = $rate['name'] . ( $rate['service'] ? ' - ' . $rate['service'] : '' );
                if( isset($rate['free_shipping']) && $rate['free_shipping'] ) {
                    $label .= ' (' . __('Free Shipping', 'shipper360') . ')';
                }

                $this->add_rate(array(
                    'id' => $this->id . ':' . $rate['carrier'],
                    'label' => $label,
                    'cost' => $rate['price'],
                    'meta_data' => array(
                        'delivery_time' => $rate['delivery_time'],
                        'carrier' => $rate['carrier'],
                    )
                ));
            }
        }

        /**
         * Prepare rate request data
         */
        private function prepare_rate_request($package) {
            $total_value = WC()->cart->get_cart_contents_total();
            $weight = 0;

            foreach ($package['contents'] as $item) {
                $product = $item['data'];
                $weight += (float)$product->get_weight() * $item['quantity'];
            }

            return array(
                'address_1' => $package['destination']['address_1'],
                'city' => $package['destination']['city'],
                'postcode' => $package['destination']['postcode'],
                'country' => $package['destination']['country'],
                'weight' => number_format((float)$weight, 2, '.', ''),
                'charge' => 2,
                'pickup' => gmdate('Y-m-d', strtotime('+1 day')),
                'insurance_amount' => number_format((float)$total_value, 2, '.', ''),
                'total_amount' => number_format((float)$total_value, 2, '.', ''),
            );
        }

        /**
         * Get rates from API
         */
        private function get_rates_from_api($request_data) {
            $endpoint = 'https://app.shipper360.io/api/shipping-rates/';
            
            $response = wp_remote_post($endpoint, array(
                'headers' => array(
                    'Authorization' => 'Token ' . $this->api_token,
                    'Content-Type' => 'application/json'
                ),
                'body' => json_encode($request_data),
                'timeout' => 30
            ));

            if (is_wp_error($response)) {
                return $response;
            }

            $body = json_decode(wp_remote_retrieve_body($response), true);

            if (isset($body['rates'])) {
                return $body['rates'];
            }

            return new WP_Error('invalid_response', __('Invalid response from shipping API', 'shipper360'));
        }

        /**
         * Disable Cash on Delivery for specific carriers
         */
        public function disable_cod_for_carriers($available_gateways) {
            if (is_admin()) {
                return $available_gateways;
            }

            // Get chosen shipping methods
            $chosen_methods = WC()->session->get('chosen_shipping_methods');
            
            if (empty($chosen_methods)) {
                return $available_gateways;
            }

            // Carriers that don't support COD
            $no_cod_carriers = array('acs_shipper'); // Add more as needed

            foreach ($chosen_methods as $method) {
                // Check if it's a shipper method
                if (strpos($method, 'shipper:') === 0) {
                    $carrier = str_replace('shipper:', '', $method);
                    
                    // If carrier doesn't support COD, remove it
                    if (in_array($carrier, $no_cod_carriers)) {
                        unset($available_gateways['cod']);
                        break;
                    }
                }
            }

            return $available_gateways;
        }

    }
}
add_action('woocommerce_shipping_init', 'shipper360_init_shipping_method');