<?php
/**
 * Plugin Name: Ship Distance
 * Description: Restrict WooCommerce shipping based on distance and display an error message if the address exceeds a specific range.
 * Version: 1.0
 * Author: Kaleem Ullah
 * Author URI: https://kaleemullah.in
 * Text Domain: ship-distance
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC tested up to: 8.2
 * License: GPLv2 or later
 * License URI: http://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

// Add HPOS and Custom Order Table compatibility
add_action('before_woocommerce_init', 'ship_distance_hpos_compatibility');
function ship_distance_hpos_compatibility() {
    if (class_exists(\Automattic\WooCommerce\Utilities\FeaturesUtil::class)) {
        // Declare HPOS compatibility
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
            'custom_order_tables', 
            __FILE__, 
            true
        );

        // Declare Remote Logging compatibility
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
            'remote_logging', 
            __FILE__, 
            true
        );
    }
}

// Additional compatibility checks during plugin activation
register_activation_hook(__FILE__, 'ship_distance_activation_compatibility_check');
function ship_distance_activation_compatibility_check() {
    // Check for WooCommerce
    if (!class_exists('WooCommerce')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('This plugin requires WooCommerce to be installed and active.', 'ship-distance'),
            esc_html__('Compatibility Error', 'ship-distance'),
            ['back_link' => true]
        );
    }

    // Check WooCommerce version
    if (version_compare(WC()->version, '7.1', '<')) {
        deactivate_plugins(plugin_basename(__FILE__));
        wp_die(
            esc_html__('This plugin requires WooCommerce version 7.1 or higher.', 'ship-distance'),
            esc_html__('Version Compatibility Error', 'ship-distance'),
            ['back_link' => true]
        );
    }
}

// Load plugin text domain
add_action('plugins_loaded', 'ship_distance_load_textdomain');
function ship_distance_load_textdomain() {
    load_plugin_textdomain('ship-distance', false, dirname(plugin_basename(__FILE__)) . '/languages/');
}

// Enqueue admin scripts and styles
add_action('admin_enqueue_scripts', 'ship_distance_enqueue_admin_scripts');
function ship_distance_enqueue_admin_scripts($hook) {
    if ($hook !== 'settings_page_ship_distance_settings') {
        return;
    }
    wp_enqueue_style('woo-dbs-admin-style', plugin_dir_url(__FILE__) . 'assets/admin.css', array(), '1.1');
    wp_enqueue_script('woo-dbs-admin-script', plugin_dir_url(__FILE__) . 'assets/admin.js', array('jquery'), '1.1', true);
}

// Add main menu item 
add_action('admin_menu', 'ship_distance_add_settings_menu');
function ship_distance_add_settings_menu() {
    add_menu_page(
        esc_html__('WooCommerce Distance Settings', 'ship-distance'), // Page title
        esc_html__('Ship Distance', 'ship-distance'), // Menu title
        'manage_options', // Capability
        'ship_distance_settings', // Menu slug
        'ship_distance_settings_page', // Callback function
        'dashicons-location-alt', // Icon (location marker)
        40 // Position in menu
    );
}

// Add "Settings" link on the Plugins page
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'ship_distance_add_settings_link');
function ship_distance_add_settings_link($links) {
    $settings_link = sprintf(
        '<a href="%s">%s</a>',
        esc_url(admin_url('options-general.php?page=ship_distance_settings')),
        esc_html__('Settings', 'ship-distance')
    );
    array_unshift($links, $settings_link);
    return $links;
}

// Encrypt data with improved security
function ship_distance_encrypt($data) {
    if (empty($data)) {
        return '';
    }
    $encryption_key = wp_salt('auth');
    return base64_encode(openssl_encrypt(
        $data, 
        'AES-256-CBC', 
        $encryption_key, 
        0, 
        substr($encryption_key, 0, 16)
    ));
}

// Decrypt data with improved security
function ship_distance_decrypt($data) {
    if (empty($data)) {
        return '';
    }
    $encryption_key = wp_salt('auth');
    return openssl_decrypt(
        base64_decode($data), 
        'AES-256-CBC', 
        $encryption_key, 
        0, 
        substr($encryption_key, 0, 16)
    );
}

// Settings page content with improved validation and sanitization
function ship_distance_settings_page() {
    // Check user capabilities
    if (!current_user_can('manage_options')) {
        wp_die(esc_html__('You do not have sufficient permissions to access this page.', 'ship-distance'));
    }

    // Save settings securely
    if (isset($_POST['ship_distance_save_settings']) && 
        check_admin_referer('ship_distance_save_settings_action', 'ship_distance_nonce')) {
        
        // Sanitize and validate inputs with unslashing
        $latitude = isset($_POST['ship_distance_latitude']) 
            ? sanitize_text_field(wp_unslash($_POST['ship_distance_latitude'])) 
            : '';
        $longitude = isset($_POST['ship_distance_longitude']) 
            ? sanitize_text_field(wp_unslash($_POST['ship_distance_longitude'])) 
            : '';
        $api_key = isset($_POST['ship_distance_api_key']) 
            ? sanitize_text_field(wp_unslash($_POST['ship_distance_api_key'])) 
            : '';
        $allowable_distance = isset($_POST['ship_distance_allowable_distance']) 
            ? abs(floatval(wp_unslash($_POST['ship_distance_allowable_distance']))) 
            : 5;
        $error_message = isset($_POST['ship_distance_error_message']) 
            ? sanitize_text_field(wp_unslash($_POST['ship_distance_error_message'])) 
            : esc_html__('Sorry, we cannot deliver to this address!', 'ship-distance');

        // Validate coordinates
        if (!preg_match('/^-?([1-8]?[1-9]|[1-9]0)\.{1}\d{1,6}$/', $latitude) || 
            !preg_match('/^-?([1-8]?[1-9]|[1-9]0)\.{1}\d{1,6}$/', $longitude)) {
            add_settings_error(
                'ship_distance_messages', 
                'ship_distance_invalid_coordinates', 
                esc_html__('Invalid latitude or longitude', 'ship-distance')
            );
        } else {
            // Encrypt sensitive fields
            update_option('ship_distance_latitude', ship_distance_encrypt($latitude));
            update_option('ship_distance_longitude', ship_distance_encrypt($longitude));
            update_option('ship_distance_api_key', ship_distance_encrypt($api_key));
            update_option('ship_distance_allowable_distance', $allowable_distance);
            update_option('ship_distance_error_message', $error_message);

            add_settings_error(
                'ship_distance_messages', 
                'ship_distance_settings_saved', 
                esc_html__('Settings saved successfully!', 'ship-distance'), 
                'updated'
            );
        }
    }

    // Retrieve and decrypt settings with default values
    $latitude = ship_distance_decrypt(get_option('ship_distance_latitude', ''));
    $longitude = ship_distance_decrypt(get_option('ship_distance_longitude', ''));
    $api_key = ship_distance_decrypt(get_option('ship_distance_api_key', ''));
    $allowable_distance = get_option('ship_distance_allowable_distance', 5);
    $error_message = get_option('ship_distance_error_message', 
        esc_html__('Sorry, we cannot deliver to this address!', 'ship-distance')
    );

    ?>
    <div class="wrap">
        <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
        <?php settings_errors(); ?>
        <form method="post" action="">
            <?php wp_nonce_field('ship_distance_save_settings_action', 'ship_distance_nonce'); ?>
            <table class="form-table">
                <tr>
                    <th scope="row">
                        <label for="ship_distance_latitude">
                            <?php echo esc_html__('Store Latitude', 'ship-distance'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text" 
                               id="ship_distance_latitude" 
                               name="ship_distance_latitude" 
                               value="<?php echo esc_attr($latitude); ?>" 
                               class="regular-text"
                               pattern="-?([1-8]?[1-9]|[1-9]0)\.{1}\d{1,6}"
                               required>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="ship_distance_longitude">
                            <?php echo esc_html__('Store Longitude', 'ship-distance'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text" 
                               id="ship_distance_longitude" 
                               name="ship_distance_longitude" 
                               value="<?php echo esc_attr($longitude); ?>" 
                               class="regular-text"
                               pattern="-?([1-8]?[1-9]|[1-9]0)\.{1}\d{1,6}"
                               required>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="ship_distance_api_key">
                            <?php echo esc_html__('Google Maps API Key', 'ship-distance'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text" 
                               id="ship_distance_api_key" 
                               name="ship_distance_api_key" 
                               value="<?php echo esc_attr($api_key); ?>" 
                               class="regular-text"
                               required>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="ship_distance_allowable_distance">
                            <?php echo esc_html__('Allowable Distance (in KM)', 'ship-distance'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="number" 
                               id="ship_distance_allowable_distance" 
                               name="ship_distance_allowable_distance" 
                               value="<?php echo esc_attr($allowable_distance); ?>" 
                               class="regular-text"
                               min="1"
                               step="0.1"
                               required>
                    </td>
                </tr>
                <tr>
                    <th scope="row">
                        <label for="ship_distance_error_message">
                            <?php echo esc_html__('Error Message', 'ship-distance'); ?>
                        </label>
                    </th>
                    <td>
                        <input type="text" 
                               id="ship_distance_error_message" 
                               name="ship_distance_error_message" 
                               value="<?php echo esc_attr($error_message); ?>" 
                               class="regular-text"
                               required>
                    </td>
                </tr>
            </table>
            <?php submit_button(esc_html__('Save Settings', 'ship-distance'), 'primary', 'ship_distance_save_settings'); ?>
        </form>
    </div>
    <?php
}

// Restrict delivery based on distance with improved error handling
add_action('woocommerce_after_checkout_validation', 'ship_distance_restrict_delivery_distance');
function ship_distance_restrict_delivery_distance($data) {
    // Retrieve and validate settings
    $latitude = ship_distance_decrypt(get_option('ship_distance_latitude', ''));
    $longitude = ship_distance_decrypt(get_option('ship_distance_longitude', ''));
    $api_key = ship_distance_decrypt(get_option('ship_distance_api_key', ''));
    $allowable_distance = get_option('ship_distance_allowable_distance', 5);
    $error_message = get_option('ship_distance_error_message', 
        esc_html__('Sorry, we cannot deliver to this address!', 'ship-distance')
    );

    // Validate required settings
    if (empty($latitude) || empty($longitude) || empty($api_key)) {
        return;
    }

    // Construct destination address
    $shipping_address = WC()->customer->get_shipping_address();
    $shipping_city = WC()->customer->get_shipping_city();
    $shipping_state = WC()->customer->get_shipping_state();
    $shipping_country = WC()->customer->get_shipping_country();
    $shipping_postcode = WC()->customer->get_shipping_postcode();

    $destination_address = implode(', ', array_filter([
        $shipping_address, 
        $shipping_city, 
        $shipping_state, 
        $shipping_country, 
        $shipping_postcode
    ]));

    // Prepare API request
    $api_url = add_query_arg([
        'origins' => "{$latitude},{$longitude}",
        'destinations' => urlencode($destination_address),
        'key' => $api_key
    ], "https://maps.googleapis.com/maps/api/distancematrix/json");

    // Remote request with error handling
    $response = wp_remote_get(esc_url_raw($api_url), [
        'timeout' => 30
    ]);

    // Check for API request errors
    if (is_wp_error($response)) {
        // Use wp_log instead of error_log for more WordPress-friendly logging
        wp_log_error('Ship Distance Distance API Error: ' . $response->get_error_message());
        return;
    }

    // Parse response
    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    // Validate API response
    if (isset($data['rows'][0]['elements'][0]['distance']['value'])) {
        $distance_in_km = $data['rows'][0]['elements'][0]['distance']['value'] / 1000;

        // Check distance against allowable limit
        if ($distance_in_km > $allowable_distance) {
            wc_add_notice(esc_html($error_message), 'error');
        }
    }
}

// Register activation hook
register_activation_hook(__FILE__, 'ship_distance_activate');
function ship_distance_activate() {
    // Perform any necessary activation tasks
    add_option('ship_distance_version', '1.1');
}

// Register deactivation hook
register_deactivation_hook(__FILE__, 'ship_distance_deactivate');
function ship_distance_deactivate() {
    // Delete all plugin-related options
    delete_option('ship_distance_latitude');
    delete_option('ship_distance_longitude');
    delete_option('ship_distance_api_key');
    delete_option('ship_distance_allowable_distance');
    delete_option('ship_distance_error_message');
    delete_option('ship_distance_version');
}