<?php
/**
 * Meta Boxes Handler
 *
 * @package Shibhu Custom Messages for WooCommerce
 */

defined( 'ABSPATH' ) || exit;

class SCMW_Meta_Boxes {
    
    private static $instance = null;
    
    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    private function __construct() {
        add_action( 'add_meta_boxes', array( $this, 'add_meta_boxes' ) );
        add_action( 'save_post', array( $this, 'save_meta_boxes' ) );
    }
    
    /**
     * Add meta boxes
     */
    public function add_meta_boxes() {
        // Message Settings
        add_meta_box(
            'scmw_message_settings',
            __( 'Message Settings', 'shibhu-custom-messages-for-woocommerce' ),
            array( $this, 'render_message_settings' ),
            'scmw_message',
            'normal',
            'high'
        );
        
        // Conditional Logic
        add_meta_box(
            'scmw_conditional_logic',
            __( 'Conditional Logic', 'shibhu-custom-messages-for-woocommerce' ),
            array( $this, 'render_conditional_logic' ),
            'scmw_message',
            'normal',
            'default'
        );
        
        // Display Settings
        add_meta_box(
            'scmw_display_settings',
            __( 'Display Settings', 'shibhu-custom-messages-for-woocommerce' ),
            array( $this, 'render_display_settings' ),
            'scmw_message',
            'side',
            'default'
        );
    }
    
    /**
     * Render Message Settings meta box
     */
    public function render_message_settings( $post ) {
        wp_nonce_field( 'scmw_save_message_settings', 'scmw_message_settings_nonce' );
        
        $message_type = get_post_meta( $post->ID, '_scmw_message_type', true );
        $message_type = $message_type ? $message_type : 'custom_message';
        
        // Get all settings
        $ticker_text = get_post_meta( $post->ID, '_scmw_ticker_text', true );
        $ticker_speed = get_post_meta( $post->ID, '_scmw_ticker_speed', true );
        $ticker_speed = $ticker_speed ? $ticker_speed : 50;
        
        $stock_show_remaining = get_post_meta( $post->ID, '_scmw_stock_show_remaining', true );
        $stock_text_template = get_post_meta( $post->ID, '_scmw_stock_text_template', true );
        $stock_text_template = $stock_text_template ? $stock_text_template : __( 'Only {stock_count} left in stock!', 'shibhu-custom-messages-for-woocommerce' );
        
        $countdown_date = get_post_meta( $post->ID, '_scmw_countdown_date', true );
        $countdown_text = get_post_meta( $post->ID, '_scmw_countdown_text', true );
        $countdown_text = $countdown_text ? $countdown_text : __( 'Offer ends in: {countdown}', 'shibhu-custom-messages-for-woocommerce' );
        
        $custom_image_id = get_post_meta( $post->ID, '_scmw_custom_image_id', true );
        $custom_image_url = $custom_image_id ? wp_get_attachment_url( $custom_image_id ) : '';
        $custom_image_link = get_post_meta( $post->ID, '_scmw_custom_image_link', true );
        
        $custom_message_text = get_post_meta( $post->ID, '_scmw_custom_message_text', true );
        
        $discount_code = get_post_meta( $post->ID, '_scmw_discount_code', true );
        $discount_description = get_post_meta( $post->ID, '_scmw_discount_description', true );
        
        $social_platforms = get_post_meta( $post->ID, '_scmw_social_platforms', true );
        $social_platforms = is_array( $social_platforms ) ? $social_platforms : array( 'facebook', 'twitter', 'pinterest' );
        
        ?>
        <div class="scmw-meta-box-wrap">
            <!-- Message Type -->
            <div class="scmw-field">
                <label for="scmw_message_type"><strong><?php esc_html_e( 'Message Type', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <select id="scmw_message_type" name="scmw_message_type" class="widefat scmw-message-type-select">
                    <option value="text_ticker" <?php selected( $message_type, 'text_ticker' ); ?>><?php esc_html_e( 'Text Ticker', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="stock_counter" <?php selected( $message_type, 'stock_counter' ); ?>><?php esc_html_e( 'Real-time Stock Counter', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="countdown" <?php selected( $message_type, 'countdown' ); ?>><?php esc_html_e( 'Countdown Timer', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="custom_image" <?php selected( $message_type, 'custom_image' ); ?>><?php esc_html_e( 'Custom Image', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="custom_message" <?php selected( $message_type, 'custom_message' ); ?>><?php esc_html_e( 'Custom Message', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="discount_code" <?php selected( $message_type, 'discount_code' ); ?>><?php esc_html_e( 'Discount Code', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="social_share" <?php selected( $message_type, 'social_share' ); ?>><?php esc_html_e( 'Social Share Buttons', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                </select>
            </div>
            
            <!-- Text Ticker Settings -->
            <div class="scmw-message-type-panel scmw-panel-text_ticker" style="<?php echo $message_type !== 'text_ticker' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label for="scmw_ticker_label"><strong><?php esc_html_e( 'Ticker Label', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_label" name="scmw_ticker_label" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_label', true ) ?: 'Breaking News' ); ?>" class="widefat" placeholder="Breaking News" />
                    <p class="description"><?php esc_html_e( 'The label text that appears on the left side of the ticker', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_label_bg"><strong><?php esc_html_e( 'Label Background Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_label_bg" name="scmw_ticker_label_bg" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_label_bg', true ) ?: '#0d6efd' ); ?>" class="scmw-color-picker" data-default-color="#0d6efd" />
                    <p class="description"><?php esc_html_e( 'Background color for the ticker label', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_label_color"><strong><?php esc_html_e( 'Label Text Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_label_color" name="scmw_ticker_label_color" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_label_color', true ) ?: '#ffffff' ); ?>" class="scmw-color-picker" data-default-color="#ffffff" />
                    <p class="description"><?php esc_html_e( 'Text color for the ticker label', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_border_color"><strong><?php esc_html_e( 'Border Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_border_color" name="scmw_ticker_border_color" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_border_color', true ) ?: '#0d6efd' ); ?>" class="scmw-color-picker" data-default-color="#0d6efd" />
                    <p class="description"><?php esc_html_e( 'Border color for the ticker container', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_text_bg"><strong><?php esc_html_e( 'Text Background Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_text_bg" name="scmw_ticker_text_bg" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_text_bg', true ) ?: '#ffffff' ); ?>" class="scmw-color-picker" data-default-color="#ffffff" />
                    <p class="description"><?php esc_html_e( 'Background color for the ticker text area', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_text_color"><strong><?php esc_html_e( 'Text Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_text_color" name="scmw_ticker_text_color" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_text_color', true ) ?: '#333333' ); ?>" class="scmw-color-picker" data-default-color="#333333" />
                    <p class="description"><?php esc_html_e( 'Color for the scrolling text', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label>
                        <input type="checkbox" name="scmw_ticker_show_dot" value="1" <?php checked( get_post_meta( $post->ID, '_scmw_ticker_show_dot', true ), '1' ); ?> />
                        <?php esc_html_e( 'Show pulsing dot animation', 'shibhu-custom-messages-for-woocommerce' ); ?>
                    </label>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_text"><strong><?php esc_html_e( 'Ticker Text', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_ticker_text" name="scmw_ticker_text" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_text', true ) ); ?>" class="widefat" />
                    <p class="description"><?php esc_html_e( 'Use shortcodes: {product_name}, {product_price}, {stock_count}', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
                
                <div class="scmw-field">
                    <label for="scmw_ticker_speed"><strong><?php esc_html_e( 'Animation Speed (pixels/second)', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="number" id="scmw_ticker_speed" name="scmw_ticker_speed" value="<?php echo esc_attr( get_post_meta( $post->ID, '_scmw_ticker_speed', true ) ?: 50 ); ?>" min="10" max="200" class="small-text" />
                </div>
                
                <div class="scmw-field" style="margin-top: 20px; padding: 15px; background: #f0f6fc; border-left: 4px solid #2271b1; border-radius: 4px;">
                    <strong><?php esc_html_e( 'Quick Presets:', 'shibhu-custom-messages-for-woocommerce' ); ?></strong>
                    <div style="margin-top: 10px; display: flex; gap: 10px; flex-wrap: wrap;">
                        <button type="button" class="button button-small scmw-ticker-preset" data-preset="blue">
                            <?php esc_html_e( 'Blue', 'shibhu-custom-messages-for-woocommerce' ); ?>
                        </button>
                        <button type="button" class="button button-small scmw-ticker-preset" data-preset="red">
                            <?php esc_html_e( 'Red', 'shibhu-custom-messages-for-woocommerce' ); ?>
                        </button>
                        <button type="button" class="button button-small scmw-ticker-preset" data-preset="green">
                            <?php esc_html_e( 'Green', 'shibhu-custom-messages-for-woocommerce' ); ?>
                        </button>
                        <button type="button" class="button button-small scmw-ticker-preset" data-preset="dark">
                            <?php esc_html_e( 'Dark', 'shibhu-custom-messages-for-woocommerce' ); ?>
                        </button>
                        <button type="button" class="button button-small scmw-ticker-preset" data-preset="orange">
                            <?php esc_html_e( 'Orange', 'shibhu-custom-messages-for-woocommerce' ); ?>
                        </button>
                    </div>
                </div>
            </div>
            
            <!-- Stock Counter Settings -->
            <div class="scmw-message-type-panel scmw-panel-stock_counter" style="<?php echo $message_type !== 'stock_counter' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label>
                        <input type="checkbox" name="scmw_stock_show_remaining" value="1" <?php checked( $stock_show_remaining, '1' ); ?> />
                        <?php esc_html_e( 'Show remaining stock count', 'shibhu-custom-messages-for-woocommerce' ); ?>
                    </label>
                </div>
                <div class="scmw-field">
                    <label for="scmw_stock_text_template"><strong><?php esc_html_e( 'Text Template', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_stock_text_template" name="scmw_stock_text_template" value="<?php echo esc_attr( $stock_text_template ); ?>" class="widefat" />
                    <p class="description"><?php esc_html_e( 'Use {stock_count} placeholder', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
            </div>
            
            <!-- Countdown Timer Settings -->
            <div class="scmw-message-type-panel scmw-panel-countdown" style="<?php echo $message_type !== 'countdown' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label for="scmw_countdown_date"><strong><?php esc_html_e( 'End Date & Time', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="datetime-local" id="scmw_countdown_date" name="scmw_countdown_date" value="<?php echo esc_attr( $countdown_date ); ?>" class="widefat" />
                </div>
                <div class="scmw-field">
                    <label for="scmw_countdown_text"><strong><?php esc_html_e( 'Display Text', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_countdown_text" name="scmw_countdown_text" value="<?php echo esc_attr( $countdown_text ); ?>" class="widefat" />
                    <p class="description"><?php esc_html_e( 'Use {countdown} placeholder for timer', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
            </div>
            
            <!-- Custom Image Settings -->
            <div class="scmw-message-type-panel scmw-panel-custom_image" style="<?php echo $message_type !== 'custom_image' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label><strong><?php esc_html_e( 'Image', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <div class="scmw-image-upload-wrapper">
                        <input type="hidden" id="scmw_custom_image_id" name="scmw_custom_image_id" value="<?php echo esc_attr( $custom_image_id ); ?>" />
                        <div class="scmw-image-preview">
                            <?php if ( $custom_image_url ) : ?>
                                <img src="<?php echo esc_url( $custom_image_url ); ?>" alt="" style="max-width: 300px; height: auto;" />
                            <?php endif; ?>
                        </div>
                        <button type="button" class="button scmw-upload-image-button"><?php esc_html_e( 'Upload Image', 'shibhu-custom-messages-for-woocommerce' ); ?></button>
                        <button type="button" class="button scmw-remove-image-button" <?php echo ! $custom_image_url ? 'style="display:none;"' : ''; ?>><?php esc_html_e( 'Remove Image', 'shibhu-custom-messages-for-woocommerce' ); ?></button>
                    </div>
                </div>
                <div class="scmw-field">
                    <label for="scmw_custom_image_link"><strong><?php esc_html_e( 'Link URL (Optional)', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="url" id="scmw_custom_image_link" name="scmw_custom_image_link" value="<?php echo esc_url( $custom_image_link ); ?>" class="widefat" />
                </div>
            </div>
            
            <!-- Custom Message Settings -->
            <div class="scmw-message-type-panel scmw-panel-custom_message" style="<?php echo $message_type !== 'custom_message' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label for="scmw_custom_message_text"><strong><?php esc_html_e( 'Message Text', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <?php
                    wp_editor( $custom_message_text, 'scmw_custom_message_text', array(
                        'textarea_rows' => 8,
                        'media_buttons' => false,
                        'teeny'         => true,
                    ) );
                    ?>
                    <p class="description"><?php esc_html_e( 'Available shortcodes: {product_name}, {product_price}, {stock_count}, {product_sku}', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
                </div>
            </div>
            
            <!-- Discount Code Settings -->
            <div class="scmw-message-type-panel scmw-panel-discount_code" style="<?php echo $message_type !== 'discount_code' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label for="scmw_discount_code"><strong><?php esc_html_e( 'Discount Code', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <input type="text" id="scmw_discount_code" name="scmw_discount_code" value="<?php echo esc_attr( $discount_code ); ?>" class="widefat" />
                </div>
                <div class="scmw-field">
                    <label for="scmw_discount_description"><strong><?php esc_html_e( 'Description', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <textarea id="scmw_discount_description" name="scmw_discount_description" rows="3" class="widefat"><?php echo esc_textarea( $discount_description ); ?></textarea>
                </div>
                
                <?php
                // Add discount style fields
                SCMW_Discount_Styles::add_discount_style_fields( $post );
                ?>
            </div>
            
            <!-- Social Share Settings -->
            <div class="scmw-message-type-panel scmw-panel-social_share" style="<?php echo $message_type !== 'social_share' ? 'display:none;' : ''; ?>">
                <div class="scmw-field">
                    <label><strong><?php esc_html_e( 'Select Platforms', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                    <p>
                        <label><input type="checkbox" name="scmw_social_platforms[]" value="facebook" <?php checked( in_array( 'facebook', $social_platforms, true ) ); ?> /> <?php esc_html_e( 'Facebook', 'shibhu-custom-messages-for-woocommerce' ); ?></label><br>
                        <label><input type="checkbox" name="scmw_social_platforms[]" value="twitter" <?php checked( in_array( 'twitter', $social_platforms, true ) ); ?> /> <?php esc_html_e( 'Twitter', 'shibhu-custom-messages-for-woocommerce' ); ?></label><br>
                        <label><input type="checkbox" name="scmw_social_platforms[]" value="pinterest" <?php checked( in_array( 'pinterest', $social_platforms, true ) ); ?> /> <?php esc_html_e( 'Pinterest', 'shibhu-custom-messages-for-woocommerce' ); ?></label><br>
                        <label><input type="checkbox" name="scmw_social_platforms[]" value="linkedin" <?php checked( in_array( 'linkedin', $social_platforms, true ) ); ?> /> <?php esc_html_e( 'LinkedIn', 'shibhu-custom-messages-for-woocommerce' ); ?></label><br>
                        <label><input type="checkbox" name="scmw_social_platforms[]" value="whatsapp" <?php checked( in_array( 'whatsapp', $social_platforms, true ) ); ?> /> <?php esc_html_e( 'WhatsApp', 'shibhu-custom-messages-for-woocommerce' ); ?></label>
                    </p>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render Conditional Logic meta box (using adapted Elegant Labels logic)
     */
    public function render_conditional_logic( $post ) {
        wp_nonce_field( 'scmw_save_conditional_logic', 'scmw_conditional_logic_nonce' );
        
        $conditional_logic = SCMW_Conditional_Logic::get_instance();
        $select_options = $conditional_logic->get_select_options();
        
        $conditions = get_post_meta( $post->ID, '_scmw_conditions', true );
        $conditions = $conditional_logic->sanitize_and_validate_conditions( $conditions );
        
        if ( empty( $conditions ) ) {
            $conditions = array(
                array(
                    array(
                        'field_type'   => 'product-status',
                        'operator'     => 'equal',
                        'value'        => '',
                        'stock_status' => 'instock',
                    ),
                ),
            );
        }
        
        ?>
        <div class="scmw-conditional-logic-wrap">
            <p class="description"><?php esc_html_e( 'Configure conditions using AND/OR logic. Conditions within the same group are joined with AND, while different groups are joined with OR.', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
            
            <div id="scmw-conditions-wrapper">
                <?php
                foreach ( $conditions as $group_index => $group ) {
                    $this->render_condition_group( $group_index, $group, $select_options, $conditional_logic );
                }
                ?>
            </div>
            
            <div class="scmw-conditions-actions">
                <button type="button" class="button button-secondary scmw-add-group"><?php esc_html_e( '+ Add OR Group', 'shibhu-custom-messages-for-woocommerce' ); ?></button>
            </div>
        </div>
        
        <!-- Hidden templates for operators -->
        <div id="scmw-operator-templates" style="display:none;">
            <?php $this->render_operator_templates( $select_options ); ?>
        </div>
        <?php
    }
    
    /**
     * Render condition group
     */
    private function render_condition_group( $group_index, $conditions, $select_options, $conditional_logic ) {
        ?>
        <div class="scmw-condition-group" data-group-index="<?php echo esc_attr( $group_index ); ?>">
            <div class="scmw-group-header">
                <h4><?php 
                        printf( 
                            /* translators: %s is a number */
                            esc_html__( 'OR Group %d', 'shibhu-custom-messages-for-woocommerce' ), 
                            absint( $group_index ) + 1 
                        );  
                    ?>                    
                </h4>
                <button type="button" class="button-link scmw-remove-group" title="<?php esc_attr_e( 'Remove Group', 'shibhu-custom-messages-for-woocommerce' ); ?>">×</button>
            </div>
            
            <div class="scmw-conditions-wrapper">
                <?php
                foreach ( $conditions as $cond_index => $condition ) {
                    $this->render_condition_row( $group_index, $cond_index, $condition, $select_options, $conditional_logic );
                }
                ?>
            </div>
            
            <div class="scmw-group-actions">
                <button type="button" class="button button-small scmw-add-condition"><?php esc_html_e( '+ Add AND Condition', 'shibhu-custom-messages-for-woocommerce' ); ?></button>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render condition row
     */
    private function render_condition_row( $group_index, $cond_index, $condition, $select_options, $conditional_logic ) {
        $condition = wp_parse_args( $condition, array(
            'field_type'   => 'product-status',
            'operator'     => 'equal',
            'value'        => '',
            'stock_status' => 'instock',
        ) );
        
        $categories = $conditional_logic->get_cached_categories();
        $tags = $conditional_logic->get_cached_tags();
        ?>
        <div class="scmw-condition-row" data-condition-index="<?php echo esc_attr( $cond_index ); ?>">
            <!-- Field Type -->
            <div class="scmw-condition-field">
                <select name="scmw_conditions[<?php echo absint( $group_index ); ?>][<?php echo absint( $cond_index ); ?>][field_type]" class="scmw-field-type-select">
                    <?php foreach ( $select_options['field_types'] as $key => $label ) : ?>
                        <option value="<?php echo esc_attr( $key ); ?>" <?php selected( $condition['field_type'], $key ); ?>><?php echo esc_html( $label ); ?></option>
                    <?php endforeach; ?>
                </select>
            </div>
            
            <!-- Operators -->
            <div class="scmw-condition-operator">
                <?php $this->render_all_operator_selects( $group_index, $cond_index, $condition, $select_options ); ?>
            </div>
            
            <!-- Values -->
            <div class="scmw-condition-value">
                <?php $this->render_all_value_inputs( $group_index, $cond_index, $condition, $select_options, $categories, $tags ); ?>
            </div>
            
            <!-- Remove Button -->
            <div class="scmw-condition-actions">
                <button type="button" class="button-link scmw-remove-condition" title="<?php esc_attr_e( 'Remove Condition', 'shibhu-custom-messages-for-woocommerce' ); ?>">×</button>
            </div>
        </div>
        <?php
    }
    
    /**
     * Render all operator selects
     */
    private function render_all_operator_selects( $group_index, $cond_index, $condition, $select_options ) {
        $current_field_type = $condition['field_type'];
        $current_operator = $condition['operator'];
        $name_base = 'scmw_conditions[' . absint( $group_index ) . '][' . absint( $cond_index ) . '][operator]';
        
        // Text operators
        echo '<select name="' . esc_attr( $name_base ) . '" class="scmw-operator-select scmw-operator-text" data-field-type="title"';
        if ( $current_field_type !== 'title' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        foreach ( $select_options['text_operators'] as $key => $label ) {
            $selected = ( $current_field_type === 'title' && $current_operator === $key ) ? ' selected="selected"' : '';
            echo '<option value="' . esc_attr( $key ) . '"' . esc_attr($selected) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Numeric operators
        echo '<select name="' . esc_attr( $name_base ) . '" class="scmw-operator-select scmw-operator-numeric" data-field-type="price,quantity"';
        if ( ! in_array( $current_field_type, array( 'price', 'quantity' ), true ) ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        foreach ( $select_options['numeric_operators'] as $key => $label ) {
            $selected = ( in_array( $current_field_type, array( 'price', 'quantity' ), true ) && $current_operator === $key ) ? ' selected="selected"' : '';
            echo '<option value="' . esc_attr( $key ) . '"' . esc_attr($selected) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Category operators
        echo '<select name="' . esc_attr( $name_base ) . '" class="scmw-operator-select scmw-operator-category" data-field-type="category"';
        if ( $current_field_type !== 'category' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        foreach ( $select_options['category_operators'] as $key => $label ) {
            $selected = ( $current_field_type === 'category' && $current_operator === $key ) ? ' selected="selected"' : '';
            echo '<option value="' . esc_attr( $key ) . '"' . esc_attr($selected) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Tags operators
        echo '<select name="' . esc_attr( $name_base ) . '" class="scmw-operator-select scmw-operator-tags" data-field-type="tags"';
        if ( $current_field_type !== 'tags' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        foreach ( $select_options['tags_operators'] as $key => $label ) {
            $selected = ( $current_field_type === 'tags' && $current_operator === $key ) ? ' selected="selected"' : '';
            echo '<option value="' . esc_attr( $key ) . '"' . esc_attr($selected) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Stock status operators
        echo '<select name="' . esc_attr( $name_base ) . '" class="scmw-operator-select scmw-operator-stock" data-field-type="product-status"';
        if ( $current_field_type !== 'product-status' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        foreach ( $select_options['stock_status_operators'] as $key => $label ) {
            $selected = ( $current_field_type === 'product-status' && $current_operator === $key ) ? ' selected="selected"' : '';
            echo '<option value="' . esc_attr( $key ) . '"' . esc_attr($selected) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
    }
    
    /**
     * Render all value inputs
     */
    private function render_all_value_inputs( $group_index, $cond_index, $condition, $select_options, $categories, $tags ) {
        $field_type = $condition['field_type'];
        $name_prefix = 'scmw_conditions[' . absint( $group_index ) . '][' . absint( $cond_index ) . ']';
        
        // Stock Status
        echo '<select name="' . esc_attr( $name_prefix ) . '[stock_status]" class="scmw-value-input scmw-stock-status-select" data-field-type="product-status"';
        if ( $field_type !== 'product-status' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        foreach ( $select_options['stock_status'] as $key => $label ) {
            $selected = ( $field_type === 'product-status' ) ? selected( $condition['stock_status'], $key, false ) : '';
            echo '<option value="' . esc_attr( $key ) . '" ' . esc_attr($selected) . '>' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Category
        echo '<select name="' . esc_attr( $name_prefix ) . '[value]" class="scmw-value-input scmw-category-select" data-field-type="category"';
        if ( $field_type !== 'category' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        echo '<option value="">' . esc_html__( 'Select Category...', 'shibhu-custom-messages-for-woocommerce' ) . '</option>';
        if ( ! empty( $categories ) && ! is_wp_error( $categories ) ) {
            foreach ( $categories as $category ) {
                $selected = ( $field_type === 'category' ) ? selected( $condition['value'], $category->term_id, false ) : '';
                echo '<option value="' . esc_attr( $category->term_id ) . '" ' . esc_attr($selected) . '>' . esc_html( $category->name ) . ' (' . absint( $category->count ) . ')</option>';
            }
        }
        echo '</select>';
        
        // Tags
        echo '<select name="' . esc_attr( $name_prefix ) . '[value]" class="scmw-value-input scmw-tags-select" data-field-type="tags"';
        if ( $field_type !== 'tags' ) {
            echo ' style="display:none;" disabled="disabled"';
        }
        echo '>';
        echo '<option value="">' . esc_html__( 'Select Tag...', 'shibhu-custom-messages-for-woocommerce' ) . '</option>';
        if ( ! empty( $tags ) && ! is_wp_error( $tags ) ) {
            foreach ( $tags as $tag ) {
                $selected = ( $field_type === 'tags' ) ? selected( $condition['value'], $tag->term_id, false ) : '';
                echo '<option value="' . esc_attr( $tag->term_id ) . '" ' . esc_attr($selected) . '>' . esc_html( $tag->name ) . ' (' . absint( $tag->count ) . ')</option>';
            }
        }
        echo '</select>';
        
        // Numeric
        echo '<input type="number" name="' . esc_attr( $name_prefix ) . '[value]" class="scmw-value-input scmw-numeric-input" data-field-type="price,quantity" min="0" step="0.01"';
        if ( in_array( $field_type, array( 'price', 'quantity' ), true ) ) {
            echo ' value="' . esc_attr( $condition['value'] ) . '"';
        } else {
            echo ' style="display:none;" disabled="disabled" value=""';
        }
        echo '>';
        
        // Text
        echo '<input type="text" name="' . esc_attr( $name_prefix ) . '[value]" class="scmw-value-input scmw-text-input" data-field-type="title"';
        if ( $field_type === 'title' ) {
            echo ' value="' . esc_attr( $condition['value'] ) . '"';
        } else {
            echo ' style="display:none;" disabled="disabled" value=""';
        }
        echo '>';
    }
    
    /**
     * Render operator templates
     */
    private function render_operator_templates( $select_options ) {
        // Text operators
        echo '<select class="scmw-operator-template" data-field-type="title">';
        foreach ( $select_options['text_operators'] as $key => $label ) {
            echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Numeric operators
        echo '<select class="scmw-operator-template" data-field-type="price,quantity">';
        foreach ( $select_options['numeric_operators'] as $key => $label ) {
            echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Category operators
        echo '<select class="scmw-operator-template" data-field-type="category">';
        foreach ( $select_options['category_operators'] as $key => $label ) {
            echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Tags operators
        echo '<select class="scmw-operator-template" data-field-type="tags">';
        foreach ( $select_options['tags_operators'] as $key => $label ) {
            echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
        
        // Stock operators
        echo '<select class="scmw-operator-template" data-field-type="product-status">';
        foreach ( $select_options['stock_status_operators'] as $key => $label ) {
            echo '<option value="' . esc_attr( $key ) . '">' . esc_html( $label ) . '</option>';
        }
        echo '</select>';
    }
    
    /**
     * Render Display Settings meta box
     */
    public function render_display_settings( $post ) {
        wp_nonce_field( 'scmw_save_display_settings', 'scmw_display_settings_nonce' );
        
        $enabled = get_post_meta( $post->ID, '_scmw_enabled', true );
        $display_position = get_post_meta( $post->ID, '_scmw_display_position', true );
        $display_position = $display_position ? $display_position : 'before_add_to_cart';
        
        $bg_color = get_post_meta( $post->ID, '_scmw_bg_color', true );
        $text_color = get_post_meta( $post->ID, '_scmw_text_color', true );
        $priority = get_post_meta( $post->ID, '_scmw_priority', true );
        $priority = $priority ? $priority : 10;
        ?>
        <div class="scmw-display-settings-wrap">
            <div class="scmw-field">
                <label>
                    <input type="checkbox" name="scmw_enabled" value="1" <?php checked( $enabled, '1' ); ?> />
                    <strong><?php esc_html_e( 'Enable this message', 'shibhu-custom-messages-for-woocommerce' ); ?></strong>
                </label>
            </div>
            
            <div class="scmw-field">
                <label for="scmw_display_position"><strong><?php esc_html_e( 'Display Position', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <select id="scmw_display_position" name="scmw_display_position" class="widefat">
                    <option value="before_title" <?php selected( $display_position, 'before_title' ); ?>><?php esc_html_e( 'Before Title', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_title" <?php selected( $display_position, 'after_title' ); ?>><?php esc_html_e( 'After Title', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="before_price" <?php selected( $display_position, 'before_price' ); ?>><?php esc_html_e( 'Before Price', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_price" <?php selected( $display_position, 'after_price' ); ?>><?php esc_html_e( 'After Price', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="before_add_to_cart" <?php selected( $display_position, 'before_add_to_cart' ); ?>><?php esc_html_e( 'Before Add to Cart', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_add_to_cart" <?php selected( $display_position, 'after_add_to_cart' ); ?>><?php esc_html_e( 'After Add to Cart', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="before_tabs" <?php selected( $display_position, 'before_tabs' ); ?>><?php esc_html_e( 'Before Tabs', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_tabs" <?php selected( $display_position, 'after_tabs' ); ?>><?php esc_html_e( 'After Tabs', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="before_description" <?php selected( $display_position, 'before_description' ); ?>><?php esc_html_e( 'Before Product Description', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_description" <?php selected( $display_position, 'after_description' ); ?>><?php esc_html_e( 'After Product Description', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="before_additional_information" <?php selected( $display_position, 'before_additional_information' ); ?>><?php esc_html_e( 'Before Additional Information', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_additional_information" <?php selected( $display_position, 'after_additional_information' ); ?>><?php esc_html_e( 'After Additional Information', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="before_reviews" <?php selected( $display_position, 'before_reviews' ); ?>><?php esc_html_e( 'Before Product Reviews', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                    <option value="after_reviews" <?php selected( $display_position, 'after_reviews' ); ?>><?php esc_html_e( 'After Product Reviews', 'shibhu-custom-messages-for-woocommerce' ); ?></option>
                </select>
            </div>
            
            <div class="scmw-field">
                <label for="scmw_priority"><strong><?php esc_html_e( 'Priority', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <input type="number" id="scmw_priority" name="scmw_priority" value="<?php echo esc_attr( $priority ); ?>" min="1" max="999" class="small-text" />
                <p class="description"><?php esc_html_e( 'Lower numbers display first', 'shibhu-custom-messages-for-woocommerce' ); ?></p>
            </div>
            
            <div class="scmw-field">
                <label for="scmw_bg_color"><strong><?php esc_html_e( 'Background Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <input type="text" id="scmw_bg_color" name="scmw_bg_color" value="<?php echo esc_attr( $bg_color ); ?>" class="scmw-color-picker" />
            </div>
            
            <div class="scmw-field">
                <label for="scmw_text_color"><strong><?php esc_html_e( 'Text Color', 'shibhu-custom-messages-for-woocommerce' ); ?></strong></label>
                <input type="text" id="scmw_text_color" name="scmw_text_color" value="<?php echo esc_attr( $text_color ); ?>" class="scmw-color-picker" />
            </div>
        </div>
        <?php
    }
    
    /**
     * Save all meta boxes
     */
    public function save_meta_boxes( $post_id ) {
        // Check nonces
        if ( ! isset( $_POST['scmw_message_settings_nonce'] ) || 
             ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['scmw_message_settings_nonce'] ) ), 'scmw_save_message_settings' ) ) {
            return;
        }
        
        // Check autosave
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        
        // Check permissions
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }
        
        // Save message settings
        if ( isset( $_POST['scmw_message_type'] ) ) {
            update_post_meta( $post_id, '_scmw_message_type', sanitize_text_field( wp_unslash( $_POST['scmw_message_type'] ) ) );
        }
        
        // Text Ticker
        if ( isset( $_POST['scmw_ticker_text'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_text', sanitize_text_field( wp_unslash( $_POST['scmw_ticker_text'] ) ) );
        }
        if ( isset( $_POST['scmw_ticker_speed'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_speed', absint( $_POST['scmw_ticker_speed'] ) );
        }

        // Add this code to class-scmw-meta-boxes.php in the save_meta_boxes() method
        // Add after the existing ticker text and speed saves (around line 450-455)

        // Text Ticker - New Fields
        if ( isset( $_POST['scmw_ticker_label'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_label', sanitize_text_field( wp_unslash( $_POST['scmw_ticker_label'] ) ) );
        }

        if ( isset( $_POST['scmw_ticker_label_bg'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_label_bg', sanitize_hex_color( wp_unslash( $_POST['scmw_ticker_label_bg'] ) ) );
        }

        if ( isset( $_POST['scmw_ticker_label_color'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_label_color', sanitize_hex_color( wp_unslash( $_POST['scmw_ticker_label_color'] ) ) );
        }

        if ( isset( $_POST['scmw_ticker_border_color'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_border_color', sanitize_hex_color( wp_unslash( $_POST['scmw_ticker_border_color'] ) ) );
        }

        if ( isset( $_POST['scmw_ticker_text_bg'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_text_bg', sanitize_hex_color( wp_unslash( $_POST['scmw_ticker_text_bg'] ) ) );
        }

        if ( isset( $_POST['scmw_ticker_text_color'] ) ) {
            update_post_meta( $post_id, '_scmw_ticker_text_color', sanitize_hex_color( wp_unslash( $_POST['scmw_ticker_text_color'] ) ) );
        }

        update_post_meta( $post_id, '_scmw_ticker_show_dot', isset( $_POST['scmw_ticker_show_dot'] ) ? '1' : '0' );
        
        // Stock Counter
        update_post_meta( $post_id, '_scmw_stock_show_remaining', isset( $_POST['scmw_stock_show_remaining'] ) ? '1' : '0' );
        if ( isset( $_POST['scmw_stock_text_template'] ) ) {
            update_post_meta( $post_id, '_scmw_stock_text_template', sanitize_text_field( wp_unslash( $_POST['scmw_stock_text_template'] ) ) );
        }
        
        // Countdown
        if ( isset( $_POST['scmw_countdown_date'] ) ) {
            update_post_meta( $post_id, '_scmw_countdown_date', sanitize_text_field( wp_unslash( $_POST['scmw_countdown_date'] ) ) );
        }
        if ( isset( $_POST['scmw_countdown_text'] ) ) {
            update_post_meta( $post_id, '_scmw_countdown_text', sanitize_text_field( wp_unslash( $_POST['scmw_countdown_text'] ) ) );
        }
        
        // Custom Image
        if ( isset( $_POST['scmw_custom_image_id'] ) ) {
            update_post_meta( $post_id, '_scmw_custom_image_id', absint( $_POST['scmw_custom_image_id'] ) );
        }
        if ( isset( $_POST['scmw_custom_image_link'] ) ) {
            update_post_meta( $post_id, '_scmw_custom_image_link', esc_url_raw( wp_unslash( $_POST['scmw_custom_image_link'] ) ) );
        }
        
        // Custom Message
        if ( isset( $_POST['scmw_custom_message_text'] ) ) {
            update_post_meta( $post_id, '_scmw_custom_message_text', wp_kses_post( wp_unslash( $_POST['scmw_custom_message_text'] ) ) );
        }
        
        // Discount Code
        if ( isset( $_POST['scmw_discount_code'] ) ) {
            update_post_meta( $post_id, '_scmw_discount_code', sanitize_text_field( wp_unslash( $_POST['scmw_discount_code'] ) ) );
        }
        if ( isset( $_POST['scmw_discount_description'] ) ) {
            update_post_meta( $post_id, '_scmw_discount_description', sanitize_textarea_field( wp_unslash( $_POST['scmw_discount_description'] ) ) );
        }

        // Save discount style options
        SCMW_Discount_Styles::save_discount_style_meta( $post_id );
        
        // Social Share
        if ( isset( $_POST['scmw_social_platforms'] ) && is_array( $_POST['scmw_social_platforms'] ) ) {
            $platforms = array_map( 'sanitize_text_field', wp_unslash( $_POST['scmw_social_platforms'] ) );
            update_post_meta( $post_id, '_scmw_social_platforms', $platforms );
        } else {
            delete_post_meta( $post_id, '_scmw_social_platforms' );
        }
        
        // Conditional Logic
        if ( isset( $_POST['scmw_conditions'] ) && is_array( $_POST['scmw_conditions'] ) ) {
            $conditional_logic = SCMW_Conditional_Logic::get_instance();
            $raw_conditions = map_deep( wp_unslash( $_POST['scmw_conditions'] ), 'sanitize_text_field' );
            $conditions = $conditional_logic->sanitize_and_validate_conditions( $raw_conditions );
            update_post_meta( $post_id, '_scmw_conditions', $conditions );
        }
        
        // Display Settings
        update_post_meta( $post_id, '_scmw_enabled', isset( $_POST['scmw_enabled'] ) ? '1' : '0' );
        
        if ( isset( $_POST['scmw_display_position'] ) ) {
            update_post_meta( $post_id, '_scmw_display_position', sanitize_text_field( wp_unslash( $_POST['scmw_display_position'] ) ) );
        }
        
        if ( isset( $_POST['scmw_priority'] ) ) {
            update_post_meta( $post_id, '_scmw_priority', absint( $_POST['scmw_priority'] ) );
        }
        
        if ( isset( $_POST['scmw_bg_color'] ) ) {
            update_post_meta( $post_id, '_scmw_bg_color', sanitize_hex_color( wp_unslash( $_POST['scmw_bg_color'] ) ) );
        }
        
        if ( isset( $_POST['scmw_text_color'] ) ) {
            update_post_meta( $post_id, '_scmw_text_color', sanitize_hex_color( wp_unslash( $_POST['scmw_text_color'] ) ) );
        }
    }
}