<?php
/**
*Frontend Display Handler
*
*@package Shibhu Custom Messages for WooCommerce
*/

defined( 'ABSPATH' ) || exit;

class SCMW_Frontend {

    private static $instance = null;

    public static function get_instance() {
        if ( self::$instance === null ) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->init_hooks();
    }

    /**
     * Initialize hooks
     */
    private function init_hooks() {
        // Hook into various product page positions
        add_action( 'woocommerce_before_single_product_summary', array( $this, 'display_before_title' ), 5 );
        add_action( 'woocommerce_single_product_summary', array( $this, 'display_after_title' ), 6 );
        add_action( 'woocommerce_single_product_summary', array( $this, 'display_before_price' ), 9 );
        add_action( 'woocommerce_single_product_summary', array( $this, 'display_after_price' ), 11 );
        add_action( 'woocommerce_single_product_summary', array( $this, 'display_before_add_to_cart' ), 29 );
        add_action( 'woocommerce_single_product_summary', array( $this, 'display_after_add_to_cart' ), 31 );
        add_action( 'woocommerce_after_single_product_summary', array( $this, 'display_before_tabs' ), 9 );
        add_action( 'woocommerce_after_single_product_summary', array( $this, 'display_after_tabs' ), 11 );
        
        // New tab content hooks
        add_action( 'woocommerce_before_product_description', array( $this, 'display_before_description' ), 5 );
        add_action( 'woocommerce_product_description_heading', array( $this, 'display_after_description' ), 15 );
        add_action( 'woocommerce_product_additional_information', array( $this, 'display_before_additional_information' ), 5 );
        add_action( 'woocommerce_product_additional_information', array( $this, 'display_after_additional_information' ), 10 );
        add_action( 'woocommerce_review_before', array( $this, 'display_before_reviews' ), 5 );
        add_action( 'woocommerce_review_before_comment_text', array( $this, 'display_after_reviews' ), 15 );
    }

    /**
     * Display messages at different positions
     */
    public function display_before_title() {
        $this->display_messages_by_position( 'before_title' );
    }

    public function display_after_title() {
        $this->display_messages_by_position( 'after_title' );
    }

    public function display_before_price() {
        $this->display_messages_by_position( 'before_price' );
    }

    public function display_after_price() {
        $this->display_messages_by_position( 'after_price' );
    }

    public function display_before_add_to_cart() {
        $this->display_messages_by_position( 'before_add_to_cart' );
    }

    public function display_after_add_to_cart() {
        $this->display_messages_by_position( 'after_add_to_cart' );
    }

    public function display_before_tabs() {
        $this->display_messages_by_position( 'before_tabs' );
    }

    public function display_after_tabs() {
        $this->display_messages_by_position( 'after_tabs' );
    }

    public function display_before_description() {
        $this->display_messages_by_position( 'before_description' );
    }

    public function display_after_description() {
        $this->display_messages_by_position( 'after_description' );
    }

    public function display_before_additional_information() {
        $this->display_messages_by_position( 'before_additional_information' );
    }

    public function display_after_additional_information() {
        $this->display_messages_by_position( 'after_additional_information' );
    }

    public function display_before_reviews() {
        $this->display_messages_by_position( 'before_reviews' );
    }

    public function display_after_reviews() {
        $this->display_messages_by_position( 'after_reviews' );
    }

    /**
     * Display messages by position
     */
    private function display_messages_by_position( $position ) {
        global $product;
        
        if ( ! $product ) {
            return;
        }
        
        $messages = $this->get_applicable_messages( $product->get_id(), $position );
        
        if ( empty( $messages ) ) {
            return;
        }
        
        foreach ( $messages as $message ) {
            $this->render_message( $message, $product );
        }
    }

    /**
     * Get applicable messages for current product
     * 
     * Note: This query uses meta_query and meta_key which can be slow on large databases.
     * However, this is acceptable because:
     * 1. The query is limited to a specific post type with publish status
     * 2. It only runs on single product pages (not on archive pages)
     * 3. The number of message posts is typically small (< 100)
     * 4. Results could be cached if performance becomes an issue
     */
    private function get_applicable_messages( $product_id, $position ) {
        // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_query -- See function docblock for justification
        // phpcs:disable WordPress.DB.SlowDBQuery.slow_db_query_meta_key -- See function docblock for justification
        $args = array(
            'post_type'      => 'scmw_message',
            'posts_per_page' => -1,
            'post_status'    => 'publish',
            'meta_query'     => array(
                'relation' => 'AND',
                array(
                    'key'   => '_scmw_enabled',
                    'value' => '1',
                ),
                array(
                    'key'   => '_scmw_display_position',
                    'value' => $position,
                ),
            ),
            'meta_key'       => '_scmw_priority',
            'orderby'        => 'meta_value_num',
            'order'          => 'ASC',
        );
        
        $messages = get_posts( $args );
        $applicable = array();
        
        $conditional_logic = SCMW_Conditional_Logic::get_instance();
        
        foreach ( $messages as $message ) {
            $conditions = get_post_meta( $message->ID, '_scmw_conditions', true );
            
            // If no conditions, show on all products
            if ( empty( $conditions ) ) {
                $applicable[] = $message;
                continue;
            }
            
            // Check if conditions match
            if ( $conditional_logic->evaluate_conditions( $conditions, $product_id ) ) {
                $applicable[] = $message;
            }
        }
        
        return $applicable;
    }

    /**
     * Render message
     */
    private function render_message( $message, $product ) {
        $message_type = get_post_meta( $message->ID, '_scmw_message_type', true );
        $bg_color     = get_post_meta( $message->ID, '_scmw_bg_color', true );
        $text_color   = get_post_meta( $message->ID, '_scmw_text_color', true );
        
        $styles = array();
        if ( $bg_color ) {
            $styles[] = 'background-color: ' . esc_attr( $bg_color );
        }
        if ( $text_color ) {
            $styles[] = 'color: ' . esc_attr( $text_color );
        }
        echo '<div class="scmw-message-container scmw-type-' . esc_attr( $message_type ) . '"';
    
        if ( ! empty( $styles ) ) {
            echo ' style="' . esc_attr( implode( '; ', $styles ) ) . '"';
        }
        
        echo '>';
        
        switch ( $message_type ) {
            case 'text_ticker':
                $this->render_text_ticker( $message, $product );
                break;
            case 'stock_counter':
                $this->render_stock_counter( $message, $product );
                break;
            case 'countdown':
                $this->render_countdown( $message, $product );
                break;
            case 'custom_image':
                $this->render_custom_image( $message, $product );
                break;
            case 'custom_message':
                $this->render_custom_message( $message, $product );
                break;
            case 'discount_code':
                $this->render_discount_code( $message, $product );
                break;
            case 'social_share':
                $this->render_social_share( $message, $product );
                break;
        }
        
        echo '</div>';
    }

    
    /**
     * Render text ticker
     */
    private function render_text_ticker( $message, $product ) {
        $ticker_text  = get_post_meta( $message->ID, '_scmw_ticker_text', true );
        $ticker_speed = get_post_meta( $message->ID, '_scmw_ticker_speed', true );
        $ticker_speed = $ticker_speed ? $ticker_speed : 50;
        
        // Get customization options
        $ticker_label       = get_post_meta( $message->ID, '_scmw_ticker_label', true );
        $ticker_label       = $ticker_label ? $ticker_label : __( 'Breaking News', 'shibhu-custom-messages-for-woocommerce' );
        $label_bg           = get_post_meta( $message->ID, '_scmw_ticker_label_bg', true );
        $label_bg           = $label_bg ? $label_bg : '#0d6efd';
        $label_color        = get_post_meta( $message->ID, '_scmw_ticker_label_color', true );
        $label_color        = $label_color ? $label_color : '#ffffff';
        $border_color       = get_post_meta( $message->ID, '_scmw_ticker_border_color', true );
        $border_color       = $border_color ? $border_color : '#0d6efd';
        $text_bg            = get_post_meta( $message->ID, '_scmw_ticker_text_bg', true );
        $text_bg            = $text_bg ? $text_bg : '#ffffff';
        $text_color         = get_post_meta( $message->ID, '_scmw_ticker_text_color', true );
        $text_color         = $text_color ? $text_color : '#333333';
        $show_dot           = get_post_meta( $message->ID, '_scmw_ticker_show_dot', true );
        
        $ticker_text = $this->replace_shortcodes( $ticker_text, $product );
        
        // Build inline styles
        $container_style = 'border-color: ' . esc_attr( $border_color ) . '; background: ' . esc_attr( $text_bg ) . ';';
        $label_style = 'background: ' . esc_attr( $label_bg ) . '; color: ' . esc_attr( $label_color ) . '; border-right-color: ' . esc_attr( $border_color ) . ';';
        $text_style = 'color: ' . esc_attr( $text_color ) . ';';
        $dot_class = $show_dot === '1' ? 'scmw-ticker-show-dot' : 'scmw-ticker-hide-dot';
        
        ?>
        <div class="scmw-ticker-container" style="<?php echo esc_attr( $container_style ); ?>">
            <div class="scmw-ticker-label <?php echo esc_attr( $dot_class ); ?>" style="<?php echo esc_attr( $label_style ); ?>">
                <?php echo esc_html( $ticker_label ); ?>
            </div>
            <div class="scmw-ticker-wrapper">
                <div class="scmw-ticker-text" data-speed="<?php echo esc_attr( $ticker_speed ); ?>" style="<?php echo esc_attr( $text_style ); ?>">
                    <?php echo wp_kses_post( $ticker_text ); ?>
                </div>
                <div class="scmw-ticker-text" data-speed="<?php echo esc_attr( $ticker_speed ); ?>" aria-hidden="true" style="<?php echo esc_attr( $text_style ); ?>">
                    <?php echo wp_kses_post( $ticker_text ); ?>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render stock counter
     */
    private function render_stock_counter( $message, $product ) {
        $show_remaining = get_post_meta( $message->ID, '_scmw_stock_show_remaining', true );
        $text_template  = get_post_meta( $message->ID, '_scmw_stock_text_template', true );
        $text_template  = $text_template ? $text_template : __( 'Only {stock_count} left in stock!', 'shibhu-custom-messages-for-woocommerce' );
        
        $stock_quantity = $product->get_stock_quantity();
        $stock_quantity = $stock_quantity !== null ? absint( $stock_quantity ) : 0;
        
        if ( $stock_quantity <= 0 && ! $show_remaining ) {
            return;
        }
        
        $text = str_replace( '{stock_count}', '<span class="scmw-stock-count">' . esc_html( $stock_quantity ) . '</span>', $text_template );
        
        ?>
        <div class="scmw-stock-counter">
            <?php echo wp_kses_post( $text ); ?>
        </div>
        <?php
    }

    /**
     * Render countdown timer
     */
    private function render_countdown( $message, $product ) {
        $countdown_date = get_post_meta( $message->ID, '_scmw_countdown_date', true );
        $countdown_text = get_post_meta( $message->ID, '_scmw_countdown_text', true );
        $countdown_text = $countdown_text ? $countdown_text : __( 'Offer ends in: {countdown}', 'shibhu-custom-messages-for-woocommerce' );
        
        if ( ! $countdown_date ) {
            return;
        }
        
        $end_timestamp = strtotime( $countdown_date );
        if ( $end_timestamp <= time() ) {
            return;
        }
        
        ?>
        <div class="scmw-countdown-container" data-end-time="<?php echo esc_attr( $end_timestamp ); ?>">
            <?php
            $parts = explode( '{countdown}', $countdown_text );
            if ( isset( $parts[0] ) ) {
                echo '<div class="scmw-countdown-label-before">' . esc_html( $parts[0] ) . '</div>';
            }
            ?>
            <div class="scmw-countdown-timer">
                <div class="scmw-countdown-item">
                    <span class="scmw-countdown-value scmw-days">00</span>
                    <span class="scmw-countdown-label"><?php esc_html_e( 'Days', 'shibhu-custom-messages-for-woocommerce' ); ?></span>
                </div>
                <div class="scmw-countdown-item">
                    <span class="scmw-countdown-value scmw-hours">00</span>
                    <span class="scmw-countdown-label"><?php esc_html_e( 'Hours', 'shibhu-custom-messages-for-woocommerce' ); ?></span>
                </div>
                <div class="scmw-countdown-item">
                    <span class="scmw-countdown-value scmw-minutes">00</span>
                    <span class="scmw-countdown-label"><?php esc_html_e( 'Mins', 'shibhu-custom-messages-for-woocommerce' ); ?></span>
                </div>
                <div class="scmw-countdown-item">
                    <span class="scmw-countdown-value scmw-seconds">00</span>
                    <span class="scmw-countdown-label"><?php esc_html_e( 'Secs', 'shibhu-custom-messages-for-woocommerce' ); ?></span>
                </div>
            </div>
            <?php
            if ( isset( $parts[1] ) ) {
                echo '<div class="scmw-countdown-label-after">' . esc_html( $parts[1] ) . '</div>';
            }
            ?>
        </div>
        <?php
    }

    /**
     * Render custom image
     */
    private function render_custom_image( $message, $product ) {
        $image_id   = get_post_meta( $message->ID, '_scmw_custom_image_id', true );
        $image_link = get_post_meta( $message->ID, '_scmw_custom_image_link', true );
        
        if ( ! $image_id ) {
            return;
        }
        
        $image_url = wp_get_attachment_url( $image_id );
        if ( ! $image_url ) {
            return;
        }
        
        ?>
        <div class="scmw-custom-image-container">
            <?php if ( $image_link ) : ?>
                <a href="<?php echo esc_url( $image_link ); ?>" target="_blank" rel="noopener noreferrer">
                    <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( get_the_title( $message->ID ) ); ?>" />
                </a>
            <?php else : ?>
                <img src="<?php echo esc_url( $image_url ); ?>" alt="<?php echo esc_attr( get_the_title( $message->ID ) ); ?>" />
            <?php endif; ?>
        </div>
        <?php
    }

    /**
     * Render custom message
     */
    private function render_custom_message( $message, $product ) {
        $message_text = get_post_meta( $message->ID, '_scmw_custom_message_text', true );
        $message_text = $this->replace_shortcodes( $message_text, $product );
        
        ?>
        <div class="scmw-custom-message">
            <?php echo wp_kses_post( $message_text ); ?>
        </div>
        <?php
    }

    /**
     * Render discount code
     */
    private function render_discount_code( $message, $product ) {
        SCMW_Discount_Styles::render_discount_code( $message->ID, $product );
    }

    /**
     * Render social share buttons
     */
    private function render_social_share( $message, $product ) {
        $platforms = get_post_meta( $message->ID, '_scmw_social_platforms', true );
        $platforms = is_array( $platforms ) ? $platforms : array( 'facebook', 'twitter', 'pinterest' );
        
        $product_url   = get_permalink( $product->get_id() );
        $product_title = $product->get_name();
        $product_image = wp_get_attachment_url( $product->get_image_id() );
        
        ?>
        <div class="scmw-social-share-container">
            <div class="scmw-social-share-title"><?php esc_html_e( 'Share this product:', 'shibhu-custom-messages-for-woocommerce' ); ?></div>
            <div class="scmw-social-buttons">
                <?php
                foreach ( $platforms as $platform ) {
                    $this->render_social_button( $platform, $product_url, $product_title, $product_image );
                }
                ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render individual social button
     */
    private function render_social_button( $platform, $url, $title, $image ) {
        $share_url = '';
        $icon = '';
        
        switch ( $platform ) {
            case 'facebook':
                $share_url = 'https://www.facebook.com/sharer/sharer.php?u=' . rawurlencode( $url );
                $icon = 'f';
                break;
            case 'twitter':
                $share_url = 'https://twitter.com/intent/tweet?url=' . rawurlencode( $url ) . '&text=' . rawurlencode( $title );
                $icon = '𝕏';
                break;
            case 'pinterest':
                $share_url = 'https://pinterest.com/pin/create/button/?url=' . rawurlencode( $url ) . '&media=' . rawurlencode( $image ) . '&description=' . rawurlencode( $title );
                $icon = 'P';
                break;
            case 'linkedin':
                $share_url = 'https://www.linkedin.com/shareArticle?mini=true&url=' . rawurlencode( $url ) . '&title=' . rawurlencode( $title );
                $icon = 'in';
                break;
            case 'whatsapp':
                $share_url = 'https://api.whatsapp.com/send?text=' . rawurlencode( $title . ' ' . $url );
                $icon = '✆';
                break;
        }
        
        if ( $share_url ) {
            echo '<a href="' . esc_url( $share_url ) . '" class="scmw-social-btn ' . esc_attr( $platform ) . '" target="_blank" rel="noopener noreferrer" aria-label="' . esc_attr( ucfirst( $platform ) ) . '">' . esc_html( $icon ) . '</a>';
        }
    }

    /**
     * Replace shortcodes in text
     */
    private function replace_shortcodes( $text, $product ) {
        $replacements = array(
            '{product_name}'  => $product->get_name(),
            '{product_price}' => wc_price( $product->get_price() ),
            '{stock_count}'   => $product->get_stock_quantity() !== null ? absint( $product->get_stock_quantity() ) : 0,
            '{product_sku}'   => $product->get_sku(),
        );
        
        return str_replace( array_keys( $replacements ), array_values( $replacements ), $text );
    }
}